(* L1 Compiler
 * IR Trees
 * Author: Kaustuv Chaudhuri <kaustuv+@cs.cmu.edu>
 * Modified: Alex Vaynberg <alv@andrew.cmu.edu>
 * Modified: Frank Pfenning <fp@cs.cmu.edu>
 *)

signature TREE =
sig

  datatype binop = ADD | SUB | MUL | DIV | MOD

  datatype exp = 
      CONST of Word32.word
    | TEMP of Temp.temp
    | BINOP of binop * exp * exp
  and stm =
      MOVE of exp * exp
    | RETURN of exp
    | SEQ of stm * stm
    | NOP

  structure Print :
  sig
    val pp_exp : exp -> PP.pp_desc
    val pp_stm : stm -> PP.pp_desc
  end
end

structure Tree :> TREE =
struct

  datatype binop = ADD | SUB | MUL | DIV | MOD

  datatype exp = 
      CONST of Word32.word
    | TEMP of Temp.temp
    | BINOP of binop * exp * exp
  and stm =
      MOVE of exp * exp
    | RETURN of exp
    | SEQ of stm * stm
    | NOP

  structure Print = 
  struct

    fun pp_binop ADD = PP.string "+"
      | pp_binop SUB = PP.string "-"
      | pp_binop MUL = PP.string "*"
      | pp_binop DIV = PP.string "/"
      | pp_binop MOD = PP.string "%"

    fun pp_exp (CONST x) = PP.string (Word32Signed.toString x)
      | pp_exp (TEMP t) = PP.string (Temp.name t)
      | pp_exp (BINOP (binop, e1, e2)) =
	  PP.hovBox (PP.PPS.Abs 2, [PP.string "(", pp_exp e1, PP.space 1,
				    pp_binop binop, PP.space 1,
				    pp_exp e2, PP.string ")"])

    and pp_stm (MOVE (e1,e2)) =
	  PP.hovBox (PP.PPS.Abs 2,
		     [pp_exp e1, PP.space 2, PP.string "<--", 
		      PP.space 2, pp_exp e2])
      | pp_stm (RETURN e) =
	  PP.hovBox (PP.PPS.Abs 2, [PP.string "return", PP.space 1, pp_exp e])
      | pp_stm (SEQ (s1,s2)) = PP.vBox(PP.PPS.Rel 0, [pp_stm s1, PP.cut, pp_stm s2])
      | pp_stm NOP = PP.string "<NOP>"
  end
end
