(* L1 Compiler
 * Abstract Syntax Trees
 * Author: Alex Vaynberg
 * Modified: Frank Pfenning <fp@cs.cmu.edu>
 *
 * Uses pretty printing library
 * structure PP  -- see util/pp.sml
 *)

signature AST =
sig
  type ident = Symbol.symbol

  datatype oper = 
     PLUS
   | MINUS
   | TIMES
   | DIVIDEDBY
   | MODULO
   | NEGATIVE			(* unary minus *)

  datatype exp =
     Var of ident
   | ConstExp of Word32.word
   | OpExp of oper * exp list
   | Marked of exp Mark.marked
  and stm =
     Assign of ident * exp
   | Return of exp

  type program = stm list

  (* print as source, with redundant parentheses *)
  structure PrintCode :
  sig
    val pp_exp : exp -> PP.pp_desc
    val pp_stm : stm -> PP.pp_desc
    val pp_program : program -> PP.pp_desc
  end

  (* print as abstract syntax tree *)
  structure PrintAst :
  sig
    val pp_exp : exp -> PP.pp_desc
    val pp_stm : stm -> PP.pp_desc
    val pp_program : program -> PP.pp_desc
  end
end

structure Ast :> AST =
struct
  type ident = Symbol.symbol

  datatype oper = 
     PLUS
   | MINUS
   | TIMES
   | DIVIDEDBY
   | MODULO
   | NEGATIVE			(* unary minus *)

  datatype exp =
     Var of ident
   | ConstExp of Word32.word
   | OpExp of oper * exp list
   | Marked of exp Mark.marked
  and stm =
     Assign of ident * exp
   | Return of exp  

  type program = stm list

  (* print programs and expressions in source form
   * using redundant parentheses to clarify precedence
   *)
  structure PrintCode =
  struct
    fun pp_ident id = PP.string (Symbol.name id)

    fun pp_oper PLUS = PP.string "+"
      | pp_oper MINUS = PP.string "-"
      | pp_oper TIMES = PP.string "*"
      | pp_oper DIVIDEDBY = PP.string "/"
      | pp_oper MODULO = PP.string "%"
      | pp_oper NEGATIVE = PP.string "-"

    fun pp_exp (Var(id)) = pp_ident id
      | pp_exp (ConstExp(c)) = PP.hBox [PP.string (Word32Signed.toString c)]
      | pp_exp (OpExp(oper, [e])) =
	  PP.hBox [pp_oper oper, PP.string "(", pp_exp e, PP.string ")"]
      | pp_exp (OpExp(oper, [e1,e2])) =
	  PP.hBox (PP.string "(" :: pp_exp e1 :: PP.space 1 :: pp_oper oper
		   :: PP.space 1 :: pp_exp e2 :: PP.string ")" :: nil)
      | pp_exp (Marked(marked_exp)) =
	  pp_exp (Mark.data marked_exp)

    fun pp_stm (Assign (id,e)) =
	  PP.hBox [pp_ident id, PP.space 1, PP.string "=", PP.space 1,
		   pp_exp e, PP.string ";"]
      | pp_stm (Return e) =
	  PP.hBox [PP.string "return ", PP.space 1, pp_exp e, PP.string ";"]

    fun pp_stms nil = nil
      | pp_stms (s::nil) = pp_stm s :: nil
      | pp_stms (s::ss) = pp_stm s :: PP.cut :: pp_stms ss

    fun pp_program ss = 
	PP.vBox (PP.PPS.Abs 0,
		 [PP.string "{", PP.cut,
		  PP.vBox (PP.PPS.Rel 2, pp_stms ss),
		  PP.cut, PP.string "}"])

  end

  (* print abstract syntax trees in internal form *)
  structure PrintAst =
  struct 
    fun pp_ident id = PP.string (Symbol.name id)

    fun pp_oper PLUS = PP.string "PLUS"
      | pp_oper MINUS = PP.string "MINUS"
      | pp_oper TIMES = PP.string "TIMES"
      | pp_oper DIVIDEDBY = PP.string "DIVIDEDBY"
      | pp_oper MODULO = PP.string "MODULO"
      | pp_oper NEGATIVE = PP.string "NEGATIVE"

    fun pp_exp (Var(id)) =
	  PP.hBox [PP.string "Var(", pp_ident id, PP.string ")"]
      | pp_exp (ConstExp c) =
	  PP.hBox [PP.string "ConstExp(", PP.string (Word32Signed.toString c), PP.string ")"]
      | pp_exp (OpExp(oper, es)) =
	  PP.hBox [pp_oper oper, PP.string "(", PP.hBox (pp_exps es), PP.string ")"]
      | pp_exp (Marked(marked_exp)) =
	  pp_exp (Mark.data marked_exp)

    and pp_exps nil = nil
      | pp_exps (e::nil) = pp_exp e :: nil
      | pp_exps (e::es) = pp_exp e::PP.string ","::PP.cut::pp_exps es

    fun pp_stm (Assign (id,e)) =
	  PP.hBox [PP.string "Assign(", pp_ident id, PP.string ",", pp_exp e, PP.string ")"]
      | pp_stm (Return e) =
	  PP.hBox [PP.string "Return(", pp_exp e, PP.string ")"]

    fun pp_stms nil = nil
      | pp_stms (s::nil) = pp_stm s :: nil
      | pp_stms (s::ss) = pp_stm s :: PP.string "," :: PP.cut :: pp_stms ss

    fun pp_program ss = 
	PP.vBox (PP.PPS.Abs 0,
		 [PP.string "[", PP.cut,
		  PP.vBox (PP.PPS.Rel 2, pp_stms ss),
		  PP.cut, PP.string "]"])

  end	      
end
