package edu.cmu.cs.cs214.lect01.trees;

import java.util.Random;

/** Represents a agent-based simulation */
public class Simulation {
	private static final int NUM_STEPS = 10;
	
	private Random rand = new Random();
	
	private int xSize;
	private int ySize;
	private Agent grid[][];
	private int agentCount=0;
	
	/** Runs the simulation for NUM_STEPS steps */
	public void simulate() {
		// too many nested loops here - better to factor as shown in timeStep(), below
		for (int i = 0; i < NUM_STEPS; ++i) {
			for (int x = 0; x < xSize; ++x) {
				for (int y = 0; y < ySize; ++y) {
					Agent a = grid[x][y]; 
					if (a != null) {
						a.timeStep(this);
						a.logState();
					}
				}
			}
			System.out.println("Agent count " + agentCount);
		}
	}
	
	/** Creates a simulation with a grid of the specified dimensions */
	public Simulation(int xs, int ys) {
		xSize = xs;
		ySize = ys;
		grid = new Agent[xs][];
		for (int x = 0; x < xs; ++x) {
			grid[x] = new Agent[ys];
		}
	}

	/** Creates a simulation with a grid sized to the defaults in the Constants class */
	public Simulation() {
		this(Constants.MAX_X, Constants.MAX_Y);
	}

	/** @return true when a random draw between 0 and 1 is less than the specified probability */
	public boolean chance(double probability) {
		return rand.nextDouble() < probability;
	}
	
	/** removes Agent a from the simulation */
	public void remove(Agent a) {
		Location l = a.getLocation();
		if (grid[l.x][l.y] != null)
			agentCount--;
		grid[l.x][l.y] = null;
	}

	/** adds Agent a to the simulation */
	public void add(Agent a) {
		Location l = a.getLocation();
		if (grid[l.x][l.y] == null)
			agentCount++;
		grid[l.x][l.y] = a;
	}

	/** Returns a randomly selected empty location on the simulation grid.
	 * Requires that there at least one empty location.
	 */
	public Location getEmptySpot() {
		while (true) {
			// should make more robust by testing for a completely full grid
			int x = rand.nextInt(xSize);
			int y = rand.nextInt(ySize);
			if (grid[x][y] == null)
				return new Location(x,y);
		}
		
	}
	
	/** Returns the set of empty locations within distance of the passed-in location. */
	// hacky to return an array
	// we'll improve it when we look at Java collections
	public Location[] getEmptySpots(Location location, int distance) {
		int arraySize=0;
		for (int x = Math.max(0,  location.x-distance); x <= Math.min(xSize-1, location.x+distance); ++x) {
			for (int y = Math.max(0,  location.y-distance); y <= Math.min(ySize-1, location.y+distance); ++y) {
				if (grid[x][y]==null) {
					arraySize++;
				}
			}			
		}
		Location[] result = new Location[arraySize];
		int spot = 0;
		for (int x = Math.max(0,  location.x-distance); x <= Math.min(xSize-1, location.x+distance); ++x) {
			for (int y = Math.max(0,  location.y-distance); y <= Math.min(ySize-1, location.y+distance); ++y) {
				if (grid[x][y]==null) {
					result[spot++] = new Location(x,y);
				}
			}			
		}
		return result;
	}

	/** Take exactly one time step.  Logs the state of each agent if withLogging is true. */
	public void timeStep(boolean withLogging) {
		for (int x = 0; x < xSize; ++x) {
			for (int y = 0; y < ySize; ++y) {
				if (grid[x][y] != null) {
					grid[x][y].timeStep(this);
					if (withLogging && grid[x][y] != null)
						grid[x][y].logState();
				}
			}
		}
	}

	/** Take numSteps time steps.  Logs the state of each agent at each time step if withLogging is true. */
	public void takeSteps(int numSteps, boolean withLogging) {
		for (int i = 0; i < numSteps; ++i) {
			timeStep(withLogging);
			if (withLogging)
				System.out.println("Agent count " + agentCount);
		}
	}
	
	/*
	 * Returns agents at least one space away, but no more than distance spaces away (inclusive).
	 */
	// hacky to return an array
	// we'll improve it when we look at Java collections
	public Agent[] getAgents(Location location, int distance) {
		int arraySize=0;
		for (int x = Math.max(0,  location.x-distance); x <= Math.min(xSize-1, location.x+distance); ++x) {
			for (int y = Math.max(0,  location.y-distance); y <= Math.min(ySize-1, location.y+distance); ++y) {
				if (grid[x][y]!=null && (x != location.x || y != location.y)) {
					arraySize++;
				}
			}			
		}
		Agent[] result = new Agent[arraySize];
		int spot = 0;
		for (int x = Math.max(0,  location.x-distance); x <= Math.min(xSize-1, location.x+distance); ++x) {
			for (int y = Math.max(0,  location.y-distance); y <= Math.min(ySize-1, location.y+distance); ++y) {
				if (grid[x][y]!=null && (x != location.x || y != location.y)) {
					result[spot++] = grid[x][y];
				}
			}			
		}
		return result;
	}
}
