package edu.cmu.cs.cs214.lect01.trees;

/** Represents an infected Lodgepole Pine tree in the simulation */
public class InfectedPine implements Agent {
	private static final int INITIAL_INFESTATION = 1;
	private static final double PROB_INFESTATION = 1.0;
	private static final int INFESTATION_DISTANCE = 10;
	private static final int MONTHLY_INFESTATION_GROWTH = 1;
	private static final int INFESTATION_DEATH_THRESHOLD = 5;
	private static final int INFESTATION_SPREAD_THRESHOLD = 2;
	private static final int MIN_INFESTATION_AGE = 1;
		
	/** Creates an infected LodgepolePine of age a at Location l */
	public InfectedPine(int a, Location l) {
		age = a;
		infestation = INITIAL_INFESTATION;
		location = l;
	}
	
	private int age;
	private int infestation;
	private final Location location;

	@Override
	public void timeStep(Simulation simulation) {
		// adjust age
		age++;
		
		// increase infestation
		infestation += MONTHLY_INFESTATION_GROWTH;
		
		// chance to die naturally
		if (simulation.chance(Constants.PROB_NATURAL_DEATH)) {
			simulation.remove(this);
		}
		
		// infestation death
		if (infestation >= INFESTATION_DEATH_THRESHOLD) {
			simulation.remove(this);
		}
		
		// chance to breed in nearby empty spots
		Location[] emptySpots = simulation.getEmptySpots(location, Constants.SEEDLING_DISTANCE);
		for (Location loc : emptySpots) {
			int distance = location.distanceTo(loc);
			if (simulation.chance(Constants.PROB_SEEDLING * (Constants.SEEDLING_DISTANCE + 1 - distance) / Constants.SEEDLING_DISTANCE)) {
				Agent seedling = new LodgepolePine(Constants.INITIAL_AGE, loc);
				simulation.add(seedling);
			}
		}
		
		// chance to infest nearby uninfested trees
		if (infestation > INFESTATION_SPREAD_THRESHOLD) {
			Agent[] nearbyAgents = simulation.getAgents(location, INFESTATION_DISTANCE);
			for (Agent a : nearbyAgents) {
				if (a.isLodgepolePine() && !a.isInfested() && a.getAge() >= MIN_INFESTATION_AGE) {
					int distance = location.distanceTo(a.getLocation());
					if (simulation.chance(PROB_INFESTATION * infestation * (INFESTATION_DISTANCE + 1 - distance) / INFESTATION_DISTANCE)) {
						Agent infectedPine = new InfectedPine(a.getAge(), a.getLocation());
						simulation.add(infectedPine);
					}
				}
			}
		}
	}

	@Override
	public boolean isLodgepolePine() {
		return true;
	}

	@Override
	public boolean isInfested() {
		return true;
	}

	@Override
	public int getAge() {
		return age;
	}

	@Override
	public int getInfestation() {
		return infestation;
	}

	@Override
	public Location getLocation() {
		return location;
	}

	@Override
	public void logState() {
		//System.out.println("Infected Lodgepole Pine age " + age + " at " + location.x + "," + location.y + " with severity " + infestation);
	}

}
