function [boxes,model,loss,ex] = detect_hier(im, model, thresh, bbox, overlap, id, label, loss)
%        [boxes,model,loss,ex] = detect_hier(im, model, thresh, bbox, overlap, id, label, loss)
% Detect objects in image using a model and a score threshold.
% Higher threshold leads to fewer detections.
%
% The function returns a matrix with one row per detected object.  The
% last column of each row gives the score of the detection.  The
% column before last specifies the component used for the detection.
% Each set of the first 4 columns specify the bounding box for a part
%
% If bbox is not empty, we pick best detection with significant overlap. 
% If label is included, we write feature vectors to a global QP structure
%
% This function updates the model (by running the QP solver) if a large hinge loss is accumulated
INF = 1e10;

if nargin > 3 && ~isempty(bbox) && label == 1
  latent = true;
  thresh = -1e10;
else
  latent = false;
end

% Compute the feature pyramid and prepare filter
pyra     = featpyramid(im,model);
interval = model.interval;
nLeaves = model.nLeaves;
levels   = 1:length(pyra.feat);

% Define global QP if we are writing features
% Randomize order to increase effectiveness of model updating
write = false;
if nargin > 5,
  global qp;
  write  = true;
  levels = levels(randperm(length(levels)));
end
if nargin < 6
  id = 0;
end
if nargin < 7
  label = 0;
end
if nargin < 8
  loss = 0;
end

% Cache various statistics derived from model
[components,filters,resp] = modelcomponents(model,pyra);
boxes     = zeros(50000, nLeaves*4+2);
ex.blocks = [];
ex.id     = [label id 0 0 0];
cnt       = 0;

% Iterate over random permutation of scales and components,
for rlevel = levels,
  % Iterate through mixture components
  for c  = randperm(length(model.components)),
    parts    = components{c};
    numparts = length(parts);

    % Skip if there is no overlap of root filter with bbox
    if latent,
      skipflag = 0;
      for k = 1:model.nLeaves
        % because all mixtures for one part is the same size, we only need to do this once
        ovmask = testoverlap(parts(k).sizx(1),parts(k).sizy(1),pyra,rlevel,bbox{k},overlap);
        if ~any(ovmask)
          skipflag = 1;
          break;
        end
      end
      if skipflag == 1
        continue;
      end
    end

    % Local scores
    for k = 1:numparts,
      f     = parts(k).filterid;
      
      level = rlevel-parts(k).scale*interval;
      if isempty(resp{level}),
        resp{level} = fconv(pyra.feat{level},filters,1,length(filters));
      end
      if ~isempty(f)
        for fi = 1:length(f)
            parts(k).score(:,:,fi) = resp{level}{f(fi)};
        end
      else
          [resp_m, resp_n, ~] = size(resp{level}{1});
          % latent node..
          parts(k).score = zeros(resp_m, resp_n, parts(k).K);
      end;
      parts(k).level = level;
	  
      if latent
        for fi = 1:length(f)
          ovmask = testoverlap(parts(k).sizx(fi),parts(k).sizy(fi),pyra,rlevel,bbox{k},overlap);
          tmpscore = parts(k).score(:,:,fi);
          tmpscore(~ovmask) = -INF;
          parts(k).score(:,:,fi) = tmpscore;
        end
      end
    end
    
    % Walk from leaves to root of tree, passing message to parent
    % tydsh: for hier model..the order is reversed..
    for k = 1:numparts-1
      par = parts(k).parent;
      assert(par > k);
      % fprintf(1, 'deal with k = %d, parent = %d\n', k, par);
      [msg,parts(k).Ix,parts(k).Iy,parts(k).Ik] = passmsg(parts(k),parts(par));
      parts(par).score = parts(par).score + msg;
    end

    % Add bias to root score
    parts(end).score = parts(end).score + parts(end).b;
    [rscore Ik]    = max(parts(end).score,[],3);

    % Zero-out invalid regions in latent mode
    % tydsh: this means in latent model (which is the mode that positive image is actually used in detect.m), 
    %        there will be at most one bounding box in each scale..
    
    if latent,
      thresh = max(thresh,max(rscore(:)));
    end

    % tydsh: !!! putting all the samples that with a score higher than threshold at a time!!! (That's the difference..)
    %        I only put the MAP solution at a time.
    %        in the latent mode (poslatent), they treat the best detection in the training set as the positive sample...
    %               (what if the detection does not fit the groundtruth well?)
    %               or just rely on the latent model to figure that out...rather than using the positive labels..
    %               the positive labels are only used for part-based training...
    
    [Y,X] = find(rscore >= thresh);
    % Walk back down tree following pointers
    % (DEBUG) Assert extracted feature re-produces score
    for i = 1:length(X)
      x = X(i);
      y = Y(i);
      k = Ik(y,x);
      [box,ex] = backtrack( x , y , k, parts , pyra , ex , latent || write);
      % check if ex gives the same score as rscore(y, x), under current w..
      %obj_ex_score(model, w, ex);
      % 
      cnt = cnt + 1;
      % only leaf node are returned.
      boxes(cnt,:) = [box c rscore(y,x)];
      if write && ~latent,
        qp_write(ex);
        % is this a bug here? C = Cneg / Cpos should depend on the label of the data. 
        loss  = loss + qp.Cneg*max(1+rscore(y,x),0);
      end
    end
    
    % If we're computing features, assert extracted feature re-produces score
    % (see qp_writ.m for computing original score)
    if write && ~latent && ~isempty(X) && qp.n < length(qp.a),
        % this only works for negative samples...
      w = -(qp.w + qp.w0.*qp.wreg) / qp.Cneg;
      assert((score(w,qp.x,qp.n) - rscore(y,x)) < 1e-5);   
    end
      
    % Optimize qp with coordinate descent, and update model
    if write && ~latent && ... 
        (qp.obj < 0 || loss/qp.obj > .01 || qp.n == length(qp.sv))
      model = optimize(model);
      [components,filters,resp] = modelcomponents(model,pyra);    
      loss = 0;
    end
  end
end

boxes = boxes(1:cnt,:);

if latent && ~isempty(boxes),
    % just pick one in the latent mode (what's that mean)?
  boxes = boxes(end,:);
  if write,
      % just write the last bbox..
    qp_write(ex);
    
    % Crucial DEBUG assertion:
    w = qp.w + qp.w0.*qp.wreg;
    if label == 1
        this_C = qp.Cpos;
    else
        this_C = -qp.Cneg;
    end;
    
    score_svm = score(w,qp.x,qp.n) / this_C;
    score_inference = boxes(1, end);
    diff = abs(score_svm - score_inference);
    
    if diff > 1e-5
        fprintf(1, 'Error!! score_svm = %f, score_inference = %f\n', score_svm, score_inference);
        %assert(0);
    end;
  end
end

% Update QP with coordinate descent
% and return the asociated model
function model = optimize(model)
  global qp;
  fprintf('.');  
  if qp.obj < 0 || qp.n == length(qp.a),
    qp_opt(1);
    qp_prune();
  else
    qp_one();
  end
  model = vec2model(qp_w,model); 

% Compute a mask of filter reponse locations (for a filter of size sizy,sizx)
% that sufficiently overlap a ground-truth bounding box (bbox) 
% at a particular level in a feature pyramid
function ov = testoverlap(sizx,sizy,pyra,level,bbox,overlap)
  scale = pyra.scale(level);
  padx  = pyra.padx;
  pady  = pyra.pady;
  [dimy,dimx,foo] = size(pyra.feat{level});
  
  bx1 = bbox(1);
  by1 = bbox(2);
  bx2 = bbox(3);
  by2 = bbox(4);
  
  % Index windows evaluated by filter (in image coordinates)
  x1 = ((1:dimx-sizx+1) - padx - 1)*scale + 1;
  y1 = ((1:dimy-sizy+1) - pady - 1)*scale + 1;
  x2 = x1 + sizx*scale - 1;
  y2 = y1 + sizy*scale - 1;
  
  % Compute intersection with bbox
  xx1 = max(x1,bx1);
  xx2 = min(x2,bx2);
  yy1 = max(y1,by1);
  yy2 = min(y2,by2);
  w   = xx2 - xx1 + 1;
  h   = yy2 - yy1 + 1;
  w(w<0) = 0;
  h(h<0) = 0;
  inter  = h'*w;
  
  % area of (possibly clipped) detection windows and original bbox
  area = (y2-y1+1)'*(x2-x1+1);
  box  = (by2-by1+1)*(bx2-bx1+1);
  
  % thresholded overlap
  ov   = inter ./ (area + box - inter) > overlap;

  
  
  
