open Printf

let pp_list_z zs =
  let rec loop = function
    | [] -> ()
    | [x] -> printf "%s" (Z.to_string x)
    | x::xs -> printf "%s " (Z.to_string x); loop xs
  in
  printf "["; loop zs; printf "]\n"

type case = { name:string; initial:int list; accept:int list; trace:int list; expected:bool; graph:string }

let read_file path =
  let ic = open_in path in
  let b = Buffer.create 1024 in
  (try while true do Buffer.add_string b (input_line ic); Buffer.add_char b '\n' done with End_of_file -> ());
  close_in ic; Buffer.contents b

let trim s =
  let is_space = function ' '| '\t' | '\r' | '\n' -> true | _ -> false in
  let a = ref 0 and b = ref (String.length s - 1) in
  while !a <= !b && is_space s.[!a] do incr a done;
  while !b >= !a && is_space s.[!b] do decr b done;
  if !a > !b then "" else String.sub s !a (!b - !a + 1)

let parse_ints s =
  s |> String.split_on_char ' ' |> List.filter ((<>) "") |> List.map int_of_string

let parse_case path : case =
  let lines = read_file path |> String.split_on_char '\n' in
  let tbl = Hashtbl.create 7 in
  let rec split_graph acc = function
    | [] -> (List.rev acc, "")
    | l::ls when trim l = "--graph--" ->
        let buf = Buffer.create 256 in
        let rec grab = function
          | [] -> Buffer.contents buf
          | l'::ls' when trim l' = "--end--" -> Buffer.contents buf
          | l'::ls' -> Buffer.add_string buf l'; Buffer.add_char buf '\n'; grab ls'
        in
        let gtxt = grab ls in
        (List.rev acc, gtxt)
    | l::ls -> split_graph (l::acc) ls
  in
  let header, graph = split_graph [] lines in
  List.iter (fun l -> match String.split_on_char ':' l with
    | [k;v] -> Hashtbl.replace tbl (trim k) (trim v)
    | _ -> () ) header;
  let get k = try Hashtbl.find tbl k with Not_found -> failwith ("missing key: "^k) in
  let name = get "name" in
  let initial = parse_ints (get "initial") in
  let accept  = parse_ints (get "accept") in
  let trace   = parse_ints (get "trace") in
  let expected = match String.lowercase_ascii (get "expect") with "true"->true | "false"->false | x->failwith ("bad expect:"^x) in
  { name; initial; accept; trace; expected; graph }

let parse_graph_text content =
  let nodes = Hashtbl.create 16 in
  let add_node i = Hashtbl.replace nodes i () in
  let edges = ref [] in
  let re_edge = Str.regexp "\\[\\([0-9]+\\)\\] *-> *\\[\\([0-9]+\\)\\]" in
  let re_node = Str.regexp "^ *\\[\\([0-9]+\\)\\]" in
  content |> String.split_on_char '\n' |> List.iter (fun l ->
    let l = trim l in
    if Str.string_match re_edge l 0 then (
      let u = int_of_string (Str.matched_group 1 l) in
      let v = int_of_string (Str.matched_group 2 l) in
      add_node u; add_node v; edges := (u,v)::!edges
    ) else if Str.string_match re_node l 0 then (
      let u = int_of_string (Str.matched_group 1 l) in add_node u
    ) else ()
  );
  let maxn = Hashtbl.fold (fun k _ m -> max m k) nodes 0 in
  let n = maxn + 1 in
  let adj = Array.make n [] in
  List.iter (fun (u,v) -> adj.(u) <- v :: adj.(u)) !edges;
  (n, adj)

let build_buchi (n,adj) (c:case) : Extracted.buchi =
  let initial = List.map Z.of_int c.initial in
  let accept  = List.map Z.of_int c.accept in
  let g = Extracted.create (Z.of_int n) initial accept in
  Array.iteri (fun u vs -> List.iter (fun v -> Extracted.add_transition g (Z.of_int u) (Z.of_int v)) (List.rev vs)) adj;
  g

let zlist_of_ints xs = List.map Z.of_int xs

let mem_z (x:Z.t) (zs:Z.t list) = List.exists (fun y -> Z.equal x y) zs

let render_diagram (graph_txt:string) : string option =
  try
    let (ic, oc, ec) = Unix.open_process_full "graph-easy --boxart" [||] in
    output_string oc graph_txt; flush oc; close_out oc;
    let buf = Buffer.create 1024 in
    (try while true do Buffer.add_string buf (input_line ic); Buffer.add_char buf '\n' done with End_of_file -> ());
    match Unix.close_process_full (ic, oc, ec) with
    | Unix.WEXITED 0 -> Some (Buffer.contents buf)
    | _ -> None
  with Unix.Unix_error (Unix.ENOENT, _, _) -> None

let print_case base_dir (c:case) : bool =
  let (n,adj) = parse_graph_text c.graph in
  let g = build_buchi (n,adj) c in
  let trace = zlist_of_ints c.trace in
  let ok = Extracted.check_accepting g trace in
  printf "- %s\n" c.name;
  printf "  initial:  ["; List.iter (fun i-> printf "%d " i) c.initial; printf "]\n";
  printf "  accept:   ["; List.iter (fun i-> printf "%d " i) c.accept;  printf "]\n";
  printf "  trace:    ["; List.iter (fun i-> printf "%d " i) c.trace;    printf "]\n";
  printf "  expected: %b\n  outcome:  %b\n" c.expected ok;
  begin match render_diagram c.graph with
  | Some d -> printf "  graph:\n%s" d
  | None -> printf "  graph:    (graph-easy unavailable or failed)\n"
  end;
  let pass = (ok = c.expected) in
  printf "  status:   %s\n\n" (if pass then "PASS" else "FAIL");
  pass

let load_cases base_dir : case list =
  let dir = Filename.concat base_dir "tests-accepting" in
  Sys.readdir dir |> Array.to_list |> List.sort compare |> List.filter (fun f-> Filename.check_suffix f ".tcase")
  |> List.map (fun f -> parse_case (Filename.concat dir f))

let run_one base_dir name =
  let cases = load_cases base_dir in
  let find nm = List.find_opt (fun t -> t.name = nm) cases in
  match find name with
  | None -> eprintf "Unknown test '%s'\n" name
  | Some t -> ignore (print_case base_dir t)

let run_all base_dir =
  let cases = load_cases base_dir in
  let totals = List.fold_left (fun (p,t) c -> let ok = print_case base_dir c in ((if ok then p+1 else p), t+1)) (0,0) cases in
  let (passed,total) = totals in
  let failed = total - passed in
  printf "Summary: %d passed, %d failed, %d total\n" passed failed total

let () =
  let exe = Sys.argv.(0) in
  let build_dir = Filename.dirname exe in
  let base_dir = Filename.dirname build_dir in
  match Array.to_list Sys.argv |> List.tl with
  | [] -> run_all base_dir
  | [arg] ->
      if arg = "all" then run_all base_dir
      else run_one base_dir arg
  | _ -> (eprintf "Usage: run_accepting [all|name]\n"; exit 2)
