;;; -*- Mode:  LISP; Package: HUH -*-

;;.@chapter{A few words about words.}

(in-package :huh)


;;.HUH represents words in ARLOtje; each word HUH recognizes is a
;;.unit defined in the @code{WORDS} package with properties relating
;;.it to both other words.  For instance, an infinitive (like
;;.@code{GO}) has a @code{PAST-TENSE} slot which points to the word
;;.@code{WENT}.  These words are all units (symbols) interned in the
;;.@code{WORDS} package which inherits from @emph{no} other packages.

(eval-when (eval load compile) 
  (defvar *words* (find-package "WORDS")
    "This is the package containing words to parse."))

;;.Most words known by HUH are sequences of alphabetic characters
;;.represented in a canonical uppercase format.  Occasionally, though,
;;.words may contain spaces or other characters: often this is a way
;;.of `cheating' on recognizing noun groups, taking phrases like
;;.``Central Square T Stop'' or ``Soviet Union'' as single words.  HUH
;;.also uses this mechanism to expand some contractions, for instance
;;.``IT'S'' expansion to ``IT IS''.

;;.The library of words is used in transforming english text into
;;.`word lists' of symbols.  The parser actually operates on these
;;.word lists.  The function @code{read-words-from-string} returns a
;;.list of the words for a text string; it is used as a preprocessor
;;.for HUH's parsing mechanisms.

;;..In fact, the variable @code{*funny-words*} contains all the
;;..strings which are known to be words with spaces; when the system
;;..attempts to extract a word from a string, it first looks for
;;..matches among @code{*funny-words*}.
(defvar *funny-words* '()
  "This is a list of `words' containing spaces.")

;;..HUH recognizes the punctuation characters stored in
;;..@code{*punctuation*} as valid word breaks and also as `atomic
;;..words' parsed as individual elements.
(defparameter *punctuation* '(#\, #\. #\? #\( #\) #\$ #\% #\')
  "A list of punctuation characters treated as separate words.")
(defparameter *whitespace* '(#\Space #\Tab #\Newline)
  "A list whitespace characters.")
(defparameter *word-breaks* (append *whitespace* *punctuation*)
  "A list of places to break words.")

;;.An occasionally useful predicate (used in HUH itself) is
;;.@code{same-word-p} which returns true if two symbols have the same
;;.print name.
(defun same-word-p (w1 w2)
  (string-equal (symbol-name w1) (symbol-name w2)))


;;.The collection @code{words} contains all the units recognized as
;;.words; a word is `registered' by calling the procedure
;;.@code{register-word} on a string (or a symbol which will be coerced
;;.into a string).  The string is interned in the @code{WORDS} package
;;.and then declared as a member of the collection @code{words}.
;;.The function @code{register-words} takes any number of words and
;;.registers them all.

(define-unit words
  (english-description "This is the collection of all words.")
  (member-of 'collections))

(defun register-word (string)
  "Declares STRING as a valid word."
  (let* ((string (if (symbolp string) (symbol-name string) string))
	 (word (intern (string-upcase string) *words*)))
    (if (find-if #'(lambda (c) (member c *word-breaks*)) string)
	(push string *funny-words*))
    (make-unit word
      (member-of 'words))))

(defun register-words (&rest words)
  (mapcar #'register-word words))

;;.Once a word has been registered, the procedure @code{identify-word}
;;.will return the corresponding unit when presented with a matching
;;.string; if the string passed to @code{identify-word} is not
;;.registered, @code{NIL} is returned.  This procedure is later extended to
;;.deal with varieties of morphemic analysis; for instance, a later
;;.version will register a word as the past tense of another if it
;;.follows the `ed' rule.

;; Later versions of this do morphemic analysis.
(defun identify-word (string)
  (find-symbol (string-upcase string) *words*))

(dolist (punc *punctuation*)
  (register-word (make-string 1 :initial-element punc)))

;;.Words can be defined in a variety of ways.  One straightforward one
;;.is the macro @code{define-word} which looks just like
;;.@code{DEFINE-UNIT} but the first argument is a string which is
;;.registered as a word.

(defmacro define-word (word &body slots)
  `(progn (if (and (find-symbol ,word *words*)
		   (get (find-symbol ,word *words*) 'aj::creation-id))
	      (kill-changes (find-symbol ,word *words*)))
	  (let ((word (register-word ,word)))
	    (record-source-file word :unit)
	    (make-unit word (member-of 'words) ,@slots))))


;;;;.Numbers into words

;;.HUH automatically transforms numbers into words with the procedure
;;.@code{numerify} which creates a new word with a
;;.@code{numeric-value} slot corresponding to the value.  These words
;;.are in the collection @code{numeric-words}.

(define-unit number-words
  (member-of 'collections)
  (supersets 'words))
(define-unit numeric-value
  (member-of 'single-valued-slots)
  (makes-sense-for 'number-words)
  (must-be 'numberp))
(defun numerify (num)
  (make-unit (fsymbol *words* "THE-NUMBER-~S" num)
    (member-of 'number-words)
    (numeric-value num)))


;;;;.Strings into words 

;;.Strings are translated into words by the procedure
;;.@code{read-words-from-string} which returns a `word list' from a
;;.string.  Words are separated by `word breaks' and certain `word
;;.breaks' count as words in themselves and are called `punctuation.'

(defun read-words-from-string (string)
  "Transforms STRING into a list of parsed and identified words."
  (let ((words '()))
    (loop (if (= (length string) 0)
	      (return (reverse words)))
      (multiple-value-bind (word remainder) (extract-word string)
	(if (and (satisfies? word 'words)
		 (not (failurep (get-value word 'text-expansion))))
	    (setq words (append (get-value word 'text-expansion) words) string remainder)
	  (setq words (cons word words) string remainder))))))

;;.Words are read from a string by repeated applications of the
;;.procedure @code{extract-word}; each word extracted is then
;;.potentially expanded into the words @code{text-expansion} (if
;;.defined).  This allows (for instance) the expansion of contractions
;;.like @code{IT'S} or @code{THEY'RE}.  It also handles an expansion
;;.of @code{his} to @code{HE'S} to indicate possessive attribution.
(define-unit text-expansion
  (member-of 'single-valued-slots)
  (makes-sense-for 'words)
  (must-be 'listp))

(define-word "his"
    (text-expansion '(words::he words::|'| words::S)))
(define-word "hers"
  (text-expansion '(words::her words::|'| words::S)))
(define-word "theirs"
  (text-expansion '(words::they words::|'| words::|S|)))
(define-word "ITS"
  (text-expansion '(words::it words::|'| words::|S|)))
(define-word "IT'S"
  (text-expansion '(words::it words::is)))

;;.The procedure @code{extract-word} takes a string and returns a word
;;.and a shorter string.  Whitespace in the original string is
;;.discarded and punctuation is returned as individual words.

;;..The procedure first checks if the string begins with a
;;..punctuation; if so, that punctuation is returned as a single word
;;..and the remainder of the string (with whitespace trimmed) is
;;..returned.  If the first character is not a punctuation character,
;;..the list of `funny words' (words containing word breaks) is
;;..searched to find if there is a match; if so, the corresponding
;;..word is returned.  Finally, if no funny word is matched, it tries
;;..to read a lisp object from the string and if this object is a
;;..number, it is appropriately @code{numerify}ed into a word and
;;..returned; otherwise, the first word break in the string is
;;..identified and the substring preceding it isolated; if this
;;..substring can be identified as a word, the corresponding word is
;;..returned; otherwise, it is simply interned as a symbol in the
;;..current package.
(defun extract-word (string)
  (if (member (char string 0) *punctuation*)
      (values (identify-word (make-string 1 :initial-element (char string 0)))
	      (string-trim *whitespace* (subseq string 1)))
    (let ((funny-word
	   (find-if #'(lambda (registered)
			(every #'char-equal registered string))
		    *funny-words*)))
      (if funny-word
	  (values (identify-word funny-word)
		  (string-trim *whitespace* (subseq string (length funny-word))))
	(multiple-value-bind (thing pos) (read-from-string string)
	  (if (numberp thing)
	      (values (numerify thing) (string-trim *whitespace* (subseq string pos)))
	    (let ((end-of-word (or (position-if #'(lambda (x) (member x *word-breaks*))
						string)
				   (length string))))
	      (values (or (identify-word (subseq string 0 end-of-word))
			  (intern (string-upcase (subseq string 0 end-of-word))))
		      (string-trim *whitespace* (subseq string end-of-word))))))))))


;;;;.Words into ARLOtje

;;.HUH's knowledge about transforming english text into ARLOtje
;;.expressions is organized by providing `interpretations' for
;;.individual words.  For instance, a word can have an
;;.@code{adjective-interpretation} slot which stores a set of
;;.slot/value pairs indicating that the adjective applied to an object
;;.means that a certain slot of the object's description has a certain
;;.value.  For instance, the adjective ``red'' has the adjective
;;.interpretation `(color red)' meaning that `the red box' should
;;.refer to a box (specified in some other way) with a @code{color}
;;.slot of @code{red}.

;;.A sort of inverse of @code{adjective-interpretation} is implemented
;;.with the @code{adjective-forms} slot of individual slots; if an
;;.adjective `adj' has an @code{adjective-interpretation} of
;;.@code{(@var{slot} @var{value})}, the corresponding slot has an
;;.@code{adjective-forms} slot of @code{(@var{`adj'} @var{value})}.

(define-unit adjective-interpretation
  (member-of 'many-valued-slots)
  (makes-sense-for 'words)
  (must-be 'listp)
  (put-demons
   '(assert-value (car %value%) adjective-forms
     (list (cadr %value%) %unit%))))

(define-unit adjective-forms
  (member-of 'many-valued-slots)
  (makes-sense-for 'slotp)
  (must-be 'listp)
  (put-demons
   '(assert-value (register-word (cadr %value%)) adjective-interpretation
     (list %unit% (car %value%)))))

;;.A similar sort of annotation is implemented with the slots
;;.@code{class-interpretation} and @code{english-class-name}.  For
;;.instance, a noun like `box' above might indicate an object in the
;;.collection @code{boxes}; this would be indicated by storing a
;;.@code{class-interpretation} slot on the word `box' (the unit
;;.@code{WORDS:BOX}) whose value is @code{boxes}.

(define-unit class-interpretation
  (member-of 'many-valued-slots)
  (makes-sense-for 'words)
  (must-be 'collections)
  (put-demons '(assert-value %value% english-class-name %unit%)))

;;.The inverse of this slot is @code{english-class-name} and often the
;;.relation between word and class is established when a class is
;;.initially defined; thus, we might have a definition of boxes like
;;.the following:@refill
;;.@example
;;.(define-unit boxes
;;.  (member-of 'collections)
;;.  (superset 'artifacts)
;;.  (english-class-name (register-word "box")))
;;.@end example
;;.when HUH is initially loaded, a few class names are set up for
;;.elements of ARLOtje's starting ontology; `woman,' `man,' `animal,'
;;.etc.  But most of this definition is reserved for individual
;;.definitions.

(define-unit english-class-name
  (member-of 'many-valued-slots)
  (put-demons '(assert-value (register-word %value%)
		class-interpretation %unit%)))

;;.English has a place for properties which are similar to ARLOtje's
;;.slots; for instance, we can speak of the `color of the box' and
;;.refer to a property which likely has a cognate in ARLOtje's
;;.representation.  The @code{slot-interpretation} slot indicates this
;;.connection; for instance, giving the word `mother' a
;;.@code{slot-interpretation} of @code{mother} allows understanding
;;.phrases like ``The mother of John'' or ``John's mother''.  The
;;.inverse of @code{slot-interpretation} is @code{english-slot-name}
;;.and this is often the way an english slot name is declared.

(define-unit slot-interpretation
  (member-of 'many-valued-slots)
  (makes-sense-for 'words)
  (must-be 'slotp)
  (put-demons '(assert-value %value% english-slot-name %unit%)))

(define-unit english-slot-name
  (member-of 'many-valued-slots)
  (put-demons '(assert-value (register-word %value%)
		slot-interpretation %unit%)))

(define-unit English-Name
  (english-description "The name in English for a concept.")
  (member-of 'many-valued-slots)
  (spec-slots 'english-class-name)
  (spec-slots 'english-slot-name))


;;;;.More Adjectives

;;.Several user level macros are provided for declaring `adjective
;;.interpretations.  The macro @code{define-adjective} takes a word
;;.(as a string), a slot name, and a value, none of which are
;;.evaluated; e.g. we can say:
;;.@example
;;.(define-adjective "big" size large)
;;.@end example
;;.to associate the word ``big'' with a @code{size} slot of @code{large}.

(defmacro define-adjective (word slot value)
  `(assertion (register-word ,word) 'adjective-interpretation '(,slot ,value)))

;;.A common cliche of slot definition is to define a slot whose values
;;.are from some fixed collection (for instance, a @code{color} slot
;;.with particular recognized values); the macro
;;.@code{define-property} provides for the specification of such slots
;;.with corresponding english interpretations.

(defmacro define-property (slot values &body slot-properties)
  "Defines how to interpret a set of words as values for SLOT.
If an element V of VALUES is a symbol, the corresponding word
corresponds to `SLOT is V' while if it is a pair (word V) the word
WORD is given the interpretation `SLOT is V'."
  `(progn
    (define-unit ,slot
      (works-like 'prototypical-slot)
      ,@slot-properties)
    (define-unit ,(fsymbol "~AS" slot)
      (member-of 'collections)
      ,@(mapcar #'(lambda (x) `(members (declare-unit ',x  :external? nil)))
		values))
    ,@(mapcar #'(lambda (x) `(define-adjective ,(symbol-name x) ,slot ,x))
       values)))

;;.For instance,
;;...@example
(define-property color
    (red blue green grey mauve purple black white gray
	 yellow silver tan brown gold orange)
  (member-of 'many-valued-slots))
;;...@end example
;;.defines a slot @code{color} and a collection @code{colors}.  The
;;.collection contains the symbols @code{red}, @code{blue}, etc; the
;;.slot @code{color} accepts values from @code{colors} (the
;;.@code{must-be} slot of @code{color} is @code{colors}).  The slot
;;.@code{colors} is additionally asserted to be a many-valued slot.

(define-property size
    (huge big large regular smaller smallish tiny)
  (member-of 'single-valued-slots))


;;;;.Special names

;;.An object's @code{Proper-names} allows the connection of particular
;;.objects to particular words or sequences of words.

(define-unit proper-name
  (english-description "These are the proper names of an object.")
  (member-of 'many-valued-slots)
  (makes-sense-for 'unitp)
  (must-be 'words))

(define-unit proper-name-of
  (english-description "These are the objects for which this is a proper name.")
  (member-of 'many-valued-slots)
  (makes-sense-for 'words)
  (must-be 'unitp)
  (inverse-slot 'proper-name))

;;.One special class of words is people's names.  HUH maintains a
;;.subset of words called @code{first-names} for recognizing proper
;;.names of human beings; associated with this is a slot
;;.@code{likely-gender} which supports assumptions made in English names
;;.of Western European extraction.  People have a @code{first-name}
;;.and @code{last-name} slot; the last name can be any symbol, but the
;;.first must be a member of the collection @code{first-names}.  But
;;.one can declare a word a first name by simply doing:
;;.@example
;;.(assertion (register-word @var{name}) 'member-of 'first-names)
;;.@end example

(define-unit first-names
  (member-of 'collections)
  (supersets 'words))

(define-unit likely-gender
  (member-of 'single-valued-slots)
  (makes-sense-for 'first-names))

(define-unit first-name
  (works-like 'prototypical-slot)
  (makes-sense-for 'animals)
  (must-be 'words))

(define-unit last-name
  (works-like 'prototypical-slot)
  (makes-sense-for 'humans)
  (must-be 'words))


;;;;.ARLOtje vocabulary

;;.In addition to vocabulary for talking about people, HUH has a
;;.limited vocabulary for describing itself and its base ontology.
;;.Words like `woman', `person', `superset', `collection' are declared
;;.in a straightforward manner:
;;...@example
(assertion 'collections 'english-class-name (register-word "collection"))
(assertion 'humans 'english-class-name (register-word "human"))
(assertion 'humans 'english-class-name (register-word "person"))
(assertion 'men 'english-class-name (register-word "man"))
(assertion 'women 'english-class-name (register-word "woman"))
(assertion 'generalizations 'english-slot-name (register-word "superset"))
(assertion 'specializations 'english-slot-name (register-word "subset"))
(assertion 'inverse-slot 'english-slot-name (register-word "inverse"))
;;...@end example


;;;;.Kinds of words

;;.Beyond the recognition of first names, HUH has more knowledge about
;;.parts of speech; it defines subsets of words for nouns, verbs,
;;.adjectives, adverbs, and prepositions.  Infinitives (`go', `walk',
;;.and the like) are tagged as such by being placed in the collection
;;.@code{infinitives} beneath verbs.

(define-unit verbs
  (member-of 'collections)
  (supersets 'words))

(define-unit adjectives
  (member-of 'collections)
  (supersets 'words))

(define-unit nouns
  (member-of 'collections)
  (supersets 'words))

(define-unit infinitives
  (member-of 'collections)
  (supersets 'verbs))

(define-unit prepositions
  (member-of 'collections)
  (supersets 'words))

(dolist (prep (register-words "in" "to" "from" "by" "for" "which"))
  (assertion prep 'member-of 'prepositions))

;;.Inferences associated with the slots
;;.@code{adjective-interpretation} and @code{class-interpretation}
;;.assert that their carriers are in the collections @code{adjectives}
;;.and @code{nouns} respectively.
(assertion 'adjective-interpretation 'inferences '(member-of adjectives))
(assertion 'class-interpretation 'inferences '(member-of nouns))


;;;;.First, A few words...

;;.A few useful words are defined at first to support parsing; words
;;.like ``the'' and ``it'' which the parser refers to explicitly.  In
;;.addition, a handful of first names are defined with corresponding
;;.genders. 

(register-words "S" "IT" "he" "her" "his" "her" "IS" "they" "the" "a" "an" "word")

(dolist (name '(John Ken Alan Barry Tod Marvin Steve Bob Robert Mike
		Michael Frank Joep Jon Eric Erik Hans Han Adam))
  (assertion (register-word name) 'member-of 'first-names)
  (assertion (register-word name) 'likely-gender 'men))
(dolist (name '(Mary Betty Diane Polly Marcia Molly Jane Kathleen
		Lisa Jean Jeanne Sarah Gloria Amy Eve Margaret))
  (assertion (register-word name) 'member-of 'first-names)
  (assertion (register-word name) 'likely-gender 'women))

;;.The ARLOtje ontology is half-heartedly extended to include
;;.restaurants, homes, schools, airports, offices, and houses; these
;;.are all subsets of @code{places} and have corresponding english
;;.class names.
(dolist (name '(restaurant school airport office home house))
  (make-unit name
    (member-of 'collections)
    (supersets 'places)
    (english-class-name (register-word name))))

;;.In addition, a few things are defined as specializations of
;;.@code{things}: cars, bicycles, trains, planes taxis, pogos,
;;.elevators, boxes, trees, and telephones.
(dolist (name '(car bicycle train plane taxi pogo elevator box table desk tree telephone))
  (make-unit name
    (member-of 'collections)
    (supersets 'things)
    (english-class-name (register-word name))))

(dolist (class.names '((restaurant "Mary Chung's" "Mary's" "Hacker Heaven")
		       (restaurant "Bertucci's")
		       (restaurant "Legal Seafoods" "Legal's")
		       (school "MIT" "the Institute")
		       (office "the Media Lab")
		       (office "the AI Lab")
		       (airport "Logan" "Logan airport")))
  (let ((unit (make-unit (gensymbol (car class.names))
		(member-of (car class.names)))))
    (dolist (name (cdr class.names))
      (assertion unit 'proper-name (register-word name)))))


;;.In addition, a few things are defined as specializations of
;;.@code{things}: cars, bicycles, trains, planes taxis, pogos,
;;.elevators, boxes, trees, and telephones.
(dolist (name '(gorilla chimpanzee dog cat llama))
  (make-unit name
    (member-of 'collections)
    (supersets 'animals)
    (english-class-name (register-word name))))

(define-unit monkey
  (member-of 'collections)
  (supersets 'animals)
  (subsets 'gorilla)
  (subsets 'chimpanzee))

(setf (get 'many-valued 'adjective-interpretation)
      '(works-like prototypical-set-slot))


;;;;.Representing prepositions.

(define-unit english-preposition
  (english-description "This is a preposition used to refer to this slot.")
  (member-of 'many-valued-slots)
  (makes-sense-for 'slotp)
  (must-be 'words))

(define-unit english-preposition-for
  (english-description "This is a preposition used to refer to this slot.")
  (member-of 'many-valued-slots)
  (makes-sense-for 'words)
  (must-be 'slotp)
  (inverse-slot 'english-preposition))


;;;;.Tense information

;;.English tenses are described in ARLOtje by slots from infinitives
;;.to their various tenses; the @code{past-tense} slot (inverse:
;;.@code{past-tense-of}) relates an infintive to its past tense;
;;.similarly the @code{passive-tense} slot contains the infinitives
;;.passive tense.  These slots have @code{to-compute-values} which
;;.follow the default morphological rules.

;;.The slot @code{verb-root} is among the @code{genl-slots} of these
;;.slot's inverses, so that the root of any verb form can be found on
;;.its @code{verb-root} slot.

(define-unit verb-root
  (member-of 'single-valued-slots)
  (makes-sense-for 'words)
  (must-be 'words))

(define-unit past-tense
  (member-of 'single-valued-slots)
  (makes-sense-for 'words)
  (must-be 'words)
  (put-demons '(assert-value %unit% member-of infinitives))
  (put-demons '(assert-value %value% member-of verbs))
  (to-compute-value 'add-ed-to-make-past-tense)
  (inverse-slot (make-unit 'past-tense-of
		  (member-of 'single-valued-slots)
		  (genl-slots 'verb-root))))
(defun add-ed-to-make-past-tense (word slot)
  (declare (ignore slot))
  (register-word (concatenate 'string (symbol-name word) "ED")))
    

(define-unit passive-tense
  (member-of 'single-valued-slots)
  (makes-sense-for 'words)
  (must-be 'words)
  (put-demons '(assert-value %unit% member-of infinitives))
  (put-demons '(assert-value %value% member-of verbs))
  (to-compute-value 'add-ing-to-make-passive-tense)
  (inverse-slot (make-unit 'passive-tense-of
		  (member-of 'single-valued-slots)
		  (genl-slots 'verb-root))))
(defun add-ing-to-make-past-tense (word slot)
  (declare (ignore slot))
  (register-word (concatenate 'string (symbol-name word) "ING")))


;;;;.More morphology

;;.The procedure @code{identify-word} is extended to handle verb
;;.tenses; if a word being identified has no current entry, it checks
;;.to see if it has the form `@var{infinitive}ed' or
;;.`@var{infinitive}ing' and if so installs a corresponding past
;;.tense.  It also is made to handle numbers by transforming them into
;;.words for the form @code{the-number-5} with a @code{numeric-value}
;;.slot indicating their real numeric value.

(defun identify-word (string)
  (flet ((firstn-word (string n)
	   (and (> (length string) n)
		(find-symbol (subseq string 0 (- (length string) n)) *words*))))
    (let ((string (string-upcase string)))
      (or (find-symbol string *words*)
	  (cond ((and (zerop (mismatch "ED" string))
		      (satisfies? (firstn-word string 2)
				  'infinitives))
		 (make-unit (register-word string)
		   (member-of 'verbs)
		   (past-tense-of (firstn-word string 2))))
		((and (zerop (mismatch "ED" string))
		      (satisfies? (firstn-word string 1) 'infinitives))
		 (make-unit (register-word string)
		   (member-of 'verbs)
		   (past-tense-of (firstn-word string 1))))
		((and (zerop (mismatch "ING" string))
		      (satisfies? (firstn-word string 3)
				  'infinitives))
		 (make-unit (register-word string)
		   (member-of 'verbs)
		   (passive-tense-of (firstn-word string 3))))
		(T nil))))))


;;;;.Case Frames

;;.HUH supports the descriptions of actions (like 'Mary threw the ball
;;.to John.') by parsing them into case frames.  These case frames
;;.describe the translation of sentences into action descriptons; in
;;.particular, they map properties of a sentence: verb, tense,
;;.subject, object, indirect object, prepositions, etc into slots of
;;.an action description.

;;.Case frames are stored on the @code{case-frames} slot of verbs;
;;.this slot `pushes through' the @code{past-tense} and
;;.@code{passive-tense} slots and are so inherited to various forms of
;;.the verb.  The general form of a case frame for a verb is:
;;.@example
;;.(@var{action-type} (@var{subject-slots})
;;.                   (@var{object-slots})
;;.                   (@var{indirect-object-slots})
;;.                   (@var{prep-1} . @var{slots-1})
;;.                   (@var{prep-2} . @var{slots-2})
;;.                   .......
;;.@end example
;;.where @var{action-type} is a collection and most of the other
;;.specifications are lists of slots.  When a case frame is applied to
;;.a sentence, the sentence's subject is stored in each of the
;;.@var{subject-slots}; the sentence's direct object in each of the
;;.@var{object-slots}, the indirect object in each of the
;;.@var{indirect-object-slots} and any modifiers based on @var{prep-1}
;;.in the slots @var{slots-1} and so forth.

;;.For instance,
;;.@example
;;.(assertion (register-word "GIVE") 'case-frames
;;.           '(BEING-MOVED (INITIATOR) (THING-MOVED) (DESTINATION)
;;.	                    (TO DESTINATION)
;;.	                    (FROM ORIGIN)
;;.	                    (BY INSTRUMENT)))
;;.@end example
;;.describes a translation of `go' verbs into events in the collection
;;.@code{being-moved} whose initiator is the subject of the sentence,
;;.whose @code{thing-moved} is the object, and whose destination is
;;.the indirect object.  It also describes the translation of the
;;.prepositions @code{to} and @code{from} to the @code{DESTINATION}
;;.and @code{ORIGIN} slots of the @code{BEING-MOVED} description.

;;.Case frames store a funny sort of inverse; the correpsonding
;;.collection (the first element of the case frame) is given a value
;;.on its @code{english-cases} slot which duplicates the case frame
;;.with the collection replaced by the verb; i.e. @code{BEING-MOVED}
;;.would have a value on its @code{english-cases} slot like:
;;.@example
;;.           '(WORDS::GO (INITIATOR) (THING-MOVED) (DESTINATION)
;;.	                  (TO DESTINATION)
;;.	                  (FROM ORIGIN)
;;.	                  (BY INSTRUMENT))
;;.@end example

(define-unit case-frames
  (member-of 'many-valued-slots)
  (makes-sense-for 'verbs)
  (must-be 'listp)
  (put-demons '(assert-value (car %value%) 'english-cases
		(cons %unit% (cdr %value%))))
  (pushes-through 'past-tense)
  (pushes-through 'passive-tense))

(define-unit english-cases
  (member-of 'many-valued-slots)
  (makes-sense-for (get-value 'processes 'subset-test))
  (must-be 'listp)
  (put-demons '(assert-value (car %value%) 'case-frames
		(cons %unit% (cdr %value%)))))


;;;;.More data.

;;.When loaded, HUH defines information about the words:
;;. go, leave, take, ask, order, give, bring, and get.

(define-word "GO"
    (member-of 'infinitives)
  (past-tense (register-word "WENT"))
  (case-frames
   '(BEING-MOVED (initiator thing-moved) () ()
     (from origin) (to destination) (by vehicle) (in vehicle))))
(define-word "LEAVE"
    (member-of 'infinitives)
  (past-tense (register-word "LEFT"))
  (case-frames
   '(BEING-MOVED (initiator thing-moved) (origin) ()
     (from origin) (to destination) (by vehicle) (in vehicle))))
(define-word "TAKE"
    (member-of 'infinitives)
  (past-tense (register-word "TOOK"))
  (case-frames
   '(BEING-MOVED (initiator) (thing-moved) ()
     (from origin) (to destination) (by vehicle) (in vehicle))))
(define-word "ASK"
    (member-of 'infinitives)
  (case-frames
   '(demanding (speaker) (listener) ()
     (for demanded-object))))
(define-word "ORDER"
    (member-of 'infinitives)
  (case-frames
   '(demanding-something (speaker demand-recipient) (demanded-object) ()
     (for demand-recipient) (from listener))))
(define-word "GIVE"
    (member-of 'infinitives)
  (past-tense (register-word "GAVE"))
  (case-frames
   '(GIVING (giver) (thing-given) (recipient)
     (to recipient) (from origin))))
(define-word "BRING"
    (member-of 'infinitives)
  (past-tense (register-word "BROUGHT"))
  (case-frames
   '(giving (giver) (thing-given) (recipient)
     (to recipient) (from origin))))
(define-word "GET"
    (member-of 'infinitives)
  (past-tense (register-word "GOT"))
  (case-frames
   '(giving (recipient) (thing-given) ()
     (from giver))))

