;;; -*- LISP -*-

(in-package "RESTAURANT")

;;;;.Defining the Restaurant Script

;;.A script is a bundled collection of expectations for a certain kind
;;.of story.  Each object expected by the script is implemented as a
;;.@emph{slot} of the script; these slots have attached inferences
;;.which describe the interrelations of that slot's value with other
;;.slots of the script.

;;.Slots of scripts are of three basic sorts: roles, settings, and
;;.scenes.  A role is a slot filled by a `thing'; a setting is a slot
;;.filled by a `place'; and a scene is a slot filled by an action or
;;.process.  In addition, these slots maybe `optional' meaning that
;;.the scene may or may not occur.

;;.For instance, in the restaurant script there is a `patron' role, a
;;.`restaurant' setting, and an `eating' scene (there are also many
;;.other slots).  The patron is the initiator and the recipient of the
;;.eating action (meaning she feeds herself); the restaurant contains
;;.all the scripts inner activities (ordering, eating, paying, etc);
;;.and is the destination and origin of the activities of leaving and
;;.entering the restaurant.  These interrelations are all implemented
;;.with @code{STRUCTURE} inferences. 

;;.When there are optional scenes, it is the case that some options
;;.exclude other options; for instance, in the restaurant script we
;;.implement here, one possible optional scene is paying first, before
;;.the food arrives, as some restaurants demand that you do.  This
;;.scene excludes the paying later scene.

;;.The first argument to @code{define-script} is the name of the
;;.script being defined; the second is a list of `keyword' strings
;;.whose presence in a story indicates that story may be of this type.
;;.Following this are a list of `slot specifications'.  Each slot
;;.specification begins with either a slot name to be defined or one
;;.of a special set of script slot types:
;;.@table @code
;;.@item role
;;.The slot contains a `thing' which is part of the script.
;;.@item setting
;;.The slot contains a `place' which is part of the script.
;;.@item scene
;;.The slot contains an `action' or `process' which is part of the script.
;;.@item optional-role
;;.The slot contains a `thing' which may be part of the script.
;;.@item optional-setting
;;.The slot contains a `place' which may be part of the script.
;;.@item optional-scene
;;.The slot contains an `action' or `process' which may be part of the script.
;;.@end table
;;.If the slot specification begins with one of these types, the
;;.second element of the specification is the name of the
;;.corresponding slot and the remaining elements are slot-value
;;.specifications (just as for @code{define-unit}) specifying
;;.additional properties of the slot.  In particular, this is where
;;.@code{structure} relations are asserted between slots.  Because
;;.these definitions are @code{bundled} together, earlier definitions
;;.can refer to later ones.

;;.The following lengthy @code{define-script} definition defines a
;;.single script for the restaurant experience; it is a LISP macro
;;.which expands into a large number of `define-unit' definitions
;;.@emph{bundled} together to enable earlier declarations to access
;;.later definitions with impunity.

;;...@example
;; This defines the `restaurant-script'.  The first argument
(define-script restaurant-script ("restaurant")
  (role patron (must-be 'humans))
  (role server (must-be 'waitrons))
  (setting establishment (must-be 'places))
  (role food (must-be 'foodstuff))
  (role payment (must-be 'money))
  (scene going-to-restaurant (must-be 'being-moved)
         (structure '(participants patron))
         (structure '(initiator patron))
         (structure '(destination establishment)))
  (scene ordering (must-be 'demanding)
         (structure '(speaker  patron))
         (structure '(listener server))
         (structure '(starts-after going-to-restaurant)))
  (scene food-arriving (must-be 'giving)
         (structure '(giver server))
         (structure '(thing-given food))
         (structure '(recipient patron))
         (structure '(demanded-in ordering))
         (structure '(starts-after ordering)))
  (scene eating-scene (must-be 'eating)
         (structure '(eater patron))
         (structure '(thing-eaten food))
         (structure '(starts-after food-arriving)))
  (scene paying-scene (must-be 'paying)
         (inferences '(excluded-slots paying-first-scene))
         (structure '(giver patron))
         (structure '(thing-given payment))
         (structure '(recipient server))
         (structure '(payment-for food-arriving))
         (structure '(starts-after eating-scene)))
  (scene leaving (must-be 'being-moved)
         (structure '(participants patron))
         (structure '(initiator patron))
         (structure '(origin establishment))
         (structure '(starts-after paying-scene)))
  (optional-scene paying-first-scene (must-be 'paying)
                  (inferences '(excluded-slots paying-scene))
                  (structure '(giver patron))
                  (structure '(thing-given payment))
                  (structure '(recipient server))
                  (structure '(payment-for food-arriving))
                  (structure '(starts-after ordering))
                  (structure '(starts-before food-arriving))))
;;...@end example

