;;;; -*- LISP -*-

(in-package :tweak)

(defpackage :tweak
    (:use :constraints :arlotje :lisp))

(defun trace-slot (slot)
  (assert-value slot 'put-demons
		`(format T "~&\; Storing ~S on ~S of ~S~&\;Assertion context:~@{~%~S~}"
		  %value% %slot% %unit% aj::*assertion-context*)))
(defun untrace-slot (slot)
  (aj::retract-value slot 'put-demons
		     `(format T "~&\; Storing ~S on ~S of ~S~&\;Assertion context:~@{~%~S~}"
		       %value% %slot% %unit% aj::*assertion-context*)))

(defvar *TWEAK-TRACE-STREAM* *TRACE-OUTPUT*
  "The stream to which object TWEAK tracing reports go.")
(defmacro TWEAK-trace (format-string &rest format-args)
  "Prints a trace from the TWEAK mechanism if tracing is activated."
  `(progn (when *TWEAK-TRACE-STREAM*
	    (format *TWEAK-TRACE-STREAM* "~&\;\;\; ")
	    (format *TWEAK-TRACE-STREAM* ,format-string ,@format-args)
	    (force-output  *TWEAK-TRACE-STREAM*))
    T))

(defun TRACE-TWEAK (&optional (stream (and (not *TWEAK-TRACE-STREAM*) *standard-output*)))
  "Turns on TWEAK tracing."
  (if *TWEAK-trace-stream*
      (cond ((eq stream *TWEAK-trace-stream*))
	    (stream (format *TWEAK-trace-stream*
			    "~&\; Switching TWEAK trace from ~S to ~S~&"
			    stream *TWEAK-trace-stream*)
		    (format stream
			    "~&\; Switching TWEAK trace from ~S to ~S~&"
			    stream *TWEAK-trace-stream*))
	    (T (format *TWEAK-trace-stream*
		       "~&\; Switching off TWEAK trace (from ~S)~&"
		       *TWEAK-trace-stream*)))
    (format stream "~&\; Tracing TWEAK on stream ~S~&" stream))
  (setq *TWEAK-TRACE-STREAM* stream))



;;;;.Identities

(define-slot distinct-from many-valued-slots annotated-valuep annotated-valuep
  (english-description "These are the values which this slot cannot be.")
  (makes-sense-for 'annotated-valuep)
  (inverse-slot 'distinct-from)
  (mutually-denies 'identical-to)
  (pushes-through 'identical-to)
  (put-demons '(assert-value %unit% 'local-constraints
		`(not-eq (get-value ,(annotated-value-unit %value%) ,(annotated-value-slot %value%)))))
  (put-demons '(assert-value %value%
		'local-constraints
		`(not-eq (get-value (get-value %unit% 'annotated-value-unit)
			  (get-value %unit% 'annotated-value-slot))))))

(defun not-eq (x y) (not (eq x y)))

(define-slot equivalences many-valued-slots slotp listp
  (english-description "These are structural inferences implemented by local constraints.")
  (put-demons '(assert-value %unit% put-demons
		`(assert-value (annotated-value %value% ,(car %value%)) identical-to
		  (annotated-value %unit% ,(cadr %value%))
		  depends-on (support-set (list (annotated-value %value% ,(car %value%))
					   (annotated-value %unit% ,(cadr %value%))))))))

(define-slot distinctions many-valued-slots slotp slotp
  (english-description "These are structural inferences implemented by local constraints.")
  (inverse-slot 'distinctions)
  (put-demons '(assert-value %unit% put-demons
		`(assert-value (annotated-value %unit% %slot%) distinct-from
		  (annotated-value %unit% ,%value%)))))

(define-slot married-to constrained-slots unitp unitp)
(define-structure marriages ()
  (husband (member-of 'constrained-slots)
	   (equivalences '(married-to wife)))
  (wife (member-of 'constrained-slots)
	(equivalences '(married-to husband))))

(define-individual my-parents (marriages)
  (husband 'ken.sr) (wife 'tineke))
(get-value 'ken.sr 'married-to) ; ===> TINEKE
(get-value 'tineke 'married-to) ; ===> KEN.SR
(assertion 'my-parents 'wife 'janet)
(get-value 'ken.sr 'married-to) ; ===> TINEKE
(get-value 'tineke 'married-to) ; ===> #<Couldn't get MARRIED-TO of TINEKE>

(defun match? (o1 o2)
  "Returns NIL of O1 and O2 are not compatabile, a list of slots which must be changed to make them compatible."
  (block matching
    (let ((t1 (first (get-value o1 'minimal-member-of)))
	  (t2 (first (get-value o2 'minimal-member-of))))
      (and (eq t1 t2)
	   (let ((props (cons 'about (get-value t1 'sensible-slots)))
		 (free  '()))
	     (dolist (prop props free)
	       (let ((v1 (get-value o1 prop)) (v2 (get-value o2 prop))
		     (a1 (annotated-value o1 prop)) (a2 (annotated-value o2 prop)))
		 (cond ((eq v1 v2) (push (list prop v1 v2) free))
		       ((and (failurep v1) (failurep v2))
			(if (query a1 'distinct-from a2) (return-from matching NIL)
			  (push (list prop v1 v2) free)))
		       ((failurep v1)
			(unless (acceptable-value o1 prop v2)
			  (return-from matching NIL)))
		       ((failurep v2)
			(unless (acceptable-value o2 prop v1)
			  (return-from matching NIL)))
		       (T (return-from matching NIL))))))))))

(defun identify (o1 o2)
  (block matching
    (let ((t1 (first (get-value o1 'minimal-member-of)))
	  (t2 (first (get-value o2 'minimal-member-of))))
      (and (eq t1 t2)
	   (let ((props (cons 'about (get-value t1 'sensible-slots)))
		 (things-to-identify '()))
	     (dolist (prop props)
	       (let ((v1 (get-value o1 prop)) (v2 (get-value o2 prop))
		     (a1 (annotated-value o1 prop)) (a2 (annotated-value o2 prop)))
		 (cond ((eq v1 v2))
		       ((and (failurep v1) (failurep v2))
			(if (query a1 'distinct-from a2) (return-from matching NIL)
			  (push (list a1 a2) things-to-identify)))
		       ((failurep v1) (assert-value o1 prop v2))
		       ((failurep v2) (assert-value o2 prop v1))
		       (T (return-from matching NIL)))))
	     (progn (dolist (identity things-to-identify)
		      (assert-value (car identity) 'identical-to (cadr identity)))
		    T))))))

(defun distinguish (o1 o2)
  (block matching
    (let ((t1 (first (get-value o1 'minimal-member-of)))
	  (t2 (first (get-value o2 'minimal-member-of))))
      (and (eq t1 t2)
	   (let ((props (cons 'about (get-value t1 'sensible-slots)))
		 (things-to-distinguish '()))
	     (dolist (prop props)
	       (let ((v1 (get-value o1 prop)) (v2 (get-value o2 prop))
		     (a1 (annotated-value o1 prop)) (a2 (annotated-value o2 prop)))
		 (cond ((or (failurep v1) (failurep v2))
			(if (query a1 'dintinct-from a2) (return-from matching T)
			  (push (list a1 a2) things-to-distinguish)))
		       ((not (eq v1 v2)) (return-from matching T)))))
	     (and things-to-distinguish
		  (assert-value (car (car things-to-distinguish)) 'distinct-from
				(cadr (car things-to-distinguish)))))))))


;;;;.Representing states

;;.Plan states are represented as a specialization of ARLOtje states.
;;.They have an `about' slot which is a spec-slot of @code{state-of}
;;.but, importantly, is a `constrained slot' defined by the constraint
;;.mechanism.  This means that its value can be partially specified
;;.without being fully identified.  Plan states are all members of the
;;.collection @code{plan-states}; individual state (like @code{on} or
;;.@code{clear}) are subsets of this this collection.

(define-structure plan-states (states)
  (about (member-of 'constrained-slots) (genl-slots 'state-of)))

;;.Plan states have the additional property of being potentially
;;.@emph{denied}; this is handled by the boolean-valued slot
;;.@code{denied}. 
(define-slot denied single-valued-slots states booleanp
  (english-description "Whether this actually describes the denial of a state."))


;;;;.Representing actions

;;.We begin by introducing a simpler version of the time relation; the
;;.relations @code{before} and @code{after} refer to the atomic
;;.actions executed by TWEAK.  Technically, these slots are not
;;.related to ARLOtje's slots to save resources, but such a connection
;;.would be equivalent to giving @code{before} the @code{genl-slots}
;;.of @code{ends-before} and @code{starts-before}.

(define-slot before transitive-slots temporal-extents temporal-extents
  (english-description "Things begining and ending after this.")
  ;; These connect TWEAK's reprsentation to ARLOtje's ontology.
; (genl-slots 'ends-before)
; (genl-slots 'starts-before)
  )

(define-slot after transitive-slots temporal-extents temporal-extents
  (english-description "Things begining and ending after this.")
  (inverse-slot 'before)
  (mutually-denies 'before))

;;.Every type of action has a set of associated slots corresponding to
;;.each precedent and consequent of the action.  These slots, usually
;;.defined by @code{define-precedent} and @code{define-consequent},
;;.are accessible as the @code{precedents} or @code{consequents} of
;;.corresponding action types.  Each individual action @emph{instance}
;;.has slots @code{pre-conditions} and @code{post-conditions} composed
;;.from these individual slots.  TWEAK's main loop does an initial
;;.takes all the pre-conditions of each step and recursively tries to
;;.satisfy these.

(define-slot precedents many-valued-slots collectionp slotp
  (english-description
   "These are the slots which containing preconditions for this type of action.")
  (inverse-slot (make-unit 'precedent-for (member-of 'single-valued-slots))))
(define-slot pre-conditions many-valued-slots actions states
  (english-description "These are the preconditions for this action.")
  (to-compute-value 'generate-pre-conditions)
  (inverse-slot (make-unit 'pre-condition-of (member-of 'single-valued-slots))))
(defun generate-pre-conditions (unit slot)
  "This generates the preconditions for a unit by getting each of the slots
in the PRECEDENTS stored on its MINIMAL-MEMBER-OF slot."
  (mapcar #'(lambda (sl) (get-value unit sl))
	  (apply #'append
		 (mapcar #'(lambda (class) (get-value class 'precedents))
			 (get-value unit 'minimal-member-of)))))

(define-slot consequents many-valued-slots unitp slotp
  (english-description
   "These are the slots which contain post-conditions for this type of action.")
  (inverse-slot (make-unit 'consequent-for (member-of 'single-valued-slots))))
(define-slot post-conditions many-valued-slots actions states
  (english-description "These are the postconditions of an action.")
  (to-compute-value 'generate-post-conditions)
  (inverse-slot (make-unit 'post-condition-of (member-of 'single-valued-slots))))
(defun generate-post-conditions (unit slot)
  "This generates the post-conditions for a unit by getting each of the slots
in the CONSEQUENTS stored on its MINIMAL-MEMBER-OF slots."
  (mapcar #'(lambda (sl) (get-value unit sl))
	  (apply #'append
		 (mapcar #'(lambda (class) (get-value class 'consequents))
			 (get-value unit 'minimal-member-of)))))

(defmacro defprecedent (name for-action &body slots)
  `(define-unit ,name
    (member-of 'single-valued-slots)
    (makes-sense-for ',for-action)
    (must-be 'states)
    (genl-slots 'pre-conditions)
    (to-compute-value 'aj::fill-in-slot)
    (precedent-for ',for-action)
    ,@slots))

(defmacro defconsequent (name for-action &body slots)
  `(define-unit ,name
    (member-of 'single-valued-slots)
    (makes-sense-for ',for-action)
    (must-be 'states)
    (genl-slots 'post-conditions)
    (to-compute-value 'aj::fill-in-slot)
    (consequent-for ',for-action)
    ,@slots))


;;;;.Constructing plans

(define-structure plans (actions)
  (objects (member-of 'many-valued-slots))
  (setup (member-of 'single-valued-slots) (must-be 'actions)
	 (genl-slots 'after)
	 (to-compute-value 'aj::fill-in-slot))
  (initial-conditions (member-of 'many-valued-slots) (must-be 'plan-states)
		      (put-demons '(get-value %unit% setup))
		      (structure '(post-condition-of setup)))
  (final-conditions (member-of 'many-valued-slots) (must-be 'plan-states)
		    (genl-slots 'pre-conditions)))

(define-slot plan single-valued-slots actions listp
  (english-description "This is the events occuring before a given
action, sorted into a unique ordering which preserves the partial
order determined by BEFORE and AFTER relations.")
  (to-compute-value 'get-sorted-precursors))
(defun get-sorted-precursors (unit slot)
  "Sorts the AFTER slot of UNIT by the BEFORE relation."
  (sort (get-value unit 'after)
	#'(lambda (x y) (query x 'before y))))

(define-unit step-summary
  (member-of 'single-valued-slots))
(define-unit state-summary
  (member-of 'single-valued-slots))

(defun step-summary (step)
  (let ((template (get-value (first (get-value step 'minimal-member-of))
			     'step-summary)))
    (if (failurep template) ""
      (apply #'format NIL (car template)
	     (mapcar #'(lambda (sl) (let ((vv (get-value step sl)))
				      (if (failurep vv) '??? vv)))
		     (cdr template))))))
(defun state-summary (step)
  (let ((template (get-value (first (get-value step 'minimal-member-of))
			     'state-summary)))
    (if (failurep template) ""
      (apply #'format NIL
	     (if (query step 'denied T)
		 (concatenate 'string "~~[" (car template) "]")
	       (car template))
	     (mapcar #'(lambda (sl) (let ((vv (get-value step sl)))
				      (if (failurep vv) '??? vv)))
		     (cdr template))))))

(defun describe-plan (plan)
  (format T "~&\; Description of solution to ~S" plan)
  (dolist (step (cdr (get-value plan 'plan)))
    (format T "~&\; ")
    (dolist (class (get-value step 'minimal-member-of))
      (format T "~S (~A)\; " step (step-summary step)))))

(define-slot possible-precursors single-valued-slots temporal-extents listp
  (english-description "These are the actions which might occur before this one.")
  (to-compute-value 'get-possible-precursors))
(defun get-possible-precursors (action sl)
  "Returns the actions which *might* occur before ACTION.  These are
computed by looking at ACTION and everything after it, collecting the
things before each of these and filtering out those known to fail the test."
  (declare (ignore sl))
  (let ((later (cons action (get-value action 'before))))
    (set-difference
     (apply #'lunion (mapcar #'(lambda (x) (get-value x 'after)) later))
     (cons action (get-value action 'before)))))


;;;;.Truth criterion

(defun asserts? (state action)
  "Returns true if ACTION asserts STATE."
  (remove-if-not #'(lambda (result)
		     (and (match? state result) (not (query result 'denied T))))
		 (get-value action 'post-conditions)))

(defun denies? (state action)
  "Returns true if ACTION denies STATE."
  (remove-if-not #'(lambda (result)
		     (and (match? state result) (query result 'denied T)))
		 (get-value action 'post-conditions)))

(defun find-establishers (state action)
  "Looks at the precedents of ACTION for establishers of STATE."
  (remove NIL
	  (mapcar #'(lambda (action) (let ((matches (asserts? state action)))
				       (and matches (cons action matches))))
		  (get-value action 'possible-precursors))))

(defun find-clobberers (state action)
  "Looks for precedents of ACTION whidh might deny STATE."
  (remove NIL
	  (mapcar #'(lambda (action) (let ((matches (denies? state action)))
				       (and matches (cons action matches))))
		  (get-value action 'possible-precursors))))

(defun find-white-knights (state for clobberer)
  "Looks for actions after clobberer yet before state which might reestablish STATE."
  (remove NIL
	  (mapcar #'(lambda (action)
		      (let ((whew (and (query action 'after clobberer) (denies? state action))))
			(and when (cons action whew))))
		  (get-value for 'possible-precursors))))

(defun tweaking (state action)
  (tweak-trace "Trying to ensure that ~S (~A) holds for ~S (~A)"
	       state (state-summary state) action (step-summary action))
  (some #'(lambda (establisher)
	    (let ((establisher (car establisher)) (asserts (cdr establisher)))
	      (some #'(lambda (consequent)
			(and (identify state consequent)
			     (tweak-trace "~S (~A) might establish ~S (~A) by establishing ~S (~A)"
					  establisher (step-summary establisher)
					  state (state-summary state)
					  consequent (state-summary consequent))
			     (suggest-value establisher 'before action)
			     (tweak-trace " if it occurs before ~S (~A)" action (step-summary action))
			     (every #'(lambda (clobberer)
					(let ((clobberer (car clobberer)) (clobbers  (cdr clobberer)))
					  (every #'(lambda (clobbered) 
						     (tweak-trace "but ~S (~A) might clobber ~S (~A) by ~S (~A)"
								  clobberer (step-summary clobberer)
								  state (state-summary state)
								  clobbered (state-summary clobbered))
						     (or (and (distinguish state clobbered)
							      (tweak-trace " but if I `~A'  it won't." (step-summary clobberer)))
							 (and (suggest-value clobberer 'after action)
							      (tweak-trace "  but I can do ~S (~A) after ~S (~A)"
									   clobberer (step-summary clobberer)
									   action (step-summary action)))
							 (some #'(lambda (white-knight)
								   (let ((white-knight (car white-knight)) (states (cdr white-knight)))
								     (some #'(lambda (saver)
									       (and (not (eq white-knight clobberer))
										    (suggest-value clobberer    'before white-knight)
										    (suggest-value white-knight 'before action)
										    (identify state saver)
										    (tweak-trace "but I can do ~S (~A) afterwards"
												 white-knight (step-summary white-knight))))
									   states)))
							       (append (find-white-knights state action clobberer)
								       (list (make-establisher state action))))))
						 clobbers)))
				    (find-clobberers state action))))
		    asserts)))
	(append (find-establishers state action)
		(list (make-establisher state action)))))
(defun tweak (plan)
  (do ((steps (cons plan (get-value plan 'after)) (cons plan (get-value plan 'after)))
       (past-steps (list plan) steps)
       (all-clear T))
      ((and all-clear (equal steps past-steps)))
    (dolist (step steps)
      (dolist (condition (get-value step 'pre-conditions))
	(setq all-clear (and all-clear (tweaking condition step)))))))

;;;;.Simple states and actions.

(define-structure clear (plan-states))
(define-structure on (plan-states)
  (atop (member-of 'constrained-slots) (member-of 'things)))
(assertion 'clear 'state-summary '("~S is clear" about))
(assertion 'on 'state-summary '("~S is on ~S" about atop))

(define-structure puton (actions)
  (upper (member-of 'constrained-slots)
	 (distinctions 'lower) (distinctions 'moved-from))
  (lower (member-of 'constrained-slots)
	 (distinctions 'upper) (distinctions 'moved-from))
  (moved-from (member-of 'constrained-slots)
	      (distinctions 'upper) (distinctions 'lower)))
(assertion 'puton 'step-summary
	   '("Moving ~S to ~S from ~S" upper lower moved-from))

(defprecedent base-supported puton
  (must-be 'on)
  (equivalences '(about upper))
  (equivalences '(atop moved-from)))
(assertion 'moved-from 'composition-of
	   '(atop base-supported))
(defprecedent lower-clear puton
  (must-be 'clear)
  (equivalences '(about lower)))
(defprecedent upper-clear puton
  (must-be 'clear)
  (equivalences '(about upper)))

(defconsequent stacked puton
  (must-be 'on)
  (equivalences '(about upper))
  (equivalences '(atop lower)))
(defconsequent unstacked puton
  (must-be 'on)
  (equivalences '(about upper))
  (equivalences '(atop moved-from))
  (put-demons '(assert-value %value% denied T)))
(defconsequent covered puton
  (must-be 'clear)
  (equivalences '(about lower))
  (put-demons '(assert-value %value% denied T)))
(defconsequent uncovered puton
  (must-be 'clear)
  (equivalences '(about moved-from)))



(define-structure newtower (actions)
  (new-base (member-of 'constrained-slots) (distinctions 'clearing))
  (clearing (member-of 'constrained-slots) (distinctions 'new-base)))
(assertion 'newtower 'step-summary
	   '("Moving ~S from ~S to the table" new-base clearing))
				 
(defprecedent new-base-supported newtower
  (must-be 'on)
  (equivalences '(about new-base))
  (equivalences '(atop clearing)))
(assertion 'new-base 'composition-of
	   '(about new-base-supported))
(assertion 'clearing 'composition-of
	   '(atop new-base-supported))

(defprecedent base-clear newtower
  (must-be 'clear)
  (equivalences '(about new-base)))

(defconsequent on-table newtower
  (must-be 'on)
  (equivalences '(about new-base))
  (value-inferences '(atop TABLE)))

(defconsequent base-denial newtower
  (must-be 'on)
  (equivalences '(about new-base))
  (equivalences '(atop clearing))
  (value-inferences '(denied T)))

(defconsequent cleared-support newtower
  (must-be 'clear)
  (equivalences '(about clearing)))



;;;;.Kludged MAKE-ESTABLISHER

(defun make-establisher (state action)
  (if (satisfies? state 'on)
      (if (eq (get-value state 'atop) 'table)
	  (let ((action (new-unit newtower
			  (new-base (get-value state 'about))) ))
	    (list action (get-value action 'on-table)))
	(let ((action
	       (new-unit puton
		 (upper (get-value state 'about))
		 (lower (get-value state 'atop)))))
	  (list action (get-value action 'stacked))))
    (let ((action
	   (new-unit newtower
	     (clearing (get-value state 'about)))))
      (list action (get-value action 'cleared-support)))))


;;;;.A test.

(define-units (table a b c d e)
  (member-of 'things))

(define-unit put-a-on-b
  (member-of 'plans)
  (final-conditions   (new-unit on (about 'a) (atop 'b)))
  (initial-conditions (new-unit on (about 'a) (atop 'table)))
  (initial-conditions (new-unit on (about 'b) (atop 'table)))
  (initial-conditions (new-unit clear (about 'a)))
  (initial-conditions (new-unit clear (about 'b))))

(define-unit put-c-on-d
  (member-of 'plans)
  (final-conditions   (new-unit on (about 'c) (atop 'd)))
  (initial-conditions (new-unit on (about 'c) (atop 'table)))
  (initial-conditions (new-unit on (about 'd) (atop 'table)))
  (initial-conditions (new-unit clear (about 'c)))
  (initial-conditions (new-unit clear (about 'd))))

(define-unit sussman-anomaly
  (member-of 'plans)
  (final-conditions   (new-unit on    (about 'a) (atop 'b)))
  (final-conditions   (new-unit on    (about 'b) (atop 'c)))
  (final-conditions   (new-unit on    (about 'c) (atop 'table)))
  (final-conditions   (new-unit clear (about 'a)))
  (initial-conditions (new-unit on    (about 'a) (atop 'table)))
  (initial-conditions (new-unit on    (about 'b) (atop 'table)))
  (initial-conditions (new-unit on    (about 'c) (atop 'a)))
  (initial-conditions (new-unit clear (about 'c)))
  (initial-conditions (new-unit clear (about 'b))))


