#+CL2 (defpackage :loader (:use common-lisp ccl))

(in-package :loader)

(export '(load-file load-source-file load-binary-file update-file recompile-file
	  kludge-pathname))

(defvar *binary-extension*
  (or #+GENERA                  "bin"           ; Lisp machines
      #+(AND LUCID MIPS)        "mbin"          ; Decstations (lucid)
      #+(AND LUCID PA)          "hbin"          ; HP's Precision Architecture (lucid)
      #+(AND LUCID SPARC)       "s4bin"         ; Sun 4s
      #+(AND LUCID SUN MC68000) "s3bin"         ; Sun 3s
      #+NEXT                    "nbin"          ; NeXTs (Allegro)
      #+KCL                     "o"             ; KCL alwyas uses this
      #+:CORAL                  "fasl"          ; MACL uses this
      "random-bin")
  "The extension for binary files.")
(defvar *binary-directory*
  (or #+GENERA                  "genera"        ; Lisp machines
      #+(AND LUCID MIPS (NOT LCL4.0)) "mbin"          ; Decstations (lucid 3.0)
      #+(AND LUCID MIPS LCL4.0) "4.0mbin"       ; Decstations (lucid 4.0)
      #+(AND LUCID PA)          "hbin"          ; HP's Precision Architecture (lucid)
      #+(AND LUCID SPARC)       "sparcbin"      ; Sun 4s
      #+(AND LUCID SUN MC68000) "sun3bin"       ; Sun 3s
      #+NEXT                    "nextbin"       ; NeXTs (Allegro)
      #+(AND KCL VAX)           "kclvax"
      #+(AND KCL MIPS)          "kclmips"
      #+(AND KCL IBMRT)            "kclrt"              
      #+(AND :CORAL :CCL-1.3)   "Macl1.3.2"
      #+(AND CORAL CCL-2)       (format nil "Macl2-~A" (lisp-implementation-version))
      "random-bin")
  "The directory in which binary files live.")

(defparameter *source-extension* "lisp"
  "The extension for source files.")
(defparameter *directory-separator*
  #+UNIX #\/ #+:CORAL #\: #+GENERA #\/
  "The separator for directory paths.")

;; Some implementations use this.
(if (boundp '*load-binary-pathname-types*)
    (pushnew *binary-extension* *load-binary-pathname-types*
	     :test 'string-equal))

#+KCL
(defun shell (&rest args) (apply 'system args))
#+NEXT
(import 'excl::shell)


;;;; Hacking pathnames.

(defun kludge-pathname (root &rest strings)
  (if (null (rest strings))
      (concatenate 'string root (car strings))
    (apply #'kludge-pathname
	   (concatenate 'string root (car strings) (list *directory-separator*))
	   (rest strings))))

(defun make-source-pathname (pathstring)
  (let ((pathstring (if (stringp pathstring) pathstring
		      (namestring pathstring))))
    (concatenate 'string pathstring "." *source-extension*)))

(defun maybe-create-directory (path)
  #+MACL
  (let ((d (make-pathname :directory (pathname-directory path))))
    (unless (probe-file d)
      (format t "~&;Creating Directory ~A~&" d)
      (ccl::create-directory d)))
  #+UNIX
  (let ((pathstring (namestring path)))
    (do ((i 1 (position #\/ pathstring :start (1+ i))))
	((null i))
      (unless (probe-file (subseq pathstring 0 i))
	(format t "~&;Creating Directory ~A~&" path)
	(shell (format nil "mkdir ~A\;chmod a+rwx ~A"
		       (subseq pathstring 0 i) (subseq pathstring 0 i)))))))

(defun make-binary-pathname (pathstring)
  (let ((pathstring (if (stringp pathstring) pathstring
		      (namestring pathstring))))
    (let* ((split (position *directory-separator* pathstring :from-end T))
	   (directory (subseq pathstring 0 (1+ split)))
	   (file (subseq pathstring (1+ split)))
           (path (kludge-pathname directory *binary-directory*
				  (concatenate 'string file "." *binary-extension*))))
      (maybe-create-directory path)
      path)))




;;;; Really gross kludges

;;; This is neccessary because LISP refuses to delete a file
;;; overriding local file protections.
(defun really-delete-file (file)
  #-UNIX(delete-file file)
  #+UNIX(shell (concatenate 'string "rm -f " (namestring file))))

;;; This is neccessary because AUS doesn't really write FASL files
;;; very well.
(defun really-compile-file (file &key
				 (output-file
				  (merge-pathnames file (make-pathname :type *binary-extension*))))
  #-MACL(compile-file file :output-file output-file)
  #+MACL(let ((temp-name (concatenate 'string "CCL;" 
                                      (pathname-name output-file)
                                      "." (pathname-type output-file))))
          (compile-file file :output-file temp-name)
          (copy-file temp-name output-file)
          (delete-file temp-name)))


;;;; Operations on files

(defun load-file (file)
  (let ((binary-path (make-binary-pathname file))
	(source-path (make-source-pathname file)))
    (if (and (probe-file binary-path)
	     (> (file-write-date binary-path) (file-write-date source-path)))
	(load binary-path)
      (if (or (not (probe-file binary-path))
	      (yes-or-no-p "\;\; ~S: Source file is newer\; load it?" binary-path))
	  (load source-path)
	(load binary-path)))))

(defun load-source-file (file)
  (let ((source-path (make-source-pathname file)))
    (load source-path)))

(defun load-binary-file (file)
  (let ((binary-path (make-binary-pathname file)))
    (load binary-path)))

(defun update-file (file)
  (let ((binary-path (make-binary-pathname file))
	(source-path (make-source-pathname file)))
    (unless (and (probe-file binary-path)
                 (> (file-write-date binary-path) (file-write-date source-path)))
      (format T "~&\; Updating (recompiling) ~A ~%\;  because ~A is newer."
	      binary-path source-path)
      ;; This gets around a protection bug for people without "umask 0"
      (if (probe-file binary-path) (really-delete-file binary-path))
      (really-compile-file source-path :output-file binary-path))
    (load binary-path)))

(defun recompile-file (file)
  (let ((binary-path (make-binary-pathname file))
	(source-path (make-source-pathname file)))
    (format T "~&\; Recompiling ~A into ~A" source-path binary-path)
    (really-compile-file source-path :output-file binary-path)
    (load binary-path)))