;;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@chapter Utilities to support ARLOtje
(in-package :arlotje)

(defvar *arlotje-package* (find-package 'arlotje))

;;.ARLOtje is built on a platform of two simple support packages and a
;;.handful of utility functions.  The two support packages are for:
;;.@itemize @bullet
;;.@item
;;.Storing series of user or program actions in restorable
;;.@dfn{sessions} maintaining cross-references to particular units.
;;.@item
;;.Doing type checking which is `optmistic'; if a predicate
;;.fails, it records and announces the failure, but does nothing
;;.further.
;;.@end itemize
;;.The utility functions have to do with generating symbol names from
;;.components in various ways.


;;;;. Sessions

;;.The problem of saving or restoring units is a tricky one; an obvious
;;.approach would be to store the properties and values which define a
;;.unit in a file corresponding to the unit name.  Then, to reload the
;;.unit one simply loads and assigns the corresponding properties.  A
;;.problem with this is that any mature representation language will
;;.infer and define a number of slot values itself and these are
;;.dependent on values defined by users or other programs.  In ARLOtje,
;;.much of the most important information is the relations between slot
;;.values which allow the retraction of a precedent to lead to the
;;.retraction of its consequents.

;;.One way to address this problem is to figure out what the
;;.`precedents' are for a given unit and to store those.  This can be
;;.done by tracing dependencies, but many properties of a unit @var{x}
;;.are consequences of properties of @var{y} and not properties of
;;.@var{x}.  The problem is that many of the properties of units
;;.consist of its relations to other units and that it is these
;;.interrelations which have an internal coherence rather than the set
;;.of properties asserted by any particular object.  For this reason,
;;.ARLOtje organizes transactions into @dfn{sessions} which are stored;
;;.units then have their properties identified by cross references to
;;.the sessions with which they are involved.

;;.All interactions with ARLOtje occur within the scope of a particular
;;.@dfn{session}.  A new session is begun each time you load ARLOtje
;;.or, depending on implementation, whenever one logs in anew or starts
;;.a stored ARLOtje image.  The session is a data structure (actually,
;;.an ARLOtje unit!) describing several properties:
;;.@enumerate
;;.@item
;;.The user responsible for the session.  Under Unix, this is gotten by
;;.reading the symbol in the .arlotje.user file in the user's home
;;.directory.
;;.@item
;;.The host machine on which the session occured.  This assumes
;;.that the session does not migrate across machines.
;;.@item
;;.The site where the host machine is located.
;;.@item
;;.The time at which the session began; this is stored at two
;;.levels of granularity: seconds since the start of the 20th century
;;.(Common LISP's `universal' time) and the internal run time of the
;;.machine (with some arbitrary offset).
;;.@item
;;.The stated purpose of the session, provided by a user.  For
;;.example, "Entering class data."
;;.@item
;;.For sessions which have been @emph{saved}, the time at which
;;.the session ended (as just a universal time).
;;.@end enumerate
;;.Time is stored at two levels of granularity so as to allow the
;;.expression of `moments' which pair an internal run time and a
;;.session identification.  This allows the parsimonious specification
;;.of very fine intervals.

;;.The variable @code{*session*} stores the current session; usually,
;;.this is accessed by a set of user functions.@findex{*session* (variable)}
(defvar *session* nil
  "This is a unit describing the current session.")

;;..The variable @code{*sessions*} stores all the sessions known of by
;;..the system. @findex{*sessions* (variable)}
(defvar *sessions* nil
  "This is a list of all sessions")

;;.Sessions are stored in a common file directory determined by the
;;.variable @code{*session-directory*} @findex{*session-directory* (variable)}.
(defvar *session-directory* nil
  "This is the directory where sessions are stored.")

;;.Sessions are cross indexed by the units they refer to.  Another
;;.common directory stores the sessions in which individual units
;;.appear.  For instance, the file @file{CLYDE} in this common
;;.directory consists of a list of sessions in which @file{CLYDE} was
;;.substantively mentioned.
(defvar *unit-directory* nil 
  "This is the directory where unit/session cross-references are stored.")

(defvar *months*
  '("JAN" "FEB" "MAR" "APR" "MAY" "JUNE"
    "JULY" "AUG" "SEPT" "OCT"
    "NOV" "DEC")
  "An ordered list of month names used in constructing session identifiers.")

;;..The function @code{make-new-session} constructs a new session
;;..description describing the current context and with a purpose given
;;..by the single argument to @code{make-new-session}.  By default,
;;..this is the uninformative "Nothing Special".
;;..@findex{make-new-session}
(defun make-new-session (&optional (purpose "Nothing special"))
  "Binds *session* to a new session description, recording site, host,
and precse time.  It also checks all the assumptions from the previous
session, clearing the assumption list."
  (check-assumptions)
  (multiple-value-bind (second minute hour date month year) (get-decoded-time)
    (let ((host (machine-instance)) (site (long-site-name)))
      (let ((name (if (symbolp purpose) (intern (symbol-name purpose) :arlotje)
		      (intern (format NIL  "~A-~D.~A.~A-~D~A~D" (string-upcase host)
				      hour (if (> minute 9) minute (format NIL "0~D" minute))
				      (if (> second 9) second (format NIL "0~D" second))
				      date (elt *months* (- month 1)) year)
			      arlotje::*arlotje-package*))))
	(push name *sessions*)
	(setf (get name 'session-start)
	      (encode-universal-time second minute hour date month year))
	(setf (get name 'session-host) host)
	(setf (get name 'session-site) site)
	(setf (get name 'session-purpose) purpose)
	(setf (get name 'session-user) (arlotje-user))
	name))))
;;.Sessions may be identified by the predicate @code{sectionp}.
;;.@findex{sectionp}
(defun sessionp (x) (get x 'session-start))

;;.A new session is initiated by the function @code{new-session}.
;;.@findex{new-session}
;;.When called, @code{new-session} saves the current session and
;;.constructs a new session with the purpose provided by its single
;;.argument.  This session is then made current by setting the value of
;;.@code{*session*} to it.
(defun new-session (&optional (purpose "Nothing Special"))
  "Creates a new session and saves the current one."
  (when (and *session* (not (get *session* 'loaded-session)))
    (save-session *session*))
  (setq *session* (make-new-session purpose)))

;;.Alternatively, the procedure @code{push-session} creates a new
;;.session without saving the current one, and provides for a future
;;.call to @code{pop-session} to restore it.  The new session is made
;;.current and the embedding session keeps a pointer to its subsession
;;.so that --- when it is restored --- it restores its subsession.
(defun push-session (&optional (purpose "Nothing Special"))
  "Makes a new session without terminating the current one.
The current one is resumed by POP-SESSION"
  (let ((new (make-new-session purpose)))
    (setf (get new 'pushed-from) *session*)
    (setq *session* new)))

;;.When a pushed session is popped, the popped session is saved; it is
;;.after this save that the embedding session records the subsession
;;.for recursive restoration.
(defun pop-session ()
  "Pops a session created by PUSH-SESSION."
  (cond ((get *session* 'pushed-from)
	 (save-session)
	 (let ((old-session *session*))
	   (setq *session* (get *session* 'pushed-from))
	   (session-event! `(restore-session ,old-session))))
	(T (error "Not currently in a pushed session!"))))

;;.The macro @code{within-session} executes its body within a pushed
;;.session.  The first argument to the macro is the purpose of the
;;.pushed session.
;;.@findex{within-session (macro)}
(defmacro within-session (purpose &body body)
  (let ((session-var (make-symbol "SESSION")))
    `(let ((,session-var (make-new-session ,purpose)))
      (let ((*session* ,session-var))
	(unwind-protect
	     (progn ,@body)
	  (save-session ,session-var))))))

;;.Some sessions are `loaded' sessions whose restoration loads a LISP
;;.file or system rather than a history of some set of operations.  The
;;.booting or ARLOtje and the loading of the HUH parser, for instance,
;;.are both implemented by loaded sessions.  The function
;;.@code{loaded-session} declares a session of
;;.this sort; its first argument is the session's purpose and its
;;.second is a LISP form which will load the session (for instance by
;;.loading a particular file or @code{REQUIR}ing a particular
;;.module).@refill
;;.@findex{loaded-session}
(defun loaded-session (purpose form)
  (new-session purpose)
  (setf (get *session* 'loaded-session) form)
  (push form (get *session* 'events)))

;;..The procedure @code{arlotje-user} returns a
;;..symbol in the ARLOtje package which describes the current user.  It
;;..is read (with @code{*package*} bound to the ARLOtje package) from
;;..the file @file{.arlotje.user} in the user's home directory under
;;..UNIX or from the current directory on the Macintosh or from the file 
;;..@file{arlotje-user} in the home directory on the Symbolics.
;;..@findex{arlotje-user}
(defun arlotje-user ()
  "Returns the current user's description."
  (let ((user-file #+UNIX "~/.arlotje.user"
		   #+GENERA (make-pathname :defaults (fs:user-homedir) :name "arlotje-user" :type "lisp")
		   #+APPLE  "Arlotje User"))
    (and (probe-file user-file)
	 (with-open-file (user-object user-file :direction :input)
	   (let ((*package* *arlotje-package*))
	     (or (read user-object nil nil) 'UNKNOWN.USER))))))

;;..The procedure @code{declare-session} declares a session, this is
;;..used in restoring session descriptions.
;;..@findex{declare-session}
(defun declare-session (name &rest props)
  (do ((props props (cddr props))
       (values (cdr props) (cddr values)))
      ((null props)
       (push name *sessions*)
       (setf (get name 'referents)
	     (mapcar #'(lambda (x)
			 (cond ((symbolp x) x)
			       ((listp x) (intern (cadr x) (or (find-package (car x))
							       (make-package (car x)))))))
		     (get name 'referents)))
       name)
    (setf (get name (first props)) (first values))))


;;;;.Fast methods for dumb compilers and other kludges

(defmacro jamnew (thing place)
  "Pushes something into a place if an EQ object isn't already there."
  `(unless (member ,thing ,place :test #'eq)
    (setf ,place (cons ,thing ,place))))

#+GENERA (import 'scl:stack-let)
#-GENERA
(defmacro stack-let (bindings &body body)
  (flet ((var-from-binding (b) (if (consp b) (car b) b)))
    `(let ,bindings
      #+CL2(declare (dynamic-extent ,@(mapcar #'var-from-binding bindings)))
      ,@body)))


;;;;. Recording events

;;.The procedure @code{SESSION-EVENT!} records
;;.an event in the current session.  The initial @var{event} parameter
;;.is a list.  When the current session is restored, the first element
;;.of the list (a symbol denoting a function) is applied to the
;;.remaining elements of the list.  The second parameter,
;;.@var{referents} is a list of symbols (units) to which the event
;;.refers.  These determine the cross-reference information stored when
;;.the session is saved.
;;.@findex{session-event!}
(defun session-event! (form &optional (referents '()))
  "Records FORM as an event in the current session with cross references to each of REFERENTS."
  (unless  (get *session* 'loaded-session)
    (dolist (referent referents)
      (when (symbolp referent)
	(jamnew referent (get *session* 'referents))))
    (push form (get *session* 'events))))


;;;;. Saving Sessions

;;.The procedure @code{save-session} saves the current session or (with
;;.an optional first argument) some other session.  The session is
;;.saved into a file in a common directory (by specifying another
;;.pathname as the second argument, though, you can save the session
;;.someplace else --- for instance in some private directory).  The
;;.format of this file consists of a list describing the session
;;.followed by the series of events recorded during the session by
;;.@code{SESSION-EVENT!}; the events are recorded in chronological
;;.order.  If a session is saved multiple times, the file is
;;.overwritten.

;;.Whenever the session is written to the common directory, a series of
;;.@emph{cross-references} are stored in another common directory for
;;.every unit declared as a @emph{referent} by @code{session-event!}.
;;.A unit can then be `reloaded' by loading the sessions in which it
;;.was defined or modified.

(defun save-session (&optional (session *session*) file)
  "Saves the session SESSION, defaulting to the current session.
Unless the second argument FILE is specified, the SESSION is stored in
the common session directory."
  (unless (null (get session 'events))
    (format T "~&\; Saving session ~S" session)
    (with-open-file (stream (or file (merge-pathnames (symbol-name session) *session-directory*))
			    :direction :output)
      (dump-object-to-stream
       `(,session
	 session-user      ,(get session 'session-user)
	 session-start     ,(get session 'session-start)
	 session-end       ,(get-universal-time)
	 session-host      ,(get session 'session-host)
	 session-site      ,(get session 'session-site)
	 session-purpose   ,(get session 'session-purpose)
	 referents ,(mapcar #'(lambda (x)
				(if (eq (symbol-package x) *arlotje-package*) x
				    (list (package-name (symbol-package x))
					  (symbol-name x))))
		     (remove-if #'(lambda (x) (null (symbol-package x)))
		      (get session 'referents))))
       stream)
      (dolist (event (reverse (get session 'events)))
	(terpri stream)
	(dump-object-to-stream event stream)))
    (unless file
      (format T "~&\; Writing cross references to session ~S" session)
      (format T "~&\; ~S" (get session 'referents))
      (dolist (unit (get session 'referents))
	(unless (or (member unit (get session 'recorded-referents))
		    (not (get unit 'creation-id)))
	  (push unit (get session 'recorded-referents))
	  (with-open-file (output (unit-filename unit )
				  :direction :output
				  :if-exists :append :if-does-not-exist :create)
	    (prin1 session output)
	    (terpri output)))))))

;;.The function @code{unit-filename} returns the filename where the
;;.cross-references for its argument @var{unit} are stored.
;;.@findex{unit-filename}
(defun unit-filename (unit &optional (directory *unit-directory*))
  "Returns the filename in which cross-references for the unit UNIT are stored."
  (if (eq (symbol-package unit) *arlotje-package*)
      (merge-pathnames (symbol-name unit) directory)
      (merge-pathnames (format NIL "~A%%~A" (package-name (symbol-package unit))
			       (symbol-name unit))
		       directory)))


;;;;. Restoring sessions

;;.Sessions are restored either directly (by an explicit
;;.@code{restore-session}) or indirectly by loading a unit which loads
;;.the sessions referring to it.

;;..The function @code{read-dump-data} reads a
;;..series of expressions from a stream and passes them to
;;..@code{FUNCALL} to be evaluated.  I.E. when an expression
;;..@code{(@var{foo} . @var{args})} is read, the function @var{foo} is
;;..applied to @var{args} (unevaluated).
;;..@findex{read-dump-data}
(defun read-dump-data (from-stream)
  "Reads and evaluates a series of dumped expressions from FROM-STREAM."
  (let ((*package* *arlotje-package*))
    (do ((input (read from-stream nil nil) (read from-stream nil nil)))
	((null input) )
      (apply (car input) (cdr input))))) 

;;.The function @code{restore-session} restores
;;.a named session.  Its argument @var{session} should be a symbol in
;;.the ARLOtje package naming a session.  It prints out a note of the form:
;;.@example
;;.;Restoring session NOISE.MEDIA.MIT.EDU-11.17.42-16June1990
;;.@end example
;;.The names of sessions consist of the name of the host on which the
;;.session occured (usually in Internet `dotty' format), the time in
;;.hours, minutes, and seconds (separated by periods), and the date in
;;.a form like `3DEC1990'.
(defun restore-session (session)
  "Loads in the occurences during the session recorded in FROM-FILE."
  (let ((session (intern (symbol-name session) 'arlotje)))
    ;; All session descriptions are in the ARLOtje package.
    (unless (get session 'events)
      (let* ((session-name (if (stringp session) session (symbol-name session)))
	     (session-file (merge-pathnames session-name *session-directory*)))
	(with-open-file (stream session-file :direction :input)
	  (let ((*package* *arlotje-package*))
	    (let ((session (apply #'declare-session (read stream))))
	      (format T "~&\; Restoring session ~A: ~A" session (get session 'session-purpose))
	      (let ((*session* session))
		(read-dump-data stream)
		session))))))))
;;.@findex{restore-session}


;;;;. Restoring units

;;..The variable @code{*unit-load-hook*} (when non-nil) is a function
;;..run on all newly loaded units.  It might be used, for instance, to
;;..do a @code{load-all-references} in the background or to notify
;;..some central arbiter that some session has accessed @code{unit}.
;;..@findex{*unit-load-hook* (variable)}

(defvar *unit-load-hook* nil
  "This is run on newly loaded units\; it might, for instance, load background data.")

;;.The function @code{try-to-load-unit} loads the session in which its
;;.argument @var{unit} was defined and records the other sessions in
;;.which @var{unit} is referred to.  These sessions are stored in a
;;.file whose name is the unit's name in a common directory; if no such
;;.file exists, the function returns @code{NIL} unless the optional
;;.keyword argument @var{declare-otherwise?} is true, in which case the
;;.unit is declared as a unit by @code{declare-unit}
;;.@pxref{Primitive functions}.  ARLOtje keeps track of whether or not a unit
;;.is loaded by whether or not it has been declared; if a unit was
;;.somehow declared without being loaded, the keyword
;;.@code{load-if-declared?} must be passed to @code{try-to-load-unit}.

(defun try-to-load-unit (unit &key (declare-otherwise? nil) (load-if-declared? nil))
  "Loads the reference data for a unit and restores the session in which it was defined."
  (unless (and (get unit 'creation-id) (not load-if-declared?))
    (if (probe-file (unit-filename unit))
	(with-open-file (input (unit-filename unit) :direction :input)
	  (let ((*package* *arlotje-package*))
	    (let ((defining-session (read input)))
	      (restore-session defining-session)
	      (do ((reference defining-session (read input nil nil))
		   (references '() (if (member reference references)
				       references (cons reference references))))
		  ((null reference)
		   (setf (get unit 'references)
			 (cons defining-session (reverse references))))))
	    (when *unit-load-hook* (funcall *unit-load-hook* unit))))
	(when declare-otherwise? (declare-unit unit))))
  unit)
;;.Note that if a session is stored in a private file (not in the
;;.common directory), the cross references to the session will not be
;;.stored in the common unit directory.  To store the cross references
;;.it is neccessary to restore the session and then explicitly save it
;;.(specifying the first argument to
;;.@code{save-session}@xref{Saving Sessions}) into the common directory.

;;.The function @code{load-all-unit-references} loads all of the
;;.sessions which refer to its argument @var{unit}.
;;.@findex{load-all-unit-references}
(defun load-all-unit-references (unit)
  "Loads all the sessions referring to UNIT."
  (dolist (session (get (try-to-load-unit unit) 'references))
    (restore-session session)))


;;;;.Dumping.

;;..The variable @code{*lisp-package*} @findex{*lisp-package* (variable)} stores
;;..the @code{LISP} package for quick references without the
;;..intermediary of @code{find-package}.
(defvar *lisp-package* (find-package 'lisp)
  "This contains the LISP package (for quick comparisons).")

;;..The function @code{dump-object-to-stream} dumps an expression to a
;;..stream in such a way that symbol names are qualified with
;;..double-colon package identifiers, which it doesn't seem possible to
;;..get Common LISP to do by itself.  This helps you win if you can't
;;..be sure of loading into a context with the same use and
;;..import/export relations as that from which you are dumping.
;;..@findex{dump-object-to-stream}
(defun dump-object-to-stream (object stream)
  (cond ((or (stringp object) (numberp object) (characterp object))
	 (format stream "~S" object))
	((null object) (format stream "()"))
	((symbolp object)
	 (if (or (eq (symbol-package object) *arlotje-package*)
		 (eq (symbol-package object) *lisp-package*))
	     (let ((*package* *arlotje-package*)) (prin1 object stream))
	   (let ((*package* (symbol-package object)))
	     (format stream "~A::~S"
		     (package-name (symbol-package object))
		     object))))
	((and (consp object) (eq (car object) 'quote))
	 (write-char #\' stream)
	 (dump-object-to-stream (cadr object) stream))
	((consp object)
	 (write-char #\( stream)
	 (dump-object-to-stream (car object) stream)
	 (do ((lst (cdr object) (cdr lst)))
	     ((not (consp lst))
	      (if (null lst)
		  (write-char #\) stream)
		  (progn (princ " . " stream)
			 (dump-object-to-stream lst stream)
			 (princ ")" stream))))
	   (write-char #\Space stream)
	   (dump-object-to-stream (car lst) stream)))
	(T (prin1 object stream)))
  object)


;;;;.Maintaining assumptions

;;.In order to bootstrap effectively, ARLOtje makes numerous
;;.assumptions; for instance, assuming that @code{to-get-value} is a
;;.slot while giving @code{to-get-value} and @code{to-get-value} slot.
;;.These assumptions are managed as part of the section mechanism.
;;.The macro @code{assuming} takes a lisp expression, a format string,
;;.and a list of arguments and pushes all of these onto the
;;.@code{assumptions} slot of the current session.  When
;;.@code{check-assumptions} (a procedure) is called, it checks all of
;;.the recorded assumptions of the current session (or another session,
;;.if provided an argument).  If any assumption is satisfied, it is
;;.removed from the assumption list; if any assumption is unsatisfied,
;;.a warning is issued based on the format string and arguments passed
;;.to @code{assuming}.  The continue options on ARLOtje type errors all
;;.record assumptions in the current session; in addition, the
;;.assumptions which shift sessions call @code{check-assumptions}.
;;.@findex{assuming (macro)}
;;.@findex{check-assumptions}
(defmacro assuming (predicate.args format-string &rest format-args)
  "Calls PREDICATE on ARGS, recording if they fail (and thus are assumed true).
When CHECK-ASSUMPTIONS is called, all those calls to ASSUMING which have
not been consumated are listed on the final output."
  (let ((predicate (first predicate.args)) (args (rest predicate.args)))
    `(push (list ',predicate (list ,@args) (list ,format-string ,@format-args))
	   (get *session* 'assumptions))))

(defmacro assuming? (predicate.args)
  "Calls PREDICATE on ARGS, recording if they fail (and thus are assumed true).
When CHECK-ASSUMPTIONS is called, all those calls to ASSUMING which have
not been consumated are listed on the final output."
  (let ((predicate (first predicate.args)) (args (rest predicate.args)))
    `(find-if #'(lambda (x) (and (eq ',predicate (car x)) (equal (cadr x) (list ,@args))))
      (get *session* 'assumptions))))

(defun check-assumptions (&optional (session *session*))
  "Checks all of the assumptions made by ASSUMING.
If any are still unconsummated, these are described.  If CLEAR is non-nil,
the assumption list is cleared."
  (setf (get session 'assumptions)
	(remove-if #'(lambda (assumption)
		       (cond ((apply (first assumption) (second assumption)) T)
			     (T (format T "~&\;\;\; Still assuming that ")
				(apply #'format T (third assumption))
				nil)))
		   (get session 'assumptions))))


;;;;. Making symbols

;;.Since symbols are used as unit names, ARLOtje and ARLOtje programs
;;.often need to construct unit names (symbols) from composite
;;.elements.  Several functions are provided for these operations.

;;.The function @code{fsymbol}constructs a symbol from a format string
;;.and a list of arguments.  It has two forms.  In one the first
;;.argument is a package or package name and the remaining arguments
;;.are a format string and arguments.  The string and arguments are
;;.used to yield a string which is interned in the package denoted by
;;.the first argument.  In the second form, the first argument is the
;;.format string and the remaining arguments are used to yield a string
;;.which is interned in the current package.
;;.@findex{fsymbol}
(defun fsymbol (string-or-pkg &rest format-args)
  "Returns a symbol constructed from FORMAT-STRING and FORMAT-ARGS."
  (let ((*print-case* :upcase))
    (cond ((null string-or-pkg)
	   (make-symbol (apply #'format nil format-args)))
	  ((or (packagep string-or-pkg) (symbolp string-or-pkg))
	   (intern (apply #'format nil format-args) string-or-pkg))
	  (T (intern (apply #'format nil string-or-pkg format-args) *package*)))))

;;.The function @code{gensymbol} is like @code{fsymbol} except that it
;;.generates numbered symbols such as @code{FOO.1}, @code{FOO.2}, etc.
;;.Its arguments are identical to those of @code{fsymbol}.  One
;;.alternate form of @code{gensymbol} takes a single argument and
;;.simply constructs an iterated symbol from that.
;;..The counter for generated versions of the symbol are stored on the
;;..@code{GENSYM-COUNTER} property of the generated symbol.
(defun gensymbol (format-string &rest format-args)
  "Returns a symbol constructed from FORMAT-STRING and FORMAT-ARGS."
  (let ((root (if (null format-args)
		  (if (symbolp format-string)
		      (or (get format-string 'gensym-root) format-string)
		    (apply #'fsymbol format-string format-args))))
	(*print-case* :upcase))
    (unless (get root 'gensym-count) (setf (get root 'gensym-count) 0))
    (let ((new-symbol
	   (intern (format NIL "~A.~D" root (incf (get root 'gensym-count))))))
      (setf (get new-symbol 'gensym-root) root)
      new-symbol)))

;;.The procedure @code{symbolize} takes any number of symbols (or
;;.numbers or strings) as arguments and returns a symbol consisting of
;;.the symbols concatenated together with periods (`.') between them.
(defun symbolize (&rest symbols)
  "Returns a symbol which concatenates each of SYMBOLS, separated by periods."
  (cond ((null symbols) (error "Must have some symbols to append!"))
	((null (rest symbols)) (first symbols))
	(T (intern (format NIL "~A~{.~A~}" (first symbols) (rest symbols))))))


;;;;.Other session operations.

(defun undo-session (session)
  "Attempts to undo all of the events in SESSION."
  (dolist (event (get session 'events))
    (if (get (car event) 'undo-function)
	(apply (get (car event) 'undo-function) (cdr event))
      (format T "~&\;!! Warning: Couldn't undo ~S" event))))
