;;; -*- LISP -*-

;;.@chapter Tests: Describing LISP predicates

;;.ARLOtje uses LISP predicates for typing the domains and ranges of
;;.slots; in addition, predicates may be useful in extensions of
;;.ARLOTtje for the preconditions of rules or actions.  To support
;;.this, ARLOtje defines a @emph{lattice} of @emph{tests} which
;;.indicate the relation which tests hold with one and other.  For
;;.instance, ARLOtje knows that any object passing the LISP predicate
;;.@code{integerp} will also pass the predicate @code{numberp}.  It
;;.describes this by asserting that @code{numberp} is a
;;.@emph{generalization} of @code{integerp} or --- conversely --- that
;;.@code{integerp} is a @emph{specialization} of @code{numberp}.

;;.ARLOtje represents the ability to @code{enforce} some tests.
;;.Consider the test that determines whether a slot has a particular
;;.value; it could be `enforced' by storing that value on the slot.
;;.Similarly, a test which conjoins two other tests can be enforced by
;;.trying to enforce the two conjuncts.  A possible extension to
;;.enforcement would be a set of coercion operations; but currently
;;.that is not implemented.  (Thus, one cannot enforce
;;.@code{integerp}.)

;;.In addition to have knowledge built-in LISP predicates, ARLOtje
;;.allows the definition of new predicates which automatically infer
;;.things about their relation to other predicates.  For instance, the
;;.system knows trivially that positive integers (the conjunction of
;;.the predicates @code{plusp} and @code{integerp}) are both
;;.specializations of @code{numberp}.

(in-package :arlotje)


;;;;.Distinguishing predicates as tests

;;.Tests are distinguished by the property @code{test} which inherits
;;.through the @code{works-like} slot.  The unit
;;.@code{prototypical-test} has a @code{test} slot of @code{t} which
;;.is inherited down to other descriptions which @code{works-like}
;;.@code{prototypical-test}.  Predicates are declared as tests by
;;.either giving them @code{works-like} slots or by explicitly just
;;.giving them non-nil @code{test} slots.

(define-unit test
  (english-description "A flag describing whether a unit is a test or not.")
  (works-like 'prototypical-primitive-slot)
  (to-get-value 'check-test))
;;.@vindex{test (slot)}
(defun check-test (unit slot)
  "Inherits the TEST property through the WORKS-LIKE hierarchy."
  (declare (ignore slot))
  (get-slot-method unit 'test))
;;.@findex{check-test}

(define-unit prototypical-test
  (english-description "A prototype for tests.")
  (test t))
;;.@vindex{prototypical-test (test prototype)}

(defun testp (x) "Returns true for tests." (check-test x 'test))
;;.@findex{testp}

(define-unit testp
  (works-like 'prototypical-test))

;;.Not all tests need be LISP predicates.  In @xref{Collections}, a
;;.family of tests is introduced which are not tests but rather finite
;;.sets of objects in which memberhip is tested.  Technically, I
;;.suppose, tests are anything which can be passed as a second
;;.argument to the @code{satisfies?} function.


;;;;.Representing the lattice

;;.ARLOtje represents the lattice of predicates with the slots
;;.@code{Generalizations} and @code{Specializations} which are
;;.transitive many-valued slots @pxref{Transitive Slots} that are also
;;.inverses of each other.  The domain and range of these are
;;.@code{testp} which simply checks for the property @code{test} on
;;.the symbol named.

(define-unit Generalizations
  (english-description "The generalizations of a test.")
  (works-like 'prototypical-set-slot)
  (kleene-plus-of 'generalizations)
  (makes-sense-for 'testp)
  (must-be 'testp))
;;.@vindex{Generalizations (slot)}

(define-unit Specializations
  (english-description "The specializations of a test.")
  (works-like 'prototypical-set-slot)
  (kleene-plus-of 'specializations)
  (makes-sense-for 'testp)
  (must-be 'testp)
  (inverse-slot 'generalizations))
;;.@vindex{Specializations (slot)}

;;.The slots @code{Subsumes} and @code{Subsumed-By} are the
;;.Kleene-stars of @code{Specializations} and @code{Generalizations};
;;.when one test subsumes another, it means that anything passing the
;;.latter test will pass the former.
(define-unit Subsumes
  (english-description "The tests subsumed by this one.")
  (works-like 'prototypical-set-slot)
  (kleene-star-of 'specializations)
  (makes-sense-for 'testp)
  (must-be 'testp))

(define-unit Subsumed-by
  (english-description "The tests which subsume this one.")
  (works-like 'prototypical-set-slot)
  (kleene-star-of 'generalizations)
  (makes-sense-for 'testp)
  (must-be 'testp))

;;.The function @code{over-generalizations} walks the lattice above
;;.a particular test.  It's first argument is a function (@code{fcn})
;;.which is applied @emph{once} to every test which is a
;;.generalization of its second argument (@code{test}).  The order of
;;.application obeys a topological sort: no test is processed until
;;.all of the tests below it are processed.
(defun over-generalizations (fcn from)
  "Applies FCN to FROM and each of its generalizations in a topological sort."
  (let ((queue (list from)) (marks '()))
    (loop (if (null queue) (return (values)))
      (let ((top (car queue)))
	(if (member top marks)
	    (setq queue (cdr queue))
	  (progn (funcall fcn top)
		 (setq marks (cons top marks))
		 (setq queue (append (get-value top 'generalizations)
				     (cdr queue)))))))))
;;.@findex{over-generalizations}


;;;;.Enforcing tests

;;;.Tests have a @code{to-enforce-test} slot which inherits through
;;;.the @code{works-like} slot.  Enforcement functions take two
;;;.arguments, an object and a test, and return non-nil if their
;;;.enforcement succeeded.  The default enforcer is
;;;.@code{cant-enforce-test} which simply returns NIL.

(define-unit to-enforce-test
  (english-description "The function for enforcing this test.")
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'inherit-through-works-like)
  (makes-sense-for 'testp)
  (must-be 'function-namep))
;;.@vindex{to-enforce-test (slot)}

(defun cant-enforce-test (thing test)
  "The default predicate enforcement function signifying nothing."
  (declare (ignore thing test))
  NIL)


;;;;.Lisp Tests

;;.The top of the lattice is @code{anythingp}.  It is specialized by
;;.@code{lisp-objectp} which is functionally identical to it.  The
;;.distinction between the two is motivated by some dubious
;;.ontological arguments.

(define-unit anythingp
  (works-like 'prototypical-test))

(deffcn lisp-objectp (x)
  (declare (ignore x))
  (annotations (works-like prototypical-test) (generalizations anythingp))
  'T)
;;.@findex{lisp-objectp (test)}

;;.Underneath @code{lisp-objectp} lies a variety of lisp tests
;;.corresponding to basic LISP data types: @code{symbolp},
;;.@code{listp}, @code{numberp}, and @code{stringp}.
(define-unit symbolp
  (works-like 'prototypical-test)
  (generalizations 'lisp-objectp))
;;.@vindex{symbolp (test)}
(define-unit listp
  (works-like 'prototypical-test)
  (generalizations 'lisp-objectp))
;;.@vindex{listp (test)}
(define-unit numberp
  (works-like 'prototypical-test)
  (generalizations 'lisp-objectp))
;;.@vindex{numberp (test)}
(define-unit stringp
  (works-like 'prototypical-test)
  (generalizations 'lisp-objectp))
;;.@vindex{stringp (test)}


;;.Several specializations of numbers are arranged in a tower:
;;.@code{complexp} and @code{realp} underneath @code{numberp}, and
;;.@code{integerp} beneath @code{rationalp} beneath @code{numberp}.
(define-unit complexp
  (works-like 'prototypical-test)
  (generalizations 'numberp))
;;.@vindex{complexp (test)}
#-CL2
(deffcn realp (x) ; Take out in up-to-date common lisp
  "Returns true for numbers which are not complex."
  (annotations (works-like prototypical-test)
	       (generalizations numberp))
  (not (complexp x)))
#+CL2
(progn (assertion 'realp 'works-like 'prototytpical-test)
       (assertion 'realp 'generalizations 'numberp))
;;.@vindex{realp (test)}
(define-unit rationalp
  (works-like 'prototypical-test)
  (generalizations 'realp))
;;.@vindex{rationalp (test)}
(define-unit integerp
  (works-like 'prototypical-test)
  (generalizations 'rationalp))
;;.@vindex{integerp (test)}

;;.The predicate @code{symbolp} is specialized into @code{function-namep}
;;.(which is identical to @code{symbolp}), @code{keywordp}, and
;;.@code{booleanp}.  The specialization @code{unitp} begins the tree
;;.of ARLOtje units and their specializations.
(define-internal-unit function-namep
  (works-like 'prototypical-test)
  (generalizations 'symbolp))
;;.@vindex{function-namep (test)}

(define-unit booleanp
  (works-like 'prototypical-test)
  (generalizations 'symbolp))
;;.@vindex{booleanp (test)}
(define-unit keywordp
  (works-like 'prototypical-test)
  (generalizations 'symbolp))
;;.@vindex{keywordp (test)}


;;;;.Types of units

(define-unit unitp
  (works-like 'prototypical-test)
  (to-enforce-test 'enforce-unithood)
  (generalizations 'lisp-objectp))
(defun enforce-unithood (thing unitp)
  (declare (ignore unitp))
  (declare-unit thing))
;;.@vindex{unitp (test)}

(define-unit whonitp
  (works-like 'prototypical-test)
  (generalizations 'unitp))
 
;;.Units (@code{unitp}) are specialized into @code{slotp}.
(defun slotp (x)
  "Returns true for all slots."
  (and (symbolp x) 
       (or (%get-default x 'to-get-value NIL)
	   (and (%get x 'works-like)
		(not (failurep (get-value x 'to-get-value)))))))
(define-unit slotp
  (works-like 'prototypical-test)
  (generalizations 'unitp)
  (to-enforce-test 'enforce-slothood))
;;.@findex{slotp}
;;.@vindex{slotp (test)}
(defun enforce-slothood (unit slotp)
  (declare (ignore slotp))
  (put-value (declare-unit unit) 'works-like 'prototypical-slot))

(define-unit sessionp
  (works-like 'prototypical-test)
  (generalizations 'unitp))

(define-unit assertionp
  (works-like 'prototypical-test)
  (generalizations 'whonitp))
;;.@vindex{assertionp (test)}

(define-unit annotated-valuep
  (works-like 'prototypical-test)
  (generalizations 'assertionp))
;;.@vindex{annotated-valuep (test)}

(define-unit elementp
  (works-like 'prototypical-test)
  (generalizations 'assertionp))
;;.@vindex{elementp (test)}

(define-unit put-demonp
  (works-like 'prototypical-test)
  (generalizations 'listp))
(define-unit lambda-definitionp
  (works-like 'prototypical-test)
  (generalizations 'listp))
(define-unit list-of-assertionsp
  (works-like 'prototypical-test)
  (generalizations 'listp))
(define-unit pair-of-slotsp
  (works-like 'prototypical-test)
  (generalizations 'listp))


;;;;.Combining tests

;;.ARLOtje provides a vocabulary of automatic coders for defining new
;;.tests in terms of existing tests.  These coders provide for
;;.automatic slot testing, logical combinations of existing tests, and
;;.so forth.

;;.These become especially useful as conditions for rules or other
;;.decision procedures as well as for defining new types to serve as
;;.domains and ranges of slots.

;;;;.Logical combinations

;;.The @code{test-intersection} coder yields a function/test which
;;.returns true if each of its @code{intersection-of} slots returns
;;.true.  
(deffcn test-intersection (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-enforce-test enforce-intersection)))
  `(lambda (x) (and ,@(mapcar #'(lambda (test) `(satisfies? x ',test))
		       (get-value type 'intersection-of)))))
;;.@findex{test-intersection}
;;.@vindex{test-intersection (coder)}
(define-unit intersection-of
  (works-like 'prototypical-set-slot)
  (genl-slots 'generalizations)
  (inferences '(works-like prototypical-test)))
;;.@vindex{intersection-of (coder slot)}
;;.For instance, one might define @code{bachelorp} thus:
;;.@example
;;.(deffcn bachelorp test-intersection
;;.  (intersection-of 'men)
;;.  (intersection-of 'unmarried))
;;.@end example
;;.to yield the classic analytic definition of bachelor.

(deffcn enforce-intersection (thing type)
  "Enforces an intersection."
  (dolist (test (get-value type 'intersection-of))
    (or (funcall test thing)
	(funcall (get-value test 'to-enforce-test) thing test))))

;;.The @code{test-union} coder yields a test which returns true if any
;;.of its component tests (on the @code{union-of} slot) return true.
(deffcn test-union (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-enforce-test enforce-union)))
  `(lambda (x) `(or ,@(mapcar #'(lambda (test) `(satisfies? x ',test))
		       (get-value ',type 'union-of)))))
;;.@findex{test-union}
;;.@vindex{test-union (coder)}
(define-unit union-of
  (works-like 'prototypical-set-slot)
  (genl-slots 'specializations)
  (inferences '(works-like prototypical-test)))
;;.@vindex{union-of (coder slot)}
;;.For instance, the logical combination `rich or smart' might be
;;.defined thus:
;;.@example
;;.(deffcn rich-or-smartp test-union
;;.  (union-of 'wealthyp)
;;.  (union-of 'intelligentp))
;;.@end example

(deffcn enforce-union (thing test)
  (dolist (test (get-value test 'union-of))
    (unless (failurep (get-value test 'to-enforce-test))
      (funcall (get-value test 'to-enforce-test) thing test)
      (if (funcall test thing) (return test)))))

;;.The @code{test-complement} coder yields a test which returns true
;;.if another test (specified in its @code{complement-of} slot)
;;.returns false.
(deffcn test-complement (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test)))
  `(lambda (x) (not (satisfies? x ',(get-value type 'complement-of)))))
;;.@findex{test-complement}
;;.@vindex{test-complement (coder)}
(define-unit complement-of
  (works-like 'prototypical-slot)
  (inferences '(works-like prototypical-test)))
;;.@vindex{complement-of (coder slot)}
;;.so that we might define `not dead yet' as:
;;.@example
;;.(deffcn not-dead-yet test-complement
;;.  (complement-of 'deadp))
;;.@end example


;;;;.Combining tests

;;.The @code{has-slot} coder yields a test which returns true if a
;;.unit has any value for a particular slot (specified by its
;;.@code{tests-slot} slot).  So we might define a predicate
;;.@code{parentp} thus:
;;.@example
;;.(deffcn parentp has-slot
;;.  (tests-slot 'children))
;;.@end example
(deffcn has-slot (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-enforce-test enforce-has-slot)))
  `(lambda (x)
     (and (satisfies? x ',(get-value (get-value type 'tests-slot) 'makes-sense-for))
	  (%get x ',(get-value type 'tests-slot)))))
;;.@findex{has-slot}
;;.@vindex{has-slot (coder)}
(define-unit tests-slot
  (works-like 'prototypical-slot)
  (must-be 'slotp)
  (put-demons '(assert-value %unit% generalizations
			     (get-value %value% makes-sense-for))))
;;.@vindex{tests-slot (coder slot)}

(deffcn enforce-has-slot (unit test)
  "Assures that UNIT has a particular slot, first by defaulting then by asking."
  (let ((slot (get-value test 'tests-slot)))
    (and (if (many-valued-slotp slot)
	     (null (get-value unit slot))
	   (failurep (get-value unit slot)))
	 (yes-or-no-p "~&\; Can you give me a value for the ~S slot of ~S? "
		      unit slot)
	 (assert-value unit slot (eu-read "~&\; What is the ~S slot of ~S?" unit slot)))))

;;.The @code{hasnt-slot} coder yields a test which returns false if a
;;.unit might legally have a value for a particaular slot (specified by its
;;.@code{tests-slot} slot) yet does not.  So we might define a predicate
;;.@code{bachelorp} thus:
;;.@example
;;.(deffcn bachelorp hasnt-slot
;;.  (tests-slot 'wife))
;;.@end example
(deffcn hasnt-slot (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-enforce-test enforce-hasnt-slot)))
  `(lambda (x) (%get-default x ',(get-value type 'tests-slot) nil)))
;;.@findex{hasnt-slot}
;;.@vindex{hasnt-slot (coder)}

(deffcn enforce-hasnt-slot (unit test)
  "Assures that UNIT has a particular slot, first by defaulting then by asking."
  (let ((slot (get-value test 'tests-slot)))
    (if (many-valued-slotp slot)
	(dolist (value (get-value unit slot))
	  (retract-value unit slot value))
      (retract-value unit slot))
    T))

;;.The @code{test-value} coder produces tests which apply another test
;;.to a particular slot.  Like @code{has-slot}, @code{test-value}
;;.checks the slot specified in the definitions @code{tests-slot}
;;.slot but only returns true if the value stored there satisfies the
;;.function specified as the @code{applies-test} slot of the description.
(deffcn test-value (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-put-value enforce-test-value)))
  (let ((slot (get-value type 'tests-slot))
	(test (get-value type 'applies-test)))
    (if (many-valued-slotp slot)
	`(lambda (x)
	   (and (satisfies? x ',(get-value (get-value type 'tests-slot) 'makes-sense-for))
		(let ((match (as-a-side-effect
			      (find-if #'(lambda (x) (get-value x ',test))
				       (get-value x ',slot)))))
		  (if match (check-value x ',slot match)
		    ;; We do a GET-VALUE to install a dependency.
		    (progn (get-value x',slot) nil)))))
      `(lambda (x) (and (satisfies? x ',(get-value (get-value type 'tests-slot) 'makes-sense-for))
			(satisfies? (get-value x ',slot) ',test))))))
;;.@findex{test-value}
;;.@vindex{test-value (coder)}
(define-unit applies-test
  (works-like 'prototypical-slot)
  (makes-sense-for 'testp)
  (must-be 'function-namep))
;;.@vindex{applies-test (coder slot)}

;;.Tests generated by @code{test-value} have an @emph{enforcement
;;.behavior} which asserts the corresponding value.  Thus, it works to
;;.assert the satisfaction of a @code{tests-value} test.  This is
;;.implemented by @code{enforce-test-value}, which is assigned as the
;;.@code{to-put-value} slot of tests generated by @code{test-value}.
;;.The function @code{enforce-test-value} recursively tries to enforce
;;.@code{enforce-test-value} on the current value of the tested slot.
(defun enforce-test-value (unit test)
  "Tries to enforce a test on a slot of an object."
  (let* ((slot (get-value test 'tests-slot))
	 (requirement (get-value test 'applies-test))
	 (enforcer    (get-value requirement 'to-enforce-test))
	 (victim      (get-value unit slot)))
    (unless (or (failurep enforcer)
		(and (many-valued-slotp slot) (null victim))
		(failurep victim))
      (if (many-valued-slotp slot)
	  (some #'(lambda (x) (funcall enforcer x requirement)) victim)
	(funcall enforcer victim requirement)))))
;;.@findex{enforce-test-value}

;;.The @code{has-value} coder yields a test which returns true if a
;;.slot has a particular value.  Like @code{has-slot} and
;;.@code{test-slot}, functions coded by @code{has-value} test the slot
;;.specified as its @code{tests-slot} slot.  They return true if the
;;.slot has the value in the @code{tests-for-value} slot of the
;;.definition.
(deffcn has-value (type)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test))
	       ((coded-definitions associated-inferences)
		(to-enforce-test enforce-has-value)))
  `(lambda (x)
     (and (satisfies? x ',(get-value (get-value type 'tests-slot) 'makes-sense-for))
	  (check-value x ',(get-value type 'tests-slot)
		       ',(get-value type 'tests-for-value)))))
;;.@findex{has-value}
;;.@vindex{has-value (coder)}

(define-unit tests-for-value
  (works-like 'prototypical-slot)
  (makes-sense-for 'testp))
;;.@vindex{tests-for-value (coder slot)}

;;.Tests generated by @code{has-value} also have an @emph{enforcement
;;.behavior} which asserts the corresponding value.  Thus, it may work
;;.to @emph{assert} the satisfaction of a @code{has-value} test.  This is
;;.implemented by @code{enforce-has-value}, which is assigned as the
;;.@code{to-put-value} slot of tests generated by @code{test-value}.
;;.The function @code{enforce-has-value} stores the coded test's
;;.@code{test-for-value} slot on the @code{tests-slot} slot of the
;;.unit it is being asserted on.
(defun enforce-has-value (unit test)
  (assert-value unit (get-value test 'tests-slot)
		(get-value test 'tests-for-value)))
;;.@findex{enforce-has-value}


;;;;.LIST-OF Combinator

;;.This combinator returns a test satisfied by lists of elements each
;;.of which satisfies the test's @code{element-test} slot.
(deffcn list-of (desc)
  (annotations ((coded-definitions associated-inferences)
		(works-like prototypical-test)))
  (let ((element-test (get-value desc 'element-test)))
    `(lambda (l) (and (listp l) (every #'(lambda (x) (satisfies? x ',element-test)) l)))))
;;.@findex{list-of (combinator)}

;;.The @code{element-test} slot contains the slot which a
;;.@code{LIST-OF} test uses to test individual elements.
(define-unit element-test
  (works-like 'prototypical-slot)
  (makes-sense-for 'testp))
;;.@vindex{element-test (slot)}


;;;;.Inferences from the lattice

;;.Some inferences can be made from the lattice.  For instance, if one
;;.slot is the inverse of another, we can assume that the domain and
;;.range (@code{makes-sense-for} and @code{must-be} slots) of the
;;.slots will be identical and thus pick out the most specialized of
;;.them to fit on each one.  This is handled by the put-demon function
;;.@code{propogate-inverse-domain-and-range}.
(defun propogate-inverse-domain-and-range (unit value)
  "Infers that the domain of the inverse is the range and the range of the inverse the domain."
  (let ((domain (get-value unit 'makes-sense-for))
	(range  (get-value unit 'must-be)))
    (unless (or (failurep domain)
		(check-value (get-value value 'must-be) 'generalizations domain))
      (assert-value value 'must-be domain))
    (unless (or (failurep range)
		(check-value (get-value value 'makes-sense-for) 'generalizations range))
      (assert-value value 'makes-sense-for range))))
;;.@findex{propogate-inverse-domain-and-range}
(put-value 'inverse-slot 'put-demons
	   '(propogate-inverse-domain-and-range %unit% %value%))
;; Yes, this is a kludge.
(let* ((demons (%get 'inverse-slot 'put-demons))
       (elts (%get-default demons 'recorded-elements '())))
  (push (car elts) (cdr (last elts)))
  (setf (%get demons 'recorded-elements) (cdr elts)))


;;;;.Test tests

;;.Several tests are predicated over tests themselves, identifying
;;.particular coders.  In particular, the tests/functions
;;.@code{test-intersectionp}, @code{test-unionp},
;;.@code{test-complementp}, @code{has-slot-testp},
;;.@code{tests-value-testp}, and @code{has-value-testp}
;;.compute the predicates corresponding to the standard predicate
;;.coders.

(deffcn test-intersectionp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'test-intersection)
  (generalizations 'testp))
;;.@findex{test-intersectionp}
;;.@vindex{test-intersectionp (test)}
(deffcn test-unionp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'test-union)
  (generalizations 'testp))
;;.@findex{test-unionp}
;;.@vindex{test-unionp (test)}
(deffcn test-complementp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'test-complement)
  (generalizations 'testp))
;;.@findex{test-complementp}
;;.@vindex{test-complementp (test)}
(deffcn has-slot-testp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'has-slot)
  (generalizations 'testp))
;;.@findex{has-slot-testp}
;;.@vindex{has-slot-testp (testp)}
(deffcn test-value-testp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'test-value)
  (generalizations 'testp))
;;.@findex{test-value-testp}
;;.@vindex{test-value-testp (test)}
(deffcn has-value-testp has-value
  (tests-slot 'lisp-coder)
  (tests-for-value 'has-slot)
  (generalizations 'testp))
;;.@findex{has-value-testp}
;;.@vindex{has-value-testp (test)}


;;;;.Enumerating tests

;;.The function @code{over-tests} iterates a function over all of the
;;.tests which an object satisfies, starting from some root (by
;;.default @code{anythingp}).

(defun over-tests (function object &optional (under (list 'anythingp)))
  "Applies FUNCTION to each test which OBJECT satisfies under UNDER."
  (let ((queue under)
	(marks '()))
    (loop (if (null queue) (return object))
      (if (or (member (car queue) marks)
	      (not (satisfies? object (car queue))))
	  (setq queue (cdr queue))
	(let ((top (car queue)))
	  (funcall function top)
	  (setq marks (cons top marks))
	  (setq queue (append (get-value (annotated-value top 'specializations)
					 'elements-as-set)
			      (cdr queue))))))))
;;.@findex{over-tests}


;;;;.Exclusion

;;.ARLOtje also provides mechanisms for describing the exclusion or
;;.mutual exclusion of collections.  The slot @code{excludes} points
;;.to the other collections explicity excluded by a given slot.  This
;;.is a many valued slot with a special @code{to-check-value} slot
;;.that tries to infer whether two collections exclude each other.
(define-unit excludes
  (english-description
   "These are the collections excluded by membership in this collection.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'testp)
  (must-be 'testp)
  (pushes-through 'specializations))
;;.@vindex{excludes (slot)}

;;.The @code{mutually-excludes} slot is a specialization of
;;.@code{excludes} which is also its own inverse.
(define-unit mutually-excludes
  (english-description
   "These are the collections mutually excluded by membership in this collection.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'testp)
  (must-be 'testp)
  (genl-slots 'excludes)
  (inverse-slot 'mutually-excludes))
;;.@vindex{mutually-excludes (slot)}

(defun excludes? (c1 c2)
  "Returns true if membership in C1 excludes membership in C2."
  (check-value c1 'excludes c2))
;;.@findex{excludes?}


;;;;.Redefining Satisfaction

;;.When tests are introduced, the @code{satisfies?} function is
;;.redefined to use @code{get-value} to resolve tests (it still will
;;.handle predicates which are not slots, but this is discouraged).
;;.In addition, for marginal purposes, a macro @code{recklessly} is
;;.defined which causes @code{satisfies?} within its body to always
;;.return true.  Its use is strongly discouraged but sometimes
;;.simplifies bootstrapping.  
(defvar *reckless?* nil
  "Whether type checking is for real.")
;;.@findex{*reckless?* (variable)}

(defmacro recklessly (&body body)
  "Executes is body with real type checking turned off --- everything satisfies everthing!"
  `(let ((*reckless?* T)) ,@body))
;;.@findex{recklessly (macro)}

(defun satisfies? (x test)
  "Returns true if X satisfies the test (slot) TEST."
  (or *reckless?* (not (fboundp test))
      (not (testp test))
      (funcall test x)))
;;.@findex{satisfies? (reckless version)}

;;.@code{Recklessly} is used when the tests relating to defining tests
;;.are employed.  For instance, tests like @code{function-namep} need to
;;.have cached @code{to-get-value} slots because otherwise, efforts to
;;.default their @code{to-get-value} slot would have to ensure that
;;.the inherited function @code{funcall-test} was indeed a function,
;;.leading to an infinite recursion.
(recklessly
 (dolist (test '(anythingp unitp annotated-valuep slotp testp numberp
		 intergerp stringp function-namep))
   (get-value test 'to-get-value)))


;;;;.Assuming membership

;;.The slot @code{assumed-to-be} indicates a test or collection in
;;.which all of its values must lie; whenever a value is stored there,
;;.it is put in the appropriate collection or given the appropriate
;;.values.

(define-unit should-be
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'testp)
  (put-demons `(assert-value %unit% 'put-demons
		`(,(get-value %value% 'to-enforce-test) %value% ,%value%))))

