;;; -*- LISP -*-

;;.@chapter The Base Ontology

(in-package :arlotje)

;;.ARLOtje's basic ontology is based on the notion of defining
;;.`contexts' which are kinds of spatial and temporal extents.  In
;;.this framework, a story like `Ken went to the restaurant.' is a
;;.context but at the same time `Ken' and `restaurant' are individual
;;.spatio-temporal contexts.


;;;;.Ontological Roots

;;.The definition of contexts begins with the collection
;;.@code{Extents} which is specialized into @code{Spatial-Extents} and
;;.@code{Temporal-Extents}.  The intersection of these is specialized
;;.into @code{things} and @code{processes}; @code{things} is further
;;.specialized into @code{processes} and further into @code{states}
;;.and @code{state-changes}.  @code{Agents} are embodiments which are
;;.both @code{things} and @code{processes} and can perform
;;.@code{actions} which are special kinds of state changes.

(define-unit Extents
  (member-of 'collections)
  (generalizations 'unitp))
;;.@vindex{Extents (collection)}

(define-unit Spatial-Extents
  (member-of 'collections)
  (supersets 'Extents))
;;.@vindex{Spatial-Extents (collection)}

(define-unit Temporal-Extents
  (member-of 'collections)
  (supersets 'Extents))
;;.@vindex{Temporal-Extents (collection)}

(define-unit Things
  (member-of 'collections)
  (supersets 'Spatial-Extents)
  (supersets 'Temporal-Extents))
;;.@vindex{Things (collection)}

(define-unit Processes
  (member-of 'collections)
  (supersets 'Spatial-Extents)
  (supersets 'Temporal-Extents))
;;.@vindex{Processes (collection)}

(define-unit States
  (member-of 'collections)
  (supersets 'things)
  (supersets 'processes))
;;.@vindex{States (collection)}

(define-unit agents
  (member-of 'collections)
  (supersets 'processes)
  (supersets 'things))
;;.@vindex{agents (collection)}

(define-unit State-Changes
  (member-of 'collections)
  (supersets 'Processes))
;;.@vindex{State-Changes (collection)}

(define-unit Actions
  (member-of 'collections)
  (supersets 'Processes))
;;.@vindex{Actions (collection)}


;;;;.Properties of extents

;;.This ontology is provisional but provides some important structure.
;;.The primary relation between extents is containment and overlap;
;;.one extent can contain some others or overlap any number of others.
;;.Extents are @code{Contained-in} another extent whose inverse is
;;.the slot @code{Contains}; each of these are transitive relations,
;;.so we can say
;;.@example
;;.(query 'kens-hand-on-4-September-1990 'contained-in 'ken.haase)
;;.@end example
;;.and get a reasonable response given an unsuprising hierarchy of
;;.containment relations.  Note that this refers only to `physical'
;;.containment and that relations like `ownership' are generally
;;.another sort of process altogether.

(define-unit Contained-In
  (member-of 'transitive-slots)
  (makes-sense-for 'Extents)
  (must-be 'Extents))
;;.@vindex{Contained-In (slot)}

(define-unit Contains
  (member-of 'transitive-slots)
  (makes-sense-for 'Extents)
  (must-be 'Extents)
  (inverse-slot 'Contained-in)
  (mutually-denies 'contained-in))
;;.@vindex{Contains (slot)}

;;.Extents overlap other extents.  This is represented by the
;;.@code{overlaps} slot which is its own inverse but is @strong{not}
;;.transitive.  @code{Overlaps} is the `genl slot' of several other
;;.slots like `enables' or `participants' or `initiator'.

(define-unit Overlaps
  (member-of 'many-valued-slots)
  (makes-sense-for 'Extents)
  (must-be 'Extents)
  (inverse-slot 'overlaps))
;;.@vindex{Overlaps (slot)}

;;.Disjointness of extents is represented by the @code{Disjoint-from}
;;.slot which is its own inverse; currently, this is not used
;;.extensively, so more shadings of meaning are likely to be
;;.neccessary in fullfilling its intended role.

(define-unit Disjoint-from
  (member-of 'many-valued-slots)
  (makes-sense-for 'Extents)
  (must-be 'Extents)
  (inverse-slot 'Disjoint-from)
  (mutually-denies 'contained-in)
  (mutually-denies 'contains))
;;.@vindex{Disjoint-from (slot)}


;;;;.Spatial extents.

;;.Spatial extents are organized into six-pronged `space frames'; a
;;.given spatial extent may be @code{above}, @code{below},
;;.@code{right-of}, @code{left-of}, @code{in-front-of}, or
;;.@code{behind} another extent.

(define-unit above
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents))
;;.@vindex{above (slot)}

(define-unit right-of
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents))
;;.@vindex{right-of (slot)}

(define-unit below
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (inverse-slot 'above)
  (mutually-denies 'above))
;;.@vindex{below (slot)}

(define-unit left-of
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (inverse-slot 'right-of)
  (mutually-denies 'right-of))
;;.@vindex{left-of (slot)}

(define-unit in-front-of
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents))
;;.@vindex{in-front-of (slot)}

(define-unit behind
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (inverse-slot 'in-front-of)
  (mutually-denies 'in-front-of))
;;.@vindex{behind (slot)}

;;.In addition, extents may be @code{surrounded-by} by another extent
;;.which has @code{genl-slots} of @code{contained-in}.
(define-unit surrounded-by
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (genl-slots 'contained-in))
;;.@vindex{surrounded-by (slot)}

(define-unit surrounds
  (member-of 'transitive-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (genl-slots 'contained-in)
  (inverse-slot 'surrounded-by)
  (mutually-denies 'surrounds))
;;.@vindex{surrounds (slot)}

;;.In addition, the relation @code{touches} applies to spatial
;;.extents.

(define-unit touches
  (member-of 'many-valued-slots)
  (makes-sense-for 'spatial-extents)
  (must-be 'spatial-extents)
  (inverse-slot 'touches))
;;.@vindex{touches (slot)}


;;;;.Temporal extents

;;.Temporal extents are currently represented by a
;;.@code{starts-before} and @code{ends-before} relations with
;;.respective inverses @code{starts-after} and @code{ends-after}.

(define-unit Starts-before
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents))
;;.@vindex{Starts-before (slot)}
(define-unit Starts-after
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents)
  (inverse-slot 'starts-before)
  (mutually-denies 'starts-before))
;;.@vindex{Starts-after (slot)}

(define-unit Ends-before
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents))
;;.@vindex{Ends-before (slot)}
(define-unit Ends-after
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents)
  (inverse-slot 'ends-before)
  (mutually-denies 'ends-before))
;;.@vindex{Ends-after (slot)}

;;.Temporal extents can have @code{occurences} which are temporal
;;.extents within it.  The inverse of @code{occurences} are
;;.@code{occurs-within} which has `genl slots' of
;;.@code{contained-in} as well as @code{starts-after} and
;;.@code{ends-before}.

(define-unit Occurs-Within
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents)
  (genl-slots 'Contained-in)
  (genl-slots 'starts-after)
  (genl-slots 'ends-before))
;;.@vindex{Occurs-Within (slot)}

(define-unit Occurences
  (member-of 'transitive-slots)
  (makes-sense-for 'Temporal-extents)
  (must-be 'Temporal-extents)
  (inverse-slot 'Occurs-Within)
  (mutually-denies 'occurs-within))
;;.@vindex{Occurences (slot)}


;;;;.States

;;.States are extents `contained in' @code{things} and the relation
;;.@code{state-of} (with genl slot @code{contained-in}) relates a
;;.state to the thing it particularizes.  The inverse of this is
;;.@code{has-state}, a many valued slot containing all the states of a
;;.particular thing.

(define-unit state-of
  (member-of 'single-valued-slots)
  (makes-sense-for 'states)
  (must-be 'things)
  (genl-slots 'contained-in)
  (inverse-slot
   (make-unit 'has-state
     (member-of 'many-valued-slots))))
;;.@vindex{state-of (slot)}


;;;;.Processes

;;.Processes are causally connected extents.  A process has a number
;;.of @code{participants} which @code{overlap} with the process; each
;;.participant is a @code{participant-in} the corresponding process.
;;.The process also has a set of @code{Conditions} which allow the
;;.process to proceed (these are states) each of which is a
;;.@code{condition-of} the process.

(define-unit Participants
  (member-of 'many-valued-slots)
  (makes-sense-for 'Processes)
  (must-be 'Things)
  (genl-slots 'overlaps)
  (inverse-slot
   (make-unit 'participant-in
     (member-of 'many-valued-slots))))
;;.@vindex{Participants (slot)}

(define-unit Conditions
  (member-of 'many-valued-slots)
  (makes-sense-for 'processes)
  (must-be 'states)
  (genl-slots 'contained-in)
  (inverse-slot
   (make-unit 'condition-of
     (member-of 'many-valued-slots))))
;;.@vindex{Conditions (slot)}

;;.Processes contain @code{sub-processes} which are other processes
;;.contained in the process; the inverse of this relation is
;;.@code{sub-process-of}.

(define-unit Sub-processes
  (member-of 'many-valued-slots)
  (makes-sense-for 'processes)
  (must-be 'things)
  (genl-slots 'contained-in)
  (inverse-slot
   (make-unit 'sub-process-of
     (member-of 'many-valued-slots))))
;;.@vindex{Sub-processes (slot)}

;;.In addition, a process can enable or disable other processes as
;;.determined by the @code{enables} and @code{disables} relations
;;.(with respective inverses @code{enabled-by} and
;;.@code{disabled-by}).

(define-unit Enables
  (member-of 'many-valued-slots)
  (makes-sense-for 'Processes)
  (must-be 'Processes)
  (inverse-slot
   (make-unit 'enabled-by
     (member-of 'many-valued-slots))))
;;.@vindex{Enables (slot)}

(define-unit Disables
  (member-of 'many-valued-slots)
  (makes-sense-for 'Processes)
  (must-be 'Processes)
  (inverse-slot
   (make-unit 'disabled-by
     (member-of 'many-valued-slots))))
;;.@vindex{Disables (slot)}


;;;;.State changes

;;.One special sort of process are the @code{state-changes} which have
;;.a @code{before-state} and an @code{after-state} which are mutually
;;.disjoint; the @code{state-change} enables the first state and
;;.disables the second.

(define-unit before-state
  (member-of 'many-valued-slots)
  (makes-sense-for 'state-changes)
  (must-be 'states)
  (genl-slots 'disables))
;;.@vindex{before-state (slot)}

(define-unit after-state
  (member-of 'many-valued-slots)
  (makes-sense-for 'state-changes)
  (must-be 'states)
  (structure '(disjoint-from before-state))
  (genl-slots 'enables))
;;.@vindex{after-state (slot)}


;;;;.Actions

;;.Actions are state changes which have an @code{initiator} and an
;;.@code{instrument}; the @code{initiator} must be one of
;;.@code{Agents} while the @code{instrument} must be one of
;;.@code{things}.  Note that unlike other representations
;;.@code{instrument} is not a generalized description of how the
;;.action was acheived but merely identifies something which overlaps
;;.the occurence and enabled it.

(define-unit Initiator
  (member-of 'single-valued-slots)
  (makes-sense-for 'Actions)
  (must-be 'Agents)
  (genl-slots 'overlaps))
;;.@vindex{Initiator (slot)}

(define-unit Instrument
  (member-of 'single-valued-slots)
  (makes-sense-for 'Actions)
  (must-be 'Things)
  (genl-slots 'overlaps))
;;.@vindex{Instrument (slot)}


;;;;.Kinds of Things

;;.The abstract ontology is made more concrete by various
;;.specializations.  @code{Spatial-extents} is specialized into
;;.@code{places} and @code{animals} and @code{robots} are introduced
;;.as specializations of @code{agents} and @code{animals} is
;;.specialized into @code{humans}.  In accord with often opressive social
;;.convention, humans are specialized into @code{men} and @code{women}
;;.and in a questionable step, @code{things} are divided into
;;.@code{natural-things} and @code{artifacts} with the introduction of
;;.@code{animals} and @code{robots} as respective specializations.

(define-unit places
  (member-of 'collections)
  (supersets 'spatial-extents))
;;.@vindex{places (collection)}

(define-unit animals
  (member-of 'collections)
  (supersets 'agents))
;;.@vindex{animals (collection)}

(define-unit robots
  (member-of 'collections)
  (supersets 'agents))
;;.@vindex{robots (collection)}

(define-unit humans
  (member-of 'collections)
  (supersets 'animals))
;;.@vindex{humans (collection)}

(define-unit Genders
  (member-of 'collections)
  (members (declare-unit 'Male))
  (members (declare-unit 'Female)))

(define-unit gender
  (member-of 'single-valued-slots)
  (makes-sense-for 'animals)
  (must-be 'genders))
;;.@vindex{gender (slot)}

(define-unit men
  (member-of 'collections)
  (supersets 'humans)
  (members-have '(gender male)))
;;.@vindex{men (collection)}

(define-unit women
  (member-of 'collections)
  (supersets 'humans)
  (members-have '(gender female))
  (mutually-excludes 'men))
;;.@vindex{women (collection)}

(define-unit artifacts
  (member-of 'collections)
  (supersets 'things))
;;.@vindex{artifacts (collection)}

(define-unit natural-things
  (member-of 'collections)
  (supersets 'things)
  (specializations 'animals)
  (mutually-excludes 'artifacts))
;;.@vindex{natural-things (collection)}



;;;;.Stating states

;;.Once varieties of @code{things} have been firmly established, a
;;.variety of states, processes and actions are introduced.
;;.@code{Being-there} is a state of a @code{thing} which involves a
;;.location (which is a spatial extent).  The location is stored in
;;.the state description's @code{being-where} slot.

(define-unit being-there
  (member-of 'collections)
  (supersets 'states))
;;.@vindex{being-there (collection)}

(define-unit being-where
  (member-of 'single-valued-slots)
  (makes-sense-for 'being-there)
  (must-be 'spatial-extents)
  (genl-slots 'overlaps)
  (inverse-slot
   (make-unit 'location-of
     (member-of 'many-valued-slots))))
;;.@vindex{being-where (slot)}

;;.@code{Moving-processes} are state changes involving a
;;.@code{thing-moved} which is the object of which its
;;.@code{before-state} and @code{after-state} are predicated.

(define-unit moving-processes
  (member-of 'collections)
  (supersets 'state-changes))
;;.@vindex{moving-processes (collection)}

(define-unit thing-moved
  (member-of 'single-valued-slots)
  (makes-sense-for 'moving-processes)
  (must-be 'things)
  (genl-slots 'participants)
  (inverse-slot (make-unit 'thing-moved-in
		  (member-of 'many-valued-slots)))
  (structure '(has-state before-state))
  (structure '(has-state after-state)))
;;.@vindex{thing-moved (slot)}

;;.Such movement proceeds from an @code{origin} to a
;;.@code{destination} and these are the @code{being-where} slots of
;;.the @code{before-state} and @code{after-state}.

(define-unit origin
  (member-of 'single-valued-slots)
  (makes-sense-for 'moving-processes)
  (must-be 'spatial-extents)
  (genl-slots 'contains)
  (inverse-slot
   (make-unit 'origin-of
     (member-of 'many-valued-slots)))
  (structure '(location-of before-state)))
;;.@vindex{origin (slot)}
;;.@vindex{origin-of (slot)}

(define-unit destination
  (member-of 'single-valued-slots)
  (makes-sense-for 'moving-processes)
  (must-be 'spatial-extents)
  (genl-slots 'contains)
  (inverse-slot
   (make-unit 'destination-of
     (member-of 'many-valued-slots)
     (makes-sense-for 'spatial-extents)
     (must-be 'moving-processes)))
  (structure '(location-of after-state)))
;;.@vindex{destination (slot)}
;;.@vindex{destination-of (slot)}

;;.The most common specialization of @code{moving-processes} is
;;.@code{being-moved} which is also an action.  The other property of
;;.@code{being-moved} is its @code{vehicle} which is also the
;;.@code{instrument} of the corresponding action.

(define-unit being-moved
  (member-of 'collections)
  (supersets 'moving-processes)
  (supersets 'actions))
;;.@vindex{being-moved (collection)}

(define-unit vehicle
  (member-of 'single-valued-slots)
  (makes-sense-for 'being-moved)
  (must-be 'artifacts)
  (genl-slots 'instrument))
;;.@vindex{vehicle (slot)}


;;;;.Abstract processes.

;;.More abstract processes involve relations between processes.  For
;;.instance, @code{perceiving} is a process involving a
;;.@code{perceiver} and a @code{perception} where both @code{overlap}
;;.the process of perceiving.  The @code{perceiver} must be an agent.

(define-unit perceiving
  (member-of 'collections)
  (supersets 'processes))
;;.@vindex{perceiving (collection)}

(define-unit perception
  (member-of 'single-valued-slots)
  (makes-sense-for 'perceiving)
  (must-be 'extents))
;;.@vindex{perception (slot)}

(define-unit perceiver
  (member-of 'single-valued-slots)
  (makes-sense-for 'perceiving)
  (must-be 'agents)
  (genl-slots 'participants))
;;.@vindex{perceiver (slot)}

;;.An action oriented version of @code{perceiving} is
;;.@code{informing} where one agent causes a perception of another
;;.agent.  The slots @code{speaker} and @code{listener} specialize
;;.@code{informing} though the `aural' connotations of the slot names
;;.are not meant to exclude other forms of communcation.

(define-unit informing
  (member-of 'collections)
  (supersets 'perceiving)
  (supersets 'actions))
;;.@vindex{informing (collection)}

(define-unit speaker
  (member-of 'single-valued-slots)
  (makes-sense-for 'informing)
  (must-be 'agents)
  (genl-slots 'initiator)
  (inverse-slot (make-unit 'speaker-in
		  (member-of 'many-valued-slots))))
;;.@vindex{speaker (slot)}

(define-unit listener
  (member-of 'single-valued-slots)
  (makes-sense-for 'informing)
  (must-be 'agents)
  (genl-slots 'perceiver)
  (inverse-slot (make-unit 'listener-in
		  (member-of 'many-valued-slots))))
;;.@vindex{listener (slot)}

;;.One special variety of `informing' is @code{demanding} where the
;;.content of the communication (the corresponding @code{perception})
;;.is an action whose @code{initiator} is the @code{listener}.  This
;;.action is the @code{demand} of the process whose inverse is stored
;;.as @code{demanded-in}.

(define-unit demanding
  (member-of 'collections)
  (supersets 'informing))
;;.@vindex{demanding (collection)}

(define-unit demand
  (member-of 'single-valued-slots)
  (makes-sense-for 'demanding)
  (must-be 'being-moved)
  (genl-slots 'perception)
  (structure '(initiator listener))
  (structure '(destination speaker)))
;;.@vindex{demand (slot)}

(define-unit demanded-in
  (member-of 'many-valued-slots)
  (makes-sense-for 'actions)
  (must-be 'demanding)
  (inverse-slot 'demand))
;;.@vindex{demanded-in (slot)}

;;.A special version of @code{demanding} is @code{demanding-something}
;;.where the demand is further specialized to be an instance of
;;.@code{moving-processes} whose @code{thing-moved} is the demand's
;;.@code{demanded-object} and whose @code{destination} is the demand's
;;.@code{demand-recipient}.

(define-unit demanding-something
  (member-of 'collections)
  (supersets 'demanding))
;;.@vindex{demanding-something (collection)}

(define-unit demanded-object
  (member-of 'single-valued-slots)
  (makes-sense-for 'demanding-something)
  (structure '(participant-in demand)))
;;.@vindex{demanded-object (slot)}

(define-unit demand-recipient
  (member-of 'single-valued-slots)
  (makes-sense-for 'demanding-something)
  (structure '(destination-of demand)))
;;.@vindex{demand-recipient (slot)}


;;;;.Ownership and giving

;;.Hegel constructed property ownership as the way the self defined
;;.itself.  While that particular derivation may be questionable,
;;.support for representing ownership is important since many stories
;;.involve issues of ownership and it is important to represent these
;;.things for the purpose of understanding stories.

;;.@code{Owning} is a kind of state which has an @code{owner} slot and
;;.an @code{owned-thing} which has a `genl slot' of @code{state-of}.
;;.The inverse of @code{owned-thing} is @code{owned-in}.

(define-unit owning
  (member-of 'collections)
  (supersets 'states))
;;.@vindex{owning (collection)}

(define-unit owner
  (member-of 'single-valued-slots)
  (makes-sense-for 'owning)
  (must-be 'agents))
;;.@vindex{owner (slot)}

(define-unit owns-in
  (member-of 'many-valued-slots)
  (makes-sense-for 'agents)
  (must-be 'owning)
  (inverse-slot 'owner))
;;.@vindex{owns-in (slot)}

(define-unit owned-thing
  (member-of 'single-valued-slots)
  (makes-sense-for 'owning)
  (must-be 'things)
  (genl-slots 'state-of))
;;.@vindex{owned-thing (slot)}

(define-unit owned-in
  (member-of 'many-valued-slots)
  (makes-sense-for 'things)
  (must-be 'owning)
  (inverse-slot 'owned-thing))
;;.@vindex{owned-in (slot)}

;;.Ownership changes in @code{giving} actions which involve both a
;;.physical and property transfer.  These actions have a @code{giver},
;;.a @code{recipient}, and a @code{thing-given} which become the
;;.corresponding slots of the hybrid `physical giving'/`transferring
;;.ownership' actions.

(define-unit giving
  (member-of 'collections)
  (supersets 'actions)
  (supersets 'being-moved))
;;.@vindex{giving (collection)}

(define-unit giver
  (member-of 'single-valued-slots)
  (makes-sense-for 'giving)
  (must-be 'agents)
  (inverse-slot (make-unit 'giver-in
		  (member-of 'many-valued-slots)
		  (must-be 'giving)
		  (makes-sense-for 'agents)))
  (genl-slots 'initiator)
  (genl-slots 'origin)
  (structure '(owns-in before-state)))
;;.@vindex{giver (slot)}

(define-unit recipient
  (member-of 'single-valued-slots)
  (makes-sense-for 'giving)
  (must-be 'agents)
  (inverse-slot (make-unit 'recipient-in
		  (member-of 'many-valued-slots)
		  (must-be 'giving)
		  (makes-sense-for 'agents)))
  (genl-slots 'destination)
  (structure '(owns-in after-state)))
;;.@vindex{recipient (slot)}

(define-unit thing-given
  (member-of 'single-valued-slots)
  (makes-sense-for 'giving)
  (must-be 'things)
  (genl-slots 'thing-moved)
  (structure '(owned-in before-state))
  (structure '(owned-in after-state)))
;;.@vindex{thing-given (slot)}






