;;; -*- LISP -*-

;;.@chapter Many valued slots

;;.ARLOtje's initial boot levels use only singled valued slots; but
;;.the ability to have slots handle their values in special ways ---
;;.introduced with annotated values --- allows the specification of
;;.values which are sets.  The simplest approach to managing such
;;.`many-valued slots' would be to have the @code{to-put-value}
;;.function of such slots do a `push' onto the slot being stored; this
;;.is how @code{depended-on-by} slots stores values (though it stores
;;.them primitively using @code{%push}).  The same thing
;;.could be done with annotated values, but the problem is that storing
;;.a new single value would invalidate any dependents on all of the
;;.current values.  For instance, suppose that a slot @code{S} of a
;;.unit @code{U} had values @code{X} and @code{Y}.  Now consider a set
;;.of values depending on just the presence of the @code{X} value; when
;;.@code{Z} is added to the set of values on @code{S} of @code{U}, this
;;.set of values is invalidated even though --- in reality --- they are
;;.still completely valid.@refill

;;.This particular case is just a special instance of the case where
;;.some values depend on a constraint (one of the values is X) rather
;;.than a precise value (the values are X, Y, and Z).  Any
;;.implementation chooses a level of `granularity' at which constraints
;;.can be specified.  ARLOtje represents many-valued slots by
;;.describing the value at two levels: the list of values itself, which
;;.is invalidated whenever a value is added or removed, and the
;;.individual values which may be explicitly invalidated.@refill

;;.Annotated descriptions for many valued slots have a slot
;;.@code{recorded-elements} into which values accumulate; these
;;.elements are merged into a single returned value by any of the
;;.slots @code{elements-as-bag}, @code{elements-as-set}, etc.  There
;;.might also be slots such as @code{elements-averaged},
;;.@code{elements-maximized}, etc.

;;.Since it is possible to refer to elements of these many valued
;;.slots individually, several iteration functions are provided which
;;.map over multiple elements of a set or slot and construct an
;;.`assertion context' @pxref{Assertion contexts} including that
;;.element; this is particularly useful with inference procedures
;;.implemented by many valued descriptions.

(in-package :arlotje)


;;;;.Implementing many valued slots.

;;.Many valued slots actually store their values in the
;;.@code{recorded-elements} slot of their value descriptions; they
;;.call the put function @code{record-element} to put values there.
(define-unit recorded-elements
  (works-like 'prototypical-primitive-slot)
  (to-put-value 'record-element)
  (to-get-value 'get-recorded-elements))
;;.@vindex{recorded-elements (slot)}

(defstruct (element (:include assertion) (:print-function print-element)
		    (:constructor make-element (in-set value)))
  in-set
  value)

(defun print-element (element stream &rest options)
  (declare (ignore options))
  (format stream "#<ELT ~S ~S ~S>"
	  (annotated-value-unit (element-in-set element))
	  (annotated-value-slot (element-in-set element))
	  (element-value element)))

;;.The function @code{record-element} looks for an element whose value
;;.is @code{EQ} to the value being stored; if such an element is
;;.found, @code{record-element} removes any potential invalidation
;;.annotation and returns the element.  Otherwise, it creates a unit
;;.describing the new member and adds this description onto the
;;.@code{recorded-elements} property of the value description.  If
;;.this occurs, it invalidates the other values depending on the value
;;.of the set as a whole (i.e. the @code{depended-on-by} slot of the
;;.set itself) and finally returns the element description which it
;;.has constructed.
(defun record-element (set slot value &rest annotations)
  "Records an element in a set description, creating a unit describing its membership."
  (declare (ignore slot annotations))
  (do ((elts (%get-default set 'recorded-elements '()) (cdr elts))
       (last-elts '() elts))
      ((or (null elts) (equal value (element-value (car elts))))
       (cond ((null elts)
	      (let ((description (make-element set value)))
		(if (null last-elts)
		    (%put set 'recorded-elements (list description))
		  (setf (cdr last-elts) (list description)))
		;; We do this because the value of the set is never
		;; invalidated itself (since we won't ever revalidate
		;; it) but only indicates that values dependent on it
		;; should be revalidated.
		(invalidate-dependents set)
		description))
	     ((assertion-invalidated (car elts))
	      (setf (assertion-invalidated (car elts)) NIL)
	      ;; We do this because the value of the set is never
	      ;; invalidated itself (since we won't ever revalidate
	      ;; it) but only indicates that values dependent on it
	      ;; should be revalidated.
	      (invalidate-dependents set)
	      (car elts))
	     (T (car elts))))))
;;.@findex{record-element}
;;.These final steps are important for maintaining the assumption made
;;.by @code{annotated-value-put} that @code{put-value} return a
;;.description of the value actually stored.

(defun get-recorded-elements (set slot)
  (%get-default set slot '()))

;;.All many valued slots have @code{recorded-elements} as 
;;.their @code{accumulates-in-slot} slot; the predicate
;;.@code{many-valued-slotp} checks exactly this property in
;;.determining the nature of its argument.
(defun many-valued-slotp (slot)
  "Returns true for many valued slots which have RECORDED-ELEMENTS as
their ACCUMULATE-IN-SLOT slot."
  (and (slotp slot)
       (eq (get-value slot 'accumulates-in-slot)
	   'recorded-elements)))
;;.@findex{many-valued-slotp}

;;.Individual element descriptions are assigned two properties: an
;;.@code{element-value} denoting their value and a
;;.@code{element-in-set} pointer to the set in which they are an element.
(define-primitively element-value
  (english-description "The assertions which this annotated value depends on.")
  (makes-sense-for 'elementp)
  (must-be 'anythingp)
  (to-get-value 'whonit-get)
  (to-put-value 'whonit-put)
  (whonit-type 'element)
  (whonit-accessor 'element-value))
(define-primitively element-in-set
  (english-description "The assertions which this annotated value depends on.")
  (makes-sense-for 'elementp)
  (must-be 'annotated-valuep)
  (to-get-value 'whonit-get)
  (to-put-value 'whonit-put)
  (whonit-type 'element)
  (whonit-accessor 'element-in-set))
;;.@vindex{element-value (slot)}
;;.@vindex{element-in-set (slot)}

;;.The standard @code{to-find-value} function for many valued slots
;;.goes and looks for a match among the value description's
;;.@code{recorded-elements}.
(defun find-recorded-element (unit slot element)
  "Returns whatever one of a value's RECORDED-ELEMENTS matches ELEMENT."
  (let* ((av (annotated-value unit slot)))
    (or (some #'(lambda (elt) (and (not (assertion-invalidated elt))
				   (equal (element-value elt) element)
				   (dependency! elt)))
	      (%get-default av 'recorded-elements '()))
	(unless (computing? get-value av 'default-members)
	  ;; If you're computing default members already, don't try
	  ;; and compute it recursively.
	  (when (member element (get-value av 'default-members))
	    (some #'(lambda (elt) (and (not (assertion-invalidated elt))
				       (equal (element-value elt) element)
				       (dependency! elt)))
		  (%get-default av 'recorded-elements '())))))))


;;;;.Iterating over many-valued descriptions

;;.Three iterating functions are provided for accessing the elements
;;.of a list.

;;.@code{Map-elements} takes a function and a description and maps
;;.the function over the values of all the @code{recorded-elements} of the
;;.description, returning a list of the results of applying the
;;.function to each @code{element-value} providing that the element has
;;.not been invalidated.
(defun map-elements (function gaggle)
  "Maps FCN over each member of GAGGLE, returning a list of the results."
  (let* ((map-result (list '())) (result-tail map-result))
    (dolist (elt (%get-default gaggle 'recorded-elements '()) (cdr map-result))
      (unless (assertion-invalidated elt)
	(push (funcall function (element-value elt)) (cdr result-tail))
	(setf result-tail (cdr result-tail))))))
;;.@findex{map-elements}

;;.The function @code{for-elements} is like @code{map-elements} but
;;.returns the original list and --- importantly --- binds the
;;.`assertion context' @pxref{Assertion contexts} to include the
;;.element description on which the per-element function is called.
;;.As a result, any assertions made by this function will depend on
;;.the validity of the corresponding value.
(defun for-elements (function gaggle)
  "Applies FUNCTION to each element of GAGGLE,
making assertions within FUNCTION depend on the corresponding element."
  (dolist (elt (%get-default gaggle 'recorded-elements '()))
    (unless (assertion-invalidated elt)
       (let ((*assertion-context* (cons elt *assertion-context*)))
	 (funcall function (element-value elt))))))
;;.@findex{for-elements}

;;.The macro @code{do-elements} is like @code{dolist} but iterates
;;.over all of the elements in a value description, making assertions
;;.in the body dependent on the individual element's presence.
(defmacro do-elements ((elt gaggle) &body body)
  "Executes BODY for each valid element ELT of GAGGLE,
making assertions in BODY dependent on ELT's presence."
  (let ((desc-var (make-symbol "ELT")))
    `(dolist (,desc-var (%get-default ,gaggle 'recorded-elements '()))
      (unless (assertion-invalidated ,desc-var)
	(let ((*assertion-context* (cons ,desc-var *assertion-context*))
	       (,elt (element-value ,desc-var)))
	  ,@body)))))
;;.@findex{do-elements (macro)}


;;;;.Many valued slots

;;.Many valued slots are implemented by annotated values whose value
;;.descriptions contain collections of elements implemented by the
;;.mechanisms above.  The @code{accumulates-in-slot} slot of such many
;;.valued slots is usually @code{recorded-elements}; depending on the
;;.particular slot, the @code{value-in-slot} slot is usually something
;;.which computes a list of values based (and dependent on) the
;;.individual elements recorded.

;;.Two prototypes for many valued slots are defined:
;;.@code{prototypical-set-slot} and @code{prototypical-bag-slot}.  The
;;.first allows reproduced elements; the second merges @code{EQUAL}
;;.elements in its returned result.  Both provide a
;;.@code{accumulates-in-slot} slot of @code{recorded-elements}.
;;.However they use the distinct slots @code{elements-as-set} and
;;.@code{elements-as-bag} respectively for their @code{value-in-slot}
;;.slot.

(define-unit prototypical-set-slot
  (english-description "The prototype of all multi-valued slots with no repeated elements.")
  (works-like 'prototypical-slot)
  (value-in-slot 'elements-as-set)
  (accumulates-in-slot 'recorded-elements)
  (to-find-value 'find-recorded-element))
;;.@vindex{prototypical-set-slot (slot prototype)}

(define-unit prototypical-bag-slot
  (english-description
   "The prototype of all multi-valued slots with potentially repeated elements.")
  (works-like 'prototypical-slot)
  (value-in-slot 'elements-as-bag)
  (accumulates-in-slot 'recorded-elements)
  (to-find-value 'find-recorded-element))
;;.@vindex{prototypical-bag-slot (slot prototype)}

;;.The slot @code{elements-as-set} returned the values of the
;;.@code{recorded elements} with duplicate appearances removed.
;;.It is a cached slot so that it is only recomputed when values are
;;.added or removed (retracted) from the value description.
(define-unit elements-as-set
  (english-description "The elements of a many valued description with duplicates removed.")
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'elements-into-set))
;;.@vindex{elements-as-set (slot)}

;;.It computes its value by calling the function @code{elements-into-set}.
(defun elements-into-set (unit slot)
  (declare (ignore slot))
  (dependency! unit)
  (as-a-side-effect (get-value unit 'default-members))
  (let ((set '()))
    (dolist (record (%get-default unit 'recorded-elements '()) set)
      (unless (assertion-invalidated record)
	(dependency! record)
	(pushnew (element-value record) set :test #'equal)))))
;;.@findex{elements-into-set}

;;.@code{Elements-as-bag} returns the values of a unit's
;;.@code{recorded-elements} with no removal of @code{EQUAL} values.
;;.As with @code{elements-as-bag}, it is a cached slot which is
;;.invalidated whenever the list of elements as a whole is modified
;;.(by addition of an element) or one of the elements is removed (by
;;.becoming invalidated).
(define-unit elements-as-bag
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'elements-into-bag))
;;.@vindex{elements-as-bag (slot)}

;;.It computes its value by the @code{elements-into-bag} function.
(defun elements-into-bag (unit slot)
  (declare (ignore slot))
  (as-a-side-effect (get-value unit 'default-members))
  (dependency! unit)
  (let ((set '()))
    (dependency! unit)
    (dolist (record (%get-default unit 'recorded-elements '()) set)
      (unless (assertion-invalidated record)
	(dependency! record)
	(push (element-value record) set)))))
;;.@findex{elements-into-bag}


;;;;.Defaults for many valued slots.

(define-unit default-members
  (works-like 'prototypical-cached-slot)
  (makes-sense-for 'annotated-valuep)
  (must-be 'listp)
  (to-compute-value 'compute-default-members)
  (to-get-value 'get-default-members)
  (to-put-value 'store-default-members))

(defun get-default-members (av sl)
  (if (failurep (get-slot-method (annotated-value-slot av) 'to-compute-value))
      '()
    (annotated-value-get av sl)))

(defun compute-default-members (av sl)
  (declare (ignore sl))
  (let ((unit (annotated-value-unit av))
	(slot (annotated-value-slot av)))
    (let ((method (get-value slot 'to-compute-value)))
      (if (failurep method) '()
	(funcall method unit slot)))))

(defun store-default-members (av sl defaults &rest properties)
  (declare (ignore sl))
  (let ((description (apply #'annotated-value-put av sl defaults
			    properties)))
    (let ((unit (annotated-value-unit av))
	  (slot (annotated-value-slot av)))
      (dolist (default defaults)
	(assert-value unit slot default 'depends-on (list description))))))

;;;;.Iterating over slot values

;;.Analgous to the `element iteration' definitions above, there are
;;.more commonly used operations which operate on units and slot.  The
;;.function @code{map-members} takes a @var{function}, a @var{unit},
;;.and a @var{slot} and applies @var{function} to each value of
;;.@var{slot} of @var{unit}, returning a list of the results.
(defun map-members (fcn unit slot)
  (if (many-valued-slotp slot)
      (map-elements fcn (%get-default unit slot '()))
    (list (funcall fcn (get-value unit slot)))))

;;.A non functional version of this iterator is @code{for-members}
;;.which takes the same arguments but returns the list of values
;;.itself and calls @code{function} in an assertion context dependent
;;.on the validity of the value being processed.
(defun for-members (fcn unit slot)
  (if (many-valued-slotp slot)
      (for-elements fcn (%get-default unit slot '()))
    (and (get unit slot) (list (funcall fcn (get-value unit slot))))))

;;.The macro @code{do-members} is like @code{do-elements} except that
;;.it takes a unit and slot rather than a `set description'.  The body
;;.is evaluated in a context dependent on the value currently being
;;.examined.
(defmacro do-members ((elt-var unit slot) &body body)
  `(if (many-valued-slotp ,slot)
    (do-elements (,elt-var (%get-default ,unit ,slot '()))
	,@body)
    (unless (%hasnt ,unit ,slot)
      (let ((,elt-var (get-value ,unit ,slot)))
	(unless (failurep ,elt-var) (let () ,@body))))))

;;.@code{dovalues} is a very useful extension of @code{doslots}
;;.@pxref{Enumerating Slots} which iterates over all of the slots of a
;;.unit and individually iterates over the elements of any many valued
;;.slot.  
(defmacro dovalues (unit (slot value &optional (desc (make-symbol "DESC")))
			 &body body)
  `(doslots ,unit (,slot ,value ,desc)
    (if (many-valued-slotp ,slot)
	(do-elements (,value ,desc) ,@body)
      (unless (failurep ,value) (let () ,@body)))))


