;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@chapter Inference Mechanisms
(in-package :arlotje)

;;.Representation languages support various sorts of @emph{inferences}
;;.about the knowledge they represent; this means that the
;;.representation language may `fill in the blanks' of partial
;;.descriptions in different ways.  In a representation language
;;.language, where the slots are descriptions, the inferences which the
;;.system makes are determined by the descriptions of slots.  For
;;.instance, the description of the @code{husband} slot might note that
;;.its inverse is the @code{wife} slot; when a @code{husband} slot is
;;.stored (say between descriptions of `Nancy' and `Ron'), the
;;.corresponding @code{wife} slot is also asserted between the
;;.descriptions of `Ron' and `Nancy'.  We saw a similar sort of
;;.inference in the @code{works-like} slot used to specify slot method
;;.inheritance.  Saying that @code{to-combine-values} @code{works-like}
;;.@code{to-compute-value} means that --- unless the system is told
;;.otherwise --- the slot methods for @code{to-combine-values} are the
;;.same as the slot methods for @code{to-compute-value}.

;;.In these two examples, we see that inferences can go @emph{forward}
;;.or @emph{backward}.  An inference like @code{inverse-slot} goes
;;.forward because asserting one relation leads to asserting another;
;;.an inference like @code{works-like} inheritance goes backwards
;;.because the @emph{demand} for a fact causes the inference
;;.(inheritance) to be made.  In general, the decision to make an
;;.inference forward or backward depends on the form of the inference's
;;.forward and backward versions.  For instance, doing the `inverse
;;.inference' as a backwards inference would involve searching all
;;.potential partners of an individual to find the one which was
;;.actually the partner.  E.G. suppose we wanted to compute @code{Wife}
;;.of `Ron' given that we knew @code{Husband} of `Nancy'; we would have
;;.to check all known women until we uncovered `Nancy' (pausing
;;.momentarily over the invalidated property of `Jane').  On the other
;;.hand, the search for @code{works-like} is direct and searches a
;;.hierarchy which is probably not going to be very deep.  Thus it is a
;;.resonable inference to do in the backwards (demand based) direction.
;;.But there are sometimes reasons to not make inferences in the
;;.forwards direction; in particular, all inferences take space so that
;;.making superfluous forward inferences will take up space to little
;;.advantage.

;;.Another decision to be made in implementing inference schemes is
;;.how to handle the retraction of the premises leading to an
;;.inference.  There are at least two possible approaches: one can use
;;.an explicit dependency network (like ARLOtje's) or one can
;;.explicitly charaterize the @emph{sorts} of inferences descending
;;.from a given assertion and remove those sorts of inferences
;;.whenever a potential premise is removed.  The problem with this is
;;.that some inferences come from a variety of sources and we do not
;;.wish to remove the result of an inference process if several other
;;.inference processes had produced the same result.  For instance, a
;;.human being might be judged intelligent both because of her
;;.demeanor and her education; when you find that the educational
;;.justification is invalid, one does not (we hope) retract the
;;.assessment deriving from her demanor.  However, if all of those
;;.constituent inference processes were invalidated, we would like the
;;.value to go away.  Because of this quandary, ARLOtje generally uses
;;.the dependency network to track individual values rather than
;;.explicitly retracing the classes of inferences which follow from
;;.particular assertions.

;;.ARLOtje uses both forward and backward inferences, many of which are
;;.taken from CYCL, the representation langauge language underlying
;;.Lenat's CYC system.  Most of them are implemented in the file
;;.`inferences.lisp' and described in the following sections.

;;.@section Forward Inferences

;;.ARLOtje's forward inferences are implemented through @dfn{put
;;.demons} which are procedures run whenever a particular slot is
;;.asserted.  Usually, the user does not see this level of description
;;.but only higher level meta-slots; inferences about these meta-slots
;;.install the put-demon inferences mechanisms which implement the
;;.actual desired semantics.  For instance, the slot @code{inferences}
;;.stores slot-value pairs to be asserted along with a particular
;;.slot; but one of the inferences about the @code{inferences} slot is
;;.to assert an appropriate @code{put-demons} slot.

;;.Some of ARLOTje's forward inferences are ideas that have been around
;;.in Artificial Intelligence for a while, but others are new.  In
;;.particular, several are stolen from the CYCL language devised by
;;.Lenat and his colleagues at MCC.


;;;;.Put demons

;;.Put demons are expressions in a LISP-like language which are
;;.evaluated whenever a slot is stored.  A put demon stored on the
;;.@var{foo} slot is evaluated whenever a value is stored as a
;;.@var{foo} slot.  This is evaluated both when the storage of the value
;;.is directly specified by a user or is the result of an assertion on
;;.the user's part.  A put demon is interpreted exactly like a LISP
;;.form except that symbols are not evaluated so that
;;.@example
;;.(assert-value fire-truck.1 color red)
;;.@end example
;;.will not try and evaluate @code{fire-truck.1} and the other
;;.symbols.  This is violated for the three special symbols
;;.@code{%UNIT%}, @code{%SLOT%}, @code{%VALUE} which evaluate
;;.(respectively) to the unit, slot, and value being stored.  These
;;.can be quoted (with single quote) to prevent their evaluation and
;;.the backquote construction works to construct lists from these
;;.elements.  Subexpressions are evaluated so that one can say:
;;.@example
;;.(assert-value (get-value fire-truck.1 'paint) color red)
;;.@end example
;;.to set a property of @code{fire-truck.1}'s paint.  For a more
;;.useful example, suppose the slot @code{painted} took a color and
;;.passed this color onto the unit's @code{paint} slot.  A put demon
;;.to implement this inference would look like:
;;.@example
;;.(assert-value (get-value %UNIT% 'paint) color %VALUE%)
;;.@end example

;;.Put demons are evaluated by the function @code{run-put-demon} which
;;.takes the demon expression and the unit, slot, and value being
;;.stored.  
(defun run-put-demon (put-demon unit slot value)
  "Evaluates PUT-DEMON with appropriate interpretations to UNIT, SLOT, and VALUE."
  (block fail
    (labels ((evaller (x)
	       (cond ((null x) x)		     
		     ((listp x)
		      (cond ((eq (car x) 'quote) (cadr x))
			    ((eq (car x) 'progn)
			     (do ((exprs (cdr x) (cdr exprs)))
				 ((null (cdr exprs)) (evaller (car exprs)))
			       (evaller (car exprs))))
			    ((macro-function (car x))
			     (evaller (macroexpand x)))
			    ((eq (car x) 'if)
			     (let ((test (evaller (cadr x))))
			       (if test (evaller (caddr x)) (evaller (cadddr x)))))
			    (t (let ((result (apply (car x) (mapcar #'evaller (cdr x)))))
				 (if (failurep result) (return-from fail result) result)))))
		     ((eq x '%UNIT%) unit)
		     ((eq x '%SLOT%) slot)
		     ((eq x '%VALUE%) value)
		     (T x))))
      (evaller put-demon))))
;;.The procedure @code{run-put-demons} is called to run the put-demons
;;.for a particular slot assertion.  The unit, slot, and value
;;.passed to this function are used as arguments for each of the put
;;.demons evaluated.  Put demons are evaluated in a context which
;;.depends on the put demon itself and the assertion being stored;
;;.this means that the establishing conditions for the put demons are
;;.also justifications for its inferences.
(defun run-put-demons (unit slot value)
  "Runs the put demons for unit, slot and value.  Demons are run in
inverse order of assertion (this sometimes matters)."
  (as-a-side-effect 
   (do-members (demon slot 'put-demons)
     (run-put-demon demon unit slot value))))

;;.Put demons operate by a variation of @code{standard-put-value}
;;.called @code{demonic-put-value} which calls the procedure
;;.@code{run-put-demons} after a value is stored.@refill
(defun demonic-put-value (unit slot value)
  "Stores a value, doing annotated value resolution and running PUT-DEMONS.
This always stores an annotated value, constructing a direct value if
none is provided."
  (check-put-value unit slot value)
  (let ((description (annotated-value-put unit slot value)))
    (let ((*assertion-context* (cons description *assertion-context*)))
      (run-put-demons unit slot value))
    description))

;;.The slot @code{put-demons} contains the put demons for a slot.  It
;;.is accessed by the procedure @code{run-put-demons}.
;;.@vindex{put-demons (slot)}
(define-unit put-demons
  (english-description "Demons run when slot values are put.")
  (works-like 'prototypical-set-slot)
  (to-put-value 'demonic-put-value)
  (makes-sense-for 'slotp)
  (must-be         'listp)
  (put-demons '(assert-value %UNIT% to-put-value demonic-put-value)))

;;.One of the inferences associated with the @code{put-demons} slot is
;;.to assign the slot a @code{to-put-value} slot of
;;.@code{demonic-put-value}.  This ensures that once a put-demon is
;;.stored it will be called.  When collections are implemented an
;;.additional inference on @code{put-demons} insures a degree of
;;.commutativity wherein put-demons are run on existing slot values.


;;;;.Unit inferences

;;.The slot @code{unit-inferences} stores particular inferences to
;;.make about the unit upon which a slot is stored.  It is a simple
;;.many-valued slot of lists having the form @code{(@var{slot}
;;.@var{value})} where the subexpressions @var{slot} and @var{value}
;;.are evaluated following the standard put demon rules.

;;.In fact, @code{unit-inferences} are implemented by a put demon
;;.which adds an @code{assert-value} put-demon to the slot; this is
;;.implemented thus:
;;...@example
(define-unit unit-inferences
  (english-description "Demons run when slot values are put.")
  (works-like 'prototypical-set-slot)
  (to-put-value 'demonic-put-value)
  (makes-sense-for 'slotp)
  (must-be         'listp)
  (put-demons '(assert-value %unit% put-demons `(assert-value %unit% ,@%value%))))
;;...@end example


;;;;.Value inferences

;;.The slot @code{value-inferences} stores particular inferences to
;;.make about the value being stored on a slot.  It is a simple
;;.many-valued slot of lists having the form @code{(@var{slot}
;;.@var{value})} where the subexpressions @var{slot} and @var{value}
;;.are evaluated following the standard put demon rules.

;;.In fact, @code{value-inferences} are implemented by a put demon
;;.which adds an @code{assert-value} put-demon to the slot; this is
;;.implemented thus:
;;...@example
(define-unit value-inferences
  (english-description "Demons run when slot values are put.")
  (works-like 'prototypical-set-slot)
  (to-put-value 'demonic-put-value)
  (makes-sense-for 'slotp)
  (must-be         'listp)
  (put-demons '(assert-value %unit% put-demons `(assert-value %value% ,@%value%))))
;;...@end example


;;;;.Local demons

;;.ARLOtje supports the specification of @dfn{local demons} which are
;;.put demons applying to one particular unit/slot pair.  For
;;.instance, one can specify a local demon to be run whenever one of the
;;.@code{members} of @code{fire-trucks} is asserted.  These demons are
;;.called local demons and are stored on a special slot of the
;;.corresponding unit and have a form identical to regular put demons.
;;.The local demons for a slot @var{foo} are stored on a special slot
;;.named something like @var{foo-demons} and recorded as the
;;.@code{local-demons} of @var{foo}.  One can use local demons by just
;;.getting the @code{local-demons} slot of another slot; specifying
;;.this slot's value will install local demons.  For example, we might
;;.make an assertion that Quakers like oatmeal by the following:
;;.@example
;;.(assertion 'quakers (get-value 'members 'local-demons)
;;.           '(assert-value %value% 'favorite-food 'oatmeal))
;;.@end example

;;.Local demons slots can be either installed by hand (they
;;.should be many-valued slot whose values are lists) or generated
;;.automatically by calling @code{get-value} of the slot and its
;;.@code{local-demons}.  This allows the use of the special
;;.syntax of @code{define-unit} so that one can say:
;;.@example
;;.(define-unit red-box
;;.  ((local-demons parts) '(assert-value %value% color red)))
;;.@end example
;;.indicating that all parts of @code{red-box} are red.

;;.@code{Run-local-demons} is the procedure whose ue as a
;;.put demon implements associated demons.  It works by using
;;.@code{do-members} to go over the list of associated demons and
;;.calling @code{run-put-demon} on each recorded demon.@refill
(defun run-local-demons (demons-slot unit slot value)
  "This makes inferences about all instances stored as values of a slot."
  (do-members (inference unit demons-slot)
    (run-put-demon inference unit slot value)))
;;.@findex{run-local-demons}

;;.One important issue with any inference mechanism is commutavity; if
;;.you assert one thing and then another, it shouldn't matter (in
;;.theory), the order in which they are asserted.  With generic
;;.inheritance, we handle this by going over all the current values of
;;.a slot whenever its associated inferences are asserted.  So if we
;;.are told that all @code{members} of @code{Quakers} have
;;.@code{favorite-food} @code{oatmeal}, we walk down the @code{members}
;;.of @code{quakers} and assert their gustatorial preference.  This is
;;.handled by the procedure @code{run-local-demons-on-current-values}
;;.which takes an asserted inference and calls @code{run-put-demon}
;;.for each current value of the corresponding slot.@refill
(defun run-local-demons-on-current-values (unit slot inference)
  "Takes a newly expressed inference and applies it to all the current values of a slot."
  (let ((inference-for (get-value slot 'local-demons-for)))
    (do-members (elt unit inference-for)
      (run-put-demon inference unit inference-for elt))))
;;.@findex{run-local-demons-on-current-values}
 
(define-unit local-demons
  (english-description "The slot containing the slot containing inferences regarding a slot.")
  (works-like 'prototypical-cached-slot)
  (makes-sense-for 'slotp)
  (unit-inferences '(put-demons `(run-local-demons ,%value% %unit% %slot% %value%)))
  (to-compute-value 'make-local-demons-slot))
;;.@vindex{local-demons (slot)}

;;.Whether created manually or automatically, the inverse slot
;;.@code{associated-inferences-for} is maintained from the inferences
;;.slot to the slot it contains inferences for.
(define-unit local-demons-for
  (english-description "The slot which this slot contains inferences for.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (unit-inferences
   '(put-demons (run-local-demons-on-current-values
		 %unit% %slot% %value%))))
;;.@vindex{local-demons-for (slot)}

;;.Local demons slots can be either created manually or created
;;.automatically by the default method @code{make-local-demons-slot}
;;.whose name is @code{LOCAL-@var{slot}-DEMONS} in the same package as
;;.@var{slot}.@refill
(defun make-local-demons-slot (unit slot)
  "Constructs a slot to store the inferences associated with UNIT."
  (declare (ignore slot))
  (make-internal-unit (fsymbol (symbol-package unit) "LOCAL-~A-DEMONS" unit)
    (works-like 'prototypical-set-slot)))
;;.@findex{make-local-demons-slot}

;;.The associated demons slot can be profitably used with the `list
;;.syntax' for slot names in macros like @code{define-unit} and
;;.@code{make-unit}.  For instance, we can say:
;;.@example
;;.(define-unit red-box
;;.  (member-of 'boxes)
;;.  (color 'red)
;;.  ((local-demons parts) '(assert-value %value% color red)))
;;.@end example
;;.asserting that all the parts of a @code{red-box} have the color
;;.red.


;;;;.Constant Inferences 

;;.Some constant inferences are associated with every slot.  For
;;.instance, whenever the @code{wife} slot is asserted on a unit
;;.@var{u}, we might also assert @code{gender male} about @var{u}
;;.(probably making heterosexist assumptions).  Inferences of this sort
;;.are stored on the @code{inferences} slot of a unit; this slot is a
;;.many valued slot whose elements are inferences of the sort described above
;;.@code{(@var{slot} @var{value} . @var{annotations})}.  These are
;;.passed (with a first argument of the unit being assigned the slot)
;;.to @code{assert-value}.  The value is made to depend on the original
;;.slot being asserted and the corresponding inference stored on
;;.@code{inferences}.

;;.These inferences are implemented by the procedure
;;.@code{infer-forward}, which is stored as a put-demon on slots with
;;.@code{inferences}.  But users never need to worry about this level
;;.of implementation because one of the inferences associated with the
;;.inferences slot is the assertion of an approriate put demon.  This
;;.strategy is used from most of ARLOtje's inference mechanisms;
;;.asserting a description of the inference (e.g. @code{gender male}
;;.follows from @code{wife}) infers the appropriate put-demon, so the
;;.user never needs to know the `put-demon' level implementation of
;;.inference schemes.

;;.The `user level' interface to these mechanisms is the
;;.@code{inferences} slot containing the list of inferences to make
;;.when a slot is asserted.
;;...@example
(define-unit inferences
  (english-description "The inferences associated with this slot.")
  (works-like 'prototypical-set-slot)
  (to-put-value 'demonic-put-value)
  (makes-sense-for 'slotp)
  (must-be 'listp)
  (put-demons '(assert-value %UNIT% PUT-DEMONS 
		`(ASSERT-VALUE %UNIT% ,(car %VALUE%) ,(cadr %VALUE%)))))
;;...@end example


;;;;.Inverses

;;.Inverses are specified by the @code{INVERSE-SLOT} slot; whenever
;;.one slot is asserted as another's inverse a put demon is set up
;;.that stores that corresponding inverse slot.  Furthermore, the
;;.@code{inverse-slot} of @code{inverse-slot} is @code{inverse-slot}
;;.meaning that the corresponding inverse inference is also installed.
;;.Thus, the following definition of @code{inverse-slot} is also an
;;.@emph{example} of @code{inverse-slot}!
;;...@example
(define-unit inverse-slot
  (english-description "The inverse of a slot.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp)
  (put-demons '(assert-value %unit% put-demons
		`(assert-value %VALUE% ,%VALUE% %UNIT%)))
  (inverse-slot 'inverse-slot))
;;...@end example

(define-unit demanded-inverse
  (english-description "The inverse of a slot.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp)
  (to-compute-value 'make-an-inverse))

(defun make-an-inverse (slot is)
  (declare (ignore is))
  (make-unit (fsymbol (symbol-package slot) "INVERSE-OF-~A" slot)
    (works-like 'prototypical-set-slot)))


;;;;.Generic Inheritance

;;.ARLOtje uses a version of the `generic inheritance' provided by
;;.CYCL.  Generic inheritance allows the specification --- for a given
;;.unit and slot --- of inferences to be made for all members of
;;.that slot.  Whenever that unit is given a value for that slot
;;.(possibly one of several values if the slot is many-valued) the
;;.corresponding inferences are made.  For instance, one common use is
;;.with the `collections' implemented by ARLOtje; we can say of a
;;.particular collection (for instance, @code{Quakers}) that all of its
;;.@code{members} have the property `@code{favorite-food oatmeal}';
;;.thus when a given value (for instance, @code{Ken}) is added to the
;;.@code{members} slot of @code{Quakers}, it is assigned the
;;.@code{favorite-food} slot @code{oatmeal}.

;;.Inferences like these are called `associated inferences'; such
;;.inferences are asserted by using the @code{associated-inferences}
;;.slot of the corresponding slot.  For instance, the
;;.@code{associated-inferences} slot of @code{members} (in ARLOtje) is
;;.the slot @code{members-have}; to assert that all Quakers have the
;;.favorite food @code{oatmeal}, we would say:
;;.@example
;;.(assertion 'quakers 'members-have '(favorite-food oatmeal))
;;.@end example

;;.Associated inferences are implemented on top of local put demons
;;.@pxref{Local demons} and have put demons which assert appropriate
;;.local put demons on the appropriate unit.

(define-unit associated-inferences
  (english-description
   "The slot containing the slot containing inferences regarding a slot.")
  (works-like 'prototypical-cached-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (to-compute-value 'make-associated-inferences-slot))
;;.@vindex{associated-inferences (slot)}

(define-unit associated-inferences-for
  (english-description "The slot which this slot contains inferences for.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (put-demons '(assert-value %unit% put-demons
		`(assert-value %unit% (get-value ,%value% local-demons)
		  `(assert-value %value% ,(car %value%) ',(cadr %value%)))))
  (inverse-slot 'associated-inferences))
;;.@vindex{associated-inferences-for (slot)}

(defun make-associated-inferences-slot (unit slot)
  "Constructs a slot to store the inferences associated with UNIT."
  (declare (ignore slot))
  (make-internal-unit (fsymbol (symbol-package unit) "INFERENCES-ABOUT-~S" unit)
    (works-like 'prototypical-set-slot)))
;;.@findex{make-associated-inferences-slot}

;;.The associated inferences slot can be profitably used with the `list
;;.syntax' for slot names in macros like @code{define-unit} and
;;.@code{make-unit}.  For instance, we can say:
;;.@example
;;.(define-unit red-box
;;.  (member-of 'boxes)
;;.  (color 'red)
;;.  ((parts associated-inferences) '(color red)))
;;.@end example
;;.asserting that all the parts of a box have the color red.  A more
;;.general way to do this would be to combine the parts inference with
;;.the members inference:
;;.@example
;;.(define-unit red-things
;;.  (member-of 'collections)
;;.  (generalizations 'colored-things)
;;.  (members-have '(color red))
;;.  (members-have '((parts associated-inferences) '(member-of 'red-things))))
;;.@end example
;;.causing the membership in red things to inherit through the parts relation.


;;;;.Slot Generalizations

;;.`Genl slots' are a useful inference mechanism stolen from CYCL.  The
;;.idea is that some slots have generalizations such that all the
;;.members of the slot are also members of the slots generalizations.
;;.Thus, @code{BROTHERS} has the `genl slot' @code{SIBLINGS}.
;;.Genl slots are specified by the @code{genl-slots} slot and
;;.the put demon on @code{genl-slots} simply adds a put demon to the
;;.subordinate slot to assert its superior.

(define-unit genl-slots
  (english-description "The slots of which this slots values are a subset.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp)
  (inverse-slot 'spec-slots)
  (put-demons '(assert-value %unit% put-demons
		`(assert-value %unit% ,%value% %value%))))
;;.@vindex{genl-slots (slot)}

(define-unit spec-slots
  (english-description "The slots of which this slots values are a superset.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp))
;;.@vindex{spec-slots (slot)}

;;.Couldn't do this above because GENL slots hadn't been defined yet.
(assertion 'demanded-inverse 'genl-slots 'inverse-slot)


;;;;.Pushes-through

;;.The @code{pushes-through} slot asserts that the value of a unit's
;;.@var{foo} slot @emph{pushes through} to be assigned to values of
;;.the unit's @var{bar} slot.  For instance, if the @code{color} slot
;;.@emph{pushes-through} the @code{looks-like} slot this means that
;;.whenever a @code{color} is asserted for a unit, the same color is
;;.asserted on each member of the unit's @code{looks-like} slot.
;;.Similarly, whenever a new entry is added to the @code{looks-like}
;;.slot of a unit, that entry is given the same @code{color} slot as
;;.@code{unit}.

(define-unit pushes-through
  (english-description
   "This stores the slot through which another slot pushes its values.")
  (works-like 'prototypical-set-slot)
  (put-demons '(assert-value %unit% put-demons
		`(assert-on-each %unit% ,%value% ,%unit% %value%))))
;;.@vindex{pushes-through (slot)}

(define-unit pushes-slots
  (english-description
   "This stores the slots which push through this slot.")
  (works-like 'prototypical-set-slot)
  (put-demons '(assert-value %unit% put-demons
		`(assert-each-value %unit% ,%value% %value% ,%value%)))
  (inverse-slot 'pushes-through))
;;.@vindex{pushes-slots (slot)}

;;.Slots which push through other slots use two special functions in
;;.the put demons they use to make inferences.  @code{Assert-on-each}
;;.takes a unit and slot (specifying a set of values) and a slot and
;;.value and asserts the latter slot and value on each of the elements
;;.of the set of values specified by the first unit and slot.
;;.@code{Assert-each-value} takes a similar unit and slot (specifying
;;.a set of values) and another unit and slot, storing each element of
;;.the set specified by the first as a value on the latter unit and slot.

;;.For example,
;;.@example
;;.(assert-on-each 'red-box 'parts 'color 'red)
;;.@end example
;;.asserts a COLOR of RED on each of the PARTS of RED-BOX while an
;;.expression like:
;;.@example
;;.(assert-on-each 'bill 'brothers 'john 'friends)
;;.@end example
;;.makes each of Bill's brothers a friend of John.

(defun assert-on-each (u s slot value)
  (do-members (elt u s)
    (assert-value elt slot value)))

(defun assert-each-value (u s unit slot)
  (do-members (elt u s)
    (assert-value unit slot elt)))


;;;;.Eager composition

(define-unit eager-composition-of
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (put-demons '(assert-value (cadr %value%) 'put-demons
		`(assert-each-value %value% ',(car %value%) %unit% ,%unit%)))
  (put-demons '(assert-value (car %value%) 'put-demons
		`(assert-on-each %unit% ,(get-value (cadr %value%) 'demanded-inverse)
		  ,%unit% %value%))))


;;;;.Structures

;;.Often we know certain knowledge which relates several slots of a
;;.single unit.  For instance, suppose we are representing a human
;;.body.  We know that the head of the body is attached to the neck,
;;.that the neck is attached to the torso, and so forth.  Similarly, if
;;.we are representing a story about going to a restaurant, we know
;;.that (usually) ordering the food precedes getting the food which
;;.precedes paying for the food.  In ARLOtje, these are said to be
;;.`part relations' about particular slots.  We can say for instance,
;;.that the @code{human-head} slot has the relation @code{connected-to}
;;.to the @code{human-neck} slot; whenever we construct a description
;;.of a human head and place it in the @code{human-head} slot, it will
;;.be inferred as @code{connected-to} the contents of the same
;;.description's @code{human-neck} slot (if it exists).  If it does
;;.not, whenever such a @code{human-neck} slot is defined, the relation
;;.will be established.

;;.Structural inferences are specified with the @code{structure} slot
;;.The format of a structure specification is a two-element list
;;.@code{(@var{relation} @var{part-slot})}.  If some slot
;;.@var{first-part} were to have such a @code{structure} specification,
;;.it would mean that asserting a @code{first-slot} of @var{v} on unit
;;.@var{u} would result in asserting on @code{v} the slot
;;.@code{relation} with a value of the @code{part-slot} of @code{u}.
;;.(This needs to be clearer, but I've got to think about it some more.)

(defun has-slotp (unit slot)
  (not (failurep (get-value unit slot))))

(define-unit structure
  (english-description "The relations this slot has to other slots.")
  (works-like 'prototypical-set-slot)
  (put-demons
   '(if (has-slotp (car %value%) inverse-slot)
     (assert-value (cadr %value%) 'structure
      `(,(get-value (car %value%) 'inverse-slot) ,%unit%))
     (assert-value (cadr %VALUE%) PUT-DEMONS
      `(ASSERT-ON-EACH %UNIT% ,%UNIT% ,(car %VALUE%) %VALUE%))))
  (put-demons
   '(assert-value %UNIT% PUT-DEMONS
     `(ASSERT-EACH-VALUE %UNIT% ,(cadr %VALUE%) %VALUE% ,(car %VALUE%)))))

;;.The procedure @code{assert-on-each} is used to implement structural
;;.inferences; it maps over all the values of a slot and stores
;;.another value on each element.


;;;;.Exclusion inferences

(define-unit denies
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (put-demons '(assert-value %unit% put-demons
		`(signal-denial ,%value% ,%unit% %unit%  %value%))))

(define-unit mutually-denies
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (genl-slots 'denies)
  (inverse-slot 'mutually-denies))

(defun signal-denial (slot from unit value)
  (when (check-value unit slot value)
    (unwind-protect (cerror "Retract ~*~S ~S ~S"
			    "Contradiction of ~S with ~S ~S ~S"
			    from unit slot value)
      (retract-value unit slot value))))


;;.@section Backwards Inferences

;;.ARLOtje's backwards inferences generally operate through the
;;.@code{to-compute-value} slots of slots.  Most are simply implemented
;;.as default methods which access parts of the slot description to
;;.figure out what to do (for instance, what slot to inherit through,
;;.which question to ask, etc).  These backwards inferences often use
;;.forward inferences at the `meta level' to set up the appropriate
;;.@code{to-compute-values}; for instance, asserting a
;;.@code{transfers-through} slot (below) will assert a
;;.@code{to-compute-value} slot of @code{transfer-slot} to use the
;;.@code{transfers-through} slot.

(put-value 'to-compute-value 'put-demons
	   '(if (not (many-valued-slotp %unit%))
	     (progn
	      (assert-value %unit% value-in-slot default-value)
	      (assert-value %unit% accumulates-in-slot default-value))))


;;;;.Per-slot defaults
 
;;.One common default method is to provide a constant default for a
;;.particular slot.  For instance, the default value of the
;;.@code{religion} slot might be @code{buddhist}.  This would be
;;.asserted by giving @code{religion} a @code{default-value} slot of
;;.@code{buddhist}.  Assigning this slot will also give the slot a
;;.@code{to-compute-value} slot of @code{use-slot-default} which simply
;;.returns the slot's @code{default-value} slot.

(define-unit value-defaults-to
  (english-description "The default for this slot, used by USE-SLOT-DEFAULT.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'anythingp)
  (inferences '(to-compute-value use-slot-default)))

(defun use-slot-default (unit slot)
  "A default method which gets the DEFAULT-VALUE slot of a slot as a default.
E.G. if INCOME were a slot with USE-SLOT-DEFAULT as a TO-COMPUTE-VALUE,
and DEFAULT-VALUE of INCOME were 20000, (get-value 'kwh 'income) would
yield a default of 20000."
  (declare (ignore unit))
  (get-value slot 'value-defaults-to))


;;;;.Auto filling slots

;;.Another simple default method is @code{fill-in-slot} which simply
;;.constructs a unit known to be a slot and makes it a @code{member-of}
;;.the type constraint on the slot in the first place.  Thus, if the
;;.@code{father} slot (which @code{must-be} a member of @code{Men})
;;.were to have @code{fill-in-slot} as its default method, the assumed
;;.father of an individual would be created on the fly as a
;;.@code{member-of} the collection @code{Men}.

(defun fill-in-slot (unit slot)
  "Creates a unit to fill in a slot, asserting its membership constraints appropriately."
  (declare (ignore unit))
  (make-unit (gensymbol slot)
    ;; This assumes the implementation of MEMBER-OF yet to come.
    (member-of (get-value slot 'must-be))))


;;;;.Mapped slots

;;.Mapped slots compute their value by mapping a function over the
;;.values of another slot and combining the results with some
;;.function.  Such slots are specified to operate on their
;;.@code{operate-on-slot} slot, applying their
;;.@code{operation-per-element} slot to each value of the
;;.corresponding slot and applying their @code{combine-results-by}
;;.slot to merge the corresponding results.  By default, the
;;.@code{operation-per-element} slot is the identity and the
;;.@code{combine-results-by} slot is the @code{list} function.
;;.The values of mapped slots are computed by the procedure
;;.@code{operate-and-combine}.

;;.Mapped slots operate on the slot in their @code{operate-on-slot}
;;.slot; this may be either a single valued or many valued slot.  If
;;.it is single valued, the combining function is applied to that
;;.single value.
(define-unit operate-on-slot
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inferences '(to-compute-value operate-and-combine)))
;;.@vindex{operate-on-slot (slot)}

;;.The operation stored in @code{operation-per-element} can be either
;;.the name of a lisp function or an ARLOtje slot; if it is a slot,
;;.that slot is fetched from each value as the corresponding operation.
(define-unit operation-per-element
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'function-namep)
  (value-defaults-to 'identity)
  (inferences '(to-compute-value operate-and-combine)))
;;.@vindex{operation-per-element (slot)}

;;.The results of each operation are combined by the n-ary function
;;.stored in the slot's @code{combine-results-by} slot.  Several
;;.useful combiners (@code{lintersection}, @code{lunion},
;;.@code{compute-average}) are provided by ARLOtje.  In addition, any
;;.of LISP's regular n-ary combiners (@code{list}, @code{+}, @code{*},
;;.etc) may also be used.
(define-unit combine-results-by
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'function-namep)
  (value-defaults-to 'list)
  (inferences '(to-compute-value operate-and-combine)))
;;.@vindex{combine-results-by (slot)}

(defun operate-and-combine (unit slot)
  "Computes a slot value by mapping an operation over a slot's values and combining the results."
  (let ((operation (get-value slot 'operation-per-element))
        (combiner  (get-value slot 'combine-results-by))
        (source    (get-value slot 'operate-on-slot)))
    (apply combiner 
           (if (slotp operation)
             (map-members #'(lambda (x) (get-value x operation)) unit source)
             (map-members operation unit source)))))
;;.@findex{operate-and-combine}

;;.For instance, suppose the slot @code{names} contained a list of
;;.strings; we could define the corresponding list of their lengths by
;;.the slot:
;;.@example
;;.(define-unit name-lengths
;;.  (works-like 'prototypical-slot)
;;.  (operate-on-slot 'names)
;;.  (operation-per-element 'length))
;;.@end example
;;.alternatively, to get the total length of all the names together,
;;.we might say:
;;.@example
;;.(define-unit total-name-lengths
;;.  (works-like 'prototypical-slot)
;;.  (operate-on-slot 'names)
;;.  (operation-per-element 'length)
;;.  (combine-results-by '+))
;;.@end example
;;.which adds the returned lengths.  Or using the ARLOtje function
;;.@code{COMPUTE-AVERAGE}:
;;...@example
(defun compute-average (&rest numbers)
  "Computes the arithmetic mean of N numbers."
  (/ (apply #'+ numbers) (length numbers)))
;;...@end example
;;.we could define the average name length by
;;.@example
;;.(define-unit name-lengths
;;.  (works-like 'prototypical-slot)
;;.  (operate-on-slot 'names)
;;.  (operation-per-element 'length)
;;.  (combine-results-by 'aj::compute-average))
;;.@end example

(defun lunion (&rest lists)
  "Returns the union (under EQL) of N lists."
  (reduce 'union lists))

(defun lintersection (&rest lists)
  "Returns the intersection (under EQL) of N lists."
  (reduce 'intersection lists))


;;;; Composed Slots

;;.Slots can be defined as the composition of other slots.  Asserting
;;.that slot is @code{composition-of} two other slots (in a list
;;.@code{(@var{f} @var{g})}) will
;;.give the slot a @code{to-compute-value} slot of
;;.@code{compute-composition} which computes the composition of the
;;.slots @code{f} and @code{g}.  I.e. it returns all the results of
;;.applying @code{f} to all the results of applying @code{g}.

(defun pair-of-slotsp (x)
  (and (listp x) (= (length x) 2) (slotp (first x)) (slotp (second x))))

(define-unit composition-of
  (english-description "The slots which this slot is a composition of.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'pair-of-slotsp)
  (inferences '(to-compute-value compute-composition)))

(defun compute-composition (unit slot)
  "As a TO-COMPUTE-VALUE this computes a composition of two other slots."
  (let* ((f.g (get-value slot 'composition-of)) (f (first f.g)) (g (second f.g)))
    (if (or (many-valued-slotp f) (many-valued-slotp g))
	(let ((results '()))
	  (do-members (middle unit g)
	    (do-members (elt middle f)
	      (pushnew elt results)))
	  results)
      (let ((middle (get-value unit g)))
	(if (failurep middle) middle (get-value middle f))))))


;;;;.Kleene Stars

;;.The Kleene star is the transitive closure of another slot.  For
;;.instance, if @code{subset-of} is a slot, the kleene star of
;;.@code{subset-of} is the set of all known sets which are supersets
;;.of (or identical to) a given one.  So if fettucine are a subset of
;;.noodles are a subset of pasta are a subset of italian food, the
;;.kleene star of @code{subset-of} of fettucine are all of these sets.
;;.You can define a kleene star manually building a slot and asserting
;;.@code{kleene-star-of} for it; or you can build one automatically by
;;.doing a @code{get-value} of @code{kleene-star} of an existing slot.

(define-unit Kleene-Star
  (english-description "The kleene star of this slot.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (to-compute-value 'make-kleene-star-slot)
  (must-be 'slotp))

(define-unit Kleene-Star-Of
  (english-description "The slot of which this slot is a kleene star.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inverse-slot 'Kleene-Star)
  (inferences '(value-in-slot kleene-star-value))
  (inferences '(to-check-value check-kleene-star)))

(define-unit kleene-star-value
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'compute-kleene-star))
(defun compute-kleene-star (unit slot)
  "Computes the transitive closure of an annotated value."
  (declare (ignore slot))
  (dependency! unit)
  (let ((unit (annotated-value-unit unit))
	(slot (get-value (annotated-value-slot unit) 'kleene-star-of)))
    (let ((closure (list unit)) (queue (list unit)))
      (loop (if (null queue) (return closure))
	(let ((top (car queue)))
	  (setq queue (cdr queue))
	  (dependency! (%get top slot))
	  (dolist (item (get-value (%get top slot) 'recorded-elements))
	    (when (not (invalidated-p item))
	      (dependency! item)
	      (let ((item (element-value item)))
		(unless (member item closure)
		  (push item closure) (push item queue))))))))))
(defun check-kleene-star (unit kslot value)
  (declare (ignore slot))
  (if (eq unit value)
      (assert-value (annotated-value unit kslot) 'recorded-elements value)
    (or (find-recorded-element unit kslot value)
	(let ((queue (list (list unit)))
	      (slot (get-value kslot 'kleene-star-of))
	      (marks '()))
	  (loop (if (null queue) (return nil))
	    (let ((top (pop queue)))
	      (push (car top) marks)
	      (let* ((dep (find-recorded-element (car top) slot value)))
		(if dep
		    (let ((local-desc
			   (make-element (%get unit kslot) value)))
		      (%push (%get unit kslot) 'recorded-elements local-desc)
		      (put-value local-desc 'depends-on (cons dep (cdr top)))
		      (return local-desc))
		  (unless (%hasnt (car top) slot)
		    (dolist (elt (%get-default (%get (car top) slot) 'recorded-elements '()))
		      (unless (or (invalidated-p elt) (member (element-value elt) marks))
			(push (list* (element-value elt) elt (cdr top))
			      queue))))))))))))

(defun make-kleene-star-slot (base-slot slot)
  "Constructs a kleene star slot for another slot."
  (declare (ignore slot))
  (make-unit (fsymbol (symbol-package base-slot) "KLEENE-STAR-OF-~A" base-slot)
    (works-like 'prototypical-set-slot)))


;;;;.Kleene Plus

;;.The Kleene plus is the transitive closure of another slot which
;;.excludes the unit from which it is extracted.  For instance, the
;;.kleene plus of @code{parents} would be @code{ancestors} You can
;;.define a kleene plus manually building a slot and asserting
;;.@code{kleene-plus-of} for it; or you can build one automatically by
;;.doing a @code{get-value} of @code{kleene-plus} of an existing slot.

(define-unit Kleene-Plus
  (english-description "The kleene plus of this slot.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (to-compute-value 'make-kleene-plus-slot)
  (must-be 'slotp))

(define-unit Kleene-Plus-Of
  (english-description "The slot of which this slot is a kleene plus.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inverse-slot 'Kleene-Plus)
  (inferences '(value-in-slot kleene-plus-value))
  (inferences '(to-check-value check-kleene-plus)))
(define-unit kleene-plus-value
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'compute-kleene-plus))
(defun compute-kleene-plus (unit slot)
  "Computes the transitive closure of an annotated value."
  (declare (ignore slot))
  (dependency! unit)
  (let ((unit (annotated-value-unit unit))
	(slot (get-value (annotated-value-slot unit) 'kleene-plus-of)))
    (let ((closure '()) (queue (list unit)))
      (loop (if (null queue) (return closure))
	(let ((top (car queue)))
	  (setq queue (cdr queue))
	  (dependency! (%get top slot))
	  (dolist (item (get-value (%get top slot) 'recorded-elements))
	    (when (not (invalidated-p item))
	      (dependency! item)
	      (let ((item (element-value item)))
		(unless (member item closure)
		  (push item closure) (push item queue))))))))))
(defun check-kleene-plus (unit kslot value)
  (declare (ignore slot))
  (or (find-recorded-element unit kslot value)
      (let ((queue (list (list unit)))
	    (slot (get-value kslot 'kleene-plus-of))
	    (marks '()))
	(loop (if (null queue) (return nil))
	  (let ((top (pop queue)))
	    (push (car top) marks)
	    (let* ((dep (find-recorded-element (car top) slot value)))
	      (if dep
		  (let ((local-desc (make-element (%get unit kslot) value)))
		    (%push (%get unit kslot) 'recorded-elements local-desc)
		    (put-value local-desc 'depends-on (cons dep (cdr top)))
		    (return local-desc))
		(unless (%hasnt (car top) slot)
		  (dolist (elt (get-value (%get (car top) slot) 'recorded-elements))
		    (unless (or (invalidated-p elt) (member (element-value elt) marks))
		      (push (list* (element-value elt) elt (cdr top))
			    queue)))))))))))

(defun make-kleene-plus-slot (base-slot slot)
  "Constructs a kleene plus slot for another slot."
  (declare (ignore slot))
  (make-unit (fsymbol (symbol-package base-slot) "KLEENE-PLUS-OF-~A" base-slot)
    (works-like 'prototypical-set-slot)))
					 

;;;;.Transfers Through

;;.One straightforward inheritance mechanism is inheritance through a
;;.hierarchy or lattice; when a slot value is needed, the graph
;;.determined by another relation is searched for objects with a value
;;.for the required slot.  These values are accumulated and returned as
;;.a value for the required slot.  When a slot @var{s} searches for values
;;.through the graph determined by the slot @var{t}, we say that
;;.@var{s} @dfn{transfers through} the slot @var{t}.@refill 

;;.In ARLOtje, we make one slot transfer through another by asserting
;;.the @code{transfers-through} property or its inverse
;;.@code{transfers-the-slot}.  These will both lead to the
;;.@code{to-compute-value} of the appropriate slot being initialized to
;;.the procedure @code{transfer-slot}.
;;.@vindex{transfers-through (slot)}
(define-unit transfers-through
  (english-description "This stores the similarity slot which a slot transfers through.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inferences '(to-compute-value transfer-slot)))

;;.@code{Transfers-through} is actually a many valued slot, so a slot
;;.may transfer through several different slots; the ordering in which
;;.they are considered is implemented by the procedure
;;.@code{transfer-slot} (or actually, by the procedure
;;.@code{maptransfers}, but who's counting...)
;;.@vindex{transfers-the-slot (slot)}
(define-unit transfers-the-slots
  (english-description "This stores the slots which transfer through this one.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inverse-slot 'transfers-through))

(defun transfer-slot (unit slot)
  "Inherits SLOT of UNIT through SLOT's TRANSFERS-THROUGH parents."
  (block transferring
    (do-members (through slot 'transfers-through)
      (do-members (parent unit through)
	(let ((match (get-value parent slot)))
	  (unless (failurep match) (return-from transferring match)))))))


;;;;.COPIES-THROUGH slots

(define-unit copies-through
  (english-description "This stores the similarity slot which a slot transfers through.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inferences '(to-compute-value copy-slot)))

(define-unit copies-the-slots
  (english-description "This stores the slots which copy through this one.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp)
  (inverse-slot 'copies-through))

(define-unit copy-map
  (english-description "A mapping of originals to copies for a similarity mapping.")
  (works-like 'prototypical-slot)
  (to-compute-value 'make-copy-map)
  (makes-sense-for 'slotp)
  (must-be 'slotp))

(defun make-copy-map (for-slot copy-map-slot)
  "Makes a copy map for structure mapping."
  (declare (ignore copy-map-slot))
  (make-unit (fsymbol (symbol-package for-slot) "~A-MAPPINGS" for-slot)
    (works-like 'prototypical-set-slot)))

(defun copy-slot (unit slot)
  "Constructs copies of inherited values."
  (let* ((through (get-value slot 'copies-through))
	 (parent (get-value unit through))
	 (transfer (if (failurep parent) parent (get-value parent slot))))
    (if (failurep transfer) transfer
      (let* ((map-slot (get-value through 'copy-map))
	     (already-copied (assoc transfer (get-value unit map-slot))))
	(if already-copied (cadr already-copied)
	  (let ((copy (make-internal-unit (gensymbol "~A-COPY" transfer))))
	    (assert-value copy through transfer)
	    (assert-value unit map-slot (list transfer copy))
	    copy))))))

;;;;.Multiple methods

(define-unit methods
  (works-like 'prototypical-set-slot)
  (must-be 'listp)
  (makes-sense-for 'slotp)
  (inferences '(to-compute-value apply-multiple-methods)))

(define-unit local-methods
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'annotated-valuep)
  (must-be 'listp))

(defmacro annotate-value (unit slot annotation-slot annotation-value)
  "Stores an annotation for SLOT of UNIT."
  (assert-value (annotated-value unit slot) annotation-slot annotation-value))

(defun run-method (method unit slot)
  "Evaluates PUT-DEMON with appropriate interpretations to UNIT, SLOT."
  (block fail
    (labels ((evaller (x)
	       (cond ((null x) x)		     
		     ((listp x)
		      (cond ((eq (car x) 'quote) (cadr x))
			    ((macro-function (car x))
			     (evaller (macroexpand x)))
			    ((eq (car x) 'if)
			     (if (evaller (cadr x)) (evaller (caddr x))
			       (evaller (cadddr x))))
			    ((eq (car x) 'get-value)
			     (let ((unit (evaller (cadr x)))
				   (slot (evaller (caddr x))))
			       (if (computing? get-value unit slot)
				   (return-from fail (fail unit slot))
				 (let ((result (get-value unit slot)))
				   (if (failurep result) (return-from fail result)
				     result)))))
			    ((eq (car x) 'query)
			     (let ((unit (evaller (cadr x)))
				   (slot (evaller (caddr x)))
				   (value (evaller (cadddr x))))
			       (if (computing? check-value unit slot value)
				   (return-from fail NIL)
				 (check-value unit slot value))))
			    (t (let ((result (apply (car x) (mapcar #'evaller (cdr x)))))
				 (if (failurep result) (return-from fail result) result)))))
		     ((eq x '%UNIT%) unit)
		     ((eq x '%SLOT%) slot)
		     (T x))))
      (evaller method))))

(defun apply-multiple-methods (unit slot)
  "Applies multiple methods --- both local and global --- to compute a slot value."
  (let ((local-value (apply-local-methods unit slot)))
    (if (failurep local-value) (apply-slot-methods unit slot)
      local-value)))

(defun apply-slot-methods (unit slot)
  "Applies a variety of methods to compute a slot value."
  (let ((attempt nil))
    (dolist (method (get-value slot 'methods) attempt)
      (setq attempt (run-method method unit slot))
      (unless (failurep attempt) (return attempt)))))

(defun apply-local-methods (unit slot)
  "Applies a the local (per-value) methods to compute a slot value."
  (let ((attempt nil))
    (dolist (method (get-value (annotated-value unit slot) 'local-methods)
	     (or attempt (fail unit slot)))
      (setq attempt (run-method method unit slot))
      (unless (failurep attempt) (return attempt)))))




;;;;.Constraint methods

;;.@dfn{Constraint methods} allow the specification of several methods
;;.for computing slot values depending on what knowledge is available.
;;.Each constraint method specifies a combiner and a set of slots; if
;;.all of the slots are currently defined, the combiner is called on
;;.their values.  If some slot is undefined or the combiner itself
;;.fails, another method is selected.

;;.Constraint methods are stored on the @code{Constraint-methods} slot
;;.and each consists of a combiner followed by a list of slot names.
(define-unit constraint-methods
  (english-description "These are methods for computing this slot.")
  (works-like 'prototypical-set-slot)
  (inferences '(to-compute-value try-multiple-methods))
; Needs to be fixed to work without recursion checking.
; (inferences '(to-put-value check-other-constraints))
  )
;;.@vindex{Constraint-Methods (slot)}

;;.These methods are used by @code{try-multiple-methods} which looks at
;;.each method and tries to evaluate it; if this succeeds, it returns
;;.the result.  @code{Try-multiple-methods} interacts with the
;;.dependency tracking mechanism in a special way; the function
;;.@code{evaluate-method} returns the result of evaluating a form and
;;.the dependencies generated in that evaluation.  If it was
;;.successful, the variable @code{*current-computation*} is bound to
;;.this list of dependencies; otherwise, the result (a failure token)
;;.is simply returned.
(defun try-multiple-methods (unit slot)
  "Selects and applies a constraint method to compute the value of a slot."
  (if *recursion-checking*
      (dolist (method (get-value slot 'constraint-methods) (fail unit slot))
	(multiple-value-bind (result dependencies) (evaluate-method unit method)
	  (unless (failurep result)
	    (setq *current-computation* dependencies)
	    (return result))))
    (let ((*recursion-checking* T))
      (computing (get-value unit slot)
	 (dolist (method (get-value slot 'constraint-methods) (fail unit slot))
	   (multiple-value-bind (result dependencies) (evaluate-method unit method)
	     (unless (failurep result)
	       (setq *current-computation* dependencies)
	       (return result))))))))
;;.@findex{try-multiple-methods}

;;.The function @code{evaluate-method} diddles with the dependency
;;.tracking mechanism by first saving the current `dependency state',
;;.then binding a new one (by fluidly binding
;;.@code{*current-computation*}) and then evaluating (using
;;.@code{careful-eval}) the method in this context.  When it returns,
;;.the saved value is restored to @code{*current-computation*} and the
;;.accumulated @code{*current-computation*} is returned as a second value.
(defun evaluate-method (unit method)
  "Evaluates METHOD to compute SLOT of UNIT."
  (block evaluating-single-method
    (let ((support *current-computation*))
      (let ((*current-computation* support))
	(flet ((evaluate-parameter (param)
		 (if (computing? get-value unit param)
		     (return-from evaluating-single-method (fail unit param))
		   (let ((value (get-value unit param)))
		     (if (failurep value)
			 (return-from evaluating-single-method value)
		       value)))))
	  (let ((result (apply (first method)
			       (mapcar #'evaluate-parameter (rest method)))))
	    (psetq support               *current-computation*
		   *current-computation* support)
	    (values result support)))))))
;;.@findex{evaluate-method}

;;.The put function for constrained slots,
;;.@code{check-other-constraints}, tries to compute a value for a slot
;;.before storing a value; if this succeeds and yields a different
;;.value, somebody's constraints are messed up and an error is
;;.signalled.
(defun check-other-constraints (unit slot value &rest annotations)
  "Checks that an asserted value does not violate any current constraints."
  (if (or (computing? get-value unit slot)
	  (failurep (get-value unit slot))
	  (not (get-value (%get unit slot) 'depends-on))
	  (equal (get-value unit slot) value))
      (apply #'annotated-value-put unit slot value annotations)
    (progn (error "Violates existing constraints which assert that ~S of ~S is ~S"
		  slot unit (get-value unit slot)))))
;;.@findex{check-other-constraints}

;;.For instance, we might define the constraints between slots
;;.describing a subtraction thus:
;;...@example
(define-internal-unit DIFF
  (works-like 'prototypical-slot)
  (constraint-methods '(- X1 X2)))
;;...@end example
;;.stores the difference of two numbers stored in the @code{X1} and
;;.@code{X2} slots of the same unit; if you need to know the
;;.@code{DIFF} slot, you can compute it by subtracting those slots:
;;...@example
(define-internal-unit X1
  (works-like 'prototypical-slot)
  (constraint-methods '(+ DIFF X2)))

(define-internal-unit X2
  (works-like 'prototypical-slot)
  (constraint-methods '(- X1 DIFF)))
;;...@end example
;;.We can then define an example:
;;...@example
(define-internal-unit DIFF1
  (DIFF 5)
  (X1 7))
;;...@end example
;;.And do
;;.@example
;;.(get-value 'diff1 'x2) ===> -2
;;.(retraction 'diff1 'diff)
;;.(get-value 'diff1 'x2) ===> #<Can't compute X1 of DIFF1>
;;.(assertion 'diff1 'x2 5)
;;.(get-value 'diff1 'x2) ===> 2
;;.@end example


;;;;.Transitive slots

;;.Transitive slots are slots which store a graph as a set of local
;;.connections; @code{get-value} on a node returns the local
;;.connections, but @code{check-value} does a search of the graph to
;;.see if two points are connected.  @code{Transitive-slot} is a
;;.prototypical slot whose @code{to-check-value} slot is the function
;;.@code{check-transitive-closure}.  It is useful for implementing
;;.various transitive relations.

(define-unit transitive-slot
  (works-like 'prototypical-set-slot)
  (value-in-slot 'transitive-closure)
  (to-check-value 'check-transitive-closure))

(define-unit transitive-closure
  (works-like 'prototypical-cached-slot)
  (to-compute-value 'compute-transitive-closure))
(defun compute-transitive-closure (unit slot)
  "Computes the transitive closure of an annotated value."
  (declare (ignore slot))
  (let ((unit (annotated-value-unit unit))
	(slot (annotated-value-slot unit)))
    (let ((closure '()) (queue (list unit)))
      (loop (if (null queue) (return closure))
	(let ((top (car queue)))
	  (setq queue (cdr queue))
	  (dependency! (%get top slot))
	  (dolist (item (get-value (%get top slot) 'recorded-elements))
	    (when (not (invalidated-p item))
	      (let ((item (get-value item 'element-value)))
		(unless (member item closure)
		  (push item closure) (push item queue))))))))))
(defun check-transitive-closure (unit slot value)
  (declare (ignore slot))
  (or (find-recorded-element unit slot value)
      (let ((queue (list (list unit)))
	    (marks '()))
	(loop (if (null queue) (return nil))
	  (let ((top (pop queue)))
	    (push (car top) marks)
	    (let ((dep (find-recorded-element unit slot value))) 
	      (if dep
		  (let ((local-desc (make-element unit value)))
		    (%push unit 'recorded-elements local-desc)
		    (put-value local-desc 'depends-on (cons dep (cdr top)))
		    (return local-desc))
		(dolist (elt (get-value (%get (car top) slot) 'recorded-elements))
		  (unless (or (invalidated-p elt) (member (element-value elt) marks))
		    (push (list* (element-value elt) elt (cdr top))
			  queue))))))))))
(put-value 'genl-slots 'kleene-star-of 'genl-slots)
(put-value 'spec-slots 'kleene-star-of 'spec-slots)


;;;;.Restricted Slots

;;.In addition to the generalizations of slots maintained by `genl
;;.slots', ARLOtje provides `restricted versions' of slots which are
;;.specialized to particular domains and ranges.  For instance,
;;.@code{emperor} is a restriction of @code{leader} for a
;;.certain class of nations (or other organizations).  The basic idea
;;.is that emperor is an `alias' for `leader' which applies only to
;;.certain objects.  The in-context relationship is specified by giving
;;.a slot a @code{restriction-of} slot which refers to the `master
;;.slot' from which it gets its value.  Conversely, the more general
;;.slot can be given the specialized implementation as one of its
;;.@code{restricted-versions}.
;;.@vindex{restriction-of (slot)}
;;.@vindex{restrictions (slot)}

(define-unit restrictions
  (english-description "The type restricted versions of this slot.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp))
(define-unit restriction-of
  (english-description "This slot is a domain restricted version of this slot.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be         'slotp)
  (inverse-slot 'restrictions)
  (inferences '(to-get-value restricted-get))
  (inferences '(to-put-value restricted-put))
  (inferences '(to-check-value restricted-check-value)))

;;.In context versions are implemented by special put, get, and check
;;.functions: @code{restricted-get}, @code{restricted-put}, and
;;.@code{restricted-find} which do type checking locally and then pass
;;.on the actual slot operation to the more general slot.
;;.@findex{restricted-get}
;;.@findex{restricted-put}
;;.@findex{restricted-check}
(defun restricted-get (unit slot)
  "Gets a domain and range restricted slot."
  (let ((domain (get-value slot 'makes-sense-for)))
    (if (and (not (failurep domain)) (not (satisfies? unit domain)))
	(error "The ~A slot doesn't make sense for ~A:~%it must pass ~A"
	       slot unit domain)))
  (let ((super-slot (get-value slot 'restriction-of))
	(range-restriction (get-value slot 'must-be)))
    (if (many-valued-slotp slot)
	(remove-if-not #'(lambda (x) (satisfies? x range-restriction))
		       (get-value unit super-slot))
      (if (many-valued-slotp super-slot)
	  (let ((values 
		 (remove-if-not #'(lambda (x) (satisfies? x range-restriction))
				(get-value unit super-slot))))
	    (if (null values) (values unit slot)
	      (if (null (cdr values)) (car values)
		(error "Too many potential values of ~S reduced to ~S from ~S"
		       super-slot slot unit))))
	(let ((value (get-value unit super-slot)))
	  (if (satisfies? value range-restriction) value
	    (fail unit slot)))))))

(defun restricted-check-value (unit slot value)
  "Stores a domain and range restricted slot."
  (check-put-value unit slot value)
  (check-value unit (get-value slot 'restriction-of) value))

(defun restricted-put (unit slot value &rest annotations)
  "Stores a domain and range restricted slot."
  (check-put-value unit slot value)
  (let ((description (apply #'assert-value unit (get-value slot 'restriction-of) value
			    annotations)))
    (run-put-demons unit slot value)
    description))


;;;;.Extracted slots

;;.Extracted slots work by extracting some component of a string
;;.stored in another slot.  The other slot is the @emph{source slot}
;;.of the extracted slot and is stored (unsuprisingly) in the
;;.@code{source-slot} slot of its slot description.  It pulls a
;;.substring out of this source slot using the patterns stored in its
;;.@code{pattern-string} slot by the function @code{string-match}
;;.(described below).  The resulting string is then passed to the
;;.slot's @code{extraction-parser} to return an actual value.

(define-unit source-slot
  (english-description
   "This is the slot from which this slot's value is extracted.")
  (works-like 'prototypical-slot-slot)
  (makes-sense-for 'slotp)
  (must-be 'slotp))
;;.@vindex{source-slot (slot)}

;;.The @code{pattern-string} of a string extraction slot is a many
;;.valued set consisting of @code{patterns}, each of which is a list
;;.of strings and symbols interpreted by @code{string-match}.  String
;;.extractions try each pattern string until a match is found and this
;;.match is then returned.
(define-unit pattern-string
  (english-description
   "These are the pattern strings which specify a substring of another slot.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'slotp)
  (must-be 'listp))
;;.@vindex{pattern-string (slot)}

;;.The @code{extraction-parser} of a slot is called on the string
;;.extracted via the @code{pattern-string}.  This is a function which
;;.returns some object from the given string.  A common version of the
;;.@code{extraction-parser} is @code{read-english-number} which reads
;;.a cardinal english number and returns the corresponding integer.
(define-unit extraction-parser
  (english-description
   "This is the function which parses a value from an extracted substring.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'function-namep)
  (transfers-through 'works-like))
;;.@vindex{extraction-parser (slot)}

;;.The prototypical extraction slot is
;;.@code{prototypical-string-extraction} which provides an
;;.@code{extraction-parser} of the @code{identity} function and a
;;.@code{to-compute-value} function of @code{string-extract} which
;;.appropriately uses the @code{source-slot}, @code{pattern-string}
;;.and @code{extraction-parser} slots.
(define-unit prototypical-string-extraction
  (works-like 'prototypical-slot)
  (to-compute-value 'string-extract)
  (extraction-parser 'identity))
;;.@vindex{prototypical-string-extraction (slot prototype)}

;;.The function @code{string-extract} is the defaulting function for
;;.many string extractions.  It gets the corresponding
;;.@code{source-slot}, searches for a @code{pattern-string} which
;;.matches and then calls the corresponding @code{extraction-parser}.
(defun string-extract (unit slot)
  "Extracts a substring (specified by SLOT's PATTERN-STRINGS) from the
SOURCE-SLOT of SLOT on UNIT and calls SLOT's EXTRACTION-PARSER on this result."
  (let ((patterns (get-value slot 'pattern-string))
	(string-to-match-with (get-value unit (get-value slot 'source-slot))))
    (if (arlotje::failurep string-to-match-with) string-to-match-with
      (dolist (pattern patterns (arlotje::fail unit slot))
	(let* ((match (string-match pattern string-to-match-with))
	       (result (funcall (get-value slot 'extraction-parser) match)))
	  (when result (return result)))))))
;;.@findex{string-extract}

;;.The procedure @code{STRING-MATCH} takes a pattern and a string.
;;.The pattern is a list of strings or symbols which are processed in
;;.order.  The symbols @code{?} and @code{*} are treated specially and
;;.match everything up to the succeeding string in the pattern;
;;.@code{?} is special in that it indicates that the matched substring
;;.is to be returned.
(defun string-match (pattern string)
  "Tries to match PATTERN against STRING."
  (cond ((and (symbolp (car pattern))
	      (equal (symbol-name (car pattern)) "?"))
	 (if (null (cdr pattern)) string
	     (and (search (cadr pattern) string)
		  (subseq string 0 (search (cadr pattern) string)))))
	((and (symbolp (car pattern))
	      (equal (symbol-name (car pattern)) "*"))
	 (and (search (cadr pattern) string)
	      (string-match (cddr pattern)
			    (subseq string (+ (search (cadr pattern) string)
					      (length (cadr pattern)))))))
	((stringp (car pattern))
	 (and (zerop (search (car pattern) string))
	      (string-match (cdr pattern)
			    (subseq string (length (car pattern))))))
	(T nil)))

(defun strip-parenthetical-comments (string)
  "Strips the parenthetical comments out of a string.  Useful for various parsing purposes."
  (let ((left (position #\( string)) (right (position #\) string)))
    (if (and right left)
	(strip-parenthetical-comments
	 (concatenate 'string
		      (subseq string 0 left)
		      (subseq string (+ right 1))))
	string)))


;;;;.Extracting Quantities

;;.The prototype @code{numerical-slot-extraction} provides an
;;.extraction parser of @code{read-english-number}.  
(define-unit numerical-slot-extraction
  (works-like 'prototypical-string-extraction)
  (extraction-parser 'read-english-number))

;;.A special version of this, @code{quantity-slot-extraction}, reads
;;.the quantity expecting to return a certain measure of value (for
;;.instance, kilometers) indicated by a keyword (e.g. @code{:kilometers}); when
;;.the value is parsed by @code{read-english-number}, corresponding
;;.measurement annotations (e.g. `inches', `meters', `furlongs') lead
;;.to corresponding conversions.
(define-unit quantity-slot-extraction
  (works-like 'prototypical-string-extraction)
  (extraction-parser 'read-english-number))

(define-unit measured-in-units
  (works-like 'prototypical-slot)
  (makes-sense-for 'slotp)
  (must-be 'keywordp))

(defun quantity-extract (unit slot)
  "Works as a string extraction which attempts to parse the
corresponding value as a numeric quantity of the type determined
by the slots @code{measured-in-units} slot."
  (let ((patterns (get-value slot 'pattern-string))
	(raw-slot (get-value unit (get-value slot 'source-slot))))
    (if (arlotje::failurep raw-slot) raw-slot
      (let ((string-to-match-with (strip-parenthetical-comments raw-slot)))
	(dolist (pattern patterns (arlotje::fail unit slot))
	  (let ((match (string-match pattern string-to-match-with)))
	    (when match
	      (return (let ((*quantity* (get-value slot 'measured-in-units)))
			(declare (special *quantity*))
			(funcall (get-value slot 'extraction-parser) match))))))))))

(defvar *quantity* nil
  "This is the quantity unit one is parsing to.")

(defvar *atomic-numbers*
  '((one 1) (two 2) (three 3) (four 4) (five 5)
    (six 6) (seven 7) (eight 8) (nine 9) (ten 10)
    (eleven 11) (twelve 12) (thirteen 13) (fourteen 14) (fifteen 15)
    (sixteen 16) (seventeen 17) (eighteen 18) (nineteen 19)
    (twenty 20) (thirty 30) (forty 40) (fifty 50)
    (sixty 60) (seventy 70) (eighty 80) (ninety 90))
  "Numbers that stand by themselves.")

(defvar *multipliers*
  '((hundred 100) (thousand 1000) (million 1000000) (billion 1000000000)
    (score 20) (tenths .1) (tenth .1) (hundredths .01) (hundredth .01)
    (thousandths .001) (thousandth .001) (millionths .000001) (millionth .000001)
    (billionths .000000001) (billionth .000000001)
    (half .5) (fourth .25) (fourths .25) (quarter .25) (quarters .25))
  "Numbers that stand by themselves.")

(defun symbol-equal (s1 s2)
  "Returns true if S1 and S2 have the same print name."
  (string-equal (symbol-name s1) (symbol-name s2)))

(defun read-number-token (string)
  "Reads a space or dash separated token from a string."
  (if (char= (char string 0) #\-) (values '- 1)
    (let* ((string (string-trim " " string))
	   (token-pos (position-if #'(lambda (x) (member x '(#\- #\Space))) string))
	   (token (read-from-string (subseq string 0 token-pos))))
      (values token token-pos))))

(defun cut-string (string start &optional end)
  "Returns a substring of STRING with dashes and spaces stripped off."
  (if (null start) ""
      (string-trim "- " (subseq string start end))))

(defun reading-english-number (string number-so-far)
  "Reads a number in cardinal format."
  (if (= (length string) 0) number-so-far
      (multiple-value-bind (token token-pos) (read-number-token string)
	(cond ((numberp token)
	       (reading-english-number (cut-string string token-pos) (+ number-so-far token)))
	      ((symbol-equal token '-) (- (reading-english-number (cut-string string token-pos) 0)))
	      ((symbol-equal token 'and) (+ number-so-far (reading-english-number (cut-string string token-pos) 0)))
	      ((and (assoc token *atomic-numbers* :test 'symbol-equal)
		    token-pos (char= (elt string token-pos) #\-))
	       (multiple-value-bind (next-token next-pos) (read-number-token (cut-string string token-pos))
		 (reading-english-number
		  (cut-string (cut-string string token-pos) next-pos)
		  (+ number-so-far
		     (cadr (assoc token *atomic-numbers* :test 'symbol-equal))
		     (cadr (assoc next-token *atomic-numbers* :test 'symbol-equal))))))
	      ((assoc token *atomic-numbers* :test 'symbol-equal)
	       (reading-english-number (cut-string string token-pos)
				       (+ number-so-far (cadr (assoc token *atomic-numbers* :test 'symbol-equal)))))
	      ((assoc token *multipliers* :test 'symbol-equal)
	       (reading-english-number (cut-string string token-pos)
				       (* number-so-far (cadr (assoc token *multipliers* :test 'symbol-equal)))))
	      ((get *quantity* (intern (symbol-name token) 'keyword))
	       (* (get *quantity* (intern (symbol-name token) 'keyword)) number-so-far))))))

(defun read-english-number (string)
  (reading-english-number (remove #\, string) 0))

(dolist (translation '((:kilometers :kilometer 1)
		       (:kilometers :km 1)
		       (:kilometers :meters 1/1000)
		       (:kilometers :meter 1/1000)
		       (:kilometers :m 1/1000)
		       (:kilometers :centimeters 1/1000)
		       (:kilometers :centimeter 1/1000)
		       (:kilometers :cm 1/1000)
		       (:kilometers :miles 1.609)
		       (:kilometers :mile 1.609)
		       (:kilometers :mi 1/1000)
		       (:kilometers :nautical-miles 1.852)
		       (:kilometers :nautical-mile 1.852)
		       (:kilometers :nm 1.852)
		       (:kilometers :feet (/ 1 3281.5))
		       (:kilometers :foot (/ 1 3281.5)
		       (:kilometers :ft (/ 1 3281.5)))))
  (setf (get (car translation) (cadr translation)) (caddr translation)))




