;;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@chapter Describing Sensible Slots
(in-package :arlotje)

;;.ARLOtje uses the inverses of @code{makes-sense-for} and
;;.@code{must-be} slots to keep track of possible slots which an
;;.object might legally have.  The slot @code{sensible-slots} is the
;;.direct inverse of @code{makes-sense-for} and the slot
;;.@code{sensible-for-slots} is the direct inverse of @code{must-be}.
;;.The first of these is used by the @code{possible-slots} slot which
;;.contains (and generates by default) the slots which are sensible
;;.for a given object.  This is accomplished by walking the lattice
;;.of tests collecting @code{sensible-slots}.



;;.The @code{sensible-slots} slot is a many valued slot which stores
;;.the inverse of the @code{makes-sense-for} annotations on individual
;;.slots.  Through a bit of bootstrapping magic, these slots are
;;.initialized before the actual inversion is installed.
(define-unit Sensible-slots
  (english-description "The  slots which make sense for members of this collection.")
  (works-like 'prototypical-set-slot))
;;.@vindex{Sensible-slots (slot)}

;;.Similarly the slot @code{sensible-for-slots} slot is a many valued
;;.slot storing the inverse of the @code{must-be} annotations on
;;.individual slots.  Again, a special bootstrapping procedure ensures
;;.that these slots are appropriately initialized before the
;;.implementation of the inverse is actually installed.
(define-unit Sensible-for-slots
  (english-description "The  slots which may have values from this collection.")
  (works-like 'prototypical-set-slot))
;;.@vindex{Sensible-for-slots (slot)}

;;.The @code{possible-slots} slot uses the @code{sensible-slots} slot
;;.to generate the set of slots which a unit can legally have.  It
;;.calls @code{for-tests} to iterate over the tests which an object
;;.satisfies and collects the corresponding sensible slots into a
;;.list.
(define-unit possible-slots
  (english-description "All the possible slots of a unit, based on its MEMBER-OF.")
  (works-like 'prototypical-slot)
  (to-compute-value 'compute-possible-slots))
;;.@vindex{possible-slots (slot)}

;;.The default method for @code{possible-slots} is the function
;;.@code{compute-possible-slots} defined thus:
;;...@example
(defun compute-possible-slots (unit slot)
  "Returns the slots which make sense for the unit UNIT."
  (declare (ignore slot))
  (let ((possibles '()))
    (over-tests #'(lambda (x) (setq possibles (append (get-value x 'sensible-slots) possibles)))
		unit)
    possibles))
;;...@end example
;;.to use the function @code{over-tests} to iterate over all the tests
;;.which the unit satisfies.
;;.@findex{compute-possible-slots (default-method)}

(defun store-makes-sense-for (unit slot value &rest annotations)
  "Stores a MAKES-SENSE-FOR slot and its inverse (SENSIBLE-SLOTS)."
  (declare (ignore annotations))
  (%put unit slot value)
  (put-value value 'sensible-slots unit))
(defun store-must-be (unit slot value &rest annotations)
  "Stores a MUST-BE slot and its inverse (SENSIBLE-FOR-SLOTS)."
  (declare (ignore annotations))
  (%put unit slot value)
  (put-value value 'sensible-for-slots unit))


;;;;.Bootstrapping

;;.ARLOtje bootstraps the representation of sensible slots by marching
;;.over all defined units (and thus slots) and appropriately inverting their
;;.@code{must-be} and @code{makes-sense-for} specifications.

(let ((unit-symbols '()) (units 0) (slots 0) (many-valued-slots 0))
  (do-symbols (symbol *arlotje-package* (mapcar 'declare-unit unit-symbols))
    (when (get symbol 'creation-id)
      (incf units) (push symbol unit-symbols)
      (when (many-valued-slotp symbol)
	(incf many-valued-slots)
	(%put symbol 'value-in-slot (get-slot-method symbol 'value-in-slot))
	(%put symbol 'accumulates-in-slot (get-slot-method symbol 'accumulates-in-slot))
	(%put symbol 'probe-in-slot (get-slot-method symbol 'probe-in-slot)))
      (when (slotp symbol)
	(incf slots)
	(%put symbol 'to-get-value (get-slot-method symbol 'to-get-value))
	(%put symbol 'to-put-value (get-slot-method symbol 'to-put-value))
	(%put symbol 'makes-sense-for (get-slot-method symbol 'makes-sense-for))
	(%put symbol 'must-be (get-slot-method symbol 'must-be))
	(unless (failurep (get-value symbol 'makes-sense-for))
	  (assert-value (get-value symbol 'makes-sense-for) 'sensible-slots symbol))
;                       'depends-on (list (get symbol 'makes-sense-for))
	(unless (failurep (get-value symbol 'must-be))
	  (assert-value (get-value symbol 'must-be) 'sensible-for-slots symbol)))))
;                       'depends-on (list (get symbol 'must-be))
  (format T "~&\;\;\; Bootstrapping with ~D units, ~D slots, ~D many-valued slots,"
	  units slots many-valued-slots))

(put-value 'makes-sense-for 'to-put-value 'store-makes-sense-for)
(put-value 'must-be         'to-put-value 'store-must-be)





