;;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@chapter Foundations
(in-package :arlotje)

;;.The foundational layer of ARLOtje consists of LISP symbols and
;;.property lists accessed by LISP's @code{GET} functions.  ARLOtje
;;.@dfn{units} are symbols with a @code{creation-id} property which
;;.indicates the @emph{session} in which they where first mentioned
;;.(declared).  Since UNITS are LISP symbols they are organized into
;;.packages by Common LISP's package system.  ARLOtje units are
;;.@dfn{internal} or @dfn{external} depending on whether the
;;.corresponding LISP symbol is exported from its package of
;;.definition.

(proclaim '(inline %get %put %get-default fail failurep))


;;;; Kludge for inline coding in dialect that don't obey INLINE declarations.

#+(and MACL CL2)
(define-compiler-macro %get (unit slot)
  "Gets the SLOT propert of UNIT, returning (FAIL unit slot) if it is not defined."
  (let ((u (gensym "unit"))
        (s (gensym "slot")))
    `(progn
       (let ((,u ,unit)
             (,s ,slot))
         (declare (optimize (speed 3) (safety 0))
                  (inline symbolp symbol-plist))
         (values (do ((props (properties ,u) (cdr (cdr props))))
                     ((or (null props) (eq (car props) ,s))
                      (if (null props) (fail ,u ,s) (car (cdr props))))))))))


#+(and MACL CL2)
(define-compiler-macro %put (unit slot value)
  "Stores VALUE on SLOT of UNIT directly."
  (let ((u (gensym "unit"))
        (s (gensym "slot"))
        (v (gensym "value")))
    `(let ((,u ,unit) (,s ,slot) (,v ,value))
      (declare (optimize (speed 3) (safety 0)))
      ;  (check-type slot symbol "a symbol (slot)")
      (do ((props (properties ,u) (cdr (cdr props))))
          ((or (null props) (eq (car props) ,s))
           (if (null props)
             (setf (properties ,u) (list* ,s ,v (properties ,u)))
             (setf (cadr props) ,v)))))))

;;;;.Basic kinds of units.

(defun symbol-unitp (x)
  (and (symbolp x) (not (%hasnt x 'creation-id))))
(deftype symbol-unit () '(satisfies symbol-unitp))

;; Anonymous units
(defstruct (whonit (:predicate whonitp) (:print-function print-whonit)
		   (:constructor make-whonit ()))
  (plist '()))

(defun print-whonit (whonit stream &rest options)
  "Prints an anonymous unit."
  (declare (ignore whonit stream options))
  (format stream "#<Anonymous unit>"))

(defun whonit-get (thing slot)
  "Gets a slot from a whonit, acting specially if the whonit caches the slot."
  (if (symbolp thing) (%get thing slot)
    (if (typep thing (%get slot 'whonit-type))
	(funcall (%get slot 'whonit-accessor) thing)
      (%get thing slot))))

(defun whonit-put (thing slot value &rest annotations)
  "Stores a value on a slot from a whonit, acting specially if the whonit caches the slot."
  (declare (ignore annotations))
  (if (symbolp thing) (%put thing slot value)
    (if (typep thing (%get slot 'whonit-type))
	(funcall (%get slot 'whonit-modifier) thing value)
      (%get thing slot))))

(defmacro properties (x)
  `(if (symbolp ,x) (symbol-plist ,x)
    (if (whonitp ,x) (whonit-plist (the whonit ,x)))))

(defsetf properties (thing) (value)
  `(if (symbolp ,thing) (setf (symbol-plist ,thing) ,value)
    (if (whonitp ,thing) (setf (whonit-plist ,thing) ,value))))

(deftype unit () '(or symbol-unit whonit))


;;;;. Describing failure

;;.ARLOtje defines a data structure called a @dfn{failure}
;;.@tindex{failure}.  Failures are identified by the predicate
;;.@code{failurep} and constructed by the function @code{fail} which
;;.takes two arguments, a @var{unit} and a @var{slot}.  Failure
;;.objects print out like:
;;.@example
;;.  #<Couldn't compute @var{slot} of @var{unit}>
;;.@end example
;;.Trying to get the value of a slot may try several methods, and a
;;.returned failure description indicates the last of such methods to
;;.fail.  Thus, the failure token returned by an attempt to get the
;;.COLOR of CLYDE might be:
;;.@example
;;.  #<Couldn't compute TO-COMPUTE-VALUE of COLOR>
;;.@end example
;;.indicating that assigning such a method would have allowed the
;;.computation to proceed further.  This is a more informative failure
;;.mechanism than simply indicating that your attempted slot access
;;.had not succeeded; it gives a hint of @emph{why} the access did not
;;.succeed.  In particular, couldn't compute ...., indicates the place
;;.where the computation had to stop because it couldn't go any
;;.further.

(defstruct (failure (:conc-name failure-) (:print-function print-failure) (:predicate failurep)
		    (:constructor make-failure (unit slot context))
		    (:include whonit))
  unit slot context)
(defun print-failure (failure stream options)
  "Prints a failure object."
  (declare (ignore options))
  (format stream "#<Couldn't get ~S of ~S~@[~**~]>"
	  (failure-slot failure) (failure-unit failure)
	  (whonit-plist failure)))
;;.@findex{failurep}

;;.The function @code{fail} returns a failure token constructed from
;;.the unit and slot passed to it as arguments.  It is used, for
;;.instance, when a @code{to-compute-value} method turns out to be
;;.unable to compute a value.
(defun fail (unit slot)
  "Returns a failure object for UNIT and SLOT."
  (declare (inline make-failure))
  (make-failure unit slot (arlotje-stack)))


;;;;.Primitive accessors

;;.The function @code{%get} attempts to get a property from a unit and
;;.returns a failure token if no such property exists.  Failure tokens
;;.are LISP structures describing a unit and slot and distinguished by
;;.the predicate @code{failurep}.
(defun %get (unit slot)
  "Gets the SLOT propert of UNIT, returning (FAIL unit slot) if it is not defined."
  (declare (optimize (speed 3) (safety 0))
	   (inline symbolp symbol-plist))
  (values (do ((props (properties unit) (cdr (cdr props))))
              ((or (null props) (eq (car props) slot))
               (if (null props) (fail unit slot) (car (cdr props)))))))
;;.@findex{%get}

(defun %get-default (unit slot default)
  "Gets the SLOT propert of UNIT, returning (FAIL unit slot) if it is not defined."
  (declare (optimize (speed 3) (safety 0)))
  (do ((props (properties unit) (cdr (cdr props))))
      ((or (null props) (eq (car props) slot))
       (if (null props) default (car (cdr props))))))

;;.The function @code{%put} stores a value on a slot of a particular
;;.unit.  It does this directly by SETF'ing a corresponding GET form.
(defun %put (unit slot value)
  "Stores VALUE on SLOT of UNIT directly."
  (declare (optimize (speed 3) (safety 0)))
;  (check-type slot symbol "a symbol (slot)")
  (do ((props (properties unit) (cdr (cdr props))))
      ((or (null props) (eq (car props) slot))
       (if (null props)
	   (setf (properties unit) (list* slot value (properties unit)))
	 (setf (cadr props) value)))))
;;.@findex{%put}
;;.@code{%put} is the @code{defsetf} of @code{%get}.
(defsetf %get %put)

;;.The function @code{%removes} a property on a symbol or whonit.
(defun %remove (unit slot)
  "Stores VALUE on SLOT of UNIT directly."
;  (check-type slot symbol "a symbol (slot)")
  (declare (optimize (speed 3) (safety 0)))
  (let ((props (properties unit)))
    (if (eq (car props) slot)
	(setf (properties unit) (cddr props))
      (do ((prop-cdrs (cdr props) (cdr (cdr prop-cdrs))))
	  ((null (cdr prop-cdrs)) NIL)
	(when (eq (cadr prop-cdrs) slot)
	  (setf (cdr prop-cdrs) (cdddr prop-cdrs))
	  (return T))))))
;;.@findex{%remove}

;;.The function @code{%hasnt} returns NIL if a particular slot does NOT
;;.exist on a symbol or whonit.
(defun %hasnt (unit slot)
  (declare (optimize (speed 3) (safety 0)))
  "Gets the SLOT propert of UNIT, returning (FAIL unit slot) if it is not defined."
  (do ((props (properties unit) (cdr (cdr props))))
      ((or (null props) (eq (car props) slot))
       (null props))))

;;.The function @code{%push} pushes a value onto the list stored as a
;;.property of a unit.  If there are no current values for the
;;.property, a one-element list is created and stored thereon.
(defun %push (unit slot value &rest annotations)
  "Pushes VALUE onto SLOT of UNIT directly."
  (declare (ignore annotations))
  (%put unit slot (cons value (%get-default unit slot '()))))
;;.@findex{%push}


;;;;.Declaring units

;;.The function @code{declare-unit} takes a LISP @findex{declare-unit}
;;.symbol and declares it as a unit; if the symbol has not already been
;;.declared as a unit, the symbol is given a @code{creation-id}
;;.property corresponding to the current session.  @code{declare-unit}
;;.takes a single keyword argument @var{external?} which determines
;;.whether the unit is exported from its package of definition; this
;;.keyword argument defaults to @code{t}.

(defvar *all-units* '()
  "This is a list of all declared units.")

(defun declare-unit (unit-name &key
			       (external? (symbol-package unit-name))
			       (ephemeral? nil))
  "Annotates UNIT-NAME with CREATION-ID and usually exports it from its package.
It exports it if it is used by the ARLOtje package itself."
  (when (%hasnt unit-name 'creation-id)
    (setf (%get unit-name 'creation-id) *session*)
    (push unit-name *all-units*)
    (unless ephemeral?
      (session-event! `(declare-unit ,unit-name) (list unit-name))))
  (when (and external?
	     (or (eq (symbol-package unit-name) *arlotje-package*)
		 (member *arlotje-package* (package-use-list (symbol-package unit-name)))))
    (let ((*package* (symbol-package unit-name))) (export (list unit-name))))
  unit-name)


;;;;.Popular put functions

;;.The function @code{%primitive-put} just stores a property on a unit.
(defun %primitive-put (unit slot value)
  (%put unit slot value))

;;.The function @code{%dont-put} has the same argument signature as a
;;.storing function: e.g. @var{.unit}, @var{slot}, @var{value}, and
;;.@var{annotations}, but does absolutely nothing.
(defun %dont-put (unit slot value)
  (declare (ignore unit slot))
  value)
;;.@findex{%dont-put}

;;.The function @code{error-to-put} signals an
;;.error if called.  It is designed to be a put function for slots
;;.which should never be stored.
(defun error-to-put (unit slot value)
  (error "Can't store ~S on ~S of ~S" value slot unit)
  value)
;;.@findex{error-to-put}

;;.The function @code{error-to-get} signals an
;;.error if called.  It is designed to be a put function for slots
;;.which should never be stored.
(defun error-to-get (unit slot)
  (error "Can't get ~S of ~S\; it's illegal" slot unit))
;;.@findex{error-to-get}


;;;;.Defining primitively

;;.The macro @code{define-primitively} declares a unit
;;.and uses @code{%put} to store values on it.  Its first
;;.argument is a unit name and its remaining arguments are slot-value
;;.pairs; for instance:
;;.@example
;;.(define-primitively to-get-value
;;.  (english-description "This stores the function used to get values of a slot.P")
;;.  (to-get-value 'standard-get-value)
;;.  (to-put-value 'standard-put-value)
;;.  (to-compute-value 'inherit-through-works-like)
;;.  (makes-sense-for 'unitp)
;;.  (must-be 'function-namep))
;;.@end example
(defmacro define-primitively (unit &body slot-specs)
  "Declares UNIT as a unit and then assigns it properties from SLOT-SPECS.
Each SLOT-SPEC is a pair (slot-name value-form) where VALUE-FORM is evaluated
and stored with %PUT."
  `(let ((unit (declare-unit ',unit)))
    (record-source-file ',unit :unit)
    ,@(mapcar #'(lambda (slot-value)
		  `(%put unit ',(first slot-value) ,(second slot-value)))
       slot-specs)
    unit))
;;.@findex{define-primitively (macro)}


;;;;. The operation stack

;;.ARLOtje maintains an operation stack for debugging purposes and to
;;.catch some of those infinite recursions to which reflective systems
;;.are prone.  The function @code{show-arlotje-stack} displays
;;.ARLOtje's current stack with a printed display like:
;;.@example
;;.; While doing GET-VALUE CLYDE COLOR
;;.;   While doing COMPUTE-DEFAULT CLYDE COLOR
;;.;     While doing GET-VALUE DUMBO COLOR
;;.;       While doing COMPUTE-DEFAULT DUMBO COLOR
;;.;         While doing GET-VALUE PROTO-PACHYDERM COLOR
;;.@end example
;;.indicating ARLOtje's current operations.  This dynamic description
;;.is set up by the macro @code{computing}.  The first argument to
;;.@code{computing} indicates the `goal' of the current operation
;;.which is maintained on an operation stack during the execution of
;;.the body of the form.  For instance,
;;.@example
;;.   (computing (COMPUTE-DEFAULT unit undefined-slot)
;;.      (funcall to-compute-value unit undefined-slot))
;;.@end example
;;.calls the value of @code{to-compute-value} on @code{unit} and
;;.@code{undefined-slot} in a context where @code{(COMPUTE-DEFAULT unit undefined-slot)}
;;.is recorded as an active operation.  The first parameter to the
;;.@code{computing} macro has the form
;;.   @code{(@var{operation} . @var{args})}
;;.where @var{operation} is an unevaluated symbol and each of
;;.@var{args} is individually evaluated.

;;.The @code{computing} macro attempts to identify circular recursions
;;.by checking that any posted operation is not already being
;;.performed.  For instance, suppose that the expression above caused a
;;.circular reference; the call to @code{to-compute-value} ended up
;;.executing the same code with similar bindings for @code{unit} and
;;.@code{undefined-slot} (for instance, the respective bindings might
;;.be @code{CLYDE} and @code{COLOR}), the following error would be signalled:
;;.@example
;;.>>> Fatal recursion while attempting COMPUTE-DEFAULT CLYDE COLOR
;;.@end example
;;.All of ARLOtje's so-called `reflexive operations' use the
;;.@code{computing} macro to post their computation and this catches
;;.most of the fatal recursions which simple self-reference engenders.

;;.The variable @code{*recursion-checking*} determines whether
;;.ARLOtje checks for fatal recursions with each slot access.  If
;;.@code{T} (the default), each slot operation is recorded on the
;;.variable @code{*operation-stack*}; if any slot operation is
;;.attempted while an identical operation is in progress, an error is
;;.signalled.
(defvar *recursion-checking* nil
  "Determines whether or not preemptive fatal recursion checking is performed.")

;;.The variable @code{*operation-stack*} keeps track of ARLOtje's
;;.current operations.  It is tested by @code{computing?} and tested
;;.and bound by @code{computing}.
(defvar *operation-stack* '()
  "This is a list of current operations used for debugging and
detecting fatal recursions.")
;;.@findex{*operation-stack* (variable)}

(defun arlotje-stack ()
  "Returns ARLOtje's current operations stack.
In some implementations, this copies a structure stored on the stack."
  (copy-tree *operation-stack*))

;;..The @code{computing} macro checks that its first parameter
;;..(describing an operation) is `current' on @code{*operation-stack*};
;;..if it is current, a fatal recursion error is signalled; if it is
;;..not current the operation description is dynamically pushed onto
;;..@code{*operation-stack*} and the body of the form is evaluated.
;;..The operation description is of the form @code{(@var{operation} . @var{args})}
;;..where @var{operation} is unevaluated and each of @var{args} is evaluated.
(defmacro computing ((operation . args) &body body)
  "Executes body in a context where OPERATION is recorded as applying to ARGS."
  `(cond (*recursion-checking*
	  (stack-let ((operation (list ',operation ,@args)))
	   (if (member operation *operation-stack* :test #'equal)
	       (error "Fatal recursion while attempting ~S" operation)
	     (stack-let ((*operation-stack* (cons operation *operation-stack*)))
			,@body))))
    (T ,@body)))
;;.@findex{computing (macro)}

;;.The macro @code{while-doing} is just like @code{computing} but does
;;.not fatal recursion checking.
(defmacro while-doing ((operation . args) &body body)
  `(stack-let ((*operation-stack* (cons (list ',operation ,@args) *operation-stack*)))
     ,@body))

;;.The stack of current operations can be queried by the macro
;;.@code{computing?}.  The parameters to @code{computing?} are 
;;.an operation and arguments as in the first parameter to the
;;.@code{computing} macro; the expression returns true if the
;;.corresponding operation is currently in progress:
;;.@example
;;.  (computing? COMPUTE-DEFAULT 'CLYDE 'COLOR)
;;.@end example
;;.returns true if one of the current operations is computing the
;;.default COLOR for CLYDE.
(defmacro computing? (operation &rest args)
  "Returns true in dynamic contexts where OPERATION is being applied to ARGS."
  `(member (list ',operation ,@args) *operation-stack* :test #'equal))
;;.@findex{computing? (macro)}

;;.Access to the stack is useful when various procedures are doing
;;.bi-directional inferences.  For instance, a slot might --- as a side
;;.effect --- assert its inverse value; storing the @code{HUSBAND} slot
;;.might store the respective @code{WIFE} property.  Taken generally,
;;.this principle will cause an infinite recursion.  However, this can
;;.be forestalled by explicitly checking if one is currently computing
;;.storing a property before asserting it.

(defun show-arlotje-stack ()
  "Describes the current state of ARLOtje's operations."
  (do ((depth 0 (1+ depth))
       (context (reverse *operation-stack*) (cdr context)))
      ((null context))
    (format T "~&; ")
    (dotimes (i (* 2 depth)) (write-char #\Space))
    (format T "While~{ ~S~}" (first context))))


;;;;. Basic Reflexive Operations

;;.One central idea in ARLOtje is the use of `reflexive' operations to
;;.perform operations on ARLOtje objects based on other ARLOtje
;;.objects.  For instance, when we get the @code{COLOR} slot from the
;;.description @code{CLYDE}, we refer to a description of @code{COLOR}
;;.to figure out how to do it.  Similarly, when we assert a color for
;;.Clyde (or assert a value for the @code{COLOR} slot of @code{CLYDE}),
;;.ARLOtje again refers to the description of @code{COLOR} to determine
;;.its actions.  Reflexive operations are the philosophical descendants
;;.of the `stored program' innovations of early computer science and the
;;.program as data innovations of early LISP developers.

;;.ARLOtje has two primary reflexive operations: @code{GET-VALUE} and
;;.@code{PUT-VALUE}.  The first of these extracts slot values and the
;;.second, @code{PUT-VALUE}, stores slot values.

;;.@code{GET-VALUE} takes the arguments @var{unit} and @var{slot} and
;;.attempts to return a value, using the description of @var{slot} to
;;.determine what methods to apply.  These methods are summarized in
;;.the @code{T0-GET-VALUE} slot of the @var{slot} being fetched.  The
;;.LISP definition of @code{GET-VALUE} looks like this:
;;...@example
(defun get-value (unit slot)
  (declare (optimize (speed 3) (safety 0)))
  "Gets the SLOT property of UNIT, based on the description of SLOT."
  (values 
   (computing (get-value unit slot)
     (let ((method (get-slot-method slot 'to-get-value)))
       (if (failurep method)
         (error "~S probably isn't a slot\; it has no TO-GET-VALUE property."
                slot)
         (funcall method unit slot))))))
;;...@end example
;;.Note the use of the @code{computing} macro to maintain the
;;.operations stack.
;;.@findex{GET-VALUE}
;;.@vindex{TO-GET-VALUE (slot)}

;;.The function @code{put-value} stores a value on a particular slot of
;;.a unit.  It has three required parameters: @var{unit}, @var{slot},
;;.and @var{value} followed by an alternating slot/value list of
;;.@var{annotations}.  These arguments are passed onto the function
;;.stored on the @code{TO-PUT-VALUE} @vindex{TO-PUT-VALUE (slot)} slot
;;.of @var{slot}.  This function may ignore the @var{annotations} or
;;.even the @var{value} (e.g. as in @code{%DONT-PUT}).  If the value
;;.is annotated @pxref{Annotated values}, the given @var{annotations}
;;.will specify properties to be stored on the annotated value
;;.describing @var{slot} of @var{unit}; but certain slots
;;.(particularly, the most primitive ones) ignore all the annotations.
(defun put-value (unit slot value)
  "Stores VALUE on the SLOT property of UNIT, based on the description of SLOT."
  (declare (optimize (speed 3) (safety 0)))
  ;; This is wrapped in a VALUES to indicate that it always returns
  ;; one value.
  (values
   (computing (put-value unit slot value)
     (let ((method (get-slot-method slot 'to-put-value)))
       (if (failurep method)
         (error "~S probably isn't a slot\; it has no TO-PUT-VALUE property."
                slot)
         (funcall method unit slot value))))))
;;.@findex{PUT-VALUE}
;;.@vindex{TO-PUT-VALUE (slot)}

(defun get-slot-method (unit slot)
  "This gets a property of a slot descrition."
  (do ((prototype (%get unit 'works-like) (%get prototype 'works-like))
       (value (%get unit slot) (%get prototype slot)))
      ((or (not (failurep value)) (failurep prototype))
       value)))


;;;;. Basic slots of slots

;;.The base semantics of slots are determined by their
;;.@code{TO-GET-VALUE} and @code{TO-PUT-VALUE} slots; the functions
;;.stored thereon refer to other slots which describe the inferences and
;;.other operations with which the slot is involved.  The power of
;;.representing one's slots in a representation language is that it
;;.allows the inference mechanisms of the representation language to be
;;.applied to checking and filling-in the descriptions of
;;.representations.  In ARLOtje, slots are organized by the relation
;;.@code{WORKS-LIKE}; and the slots @code{TO-GET-VALUE} and
;;.@code{TO-PUT-VALUE} inherit through this relation.  This means that
;;.if we say the slot @code{TO-PRINT-VALUE} has a @code{WORKS-LIKE}
;;.slot of @code{TO-GET-VALUE}, this means that the
;;.@code{TO-PRINT-VALUE} slot will fetch and store its values using
;;.the same mechanism as @code{TO-GET-VALUE}.
;;.@vindex{WORKS-LIKE (slot)} 
;;.A review note on ARLOtje's gradual construction: the most common
;;.getting and putting functions --- at least in ARLOtje's description
;;.of itself --- are the function @code{STANDARD-GET-VALUE} and
;;.@code{STANDARD-PUT-VALUE}.  As ARLOtje is extended in its
;;.incremental loading, these functions (along with several others) are
;;.repeatedly redefined to include new capabilities (for instance,
;;.support of annotated values, put demons, domain and range typing, etc.)
(loaded-session '|Booting-ARLOtje| '(aj::load-arlotje))

;;.The @code{TO-GET-VALUE} slot stores a function which is used to get
;;.the value of the slot on which it is stored.  It is thus a two
;;.argument function whose `standard' value is STANDARD-GET-VALUE.
(define-primitively to-get-value
  (english-description "This stores the function used to get values of a slot.")
  (works-like 'prototypical-slot-slot)
  (to-get-value 'get-slot-method)
  (to-put-value '%primitive-put)
  (makes-sense-for 'unitp)
  (must-be 'function-namep))
;;.@vindex{TO-GET-VALUE (slot)}

;;.The @code{TO-PUT-VALUE} slot stores the function which is used to
;;.assert values for the slot on which it is stored.  It must be a
;;.function which requires three or more arguments.
(define-primitively to-put-value
  (english-description "This stores the function used to get values of a slot.")
  (works-like 'prototypical-slot-slot)
  (to-get-value 'get-slot-method)
  (to-put-value '%put)
  (makes-sense-for 'slotp)
  (must-be 'function-namep))
;;.@vindex{TO-PUT-VALUE (slot)}

;;.The @code{WORKS-LIKE} slot is the inheritance relation for
;;.functional slot properties.  As well as being used for inheritance
;;.of the getting and putting functions, it is eventually used for
;;.inheritance of slot domain and range information, printing and
;;.description methods, etc.
(define-primitively works-like
  (english-description "This is the slot which this slot works like.")
  (works-like 'prototypical-primitive-slot)
  (to-get-value '%get)
  (to-put-value '%put)
  (must-be 'slotp))
;;.@vindex{WORKS-LIKE (slot)} 

(defun inherit-through-works-like (unit slot)
  "Looks on a UNIT's WORKS-LIKE slot for an inherited slot value."
  (let ((works-like (get-value unit 'works-like)))
    (if (failurep works-like) (fail unit slot)
	(get-value works-like slot))))
;;.@findex{inherits-through-works-like}

;;.The slot @code{PROTOTYPICAL-PRIMITIVE-SLOT} is used as a @emph{prototoype};
;;.other slots inherit from it (through @code{WORKS-LIKE}) but it has
;;.no other meaning.
(define-primitively prototypical-primitive-slot
  (english-description "This is the prototypical single valued slot.")
  (to-get-value '%get)
  (to-put-value '%put)
  (makes-sense-for 'unitp)
  (must-be 'anythingp))
;;.@vindex{PROTOTYPICAL-PRIMITIVE-SLOT (slot prototype)}

;;.The slot @code{PROTOTYPICAL-SLOT-SLOT} is used as a
;;.@emph{prototoype} for slots which describe slots; other slots
;;.inherit from it (through @code{WORKS-LIKE}) but it has no
;;.descriptions itself.
(define-primitively prototypical-slot-slot
  (english-description "This is the prototypical single valued slot-describing slot.")
  (to-get-value 'get-slot-method)
  (to-put-value '%put)
  (makes-sense-for 'slotp)
  (must-be 'anythingp))
;;.@vindex{PROTOTYPICAL-SLOT-SLOT (slot prototype)}


;;;;. Basic slots of units

;;.A few primitive slots are possible for any sort of unit.  These
;;.provide english descriptions, definition information, etc.

;;.The @code{CREATION-ID} slot stores the @emph{session} in which a
;;.unit was originally @emph{declared}.  The session description
;;.stored here (detailed in ...) records information on the host,
;;.site, time, and active user in the unit's declaration context.
(define-primitively creation-id
  (english-description
   "This is the session in which this unit was created (declared).")
  (works-like 'prototypical-primitive-slot)
  (to-get-value '%get)
  (to-put-value '%dont-put)
  (makes-sense-for 'unitp)
  (must-be 'sessionp))
;;.@vindex{CREATION-ID (slot)}

;;.The @code{ENGLISH-DESCRIPTION} slot stores an english string
;;.describing a unit.  This is mainly useful for humans' perusal;
;;.ideally, this string would be generated automatically by ARLOtje,
;;.but currently no such mechanisms exist.
(define-primitively english-description
  (english-description
   "This is the english-descrition of a unit.")
  (works-like 'prototypical-primitive-slot)
  (makes-sense-for 'unitp)
  (must-be 'stringp))
;;.@vindex{ENGLISH-DESCRIPTION (slot)} 


;;;;. Basic predicates

;;.The predicate function @code{ANYTHINGP} returns true for anything.
(defun anythingp (x) "Returns true." (declare (ignore x)) T)
;;.@findex{ANYTHINGP (predicate)}

;;.The predicate function @code{BOOLEANP} returns true for only the tokens @code{T} and @code{NIL}.
(defun booleanp (x) "Returns true for T or NIL" (or (eq x t) (eq x nil)))
;;.@findex{BOOLEANP (predicate)}

;;.The predicate function @code{UNITP} returns true for any symbol.  It should really check
;;.for a @code{CREATION-ID}, but it currently assumes that any symbol is a valid unit.
(defun unitp (x)
  "Returns true for all units."
  (or (and (symbolp x) (if (%hasnt x 'creation-id) (declare-unit x) T))
      (whonitp x)))
;;.@findex{UNITP (predicate)}

;;.The predicate function @code{SLOTP} returns true for slots; a symbol is a slot if it has
;;.(directly or by inheritance) a @code{TO-GET-VALUE} slot.
(defun slotp (x)
  "Returns true for all slots."
  (and (symbolp x)
       (or (get x 'to-get-value)
	   (and (get x 'works-like) (not (failurep (get-value x 'to-get-value))))
	   (assuming (slotp x) "~A is a valid slot." x))))
;;.@findex{SLOTP (predicate)}

(defun function-namep (x) (symbolp x))


;;;;.Iterating over property lists.

;;.@code{Doproplist} iterates over all of the property/value pairs on
;;.a symbol.  It is like @code{dolist}, but with two element
;;.specifications; the first is the variable to contain the property
;;.on each interaction the second is the variable to contain the
;;.corresponding value.
(defmacro doproplist ((prop value symbol &optional return-form) &body body)
  "Iterates over all of the property/value pairs of a symbol."
  (let ((prop-var (make-symbol "PROPERTIES")))
    `(do ((,prop-var (properties ,symbol) (cddr ,prop-var)))
      ((null ,prop-var) ,return-form)
      (let ((,prop (car ,prop-var)) (,value (cadr ,prop-var)))
	,@body))))
;;.@findex{doproplist (macro)}

;;.For instance, the simplest unit describer is implemented as:
;;...@example
(defun du (unit &rest other-args)
  "Describes a unit in the simplest way possible."
  (declare (ignore other-args))
  (format T "~&Description of ~S" unit)
  (doproplist (slot value unit)
    (format T "~&~S:~40T~S" slot value)))
;;...@end example
;;.@findex{du (simplest version)}
;;.More extended versions of @code{du} are specified as more of
;;.ARLOtje is defined and initialized.


