;;; $Id: scan-objects.scm,v 1.3 1993/09/13 17:24:05 queinnec Exp $
;;; Copyright (c) 1990-93 by Christian Queinnec. All rights reserved.
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;;                        LiSP2TeX
;;;   Christian Queinnec             or to:  Christian Queinnec
;;;   <queinnec@polytechnique.fr>            <Christian.Queinnec@inria.fr>
;;;   Laboratoire d'Informatique de l'X      INRIA -- Rocquencourt
;;;   Ecole Polytechnique                    Domaine de Voluceau, BP 105
;;;   91128 Palaiseau                        78153 Le Chesnay Cedex
;;;   France                                 France
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This program is distributed in the hope that it will be useful.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted, so long as the following
;;; conditions are met:
;;;      o credit to the authors is acknowledged following current
;;;        academic behaviour
;;;      o no fees or compensation are charged for use, copies, or
;;;        access to this software
;;;      o this copyright notice is included intact.
;;; This software is made available AS IS, and no warranty is made about
;;; the software or its performance.

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; This file defines all the objects that are needed to scan files.

;;; A scanned file is a file completely processed ie where every form
;;; is associated with a substring where it appears.
;;; scanned-expressions is a vector of scanned-expression(s)
;;; string contains the verbatim content of the file.
(define (make-scanned-file filename scanned-expressions string)
  (vector 'scanned-file
          filename scanned-expressions string ) )
(define (scanned-file-filename sf)
  (vector-ref sf 1) )
(define (scanned-file-scanned-expressions sf)
  (vector-ref sf 2) )
(define (scanned-file-string sf)
  (vector-ref sf 3) )


;;; A scanned expression is associated to its bounds in the string
;;; as well as its definer and its name (both are #f if not recognizable).
;;; expression is the original Sexpression
;;; type is the string representing the definer (for ex "define")
;;; key is the string  representing the defined name
;;; start marks where starts the associated substring
;;; real-start where is the opening parenthesis
;;; end where is the closing parenthesis.
(define (make-scanned-expression expression type key start real-start end)
  (vector 'scanned-expression
          expression type key start real-start end ) )
(define (scanned-expression-original se)
  (vector-ref se 1) )
(define (scanned-expression-type se)
  (vector-ref se 2) )
(define (scanned-expression-key se)
  (vector-ref se 3) )
(define (scanned-expression-start se)
  (vector-ref se 4) )
(define (scanned-expression-real-start se)
  (vector-ref se 5) )
(define (scanned-expression-end se)
  (vector-ref se 6) )


;;; Counting-streams are like streams but they maintain an index of
;;; the next position to read and the number of read lines.
;;; The eof-object to return when the string is exhausted is also kept.
;;; string is the normal string where is recorded the file
;;; size is the total size of the file
;;; position is the current position within the stream

;;; The file is first read as a whole into a string and then reread
;;; in memory (from this string) to assign bounds to Sexpressions. 
;;; It allows to cache files and not to reread them if more than one
;;; definition is needed from a file.
(define (make-counting-stream filename string size position eof-object line)
  (vector 'counting-stream
          filename string size position eof-object line ) )
(define (counting-stream? o)
  (and (vector? o) (eq? 'counting-stream (vector-ref o 0))) )
(define (counting-stream-filename cs)
  (vector-ref cs 1) )
(define (counting-stream-string cs)
  (vector-ref cs 2) )
(define (counting-stream-size cs)
  (vector-ref cs 3) )
(define (counting-stream-position cs)
  (vector-ref cs 4) )
(define (counting-stream-eof-object cs)
  (vector-ref cs 5) )
(define (set-counting-stream-position! cs position)
  (vector-set! cs 4 position) )
(define (counting-stream-line cs)
  (vector-ref cs 6) )
(define (set-counting-stream-line! cs n)
  (vector-set! cs 6 n) )

;;; Read a character from a counting stream
(define (counting-stream-read-char stream)
  (let ((position (counting-stream-position stream))
        (char 'wait) )
    (if (< position (counting-stream-size stream))
        (begin
          (set-counting-stream-position! stream (+ 1 position))
          (set! char (string-ref (counting-stream-string stream) position))
          (when (char=? char #\newline)
                (set-counting-stream-line!
                 stream (+ 1 (counting-stream-line stream)) ) )
          char )
        (counting-stream-eof-object stream) ) ) )

;;; Peek a character from a counting stream
(define (counting-stream-peek-char stream)
  (let ((position (counting-stream-position stream)))
    (if (< position (counting-stream-size stream))
        (string-ref (counting-stream-string stream) position)
        (counting-stream-eof-object stream) ) ) )

;;; Return the content of the last line to be read upto the current position.
;;; This is useful to locate the context of errors.
(define (counting-stream-current-read-line stream)
  (let ((position (counting-stream-position stream))
        (string (counting-stream-string stream)) )
    (do ((i (max 0 (- position 1)) (- i 1)))
        ((or (= i 0)
             (char=? #\newline (string-ref string i)) )
         (substring string i position) )
      ) ) )


;;; Numbering streams are like streams except that they maintain the line
;;; number. They are based on a Scheme port and are used for filtering.

(define (make-numbering-stream port line string poken-char)
  (vector 'numbering-stream 
          port line string poken-char ) )

;;; Another creator with default options
(define (create-numbering-stream port)
  (make-numbering-stream port 1 (make-extensible-string 80) #f) )

(define (numbering-stream? o)
  (and (vector? o) (eq? 'numbering-stream (vector-ref o 0))) )

(define (numbering-stream-port ns)
  (vector-ref ns 1) )
(define (numbering-stream-line ns)
  (vector-ref ns 2) )
(define (set-numbering-stream-line! ns n)
  (vector-set! ns 2 n) )
(define (numbering-stream-string ns)
  (vector-ref ns 3) )
(define (numbering-stream-poken-char ns)
  (vector-ref ns 4) )
(define (set-numbering-stream-poken-char! ns char)
  (vector-set! ns 4 char) )

;;; Read a character from a numbering stream.
(define (numbering-stream-read-char stream)
  (let ((char (if system-can-peek-char?
                  (read-char (numbering-stream-port stream))
                  (let ((char (or (numbering-stream-poken-char stream)
                                  (read-char (numbering-stream-port stream)) )))
                    (set-numbering-stream-poken-char! stream #f)
                    char ) )))
    (cond ((eof-object? char) char)
          ((char=? char #\newline)
           (set-numbering-stream-line! 
            stream (+ 1 (numbering-stream-line stream)) )
           (set-extensible-string-index! (numbering-stream-string stream) 0)
           char )
          (else 
           (increment-string char (numbering-stream-string stream))
           char ) ) ) )

;;; PC Scheme does not have peek-char, so simulate it.  Others have
;;; peek-char so follow the global flag system-can-peek-char?. 

(define (numbering-stream-peek-char stream)
  (if system-can-peek-char?
      (peek-char (numbering-stream-port stream))
      (or (numbering-stream-poken-char stream)
          (let ((char (read-char (numbering-stream-port stream))))
            (set-numbering-stream-poken-char! stream char)
            char ) ) ) )

(define (numbering-stream-current-read-line stream)
  (finalize-string (numbering-stream-string stream)) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; A little test (see reading.test file which contains them)

(define (test-read file)
  (call-with-input-file file
    (lambda (port)
      (let ((stream (create-numbering-stream port)))
	(define (loop e)
	  (if (eof-object? e) 'done
	      (begin (display e)
		     (newline)
		     (loop (numbering-read stream)) ) ) )
	(loop (numbering-read stream)) ) ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; An equal that works case-insensitively. This is useful to compare
;;; expressions read with different case sensitivity status.

(define (equal-ci? e1 e2)
  (cond ((symbol? e1) 
         (and (symbol? e2)
              (string-ci=? (symbol->string e1) (symbol->string e2)) ) )
        ((pair? e1)
         (and (pair? e2)
              (equal-ci? (car e1) (car e2))
              (equal-ci? (cdr e1) (cdr e2)) ) )
        ((vector? e1)
         (and (vector? e2)
              (= (vector-length e1) (vector-length e2))
              (let loop ((i 0))
                (if (< i (vector-length e1))
                    (and (equal-ci? (vector-ref e1 i) (vector-ref e2 i))
                         (loop (+ i 1)) )
                    #t ) ) ) )
        ((atom? e1) 
         (equal? e1 e2) ) ) )

;;; end of scan-objects.scm
