;;; $Id: pp.scm,v 1.14 1994/02/03 17:36:14 queinnec Exp $
;;; Copyright (c) 1990-93 by Christian Queinnec. All rights reserved.
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;;                        LiSP2TeX
;;;   Christian Queinnec             or to:  Christian Queinnec
;;;   <queinnec@polytechnique.fr>            <Christian.Queinnec@inria.fr>
;;;   Laboratoire d'Informatique de l'X      INRIA -- Rocquencourt
;;;   Ecole Polytechnique                    Domaine de Voluceau, BP 105
;;;   91128 Palaiseau                        78153 Le Chesnay Cedex
;;;   France                                 France
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This program is distributed in the hope that it will be useful.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted, so long as the following
;;; conditions are met:
;;;      o credit to the authors is acknowledged following current
;;;        academic behaviour
;;;      o no fees or compensation are charged for use, copies, or
;;;        access to this software
;;;      o this copyright notice is included intact.
;;; This software is made available AS IS, and no warranty is made about
;;; the software or its performance.

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;;                       A very naive pretty-printer

;;; Record the current form (this eases reporting errors).

(define *the-current-form* 'wait)

;;; Error handling while pretty-printing.

(define (pp-error msg . culprits)
  (command-error *pp-error-format* msg *the-current-form* culprits) )

(define *pp-error-format* 
  "LiSP2TeX (PrettyPrint) error: ~A
while pretty-printing ~A
Culprits are: ~A~%" )

;;; A hook to expand expressions to be pretty-printed if wished.
;;; Allows the user to redefine it (funcall1 hack below).

(define (pp-expand e) e)
(set! pp-expand pp-expand)

;;; The entry point of the pretty-printer viewed from pp-format.

(define (pp-greekify e . out)
  (let ((out (if (pair? out) (car out) (current-output-port))))
    (set! *the-current-form* e)
    (pp-format out *pretty-print-prologue*)
    (pp-print (pp-expand e) *environment* out)
    (pp-format out *pretty-print-epilogue*) ) )

;;; The real printing engine. It is driven by the environment so it
;;; explores the environment to find an appropriate method.

(define (pp-print e r out)
  (define (lookup rr)
    (if (pair? rr)
        (if (procedure? (car rr))
            (let ((bool ((car rr) e)))
              (if bool 
                  (bool e (context-extend r e) out)
                  (lookup (cdr rr)) ) )
            (lookup (cdr rr)) )
        (pp-error "Do not know how to pretty-print" e) ) )
  (lookup r) )

;;; The CONTEXT records, in the environment, the list of enclosing forms.
;;; It can be retrieved through pp-context.

(define (context-extend rr e)
  (let scan ((r rr))
    (if (pair? r)
        (if (and (pair? (car r)) (eq? (caar r) 'CONTEXT))
            (pp-extend rr (cons 'CONTEXT (cons e (cdar r))))
            (scan (cdr r)) )
        (pp-extend rr (list 'CONTEXT e)) ) ) )

;;; By default, retrieve the current context from the current environment.

(define (pp-context . r)
  (let scan ((r (if (pair? r) (car r) *environment*)))
    (if (pair? r)
        (if (and (pair? (car r)) (eq? (caar r) 'CONTEXT))
            (cdar r)
            (scan (cdr r)) )
        '() ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; The default environment contains the default printer which
;;; just writes the expression. The default-environment is a list
;;; of things among which can be
;;;  1) methods with signature
;;;      (lambda (e) ...) returns #f if it is not an appropriate method
;;;                       returns a printer if appropriate
;;;             where a printer is (lambda (e r out) (display...))
;;;  2) symbols like INDENT indicating the indentation level
;;;  3) (VARIABLE image name) indicating a bound lexical variable and
;;;             its associated image (a string).
;;;  4) (IMAGE string . names) indicating the images of some variables.
;;;  5) (CONTEXT . forms...) containing the list of embedding forms.
;;;  6) (INDEXING-MODE mode . names) defines how to disambiguate the images
;;;             of the names.
;;; Other terms can belong to the environment provided they do not 
;;; interfere with these terms.

;;; One can extend the environment at one's risk. LiSP2TeX pushes
;;; functions and lists on the environment and will never pushes other
;;; types of data (ie vectors) so vectors can be freely used by users.

;;; Extend an environment with a term, return an environment.

(define (pp-extend environment term)
  (cons term environment) )

;; This function is the ultimate and by default accepts anything and
;; just write it.

(define *environment*
  (list (lambda (e) pp-default)) )

;;; This function returns the appropriate method if it exist. It also
;;; returns its rank in the environment. Useful for debug. Might be
;;; interesting to have a call-next-method ? [But it can be obtained
;;; using (pp-method e (list-tail r n)).

(define (pp-method e . rr)
  (let ((rr (if (pair? rr) (car rr) *environment*)))
    (let scan ((r rr)
               (index 0) )
      (if (pair? r)
          (let ((method (and (procedure? (car r))
                             ((car r) e) )))
            (if method
                (cons (cons index (length rr)) method)
                (scan (cdr r) (+ 1 index)) ) )
          #f ) ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; All macros enrich the current environment.

;;; Define that some variables must be displayed as images.
;;; Check that all variables are symbols.

(define-lisp2tex-macro (def-image image . variables)
  (let check ((variables variables))
    (or (null? variables)
        (if (pair? variables)
            (if (symbol? (car variables))
                (check (cdr variables))
                (command-error *def-image-error1* (car variables)) )
            (command-error *def-image-error2* variables) ) ) )
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (cons 'IMAGE 
                            (cons ',image  ',variables) ) ) )
     ,image ) )

(define *def-image-error1*
  "~%LiSP2TeX def-image error: not a symbol" )
(define *def-image-error2*
  "~%LiSP2TeX def-image error: not a list of symbols" )

;;; Associate method to form that begin with keyword. Useful for
;;; special forms. Check that keyword is a symbol.

(define-lisp2tex-macro (def-form-printer keyword method)
  (unless (symbol? keyword)
    (command-error *def-form-printer-error* keyword) )
  `(begin
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e)
                        (if (and (pair? e)
                                 (symbol? (car e))
                                 (string-ci=? 
                                  (symbol->string (car e))
                                  ',(symbol->string keyword) ) )
                            ,method
                            #f ) ) ) )
     ',keyword ) )

(define *def-form-printer-error*
  "~%LiSP2TeX def-form-printer error: Keyword is not a symbol" )

;;; Associate a method to a type.

(define-lisp2tex-macro (def-type-printer predicate method)
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e) 
                        (if (,predicate e)
                            ,method
                            #f ) ) ) )
     ',predicate ) )

;;; Associate an indexing mode to a set of variables.

(define-lisp2tex-macro (def-indexing-mode mode . variables)
  `(begin 
     (set! *environment*
           (pp-extend *environment* (cons 'INDEXING-MODE
                                          (cons ',mode ',variables) )) )
     ',mode ) )                      

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; These functions define how to print some data structures. They 
;;; are named so they can easily be redefined.

;;; The default printer:

(define (pp-default e r out)
  (pp-format out "~A" e) ) 

;;; Booleans appear as bold-face words.

(define (pp-boolean e r out)
  (pp-format out (if e pp-true-format 
                     pp-false-format )) )

(define pp-true-format "\\mbox{\\bf true}")
(define pp-false-format "\\mbox{\\bf false}")

;;; Prints characters surrounded with ` and '.

(define (pp-char e r out)
  (pp-format out "\\mbox{`{~U}'}" (make-string 1 e)) )

;;; Prints (f x y) as f(x,y)

(define (pp-pair e r out)
  (pp-print (car e) r out)
  (pp-print-terms (cdr e) r out "(" "," ")") )

;;; Prints (f x y) as (f x y)

(define (pp-list e r out)
  (pp-print-terms e r out "(" pp-whitespace-format ")") )

(define pp-whitespace-format "\\:")

;;; Many keywords are surrounded by this space, this is the function
;;; to wrap with this space.

(define (pp-wrap-with-spaces fmt)
  (string-append pp-whitespace-format fmt pp-whitespace-format) )

;;; Display each term of a list with a prefix, a separator and a suffix.
;;; The list can be empty in this case it is printed as prefix-suffix.

(define (pp-print-terms e* r out beg mid end)
  (define (iterate e*)
    (if (pair? (cdr e*))
        (begin (pp-print (car e*) r out)
               (pp-format out mid)
               (iterate (cdr e*)) )
        (pp-print (car e*) r out) ) )
  (pp-format out beg)
  (when (pair? e*) (iterate e*))
  (pp-format out end) )

;;; Displays (begin form) as form. Warns you if it is not the
;;; case. This is useful in denotational style where no side effect
;;; can appear \ie no begin forms with more than one term can
;;; appear. Begin forms can still appear due to macro expansion.

(define (pp-begin e r out)
  (if (pair? (cdr e))
      (if (pair? (cddr e))
          (pp-error "BEGIN with more than one form: " e)
          (pp-print (cadr e) r out) )
      (pp-error "BEGIN with less than one form: " e) ) )

;;; Display a list of bindings (as in a let or letrec). Use NEWR for
;;; the names of the variables, use OLDR for the associated value.
;;; This eases to print with the same function let or letrec forms.
;;; Display ((a 1)(b (f x y))) as:       a = 1
;;;                                  and b = f(x,y)

(define (pp-print-bindings bindings oldr newr out)
  (when (pair? bindings)
     (pp-print (caar bindings) newr out)
     (pp-format out pp-within-bindings-format)
     (pp-print (cadr (car bindings)) oldr out)
     (when (pair? (cdr bindings)) 
           (pp-newline newr out)
           (pp-format out pp-between-bindings-format) )
     (pp-print-bindings (cdr bindings) oldr newr out) ) )

(define pp-between-bindings-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{and}}") )
(define pp-within-bindings-format " = ")

;;; Extend R with variables stating what their images are. When the
;;; preferred indexing method is NOTHING, then variables are checked
;;; not to clash ie to be simultaneously introduced and associated to
;;; a same image. Otherwise, the indexing mode disambiguates them.

(define (extend-with-variables r variables)
  (define (search-image v r)
    (if (pair? r)
        (if (and (pair? (car r))
                 (eq? (caar r) 'IMAGE)
                 (member-ci? v (cddr (car r))) )
            (cadr (car r))
            (search-image v (cdr r)) )
        (texify-symbol v) ) )
  (define (clash? image bindings)
    (and (pair? bindings)
         (or (equal? image (car (car bindings)))
             (clash? image (cdr bindings)) ) ) )
  ;; bindings =  (  ( image variable ) ... )
  (let accum ((variables variables)
              (bindings  '()) )
    (if (pair? variables)
        (let* ((v     (car variables))
               (image (search-image v r)) )
          (if (and (eq? (search-indexing-mode v r) 'nothing)
                   (clash? image bindings) )
              (pp-error "Variable ambiguously imagified" v image)
              (accum (cdr variables) (cons (list image v) bindings)) ) )
        (let add ((bindings bindings))
          (if (pair? bindings)
              (pp-extend (add (cdr bindings)) `(VARIABLE . ,(car bindings)))
              r ) ) ) ) )

;;; Convert a symbol into a TeX string.

(define (texify-symbol v)
  (let ((str (string-lowercase (symbol->string v))))
    (string-append "\\mbox{\\it " str "\\/}") ) )
;;; Displays a symbol. 

(define (pp-symbol e r out)
  (define (search-image-collision image r)
    (if (pair? r)
        (if (and (pair? (car r)) 
                 (eq? (caar r) 'VARIABLE)
                 (equal? image (cadr (car r))) )
            (+ 1 (search-image-collision image (cdr r)))
            (search-image-collision image (cdr r)) )
        0 ) )
  (define (search-image e r)
    (if (pair? r)
        (if (pair? (car r)) 
            (cond ((and (eq? (caar r) 'VARIABLE)
                        ;; member-ci? since I do not write eq-ci?
                        (member-ci? e (cddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image (search-image-collision image (cdr r))) ) )
                  ((and (eq? (caar r) 'IMAGE)
                        (member-ci? e (cddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image 0) ) )
                  (else (search-image e (cdr r))) )
            (search-image e (cdr r)) )
        (cons (texify-symbol e) 0) ) )
  (let* ((image+index (search-image e r))
         (image (car image+index))
         (index (cdr image+index)) )
    (pp-indexed-symbol image index (search-indexing-mode e r) r out) ) )

;;; Search if there is a special mode to index a variable. By default
;;; it is the value of the global variable *preferred-index*.

(define (search-indexing-mode e r)
  (if (pair? r)
      (if (pair? (car r))
          (if (and (eq? (caar r) 'INDEXING-MODE)
                   ;; member-ci? since I do not write eq-ci?
                   (member-ci? e (cddr (car r))) )
              (cadr (car r))
              (search-indexing-mode e (cdr r)) )
          (search-indexing-mode e (cdr r)) )
      *preferred-index* ) )                  
  
;;; There is a special member version that compares symbols
;;; case-insensitively since def-form-printer or def-image forms are
;;; read by the underlying Scheme and the symbols that appear in them
;;; must be recognized by the pretty-printer.

(define (member-ci? s s*)
  (let ((name (symbol->string s)))
    (let scan ((s* s*))
      (and (pair? s*)
           (or (string-ci=? name (symbol->string (car s*)))
               (scan (cdr s*)) ) ) ) ) )            

;;; Four modes exist to resolve conflicts. When two variables have a
;;; similar image then they are disambiguated:
;;;    -- prime mode: symbols are suffixed with ' '' ''' etc.
;;;    -- index mode: symbols are indexed with 0, 1, 2 etc.
;;;    -- then-index: the first occurrence appears without index, the following
;;;                   are indexed 1, 2 etc.
;;;    -- nothing:    nothing is prefixed or suffixed. In that case, variables
;;;                   are checked not to clash (see extend-with-variables).

(define *preferred-index* 'then-index)

(define (pp-indexed-symbol image index indexing-mode r out)
  (case indexing-mode
    ((index) (pp-format out "{~U}_{~A}" image index))
    ((then-index)
     (if (= index 0)
         (pp-format out "{~U}" image)
         (pp-format out "{~U}_{~A}" image index) ) )
    ((prime) (pp-format out "{~U}" image)
             (do ((i 0 (+ 1 i)))
                 ((>= i index))
               (pp-format out "'") ) )
    ((nothing) (pp-format out "{~U}" image))
    (else (pp-error "Unknown indexing mode" indexing-mode)) ) )

;;;;ooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Some usual special forms.

(define (pp-if e r out)
  (let ((r (pp-incr-indentation r out)))
    (pp-format out pp-if-if-format)
    (pp-print (cadr e) r out)
    (pp-newline r out)
    (pp-format out pp-if-then-format)
    (pp-print (caddr e) r out)
    (pp-newline r out)
    (pp-format out pp-if-else-format)
    (pp-print (cadddr e) r out)
    (pp-newline r out)
    (pp-format out pp-if-endif-format)
    (pp-decr-indentation r out) ) )

(define pp-if-if-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{if}}") )
(define pp-if-then-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{then}}") )
(define pp-if-else-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{else}}") )
(define pp-if-endif-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{endif}}") )

(define (pp-let e r out)
  (if (symbol? (cadr e))
      (let ((loop     (cadr e))
            (bindings (caddr e))
            (body     (cdddr e)) )
        (pp-print 
         `(letrec ((,loop (lambda ,(map car bindings) . ,body)))
            (,loop . ,(map cadr bindings)) ) 
         r
         out ) )
      (let* ((new1r (pp-incr-indentation r out))
             (new2r (extend-with-variables new1r (map car (cadr e)))) )
        (pp-format out pp-let-let-format)
        (pp-print-bindings (cadr e) r new2r out)
        (pp-newline new2r out)
        (pp-format out pp-let-in-format)
        (pp-print `(begin . ,(cddr e)) new2r out)
        (pp-decr-indentation r out) ) ) )

(define pp-let-let-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{let}}") )
(define pp-let-in-format 
  (pp-wrap-with-spaces "\\mbox{\\bf{in}}") )

(define (pp-letrec e r out)
  (let* ((new1r (pp-incr-indentation r out))
         (new2r (extend-with-variables new1r (map car (cadr e)))) )
    (pp-print `(begin . ,(cddr e)) new2r out)
    (pp-newline new2r out)
    (pp-format out pp-letrec-whererec)
    (pp-print-bindings (cadr e) new2r new2r out)
    (pp-decr-indentation r out) ) )

(define pp-letrec-whererec 
  (pp-wrap-with-spaces "\\mbox{\\bf{whererec}}") )

(define (pp-let* e r out)
  (if (pair? (cadr e))
      (pp-print `(let (,(car (cadr e))) 
                   (let* ,(cdr (cadr e)) . ,(cddr e)) )
                r out )
      (pp-print `(begin . ,(cddr e)) r out) ) )

;;; Prints a lambda definition.

(define (pp-lambda e r out)
  (let ((r (extend-with-variables r (cadr e))))
    (pp-format out "\\lambda ")
    (pp-print-terms (cadr e) r out "" "" " . ")
    (pp-print `(begin . ,(cddr e)) r out) ) )

;;; Prints a define form.

(define (pp-define e r out)
  (pp-format out "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Definition of ~U~%" 
                  (find-scanned-expression-key e) )
  (let ((r (if (pair? (cadr e))
               (extend-with-variables r (cdadr e))
               r )))
    (pp-print (cadr e) r out)
    (pp-format out pp-define-equal-format)
    (collect-definitions
     (cddr e)
     '()
     (lambda (definitions rest)
       (pp-print `(begin . ,rest) r out)
       (when (pair? definitions)
         (pp-format out pp-define-wherec-format)
         (let ((r (pp-incr-indentation r out)))
           (for-each (lambda (e)
                       (pp-print e r out)
                       (pp-format out pp-define-between-format) )
                     definitions ) ) ) ) ) ) )

(define pp-define-equal-format " = \\newline~%")
(define pp-define-wherec-format "\\newline~% \\mbox{\\bf{whererec}}\\:")
(define pp-define-between-format " \\newline~%")

;;; This function collects internal define to prepare printing them
;;; with a whererec clause.

(define (collect-definitions e* defs k)
  (if (pair? e*)
      (if (and (pair? (car e*))
               (eq? (caar e*) 'define) )
          (collect-definitions (cdr e*)
                               (cons (car e*) defs)
                               k )
          (k (reverse! defs) e*) )
      (pp-error "No body for this definition" defs) ) )

;;;ooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Utility functions to print various things. Useful in customization files.
;;; The pp-check function checks the length of the expression to print.

(define (pp-check-length e size)
  (unless (= size (length e))
    (pp-error "Expression with a wrong size" size e) ) )

(define (pp-nullary image)
  (lambda (e r out)
    (pp-check-length e 1)
    (pp-format out image) ) )

(define (pp-unary beg end)
  (lambda (e r out)
    (pp-check-length e 2)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out end) ) )

(define (pp-binary beg mid end)
  (lambda (e r out)
    (pp-check-length e 3)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out mid)
    (pp-print (caddr e) r out)
    (pp-format out end) ) )

(define (pp-ternary beg mid1 mid2 end)
  (lambda (e r out)
    (pp-check-length e 4)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out mid1)
    (pp-print (caddr e) r out)
    (pp-format out mid2)
    (pp-print (cadddr e) r out)
    (pp-format out end) ) )

(define (pp-nary beg mid end)
  (lambda (e r out)
    (pp-print-terms (cdr e) r out beg mid end) ) )

(define (pp*-nary . fmt*)
  (lambda (e r out)
    (do ((e* (cdr e) (cdr e*))
         (fmt* fmt* (cdr fmt*)) )
        ((or (null? fmt*) (null? e*))
         (if (null? fmt*)
             (unless (null? e*)
               (pp-error "Not enough formats" 'pp*-nary e*) )
             (begin
               (pp-format out (car fmt*))
               (unless (null? (cdr fmt*))
                 (pp-error "Too much formats" 'pp*-nary fmt*) ) ) ) )
      (pp-format out (car fmt*))
      (pp-print (car e*) r out) ) ) )

(define pp-semantical-lbracket "\\lbrack\\!\\lbrack ")
(define pp-semantical-rbracket "\\rbrack\\!\\rbrack ")

(define (pp-meaning letter beg mid end)
  (lambda (e r out)
    (pp-format out letter)
    (pp-format out pp-semantical-lbracket)
    (pp-format out beg)
    (pp-print-terms (cdr e) r out "" mid "")
    (pp-format out end) 
    (pp-format out pp-semantical-rbracket) ) )

(define (pp*-meaning letter . fmt*)
  (lambda (e r out)
    (pp-format out letter)
    (pp-format out pp-semantical-lbracket)
    (do ((e* (cdr e) (cdr e*))
         (fmt* fmt* (cdr fmt*)) )
        ((or (null? fmt*) (null? e*))
         (if (null? fmt*)
             (unless (null? e*)
               (pp-error "Not enough formats" 'pp*-meaning e*) )
             (begin
               (pp-format out (car fmt*))
               (unless (null? (cdr fmt*))
                 (pp-error "Too much formats" 'pp*-meaning fmt*) ) ) ) )
      (pp-format out (car fmt*))
      (pp-print (car e*) r out) )
    (pp-format out pp-semantical-rbracket) ) )

;;; Sometimes one wants to compose various effects.  USEFUL ?

(define (pp-compose . fns)
  (lambda (e r out)
    (for-each (lambda (f) (f e r out))
              fns ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Indentation is usually done via the tabbing environment of LaTeX,
;;; so nothing needs to be recorded in R. Nevertheless a INDENT is
;;; pushed onto R so it is sufficient to count the number of INDENT in
;;; R to know the indentation level.

;;; Indentation is performed by the tabbing environment of TeX by
;;; means of three macros:
;;;    setandincrindent: set the left margin as a new indentation level
;;;    newline: go to the next line to the current left margin
;;;    decrindent: reset the left margin to what it was before the
;;;    last matching setandincrindent.
;;; The following prologue and epilogue set up the appropriate TeX
;;; environment and the definitions of these three macros. Moreover,
;;; to ease fixing indentation in a paper, they depend on a TeX
;;; boolean named ifdenotationinline which controls if they are active
;;; or not. Insert denotationinlinetrue if you want to see your
;;; denotation on a single line, set denotationinlinefalse if you want
;;; it to appear on multiple lines.

;;; set a new margin and return a new environment.

(define (pp-incr-indentation r out)
  (pp-format out " \\setandincrindent~%")
  (pp-extend r 'indent) )

;;; Go to the newline with the current indentation.

(define (pp-newline r out)
  (pp-format out " \\newline~%") )

;;; Reset the margin to the previous indentation. Does not return a new
;;; environment since this is usually restaured through lexical scoping
;;; in the methods which use it (see the method for if).

(define (pp-decr-indentation r out)
  (pp-format out " \\decrindent~%") )

;;; The TeX condition \DenotationInline{true,false} controls whether
;;; lines are broken or not. So a common idiom is:
;;;   \DenotationInlinetrue \PrettyPrint (..)
;;;   to obtain a pretty printed version of a function in a single line.
;;; See for more details LiSP2TeX.sty

(define *pretty-print-prologue* "%%%~%\\DenotedLisp~%")

(define *pretty-print-epilogue* " %%%~%\\EndDenotedLisp ")

;;; end of pp.scm
