;;; $Id: pp.scm,v 1.5 1993/06/27 13:01:59 queinnec Exp $
;;; Copyright (c) 1990-93 by Christian Queinnec. All rights reserved.
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;;                        LiSP2TeX
;;;   Christian Queinnec             or to:  Christian Queinnec
;;;   <queinnec@polytechnique.fr>            <Christian.Queinnec@inria.fr>
;;;   Laboratoire d'Informatique de l'X      INRIA -- Rocquencourt
;;;   Ecole Polytechnique                    Domaine de Voluceau, BP 105
;;;   91128 Palaiseau                        78153 Le Chesnay Cedex
;;;   France                                 France
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This program is distributed in the hope that it will be useful.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted, so long as the following
;;; conditions are met:
;;;      o credit to the authors is acknowledged following current
;;;        academic behaviour
;;;      o no fees or compensation are charged for use, copies, or
;;;        access to this software
;;;      o this copyright notice is included intact.
;;; This software is made available AS IS, and no warranty is made about
;;; the software or its performance.

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;;                       A very naive pretty-printer

;;; Record the current form (this eases reporting errors).

(define *the-current-form* 'wait)

;;; Error handling while pretty-printing.

(define (pp-error msg . culprits)
  (command-error *pp-error-format* msg *the-current-form* culprits) )

(define *pp-error-format* 
  "LiSP2TeX (PrettyPrint) error: ~A
while pretty-printing ~A
Culprits are: ~A~%" )

;;; The entry point of the pretty-printer viewed from pp-format.

(define (pp-greekify e . out)
  (let ((out (if (pair? out) (car out) (current-output-port))))
    (set! *the-current-form* e)
    (pp-format out *pretty-print-prologue*)
    (pp-print e *environment* out)
    (pp-format out *pretty-print-epilogue*) ) )

;;; The real printing engine. It is driven by the environment so it
;;; explores the environment to find an appropriate method.

(define (pp-print e r out)
  (define (lookup rr)
    (if (pair? rr)
        (if (procedure? (car rr))
            (let ((bool ((car rr) e)))
              (if bool 
                  (bool e (context-extend r e) out)
                  (lookup (cdr rr)) ) )
            (lookup (cdr rr)) )
        (pp-error "Do not know how to pretty-print" e) ) )
  (lookup r) )

;;; The CONTEXT records, in the environment, the list of enclosing forms.
;;; It can be retrieved through pp-context.

(define (context-extend rr e)
  (let scan ((r rr))
    (if (pair? r)
        (if (and (pair? (car r)) (eq? (caar r) 'CONTEXT))
            (pp-extend rr (cons 'CONTEXT (cons e (cdar r))))
            (scan (cdr r)) )
        (pp-extend rr (list 'CONTEXT e)) ) ) )

;;; By default, retrieve the current context from the current environment.

(define (pp-context . r)
  (let scan ((r (if (pair? r) (car r) *environment*)))
    (if (pair? r)
        (if (and (pair? (car r)) (eq? (caar r) 'CONTEXT))
            (cdar r)
            (scan (cdr r)) )
        '() ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; The default environment contains the default printer which
;;; just writes the expression. The default-environment is a list
;;; of things among which can be
;;;  1) methods with signature
;;;      (lambda (e) ...) returns #f if it is not an appropriate method
;;;                       returns a printer if appropriate
;;;             where a printer is (lambda (e r out) (display...))
;;;  2) symbols like INDENT indicating the indentation level
;;;  3) (VARIABLE image name) indicating a bound lexical variable and
;;;             its associated image (a string).
;;;  4) (IMAGE string . names) indicating the images of some variables.
;;;  5) (CONTEXT . forms...) containing the list of embedding forms.
;;; Other terms can belong to the environment provided they do not 
;;; interfere with these terms.

;;; One can extend the environment at one's risk. LiSP2TeX pushes
;;; functions and lists on the environment and will never pushes other
;;; types of data (ie vectors) so vectors can be freely used by users.

;;; Extend an environment with a term, return an environment.
(define (pp-extend environment term)
  (cons term environment) )

;; This function is the ultimate and by default accepts anything and
;; just write it.

(define *environment*
  (list (lambda (e) pp-default)) )

;;; This function returns the appropriate method if it exist. It also
;;; returns its rank in the environment. Useful for debug. Might be
;;; interesting to have a call-next-method ?

(define (pp-method e . rr)
  (let ((rr (if (pair? rr) (car rr) *environment*)))
    (let scan ((r rr)
               (index 0) )
      (if (pair? r)
          (let ((method (and (procedure? (car r))
                             ((car r) e) )))
            (if method
                (cons (cons index (length rr)) method)
                (scan (cdr r) (+ 1 index)) ) )
          #f ) ) ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; All macros enrich the current environment.

;;; Define that some variables must be displayed as images.
;;; Check that all variables are symbols.

(define-pervasive-macro (def-image image . variables)
  (let check ((variables variables))
    (or (null? variables)
        (if (pair? variables)
            (if (symbol? (car variables))
                (check (cdr variables))
                (command-error *def-image-error1* (car variables)) )
            (command-error *def-image-error2* variables) ) ) )
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (cons 'IMAGE 
                            (cons ',image  ',variables) ) ) )
     ,image ) )

(define *def-image-error1*
  "~%LiSP2TeX def-image error: not a symbol" )
(define *def-image-error2*
  "~%LiSP2TeX def-image error: not a list of symbols" )

;;; Associate method to form that begin with keyword. Useful for
;;; special forms. Check that keyword is a symbol.

(define-pervasive-macro (def-form-printer keyword method)
  (unless (symbol? keyword)
    (command-error *def-form-printer-error* keyword) )
  `(begin
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e)
                        (if (and (pair? e)
                                 (symbol? (car e))
                                 (string-ci=? 
                                  (symbol->string (car e))
                                  ',(symbol->string keyword) ) )
                            ,method
                            #f ) ) ) )
     ',keyword ) )

(define *def-form-printer-error*
  "~%LiSP2TeX def-form-printer error: Keyword is not a symbol" )

;;; Associate a method to a type.

(define-pervasive-macro (def-type-printer predicate method)
  `(begin 
     (set! *environment*
           (pp-extend *environment*
                      (lambda (e) 
                        (if (,predicate e)
                            ,method
                            #f ) ) ) )
     ',predicate ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; These functions define how to print some data structures. They 
;;; are named so they can easily be redefined.

;;; The default printer:

(define (pp-default e r out)
  (pp-format out "~A" e) ) 

;;; Booleans appear as bold-face words.

(define (pp-boolean e r out)
  (pp-format out (if e "\\mbox{\\bf true}"
                     "\\mbox{\\bf false}" )) )

;;; Prints characters surrounded with ` and '.

(define (pp-char e r out)
  (pp-format out "`{~U}'" (make-string 1 e)) )

;;; Prints (f x y) as f(x,y).

(define (pp-pair e r out)
  (pp-print (car e) r out)
  (pp-print-terms (cdr e) r out "(" "," ")") )

;;; Prints (f x y) as (f x y)

(define (pp-list e r out)
  (pp-print-terms e r out "(" "\\mbox{\\ }" ")") )

;;; Display each term of a list with a prefix, a separator and a suffix.
;;; The list can be empty in this case it is printed as prefix-suffix.

(define (pp-print-terms e* r out beg mid end)
  (define (iterate e*)
    (if (pair? (cdr e*))
        (begin (pp-print (car e*) r out)
               (pp-format out mid)
               (iterate (cdr e*)) )
        (pp-print (car e*) r out) ) )
  (pp-format out beg)
  (when (pair? e*) (iterate e*))
  (pp-format out end) )

;;; Displays (begin form) as form.
;;; Warns you if it is not the case.

(define (pp-begin e r out)
  (if (pair? (cdr e))
      (if (pair? (cddr e))
          (pp-error "BEGIN with more than one form: " e)
          (pp-print (cadr e) r out) )
      (pp-error "BEGIN with less than one form: " e) ) )

;;; Display a list of bindings (as in a let or letrec). Use NEWR for
;;; the names of the variables, use OLDR for the associated value.
;;; This eases to print with the same function let or letrec forms.
;;; Display ((a 1)(b (f x y))) as:
;;;       a = 1
;;;   and b = f(x,y)

(define (pp-print-bindings bindings oldr newr out)
  (when (pair? bindings)
     (pp-print (caar bindings) newr out)
     (pp-format out " = ")
     (pp-print (cadr (car bindings)) oldr out)
     (when (pair? (cdr bindings)) 
           (pp-newline newr out)
           (pp-format out "\\mbox{\\bf\\ and\\ }") )
     (pp-print-bindings (cdr bindings) oldr newr out) ) )

;;; Extend R with variables stating what their images are.

(define (extend-with-variables r variables)
  (define (search-image v r)
    (if (pair? r)
        (if (and (pair? (car r))
                 (eq? (caar r) 'IMAGE)
                 (member-ci? v (cddr (car r))) )
            (cadr (car r))
            (search-image v (cdr r)) )
        (string-append "\\mbox{\\it "
                       (string-lowercase (symbol->string v))
                       "\\/}" ) ) )
  (if (pair? variables)
      (let* ((v (car variables))
             (image (search-image v r)) )
        (extend-with-variables (pp-extend r `(VARIABLE ,image ,v))
                               (cdr variables) ) )
      r ) )

;;; Displays a symbol. 

(define (pp-symbol e r out)
  (define (search-image-collision image r)
    (if (pair? r)
        (if (and (pair? (car r)) 
                 (eq? (caar r) 'VARIABLE)
                 (equal? image (cadr (car r))) )
            (+ 1 (search-image-collision image (cdr r)))
            (search-image-collision image (cdr r)) )
        0 ) )
  (define (search-image e r)
    (if (pair? r)
        (if (pair? (car r)) 
            (cond ((and (eq? (caar r) 'VARIABLE)
                        (member-ci? e (cddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image (search-image-collision image (cdr r))) ) )
                  ((and (eq? (caar r) 'IMAGE)
                        (member-ci? e (cddr (car r))) )
                   (let ((image (cadr (car r))))
                     (cons image 0) ) )
                  (else (search-image e (cdr r))) )
            (search-image e (cdr r)) )
        (let ((image (string-append 
                      "\\mbox{\\it "
                      (string-lowercase (symbol->string e))
                      "\\/}" )))
          (cons image 0) ) ) )
  (let* ((image+index (search-image e r))
         (image (car image+index))
         (index (cdr image+index)) )
    (pp-indexed-symbol image index r out) ) )

;;; There is a special member version that compares symbols
;;; case-insensitively since def-form-printer or def-image forms are
;;; read by the underlying Scheme and the symbols that appear in them
;;; must be recognized by the pretty-printer.

(define (member-ci? s s*)
  (let ((name (symbol->string s)))
    (let scan ((s* s*))
      (and (pair? s*)
           (or (string-ci=? name (symbol->string (car s*)))
               (scan (cdr s*)) ) ) ) ) )            

;;; Four modes exist to resolve conflicts. When two variables have a
;;; similar image then they are disambiguated:
;;;    -- prime mode: symbols are suffixed with ' '' ''' etc.
;;;    -- index mode: symbols are indexed with 0, 1, 2 etc.
;;;    -- then-index: the first occurrence appears without index, the following
;;;                   are indexed 1, 2 etc.
;;;    -- complain mode: signals an error if it there is an ambiguity.

(define *preferred-index* 'then-index)

(define (pp-indexed-symbol image index r out)
  (case *preferred-index*
    ((index) (pp-format out "{~U}_{~A}" image index))
    ((then-index)
     (if (= index 0)
         (pp-format out "{~U}" image)
         (pp-format out "{~U}_{~A}" image index) ) )
    ((prime) (pp-format out "{~U}" image)
             (do ((i 0 (+ 1 i)))
                 ((>= i index))
               (pp-format out "'") ) )
    ((complain) (if (= index 0)
                    (pp-format out "{~U}" image)
                    (pp-error "Ambiguity on ~A." image) ))
    (else (pp-error "Unknown indexing mode" *preferred-index*)) ) )

(define (pp-if e r out)
  (let ((r (pp-incr-indentation r out)))
    (pp-format out "\\mbox{\\bf\\ if\\ } ")
    (pp-print (cadr e) r out)
    (pp-newline r out)
    (pp-format out "\\mbox{\\bf\\ then\\ } ")
    (pp-print (caddr e) r out)
    (pp-newline r out)
    (pp-format out "\\mbox{\\bf\\ else\\ } ")
    (pp-print (cadddr e) r out)
    (pp-newline r out)
    (pp-format out "\\mbox{\\bf\\ endif\\ } ")
    (pp-decr-indentation r out) ) )

(define (pp-let e r out)
  (let* ((new1r (pp-incr-indentation r out))
         (new2r (extend-with-variables new1r (map car (cadr e)))) )
    (pp-format out "\\mbox{\\bf\\ let\\ }")
    (pp-print-bindings (cadr e) r new2r out)
    (pp-newline new2r out)
    (pp-format out "\\mbox{\\bf\\ in\\ }")
    (pp-print `(begin . ,(cddr e)) new2r out)
    (pp-decr-indentation r out) ) )

(define (pp-letrec e r out)
  (let* ((new1r (pp-incr-indentation r out))
         (new2r (extend-with-variables new1r (map car (cadr e)))) )
    (pp-print `(begin . ,(cddr e)) new2r out)
    (pp-newline new2r out)
    (pp-format out "\\mbox{\\bf\\ whererec\\ }")
    (pp-print-bindings (cadr e) new2r new2r out)
    (pp-decr-indentation r out) ) )

(define (pp-let* e r out)
  (if (pair? (cadr e))
      (pp-print `(let (,(car (cadr e))) 
                   (let* ,(cdr (cadr e)) . ,(cddr e)) )
                r out )
      (pp-print `(begin . ,(cddr e)) r out) ) )

;;; Prints a lambda definition.

(define (pp-lambda e r out)
  (let ((r (extend-with-variables r (cadr e))))
    (pp-format out "\\lambda ")
    (pp-print-terms (cadr e) r out "" "" " . ")
    (pp-print `(begin . ,(cddr e)) r out) ) )

;;; Prints a define form.

(define (pp-define e r out)
  (pp-format out "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Definition of ~U~%" 
                  (find-scanned-expression-key e) )
  (let ((r (if (pair? (cadr e))
               (extend-with-variables r (cdadr e))
               r )))
    (pp-print (cadr e) r out)
    (pp-format out " = \\newline~%")
    (collect-definitions
     (cddr e)
     '()
     (lambda (definitions rest)
       (pp-print `(begin . ,rest) r out)
       (when (pair? definitions)
         (pp-format out "\\newline~% \\mbox{\\bf whererec} ")
         (let ((r (pp-incr-indentation r out)))
           (for-each (lambda (e)
                       (pp-print e r out)
                       (pp-format out " \\newline~%") )
                     definitions ) ) ) ) ) ) )

;;; This function collects internal define to prepare printing them
;;; with a whererec clause.

(define (collect-definitions e* defs k)
  (if (pair? e*)
      (if (and (pair? (car e*))
               (eq? (caar e*) 'define) )
          (collect-definitions (cdr e*)
                               (cons (car e*) defs)
                               k )
          (k (reverse! defs) e*) )
      (pp-error "No body for this definition" defs) ) )

;;;ooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Utility functions to print various things. Useful in customization files.

(define (pp-unary beg end)
  (lambda (e r out)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out end) ) )

(define (pp-binary beg mid end)
  (lambda (e r out)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out mid)
    (pp-print (caddr e) r out)
    (pp-format out end) ) )

(define (pp-ternary beg mid1 mid2 end)
  (lambda (e r out)
    (pp-format out beg)
    (pp-print (cadr e) r out)
    (pp-format out mid1)
    (pp-print (caddr e) r out)
    (pp-format out mid2)
    (pp-print (cadddr e) r out)
    (pp-format out end) ) )

(define (pp-nary beg mid end)
  (lambda (e r out)
    (pp-print-terms (cdr e) r out beg mid end) ) )

(define (pp-meaning letter beg mid end)
  (lambda (e r out)
    (pp-format out letter)
    (pp-format out "\\lbrack\\!\\lbrack ")
    (pp-format out beg)
    (pp-print-terms (cdr e) r out "" mid "")
    (pp-format out end) 
    (pp-format out "\\rbrack\\!\\rbrack ") ) )

(define (pp*-meaning letter . fmt*)
  (lambda (e r out)
    (pp-format out letter)
    (pp-format out "\\lbrack\\!\\lbrack ")
    (do ((e* (cdr e) (cdr e*))
         (fmt* fmt* (cdr fmt*)) )
        ((or (null? fmt*) (null? e*))
         (if (null? fmt*)
             (unless (null? e*)
               (pp-error "Not enough formats" 'pp*-meaning e*) )
             (begin
               (pp-format out (car fmt*))
               (unless (null? (cdr fmt*))
                 (pp-error "Too much formats" 'pp*-meaning fmt*) ) ) ) )
      (pp-format out (car fmt*))
      (pp-print (car e*) r out) )
    (pp-format out "\\rbrack\\!\\rbrack ") ) )

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;; Indentation is usually done via the tabbing environment of LaTeX,
;;; so nothing needs to be recorded in R. Nevertheless a INDENT is
;;; pushed onto R so it is sufficient to count the number of INDENT in
;;; R to know the indentation level.

;;; Indentation is performed by the tabbing environment of TeX by
;;; means of three macros:
;;;    setandincrindent: set the left margin as a new indentation level
;;;    newline: go to the next line to the current left margin
;;;    decrindent: reset the left margin to what it was before the
;;;    last matching setandincrindent.
;;; The following prologue and epilogue set up the appropriate TeX
;;; environment and the definitions of these three macros. Moreover,
;;; to ease fixing indentation in a paper, they depend on a TeX
;;; boolean named ifdenotationinline which controls if they are active
;;; or not. Insert denotationinlinetrue if you want to see your
;;; denotation on a single line, set denotationinlinefalse if you want
;;; it to appear on multiple lines.

;;; set a new margin and return a new environment.

(define (pp-incr-indentation r out)
  (pp-format out " \\setandincrindent~%")
  (pp-extend r 'indent) )

;;; Go to the newline with the current indentation.

(define (pp-newline r out)
  (pp-format out " \\newline~%") )

;;; Reset the margin to the previous indentation. Does not return a new
;;; environment since this is usually restaured through lexical scoping
;;; in the methods which use it (see the method for if).

(define (pp-decr-indentation r out)
  (pp-format out " \\decrindent~%") )

;;; The TeX condition \denotationinline{true,false} controls whether
;;; lines are broken or not. So a common idiom is:
;;;   \denotationinlinetrue \PrettyPrint (..)
;;;   to obtain a pretty printed version of a function in a single line.
;;; See for more details LiSP2TeX.sty

(define *pretty-print-prologue* "%%%~%\\DenotedLisp~%")

(define *pretty-print-epilogue* " %%%~%\\EndDenotedLisp ")

;;; end of pp.scm
