;;; type-constructors.s

;;; define-record-type and define-type-constructor

; The following code defines a macro for defining record types with variant
; records in type-checked code, much like the types in Hope or Standard ML.
; The record types are in fact type-constructors, so one can define
; parameterized types like (stack x).  One declares a global constructor
; name for each variant, along with the types of the components.
; These are exported, along with a case function (eg stack-case),
; which decomposes any instance of the record and sends 
; the component to the appropriate function.  Examples are below.


;;; this code is OK, with a couple of exceptions:

;;; 1.  The construction of the types should be done with abstract
;;; rather than concrete syntax.

;;; 2.  What should be exported is a decomposition MACRO, like
;;; record-case, rather than a function.

;;; ****************************************************************

(extend-syntax (define-record-type)
   [(define-record-type
       (type-name type-var ...)
       [constructor-name  component-types]
       ...)
    (with ([name-for-case-function
	     (string->symbol
	       (string-append
		 (symbol->string 'type-name)
		 "-case"))])
      (list
	(define-unchecked constructor-name
	  (generic (type-var ...)
	    (-> (seq . component-types) (type-name type-var ...)))
	  (lambda x (cons 'constructor-name x)))
	...
	(define-unchecked name-for-case-function
	  (generic (target-type type-var ...)
	    (-> (seq (type-name type-var ...)
		     (-> (seq . component-types) target-type)
		     ...)
		target-type))
	  (lambda (x . fnlist)
	    (let loop
	      ([tagnames '(constructor-name ...)]
	       [functions fnlist])
	      (cond
		[(null? tagnames) 
		 (report-error (list "Bad data to struct selector" x))]
		[(eq? (car x) (car tagnames))
		 (apply (car functions) (cdr x))]
		[t (loop (cdr tagnames) (cdr functions))]))))))])



; some simple-minded examples:

'(define-record-type (stack x)	; how could we not have a stack example?
   (empty-stack ())
   (push-stack (x (stack x))))

;; exports: empty-stack () -> (stack x), push-stack x * (stack x) ->
;; stack x, and stack-case: (stack x) * (()->y) * (x*(stack x)->y) -> y.

; one could use this facility to build lists, pairs, and unions, albeit
; somewhat less efficiently than the existing facilities:

'(define-record-type (zlist x)
   (znil ())
   (zcons (x (zlist x))))

'(define-record-type (zpair x y)
   (zpair (x y)))

; a simple-minded test tag:

'(define-checked test-zlist
   (-> (seq) (list int))
   (lambda ()
      (let loop ([x (zcons 1 (zcons 2 (zcons 3 (znil))))])
	   (zlist-case x
	      (lambda () nil)
	      (lambda (a d) (cons a (loop d)))))))

;;; ****************************************************************

;;; define-type-constructor

; type-definition facility.  The functions are defined globally, but the
; representation (rep) is hidden.  

; the syntax has changed slightly from old versions:  it declares the
; global generics the same way as define-record-type, and individual
; functions must have all their generics declared locally with
; (generic (x y) ...)

(extend-syntax (define-type-constructor)
  [(define-type-constructor
     (name type-var ...)
     rep-type
     (fn-name fn-type value) ...)
   (with ([(type-scheme ...)
	   (map external-type->type-scheme  '(fn-type ...))]
	  [group-prefix (make-group-prefix 'name '(type-var ...)
			  'rep-type '(fn-name ...) '(fn-type ...))])
     (with
       ([(ans ...)
	 (map
	   (lambda (fn-name fn-type value)
	     (check-operation fn-name fn-type 'group-prefix value))
	   '(fn-name ...) '(fn-type ...) '(value ...))])
       (list
	 (if 'ans
	   (begin
	     (change-global-prefix! 'fn-name 'type-scheme)
	     (set! fn-name
	       (let ((dec (lambda (x) x))
		     (enc (lambda (x) x)))
		 value))
	     (list 'fn-name 'fn-type))
	   (list 'fn-name ': 'type-error))
	 ...)))])

(define make-group-prefix
  (lambda (name type-vars rep-type fn-names fn-types)
    (let
      ([type (make-type name type-vars)])
      (ext-prefix
	   (list 'dec 'enc)
	   (list
	     (make-type-scheme type-vars
	       (make-functional-type (list type) rep-type))
	     (make-type-scheme type-vars
	       (make-functional-type (list rep-type) type)))
	   (ext-prefix
	     fn-names
	     (map external-type->type-scheme fn-types)
	     empty-prefix)))))

(define check-operation
  (lambda (fn-name fn-type group-prefix value)
    (call/cc
      (lambda (return)
	(let* ((dummy-name (gen-fcn-sym))
	       (type-scheme (external-type->type-scheme fn-type))
	       (generics (type-scheme->generics type-scheme))
	       (type     (type-scheme->type type-scheme)))
	  ;;(add-undefined-types-to-badlist external-type generics)
	  ;;(printf "check-operation: fn-name = ~s fn-type = ~s ~%"
	  ;;fn-name fn-type)
	  ;;(printf "type-scheme = ~s~%group-prefix =~%" type-scheme)
	  ;;(pretty-print group-prefix)
	  (checker
	    (ext-prefix 
	      (list dummy-name)
	      (list
		(make-type-scheme
		  '()
		  (make-functional-type
		    (list (subst-consts-for-generics type-scheme))
		    (gen-type-var))))
	      group-prefix)
	    (make-application dummy-name (list value))
	    empty-type-env
	    (lambda (msg)
		(printf "define-type-constructor: error in ~s:~%" fn-name)
		(pretty-print msg)
		(return '()))))))))





