;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                 This is the file VM.SCM                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; VIRTUAL MACHINE: DATA REPRESENTATION AND MEMORY LAYOUT

; Handy utility for internal consistency checking

(define (assert truth where)
  (if (not truth)
      (error "assertion failed" where)))

; Statistics counters

(define *counters* '())

(define (make-counter name)
  (let ((counter (vector 0)))
    (set! *counters* (cons (list name counter) *counters*))
    counter))

(define (add-to-counter counter how-much)
  (vector-set! counter 0 (+ (vector-ref counter 0) how-much)))

(define (reset-counters)
  (for-each (lambda (name+counter)
              (vector-set! (cadr name+counter) 0 0))
            *counters*))

(define (show-counters)
  (for-each (lambda (name+counter)
              (newline)
              (display (car name+counter))
              (display " ")
              (write (vector-ref (cadr name+counter) 0)))
            *counters*)
  (newline))

; Registers

(define (make-register initial-contents)
  (vector 'register initial-contents))

(define (fetch register)
  (vector-ref register 1))

(define (store! register new-value)
  (vector-set! register 1 new-value))

(define *descriptor-registers* '())

(define (make-descriptor-register)
  (let ((register (make-register vm-unspecified)))
    (set! *descriptor-registers*
          (cons register *descriptor-registers*))
    register))

(define (clear-descriptor-registers)
  (for-each (lambda (reg) (store! reg vm-unspecified))
            *descriptor-registers*))

; Descriptors
;  A descriptor describes a Scheme object.

(define tag-limit 8)

(define (make-descriptor tag data)
  (assert (and (integer? data) (integer? tag) (>= tag 0) (< tag tag-limit))
          'make-descriptor)
  (+ tag (* data tag-limit)))

(define (descriptor-tag descriptor) (remainder descriptor tag-limit))
(define (descriptor-data descriptor) (quotient descriptor tag-limit))

(define same-descriptor? =)

; Tags

(define small-integer-tag 0)
(define misc-tag          1)
; tag 2 is reserved for future expansion
(define pair-tag          3)
(define vector-tag          4)
(define symbol-tag          5)
(define procedure-tag          6)
(define ref-tag                  7)

(define least-stored-object-tag pair-tag)

(define (stored-object? descriptor)
  (>= (descriptor-tag descriptor) least-stored-object-tag))

(define (vm-small-integer? descriptor)
  (= (descriptor-tag descriptor) small-integer-tag))

(define (vm-pair? descriptor)
  (= (descriptor-tag descriptor) pair-tag))

(define (vm-vector? descriptor)
  (= (descriptor-tag descriptor) vector-tag))

(define (vm-symbol? descriptor)
  (= (descriptor-tag descriptor) symbol-tag))

(define (vm-procedure? descriptor)
  (= (descriptor-tag descriptor) procedure-tag))

(define (vm-ref? descriptor)
  (= (descriptor-tag descriptor) ref-tag))

; Small integers

(define (vm-make-small-integer integer)
  (make-descriptor small-integer-tag integer))

(define (vm-small-integer-value descriptor)
  (assert (vm-small-integer? descriptor) 'vm-small-integer-value)
  (descriptor-data descriptor))

; Miscellaneous objects: booleans, empty list, unspecified

(define vm-false       (make-descriptor misc-tag 0))
(define vm-true               (make-descriptor misc-tag 1))
(define vm-empty-list  (make-descriptor misc-tag 2))
(define vm-unspecified (make-descriptor misc-tag 3))
(define vm-undefined   (make-descriptor misc-tag 4))

(define (vm-make-boolean boolean)
  (if boolean vm-true vm-false))

(define (vm-boolean? descriptor)
  (or (same-descriptor? descriptor vm-false)
      (same-descriptor? descriptor vm-true)))

(define (vm-boolean-value descriptor)
  (assert (vm-boolean? descriptor) 'vm-boolean-value)
  (not (same-descriptor? descriptor vm-false)))

(define (vm-empty-list? descriptor)
  (same-descriptor? descriptor vm-empty-list))

; Descriptors for stored objects

(define (make-stored-object-descriptor tag length-address)
  (assert (>= tag least-stored-object-tag) 'make-stored-object-descriptor)
  (make-descriptor tag length-address))

(define (address-of-length-cell descriptor)
  (assert (stored-object? descriptor) 'address-of-length-cell)
  (descriptor-data descriptor))

; (No uses of DESCRIPTOR-DATA should follow this point.)



; Memory is a vector of cells.  Each cell can hold one descriptor.

(define *memory* #f)
(define *heap-size* 5000)
(define *stack-size* 5000)

(define (initialize-memory size)
  (set! *memory* (make-vector size)))

; Memory is subdivided into "areas".

(define (make-area start size) (vector 'area start (+ start size) start))
(define (area-begin area) (vector-ref area 1))
(define (area-end area) (vector-ref area 2))
(define (next-available area) (vector-ref area 3))
(define (set-next-available! area new-value)
  (vector-set! area 3 new-value))

(define (in-area? descriptor area)
  (assert (stored-object? descriptor) 'in-area?)
  (let ((address (descriptor-data descriptor)))
    (and (>= address (area-begin area))
         (< address (area-end area)))))

(define *current-area* #f)
(define *other-area* #f)
(define *stack-area* #f)

(define (initialize-areas)
  (set! *current-area* (make-area 0 *heap-size*))
  (set! *other-area*   (make-area (area-end *current-area*) *heap-size*))
  (set! *stack-area*   (make-area (area-end *other-area*) *stack-size*)))

; Stored objects

(define (length-or-forwarded-to descriptor)
  (assert (stored-object? descriptor) 'length-or-forwarded-to)
  (vector-ref *memory* (address-of-length-cell descriptor)))

(define (set-length-or-forwarded-to! descriptor new)
  (assert (stored-object? descriptor) 'set-length-or-forwarded-to!)
  (vector-set! *memory* (address-of-length-cell descriptor) new))

(define (forwarded? descriptor)
  (not (vm-small-integer? (length-or-forwarded-to descriptor))))

(define forwarded-to length-or-forwarded-to)

(define set-forwarded-to! set-length-or-forwarded-to!)

(define (stored-object-length descriptor)
  (assert (not (forwarded? descriptor)) 'stored-object-length)
  (vm-small-integer-value (length-or-forwarded-to descriptor)))

(define (stored-object-ref descriptor offset)
  (assert (and (stored-object? descriptor)
               (>= offset 0)
               (< offset (stored-object-length descriptor)))
          'stored-object-ref)
  (vector-ref *memory* (+ (address-of-length-cell descriptor) 1 offset)))

(define (stored-object-set! descriptor offset new-component)
  (assert (and (stored-object? descriptor)
               (>= offset 0)
               (< offset (stored-object-length descriptor)))
          'stored-object-set!)
  (vector-set! *memory*
               (+ (address-of-length-cell descriptor) 1 offset)
               new-component))

(define heap-allocation-counter (make-counter "Heap allocated"))

(define (allocate-stored-object tag size)
  (assert (>= size 0)
          'allocate-stored-object)
  (cond ((> (+ (next-available *current-area*) size 1)
            (area-end *current-area*))
         (collect)
         (if (> (+ (next-available *current-area*) size 1)
                (area-end *current-area*))
             (error "out of memory" tag size))))
  (let ((descriptor
         (make-stored-object-descriptor tag (next-available *current-area*))))
    (set-length-or-forwarded-to! descriptor (vm-make-small-integer size))
    (set-next-available! *current-area*
                         (+ (next-available *current-area*) size 1))
    (add-to-counter heap-allocation-counter (+ size 1))
    descriptor))

; Pairs

(define (vm-make-pair)
  (allocate-stored-object pair-tag 2))

(define (vm-car descriptor)
  (assert (vm-pair? descriptor) 'vm-car)
  (stored-object-ref descriptor 0))

(define (vm-cdr descriptor)
  (assert (vm-pair? descriptor) 'vm-cdr)
  (stored-object-ref descriptor 1))

(define (vm-set-car! descriptor new-car)
  (assert (vm-pair? descriptor) 'vm-set-car!)
  (stored-object-set! descriptor 0 new-car))

(define (vm-set-cdr! descriptor new-cdr)
  (assert (vm-pair? descriptor) 'vm-set-cdr!)
  (stored-object-set! descriptor 1 new-cdr))

; Refs are just like pairs except that they only have one component.

(define (vm-make-ref)
  (allocate-stored-object ref-tag 1))

(define (vm-get descriptor)
  (assert (vm-ref? descriptor) 'vm-get)
  (stored-object-ref descriptor 0))

(define (vm-put! descriptor new-get)
  (assert (vm-ref? descriptor) 'vm-put!)
  (stored-object-set! descriptor 0 new-get))

; Vectors

(define (vm-make-vector size)
  (allocate-stored-object vector-tag size))

(define (vm-vector-length descriptor)
  (assert (vm-vector? descriptor) 'vm-vector-length)
  (stored-object-length descriptor))

(define (vm-vector-ref descriptor index)
  (assert (vm-vector? descriptor) 'vm-vector-ref)
  (stored-object-ref descriptor index))

(define (vm-vector-set! descriptor index new-component)
  (assert (vm-vector? descriptor) 'vm-vector-set!)
  (stored-object-set! descriptor index new-component))

(define (vm-vector-fill! descriptor initial-value)
  (let ((len (vm-vector-length descriptor)))
    (do ((i 0 (+ i 1)))
        ((>= i len))
      (vm-vector-set! descriptor i initial-value))))

; Symbols

(define *symbol-table* (make-descriptor-register))

(define (reset-symbol-table)
  (store! *symbol-table* vm-empty-list))

(define (find-or-make-symbol symbol)
  (let loop ((l (fetch *symbol-table*)))
    (cond ((vm-empty-list? l)
           (push (vm-make-pair))
           (let ((new-symbol (copy-symbol-to-heap symbol)))
             (let ((pair (pop)))
               (vm-set-car! pair new-symbol)
               (vm-set-cdr! pair (fetch *symbol-table*))
               (store! *symbol-table* pair)
               new-symbol)))
          ((same-as-vm-symbol? symbol (vm-car l))
           (vm-car l))
          (else
           (loop (vm-cdr l))))))

(define (same-as-vm-symbol? symbol descriptor)
  (let ((name (symbol->string symbol)))
    (let ((len (string-length name)))
      (and (= len (stored-object-length descriptor))
           (let loop ((i 0))
             (cond ((>= i len) #t)
                   ((not (= (vm-small-integer-value
                              (stored-object-ref descriptor i))
                            (char->integer (string-ref name i))))
                    #f)
                   (else (loop (+ i 1)))))))))

(define (copy-symbol-to-heap symbol)
  (let ((name (symbol->string symbol)))
    (let ((len (string-length name)))
      (let ((descriptor (allocate-stored-object symbol-tag len)))
        (do ((i 0 (+ i 1)))
            ((>= i len) descriptor)
          (stored-object-set! descriptor
                              i
                              (vm-make-small-integer
                                (char->integer (string-ref name i)))))))))

(define (extract-symbol-from-heap descriptor)
  (let ((len (stored-object-length descriptor)))
    (do ((i (- len 1) (- i 1))
         (l '() (cons (integer->char (vm-small-integer-value
                                       (stored-object-ref descriptor i)))
                      l)))
        ((< i 0) (string->symbol (list->string l))))))

; Procedures
;  Procedures are represented as (code, environment) pairs.

(define (vm-procedure-code descriptor)
  (assert (vm-procedure? descriptor) 'vm-procedure-code)
  (stored-object-ref descriptor 0))

(define (vm-procedure-environment descriptor)
  (assert (vm-procedure? descriptor) 'vm-procedure-environment)
  (stored-object-ref descriptor 1))

(define (vm-set-procedure-code! descriptor code)
  (assert (vm-procedure? descriptor) 'vm-set-procedure-code!)
  (stored-object-set! descriptor 0 code))

(define (vm-set-procedure-environment! descriptor environment)
  (assert (vm-procedure? descriptor) 'vm-set-procedure-environment!)
  (stored-object-set! descriptor 1 environment))

(define (vm-make-procedure)
  (allocate-stored-object procedure-tag 2))

;;; THE STACK

; The stack grows toward low memory, and the stack pointer always points at the
; location from which a POP will fetch.

(define *sp* (make-register 0))

(define (reset-stack-pointer)
  (store! *sp* (area-end *stack-area*)))

(define stack-allocation-counter (make-counter "Stack allocated"))

(define (push descriptor)
  (if (<= (fetch *sp*) (area-begin *stack-area*))
      (error "stack overflow" descriptor))
  (store! *sp* (- (fetch *sp*) 1))
  (add-to-counter stack-allocation-counter 1)
  (vector-set! *memory* (fetch *sp*) descriptor))

(define (pop)
  (let ((descriptor (vector-ref *memory* (fetch *sp*))))
    (store! *sp* (+ (fetch *sp*) 1))
    descriptor))

(define (make-stack-vector len)
  (push (vm-make-small-integer len))
  (make-stored-object-descriptor vector-tag (fetch *sp*)))



;;; VIRTUAL MACHINE: INTERPRETER

(define *val* (make-descriptor-register))
(define *env* (make-descriptor-register))
(define *cont* (make-descriptor-register))
(define *code* (make-descriptor-register))
(define *pc* (make-register -1))
(define *nargs* (make-register -1))

(define (push-state pc size)
  (push (fetch *env*))
  (push (vm-make-small-integer pc))
  (push (fetch *code*))
  (push (fetch *cont*))
  (store! *cont*
          (make-stack-vector (+ size 4))))

(define (pop-state cont)
  (store! *sp* (+ (address-of-length-cell cont) 1))
  (store! *cont* (pop))
  (store! *code* (pop))
  (store! *pc* (vm-small-integer-value (pop)))
  (store! *env* (pop)))

(define (next-instruction-word)
  (let ((word (vm-vector-ref (fetch *code*) (fetch *pc*))))
    (store! *pc* (+ (fetch *pc*) 1))
    word))

(define (next-instruction-integer)
  (vm-small-integer-value (next-instruction-word)))

(define *trace?* #f)

(define (interpret)
  (let interpret-loop ()
    (let ((op (next-instruction-integer)))
      (if *trace?* (begin (newline) (write (opcode-name op))))
      ((vector-ref instruction-dispatch-vector op))
      (if (>= (fetch *pc*) 0)
          (interpret-loop)))))

(define instruction-dispatch-vector
  (make-vector number-of-opcodes (lambda () (error "unimplemented opcode"))))

(define (define-opcode name procedure)
  (vector-set! instruction-dispatch-vector (opcode name) procedure))

(define-opcode 'check-nargs
  (lambda ()
    (let ((desired-nargs (next-instruction-integer)))
      (if (not (= (fetch *nargs*) desired-nargs))
          (error "wrong number of arguments" desired-nargs (fetch *nargs*))))))

(define-opcode 'make-environment
  (lambda ()
    (let ((nargs (next-instruction-integer)))
      (let ((v (vm-make-vector (+ nargs 1))))
        (vm-vector-set! v 0 (fetch *env*))
        (do ((i 1 (+ i 1)))
            ((> i nargs) (store! *env* v))
          (vm-vector-set! v i (pop)))))))

(define-opcode 'load-constant
  (lambda ()
    (store! *val* (next-instruction-word))))

(define (nth-outer-rib n)
  (let loop ((n n) (rib (fetch *env*)))
    (if (<= n 0) rib (loop (- n 1) (vm-vector-ref rib 0)))))

(define-opcode 'load-variable
  (lambda ()
    (let ((rib (nth-outer-rib (next-instruction-integer))))
      (let ((val (vm-vector-ref rib (next-instruction-integer))))
        (if (same-descriptor? val vm-undefined)
            (error "undefined variable"))
        (store! *val* val)))))

(define-opcode 'set-variable
  (lambda ()
    (let ((rib (nth-outer-rib (next-instruction-integer))))
      (vm-vector-set! rib (next-instruction-integer) (fetch *val*)))))

(define-opcode 'make-procedure
  (lambda ()
    (let ((proc (vm-make-procedure)))
      (store! *val* proc)
      (vm-set-procedure-code! proc (next-instruction-word))
      (vm-set-procedure-environment! proc (fetch *env*)))))

(define-opcode 'push
  (lambda ()
    (push (fetch *val*))))

(define-opcode 'call
  (lambda ()
    (let ((nargs (next-instruction-integer)))
;      (assert (= (+ (fetch *sp*) nargs)
;                 (address-of-length-cell (fetch *cont*)))
;              'call)
      (start-call (fetch *val*) nargs))))

(define (start-call proc nargs) (store! *nargs* nargs) (store! *code*
(vm-procedure-code proc)) (store! *env* (vm-procedure-environment
proc)) (store! *pc* 0))

(define-opcode 'make-continuation
  (lambda ()
    (let ((return-pc (next-instruction-integer)))
      (push-state return-pc
                  (next-instruction-integer)))))

(define-opcode 'return
  (lambda ()
    (pop-state (fetch *cont*))))

(define-opcode 'leave-environment
  (lambda ()
    (store! *env* (vm-vector-ref (fetch *env*) 0))))

(define-opcode 'jump-if-false
  (lambda ()
    (let ((offset (next-instruction-integer)))
      (if (not (vm-boolean-value (fetch *val*)))
          (store! *pc* offset)))))

(define-opcode 'jump
  (lambda ()
    (store! *pc* (next-instruction-integer))))



; Handy combinators for making primitives

(define (no-arg-primitive proc)
  (lambda ()
    (store! *val* (proc))))

(define (one-arg-primitive proc)
  (lambda ()
    (store! *val* (proc (fetch *val*)))))

(define (two-arg-primitive proc)
  (lambda ()
    (store! *val* (proc (fetch *val*) (pop)))))

(define (three-arg-primitive proc)
  (lambda ()
    (let ((second-arg (pop)))
      (store! *val* (proc (fetch *val*) second-arg (pop))))))

(define (one-arg-predicate-primitive proc)
  (one-arg-primitive (lambda (d) (vm-make-boolean (proc d)))))

(define (two-arg-predicate-primitive proc)
  (two-arg-primitive (lambda (d1 d2) (vm-make-boolean (proc d1 d2)))))

(define (no-arg-effect-primitive proc)
  (no-arg-primitive (lambda () (proc) vm-unspecified)))

(define (one-arg-effect-primitive proc)
  (one-arg-primitive (lambda (d) (proc d) vm-unspecified)))

(define (two-arg-effect-primitive proc)
  (two-arg-primitive (lambda (d1 d2) (proc d1 d2) vm-unspecified)))

(define (arithmetic-primitive proc)
  (two-arg-primitive
   (lambda (d1 d2)
     (vm-make-small-integer (proc (vm-small-integer-value d1)
                                  (vm-small-integer-value d2))))))

(define (arithmetic-comparison-primitive proc)
  (two-arg-predicate-primitive (lambda (d1 d2)
                                 (proc (vm-small-integer-value d1)
                                       (vm-small-integer-value d2)))))

; Primitives

(define-opcode 'eq?
  (two-arg-predicate-primitive same-descriptor?))

(define-opcode 'integer?
  (one-arg-predicate-primitive vm-small-integer?))

(define-opcode '+         (arithmetic-primitive +))
(define-opcode '-          (arithmetic-primitive -))
(define-opcode '*          (arithmetic-primitive *))
(define-opcode 'quotient  (arithmetic-primitive quotient))
(define-opcode 'remainder (arithmetic-primitive remainder))

(define-opcode '= (arithmetic-comparison-primitive =))
(define-opcode '< (arithmetic-comparison-primitive <))

(define-opcode 'pair? (one-arg-predicate-primitive vm-pair?))

(define-opcode 'cons
  (lambda ()
    (let ((pair (vm-make-pair)))
      (vm-set-car! pair (fetch *val*))
      (vm-set-cdr! pair (pop))
      (store! *val* pair))))

(define-opcode 'car (one-arg-primitive vm-car))
(define-opcode 'cdr (one-arg-primitive vm-cdr))
(define-opcode 'set-car! (two-arg-effect-primitive vm-set-car!))
(define-opcode 'set-cdr! (two-arg-effect-primitive vm-set-cdr!))

(define-opcode 'ref? (one-arg-predicate-primitive vm-ref?))
(define-opcode 'ref
  (lambda ()
    (let ((the-ref (vm-make-ref)))
      (vm-put! the-ref (fetch *val*))
      (store! *val* the-ref))))
(define-opcode 'get (one-arg-primitive vm-get))
(define-opcode 'put! (two-arg-effect-primitive vm-put!))

(define-opcode 'vector? (one-arg-predicate-primitive vm-vector?))

(define-opcode 'make-vector
  (lambda ()
    (let ((vec (vm-make-vector (vm-small-integer-value (fetch *val*)))))
      (vm-vector-fill! vec (pop))
      (store! *val* vec))))

(define-opcode 'vector-length (one-arg-primitive vm-vector-length))

(define-opcode 'vector-ref
  (two-arg-primitive (lambda (d1 d2)
                       (vm-vector-ref d1 (vm-small-integer-value d2)))))

(define-opcode 'vector-set!
  (three-arg-primitive (lambda (d1 d2 d3)
                         (vm-vector-set! d1 (vm-small-integer-value d2) d3)
                         vm-unspecified)))

(define-opcode 'read
  (no-arg-primitive (lambda () (copy-to-heap (read)))))

(define-opcode 'write
  (one-arg-effect-primitive (lambda (d)
                              (write (extract-from-heap d)))))

(define-opcode 'display 
  (one-arg-effect-primitive (lambda (d)
                              (display (extract-from-heap d)))))

(define-opcode 'newline
  (no-arg-effect-primitive newline))



;;; Maybe should redo opcodes here just in case.
(define (initialize)
  (clear-descriptor-registers)
  (initialize-memory (+ (* 2 *heap-size*) *stack-size*))
  (initialize-areas)
  (reset-stack-pointer)
  (reset-symbol-table))

; Entry to interpreter

(define *global-env* (make-descriptor-register))
(define *global-env-size* 200)

(define (validate-global-environment)
  (cond ((not (vm-vector? (fetch *global-env*)))
         (set! *global-variables* '())
         (store! *global-env*
                 (vm-make-vector *global-env-size*))
         (vm-vector-fill! (fetch *global-env*) vm-undefined))))

(define (vm-define var descriptor)
  (vm-vector-set! (fetch *global-env*)
                  (global-variable-index var)
                  descriptor))

(define (run exp)                        ;don't forget to (initialize)
  (run! exp)
  (extract-from-heap (fetch *val*)))

(define *show-counters?* #t)

(define (run! exp)
  (let ((code (compile `(lambda () ,exp))))
    (reset-stack-pointer)                ;in case it overflowed
    (validate-global-environment)
    (push (copy-to-heap code))
    (let ((proc (vm-make-procedure)))
      (vm-set-procedure-code! proc (pop))
      (vm-set-procedure-environment! proc (fetch *global-env*))
      (reset-counters)
      (start-interpreter proc)
      (if *show-counters?* (show-counters)))))

(define (start-interpreter thunk)
  (push-state -1 0)
  (start-call thunk 0)
  (interpret))



; Read-evaluate-print loop.

(define (repl)                                ;don't forget to (initialize)
  (let loop ()
    (newline)
    (display "PLScheme==> ")
    (let ((form (read)))
      (cond ((definition? form)
             (newline)
             (let ((var (definition-lhs form))
                   (exp (definition-rhs form)))
               (run! exp)
               (vm-define var (fetch *val*))
               (write var)
               (display " defined.")))
            (else
             (newline)
             (run! form)
             (write (extract-from-heap (fetch *val*))))))
    (loop)))

; Test routines.
; You must do (initialize) before calling these.

(define (test-fact n)
  (run `(letrec ((fact (lambda (n) (if (< n 2) n (* n (fact (- n 1)))))))
          (fact ,n))))

(define (test-ifact n)
  (run `(let ((ifact (lambda (n)
                       (let loop ((i n) (a 1))
                         (if (< i 2) a (loop (- i 1) (* i a)))))))
          (ifact ,n))))

(define (cons-a-lot)  ;This exercises the garbage collector.
  (run '(let loop ((n 2000))
          (if (< n 0)
              'done
              (begin (cons 'this 'that) (loop (- n 1)))))))



;;; VIRTUAL MACHINE: GARBAGE COLLECTOR

(define *gc-noisily?* #t)

(define (collect)
  (if *gc-noisily?* (begin (newline) (write 'gc)))
  (let ((before (- (area-end *current-area*)
                   (next-available *current-area*))))
    (flip)
    (scan-registers)
    (scan-stack)
    (scan-transitively)
    (discard)
    (let ((after (- (area-end *current-area*)
                    (next-available *current-area*))))
      (if *gc-noisily?*
          (begin (display " reclaimed ")
                 (write (- after before))
                 (newline))))))

(define (flip)
  (let ((full-area *current-area*))
    (set! *current-area* *other-area*)
    (set! *other-area* full-area)))

(define (discard)
  (set-next-available! *other-area* (area-begin *other-area*)))

(define (scan-registers)
  (for-each (lambda (register)
              (store! register (maybe-copy (fetch register))))
            *descriptor-registers*))

(define (scan-stack)
  (let ((end (area-end *stack-area*)))
    (let scan-loop ((scan-pointer (fetch *sp*)))
      (cond ((< scan-pointer end)
             (vector-set! *memory*
                          scan-pointer
                          (maybe-copy (vector-ref *memory*
                                                  scan-pointer)))
             (scan-loop (+ scan-pointer 1)))))))

(define (scan-transitively)
  (let scan-loop ((scan-pointer (area-begin *current-area*)))
    (cond ((< scan-pointer (next-available *current-area*))
           (vector-set! *memory*
                        scan-pointer
                        (maybe-copy (vector-ref *memory* scan-pointer)))
           (scan-loop (+ scan-pointer 1))))))



(define (maybe-copy descriptor)
  (cond ((not (stored-object? descriptor)) descriptor)
        ((forwarded? descriptor) (forwarded-to descriptor))
        ((in-area? descriptor *other-area*)
         (let ((len (stored-object-length descriptor)))
           (let ((new (allocate-stored-object (descriptor-tag descriptor)
                                              len)))
             (do ((i 0 (+ i 1)))
                 ((>= i len))
               (stored-object-set! new
                                   i
                                   (stored-object-ref descriptor i)))
             (set-forwarded-to! descriptor new)
             new)))
        (else descriptor)))



;;; VIRTUAL MACHINE: BOOTSTRAPPING AND DEBUGGING

(define (copy-to-heap object)
  (cond ((integer? object) (vm-make-small-integer object))
        ((null? object) vm-empty-list)
        ((equal? object true) vm-true)
        ((equal? object false) vm-false)
        ((equal? object unspecified) vm-unspecified)
        ((pair? object)
         (push (copy-to-heap (car object)))
         (push (copy-to-heap (cdr object)))
         (let ((pair (vm-make-pair)))
           (vm-set-cdr! pair (pop))
           (vm-set-car! pair (pop))
           pair))
        ((vector? object)
         (let ((len (vector-length object)))
           (let ((v (vm-make-vector len)))
             (vm-vector-fill! v vm-unspecified)
             (push v))
           (do ((i 0 (+ i 1)))
               ((>= i len) (pop))
             (let ((component (copy-to-heap (vector-ref object i))))
               (let ((v (pop)))
                 (vm-vector-set! v i component)
                 (push v))))))
        ((symbol? object)
         (find-or-make-symbol object))
        (else
         (error "don't know how to copy this into heap"
                object))))



(define (extract-from-heap descriptor)
  (let loop ((descriptor descriptor)
              (trail '()))                ;don't blow up on circular objects
    (if (member descriptor trail)
        '<cycle>
        (let ((trail (cons descriptor trail)))
          (cond ((vm-small-integer? descriptor)
                 (vm-small-integer-value descriptor))
                ((vm-empty-list? descriptor) '())
                ((same-descriptor? descriptor vm-false) false)
                ((same-descriptor? descriptor vm-true) true)
                ((same-descriptor? descriptor vm-unspecified) unspecified)
                ((same-descriptor? descriptor (fetch *global-env*))
                 '<global-env>)
                ((vm-pair? descriptor)
                 (cons (loop (vm-car descriptor) trail)
                       (loop (vm-cdr descriptor) trail)))
                ((vm-vector? descriptor)
                 (let ((len (vm-vector-length descriptor)))
                   (let ((v (make-vector len)))
                     (do ((i 0 (+ i 1)))
                         ((>= i len) v)
                       (vector-set! v i
                                    (loop (vm-vector-ref descriptor i)
                                           trail))))))
                ((vm-symbol? descriptor)
                 (extract-symbol-from-heap descriptor))
                ((vm-ref? descriptor)
                 ;; extract?
                 (vector '<ref> (loop (vm-get descriptor) trail)))
                ((vm-procedure? descriptor)
                 (vector '<procedure>
                         (loop (vm-procedure-code descriptor) trail)
                         (loop (vm-procedure-environment descriptor) trail)))
                (else
                 (vector 'unknown-type descriptor)))))))

