#| -*-Scheme-*-

$Header$

Copyright (c) 1989 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Data Structures

(declare (usual-integrations))

;;;; Constraint Types

(define-structure (constraint-type
		   (constructor %make-constraint-type)
		   (conc-name constraint-type/)
		   (print-procedure
		    (unparser/standard-method 'constraint-type
		      (lambda (state type)
			(unparse-object state (constraint-type/name type))))))
  (name false read-only true)
  (symbol false read-only true)
  (variables false read-only true)
  (add-rules false read-only true)
  (forget-rules false read-only true)
  (nogood-rules false read-only true))

(define (make-constraint-type name symbol variables)
  (let* ((variables (list->vector variables))
	 (n-variables (vector-length variables)))
    (%make-constraint-type name
			   symbol
			   variables
			   (make-vector n-variables '())
			   (make-vector n-variables '())
			   (make-vector n-variables '()))))

;;;; Constraints

(define-structure (constraint
		   (constructor %make-constraint)
		   (conc-name constraint/)
		   (print-procedure
		    (unparser/standard-method 'constraint
		      (lambda (state constraint)
			(unparse-object
			 state
			 (constraint-type/name
			  (constraint/type constraint)))))))
  (type false read-only true)
  (cells false read-only true)
  queued-rules
  (properties false read-only true))

(define (*make-constraint type)
  (let* ((variables (constraint-type/variables type))
	 (n-variables (vector-length variables))
	 (cells (make-vector n-variables))
	 (constraint (%make-constraint type cells 0 (make-1d-table))))
    (for-each-index n-variables
      (lambda (i)
	(vector-set! cells i (make-cell i constraint))))
    (for-each-vector-element (constraint-type/forget-rules type)
      (lambda (rules)
	(for-each (lambda (rule)
		    (if (null? (rule/inputs rule))
			(enqueue-rule/forget! rule constraint)))
		  rules)))
    constraint))

(define (constraint/find-cell constraint name)
  (let ((names (constraint-type/variables (constraint/type constraint))))
    (let ((end (vector-length names)))
      (let loop ((n 0))
	(and (< n end)
	     (if (eq? name (vector-ref names n))
		 (vector-ref (constraint/cells constraint) n)
		 (loop (1+ n))))))))

(define (constraint/cell constraint name)
  (or (constraint/find-cell constraint name)
      (error "no cell of this name" name constraint)))

(define-integrable (constraint/rule-queued? constraint rule)
  (odd? (quotient (constraint/queued-rules constraint) (rule/id-bit rule))))

(define (constraint/rule-queued! constraint rule)
  (set-constraint/queued-rules! constraint
				(+ (constraint/queued-rules constraint)
				   (rule/id-bit rule))))

(define (constraint/rule-dequeued! constraint rule)
  (set-constraint/queued-rules! constraint
				(- (constraint/queued-rules constraint)
				   (rule/id-bit rule))))

(define (constraint/id constraint)
  (intern
   (string-append
    (symbol->string (constraint-type/name (constraint/type constraint)))
    "-"
    (number->string (hash constraint)))))

(define-integrable (constraint/info constraint)
  (1d-table/get (constraint/properties constraint) 'INFO false))

(define-integrable (set-constraint/info! constraint info)
  (1d-table/put! (constraint/properties constraint) 'INFO info))

(define (constraint/inputs constraint rule)
  (let ((cells (constraint/cells constraint)))
    (map (lambda (input) (vector-ref cells input))
	 (rule/inputs rule))))

(define (constraint/output constraint rule)
  (let ((output (rule/output rule)))
    (and output
	 (vector-ref (constraint/cells constraint) output))))

;;;; Rules

(define unparse-rule
  (unparser/standard-method 'rule
    (lambda (state rule)
      (let ((type (rule/type rule))
	    (inputs (rule/inputs rule))
	    (output (rule/output rule)))
	(let ((unparse-pin
	       (let ((variables (constraint-type/variables type)))
		 (lambda (pin)
		   (unparse-object state (vector-ref variables pin))))))
	  (unparse-object state (constraint-type/name type))
	  (unparse-string state " (")
	  (if (not (null? inputs))
	      (begin
		(unparse-pin (car inputs))
		(let loop ((inputs (cdr inputs)))
		  (if (not (null? inputs))
		      (begin
			(unparse-string state ",")
			(unparse-pin (car inputs))
			(loop (cdr inputs)))))))
	  (unparse-string state ")")
	  (if output
	      (begin
		(unparse-string state "->")
		(if (rule/assumption? rule)
		    (begin
		      (unparse-string state "(")
		      (unparse-pin output)
		      (if (rule/nogood? rule)
			  (unparse-string state " nogood"))
		      (if (rule/nogoodbeg? rule)
			  (unparse-string state " nogoodbeg"))
		      (unparse-string state ")"))
		    (unparse-pin output)))))))))

(define-structure (rule
		   (conc-name rule/)
		   (print-procedure unparse-rule))
  (type false read-only true)
  (inputs false read-only true)
  (output false read-only true)
  (procedure false read-only true)
  (assumption? false read-only true)
  (id-bit false read-only true))

(define-integrable (rule/nogood? rule)
  (eq? (rule/assumption? rule) 'NOGOOD))

(define-integrable (rule/nogoodbeg? rule)
  (eq? (rule/assumption? rule) 'NOGOODBEG))

(define rule-result:dismiss "dismiss")
(define rule-result:contradiction "contradiction")

(define (rule/id rule)
  (intern
   (string-append (symbol->string (constraint-type/name (rule/type rule)))
		  "-rule-"
		  (number->string (hash rule)))))

(define (define-rules type . definitions)
  (let loop ((definitions definitions) (id-bit 1))
    (if (not (null? definitions))
	(begin
	  (let ((definition (car definitions)))
	    (define-rule type
	      (car definition)
	      (cadr definition)
	      (caddr definition)
	      id-bit
	      (cadddr definition)))
	  (loop (cdr definitions) (* id-bit 2))))))

(define (define-rule type inputs output assumption? id-bit procedure)
  (let ((name->index
	 (let ((variables (constraint-type/variables type)))
	   (lambda (name)
	     (let loop ((i (-1+ (vector-length variables))))
	       (cond ((negative? i) (error "Unknown name" name))
		     ((eq? (vector-ref variables i) name) i)
		     (else (loop (-1+ i)))))))))
    (let ((inputs (map name->index inputs))
	  (output (and output (name->index output))))
      (let ((rule
	     (make-rule type inputs output procedure assumption? id-bit)))
	(let ((add-rule!
	       (lambda (rules)
		 (lambda (index)
		   (vector-set! rules
				index
				(cons rule (vector-ref rules index)))))))
	  (for-each (add-rule! (constraint-type/add-rules type)) inputs)
	  (if output
	      ((add-rule! (constraint-type/forget-rules type)) output))
	  (if assumption?
	      ((add-rule! (constraint-type/nogood-rules type)) output))))))
  unspecific)

(define (make-dummy-rule)
  (make-rule false
	     '()
	     false
	     (lambda values (error "invoked dummy rule" values))
	     false
	     1))

(define parameter-rule (make-dummy-rule))
(define default-rule (make-dummy-rule))
(define constant-rule (make-dummy-rule))

(define-integrable (rule/parameter? rule) (eq? rule parameter-rule))
(define-integrable (rule/default? rule) (eq? rule default-rule))
(define-integrable (rule/constant? rule) (eq? rule constant-rule))

(define (rule/fixed? rule)
  (or (rule/parameter? rule)
      (rule/default? rule)
      (rule/constant? rule)))

;;;; Cells

(define unparse-cell
  (unparser/standard-method 'cell
    (lambda (state cell)
      (let ((owner (cell/owner cell)))
	(if owner
	    (unparse-object state (cell/good-name cell))
	    (unparse-object state (list (cell/name cell)))))
      (unparse-string state " ")
      (unparse-object state (cell/state cell))
      (let ((unparse-supplier-value
	     (lambda ()
	       (unparse-string state " ")
	       (unparse-object
		state
		(let ((supplier (node/supplier cell)))
		  (if (cell/king? supplier)
		      (node/value cell)
		      (list 'BAD-SUPPLIER (cell/state supplier))))))))
	(case (cell/state cell)
	  ((KING FRIEND)
	   (if (positive? (node/n-contradicting cell))
	       (unparse-string state " [opposed]"))
	   (unparse-string state " ")
	   (unparse-object state (cell/value cell)))
	  ((SLAVE)
	   (let ((supplier (node/supplier cell)))
	     (if (not (cell/puppet? supplier))
		 (unparse-supplier-value))))
	  ((REBEL DUPE)
	   (unparse-string state " ")
	   (unparse-object state (cell/value cell))
	   (unparse-string state " against")
	   (unparse-supplier-value))
	  ((PUPPET)
	   unspecific))))))

(define (cell/id cell)
  (intern
   (string-append (if (cell/owner cell)
		      "cell"
		      (symbol->string (cell/name cell)))
		  "-"
		  (number->string (hash cell)))))

(define (cell/normal-name cell)
  (if (cell/global? cell)
      (cell/id cell)
      (vector-ref
       (constraint-type/variables (constraint/type (cell/owner cell)))
       (cell/name cell))))

(define (cell/good-name cell)
  (cond ((cell/global? cell)
	 (cell/name cell))
	((let ((rule (cell/rule cell)))
	   (and rule
		(rule/fixed? rule)))
	 (cell/id cell))
	(else
	 `(THE ,(cell/normal-name cell) ,(constraint/id (cell/owner cell))))))

(define-structure (cell
		   (constructor %make-cell)
		   (conc-name cell/)
		   (print-procedure unparse-cell))
  (name false read-only true)
  (owner false read-only true)
  (state false)
  (contents false)
  (rule false)
  (repository false)
  (mark false)
  (equivalents false)
  (link false))

(define (make-cell name owner)
  (let ((repository (make-repository)))
    (let ((cell
	   (%make-cell name owner 'PUPPET false false repository 'UNMARKED '()
		       false)))
      (set-repository/cells! repository (list cell))
      (set-repository/supplier! repository cell)
      cell)))

(define (make-initialized-cell name rule contents)
  (let ((cell (make-cell name false)))
    (set-cell/state! cell 'KING)
    (set-cell/contents! cell contents)
    (set-cell/rule! cell rule)
    cell))

(define (reset-cell/state! cell state)
  (set-cell/state! cell state)
  (set-cell/contents! cell false)
  (if (not (rule/fixed? (cell/rule cell)))
      (set-cell/rule! cell false)))

(define-integrable (cell/king? cell)
  (eq? (cell/state cell) 'KING))

(define-integrable (cell/friend? cell)
  (eq? (cell/state cell) 'FRIEND))

(define-integrable (cell/rebel? cell)
  (eq? (cell/state cell) 'REBEL))

(define-integrable (cell/slave? cell)
  (eq? (cell/state cell) 'SLAVE))

(define-integrable (cell/dupe? cell)
  (eq? (cell/state cell) 'DUPE))

(define-integrable (cell/puppet? cell)
  (eq? (cell/state cell) 'PUPPET))

(define (cell/global? cell)
  (and (not (cell/owner cell))
       (not (cell/rule cell))))

(define (cell/true-supplier cell)
  (case (cell/state cell)
    ((KING FRIEND REBEL PUPPET) cell)
    ((SLAVE) (node/supplier cell))
    ((DUPE) (cell/contents cell))
    (else (error "bad cell state" cell))))

(define (cell/value cell)
  (case (cell/state cell)
    ((KING FRIEND REBEL) (cell/contents cell))
    ((SLAVE) (node/value cell))
    ((DUPE)
     (let ((c (cell/contents cell)))
       (if (not (and (cell/rebel? c) (node=? c cell)))
	   (error "bad DUPE indirection (from/to)" cell c))
       (cell/contents c)))
    (else
     (error "cell state incorrect for value" cell))))

(define (cell/ancestor? x y)
  (or (node=? x y)
      (let ((check-inputs
	     (lambda (y)
	       (for-all? (cell/inputs y)
		 (lambda (input)
		   (cell/ancestor? x input))))))
	(case (cell/state y)
	  ((KING REBEL) (check-inputs y))
	  ((FRIEND) (check-inputs (node/supplier y)))
	  ((DUPE) (check-inputs (cell/contents y)))
	  ((SLAVE) (and (node/bound? y) (check-inputs (node/supplier y))))
	  ((PUPPET) false)
	  (else (error "bad cell state" y))))))

(define (cell/inputs cell)
  (let ((inputs (rule/inputs (cell/rule cell))))
    (if (null? inputs)
	'()
	(let ((cells (constraint/cells (cell/owner cell))))
	  (map (lambda (input) (vector-ref cells input))
	       inputs)))))

;;;; Repositories

(define-structure (repository
		   (constructor %make-repository)
		   (conc-name repository/))
  (number false)
  (cells false)
  (supplier false)
  (nogoods false)
  (n-contradicting false))

(define (make-repository)
  (let ((repository (%make-repository false '() false '() 0)))
    (set-repository/number! repository (hash repository))
    repository))

(define-integrable (repository<? x y)
  (< (repository/number x) (repository/number y)))

(define-integrable (reset-repository/n-contradicting! repository)
  (set-repository/n-contradicting! repository 0))

(define (increment-repository/n-contradicting! repository)
  (set-repository/n-contradicting!
   repository
   (1+ (repository/n-contradicting repository))))

(define (decrement-repository/n-contradicting! repository)
  (set-repository/n-contradicting!
   repository
   (-1+ (repository/n-contradicting repository))))

;;;; Nodes

(define-integrable (node=? x y)
  (eq? (cell/repository x) (cell/repository y)))

(define-integrable (node<? x y)
  (repository<? (cell/repository x) (cell/repository y)))

(define-integrable (node/cells cell)
  (repository/cells (cell/repository cell)))

(define-integrable (set-node/cells! cell cells)
  (set-repository/cells! (cell/repository cell) cells))

(define-integrable (node/supplier cell)
  (repository/supplier (cell/repository cell)))

(define-integrable (set-node/supplier! cell supplier)
  (set-repository/supplier! (cell/repository cell) supplier))

(define-integrable (node/nogoods cell)
  (repository/nogoods (cell/repository cell)))

(define-integrable (set-node/nogoods! cell nogoods)
  (set-repository/nogoods! (cell/repository cell) nogoods))

(define-integrable (node/value cell)
  (cell/contents (node/supplier cell)))

(define-integrable (node/rule cell)
  (cell/rule (node/supplier cell)))

(define-integrable (node/n-contradicting cell)
  (repository/n-contradicting (cell/repository cell)))

(define-integrable (reset-node/n-contradicting! cell)
  (reset-repository/n-contradicting! (cell/repository cell)))

(define-integrable (increment-node/n-contradicting! cell)
  (increment-repository/n-contradicting! (cell/repository cell)))

(define-integrable (decrement-node/n-contradicting! cell)
  (decrement-repository/n-contradicting! (cell/repository cell)))

(define (node/bound? cell)
  (let ((supplier (node/supplier cell)))
    (case (cell/state supplier)
      ((KING) true)
      ((PUPPET) false)
      (else (error "bad supplier state" supplier)))))