/*
 * set_state.c
 * Front end for graph generation and state analysis
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	5/27/88 - Created DAR
 *	7/01/88 - split off state.c DAR
 *	8/01/88 - split into set_state.c and graph.c DAR
 */

/* headers */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "state.h"
#include "lattice.h"

/* macros */
#include "macros.c"
#include "pgm_macros.c"
#include "gpmacros.c"

/* function prototypes */
#include "protos/callpat.p"
#include "protos/state.p"
#include "protos/lattice.p"

/* debugging flags */
/*#define HIBUG	1*/		/* high level debugging */
/*#define LOBUG	1*/		/* low level debugging */

extern NameEntry 	*NmTable[], Names[];
extern Procedure 	procedures[];
extern SHORT 		from[], to[], litcount, QueryFunc;
extern SLOT 		*pos[], ListFunc, NilAtom;
extern SLOT		*tuple;

/* globals */
static STATETBL		*state[MaxNodes];
static SHORT		ref_count[MaxNodes], arc_index;

/*
 * void init_ref_count(SHORT nodes)
 * Initialize the reference count array.
 * A count of how many arcs lead into each node
 * is necessary so we can tell whether or not a
 * node is eligible for analysis
 */

void init_ref_count(nodes)
     SHORT nodes;
{
  SHORT	i;

  /* zero ref_counts */
  for (i=0; i < nodes; i++)
    ref_count[i] = 0;

  /* Even though arrays are allocated from 0 to max element,
   * pos/from/to only uses elements 1 to max element.  Hence,
   * we make init_ref_count only search the to array from 1
   * to the number of nodes allocated
   */
  for (i=1; i < litcount; i++)
    ref_count[ to[i] ]++;
  
#ifdef LOBUG
  printf("REFERENCE COUNTS\nNode\tCount\n");
  for (i=0; i < nodes; i++)
    printf("%d\t%d\n", i, ref_count[i]);
#endif
} /* end init_ref_count */

/*
 * SHORT select_node(SHORT num_nodes)
 * Finds a node whose reference count is zero
 * Returns the node number.  If no such node exists,
 * -1 is returned
 */
SHORT select_node(num_node)
     SHORT	num_node;
{
  SHORT	i;

  for (i=0; i < num_node; i++)
    if (ref_count [i] == 0) {
      ref_count [i] = -1;
      return i;
    }
  return -1;
}
/*
 * init_select_arc()
 * Initializes the select arc routine
 * Must be called each time user wishes to change the node
 * that arcs are to be selected from
 */
void init_select_arc()
{
  arc_index = 1;	/* don't start from 0 as from/to start from 1 */
}

/*
 * SHORT select_arc(SHORT node)
 * Returns an arc emanating from node.
 * If there are no more arcs, -1 is returned
 *   otherwise the arc is returned and the reference count of
 *   the node it leads to is decremented
 * last is the last index to check in the from/to/pos table
 */
SHORT select_arc(node)
     SHORT	node;
{

  for (; arc_index < litcount; arc_index++)
    if (from[arc_index] == node) {	/* if arc emanates from node */
      ref_count[to[arc_index]]--;	/* decrement ref count */
      return arc_index++;		/* return the value, then increment it */
    }
  return -1;
} /* end select_arc */

/*
 * void set_and_state(SLOT *head, CLAUSE *clause)
 * Performs state analysis and generates graphs for all
 * set {} notation.
 */
void set_and_state(head, clause)
     SLOT	*head;
     CLAUSE	*clause;
{
  SHORT 	i, new_next_node, back_patch, forward_patch, final_patch;
  SHORT		work_node, work_arc, num_vars;
  SLOT		*term, *next, tmp;
  STATETBL	*work_state;


  /* for now, return if top level fucntor */
  DEREFPTR(head);
  if (GET_FUNCTOR(head) == QueryFunc) {
    clause->states = NULL;
    return;
  }

  new_next_node = clause->num_nodes;
  num_vars = clause->num_vars;
#ifdef HIBUG
  printf("Next node %d\n", new_next_node);
  printf("TABLE BEFORE set_and_state\n");
  printf("\tINDEX\tFROM\tTO\tPROCEDURE\n");
  for (i=1; i < litcount; i++) {
    term = pos[i];
    DEREFPTR(term);
    tmp = GET_FUNCTOR(term);
    printf("\t%d\t%d\t%d\t%s/%d\n", i, from[i], to[i],
	   procedures[tmp].head, procedures[tmp].arity);
  }
#endif

  /* get state for node 0 --- after head unification */
  state[0] = headmap(head, num_vars);
  for (i=1; i < new_next_node; i++)	/* init other states */
    state[i] = NULL;
  
  /* Initialize reference counts, working variables */
  init_ref_count(new_next_node);
  work_state = NULL;
  
  /* while there are nodes that have not been processed */
  while ((work_node = select_node(new_next_node)) > -1) {
#   ifdef HIBUG
    printf("Node selected for analysis %d\n", work_node);
#   endif

    /* while there are arcs emanating from the selected node that
     * have not been processed */
    init_select_arc();
    while ((work_arc = select_arc(work_node)) > -1) {
#     ifdef HIBUG
      printf("Arc selected for expansion %d\n", work_arc);
#     endif
      /* As we are working with an arc going from work_node
       * to node to[work_node], copy the state of work_node
       */
      work_state = copy_statetbl(state[work_node], num_vars); 

#     ifdef LOBUG
      printf("work_state init'd as:\n");
      print_state(work_state);
#     endif

      if (CONS_CELL(pos[work_arc])) {
	/* arc is set notation, must build a graph */
	process_set(pos, work_state, state, clause, work_arc);
      }
      else {
	/* normal arc, determine state afterwards */
#	ifdef HIBUG
	printf("processing arc - text:  ");
	PrintTerm(pos[work_arc], tuple);
	OsPrint("\n");
#	endif
	arcmap(pos[work_arc], work_state); 	/* determine state after arc
						   finishes executing */
#	ifdef LOBUG
	printf("State after arcmap\n");
	print_state(work_state);
#	endif
      } /* end if (CONS_CELL(... then/else */

      /* update the node's state */
      if (state[to[work_arc]] == NULL) {	/* if first arc to node */
	state[to[work_arc]] = work_state;
	work_state = NULL;
      } else {	/* merge variable states of this arc with others leading
		   to the same node */
	further_bind(state[to[work_arc]], work_state);
      } /* end if (state[to[work_arc]] == NULL) then/else */
      
#     ifdef HIBUG
      printf("State %d updated to:  \n", to[work_arc]);
      print_state(state[to[work_arc]]);
#     endif
    } /* end while ((work_arc =...) > -1) */
  } /* end while ((work_node = ...) > -1) */

#ifdef HIBUG
  printf("Next node %d\n", new_next_node);
  printf("TABLE BEFORE set_and_state\n");
  printf("\tINDEX\tFROM\tTO\tPROCEDURE\n");
  for (i=1; i < litcount; i++) {
    term = pos[i];
    DEREFPTR(term);
    tmp = GET_FUNCTOR(term);
    printf("\t%d\t%d\t%d\t%s/%d\n", i, from[i], to[i],
	   procedures[tmp].head, procedures[tmp].arity);
  } /* end for (i=1...) */
#endif

  /* copy states to CLAUSE structure */
  if ((clause->states = (STATETBL **) malloc(sizeof(STATETBL *) * new_next_node))
      == NULL)
    cmperror(5, "Unable to allocate memory for states in set_and_state");
  else {
    for (i=0; i < new_next_node; i++) {
      clause->states[i] = state[i];
    }
  } /* end if ((clause->states...)) else ... */

  /* update number of nodes:
     Will only have changed if sets have been expanded */
  clause->num_nodes = new_next_node;
} /* end set_and_state */
