/*
 * lattice.c
 * Contains abstract type lattices and access routines
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	7/20/88 - Created DAR
 *
 */

/* includes */
#include "typedefs.h"
#include "pgm_typedefs.h"
#include "lattice.h"

/* externals */
extern char	*cmperrmsg;

/* globals */
/* none */

/* type table */
static TYPEENTRY typetable[] = {
  "g",	GROUND,
  "s",  SCALAR,
  "fs", FREE_SCALAR,
  "pg", PARTIAL_GROUND,	/* parental guidance suggested */
  "p",	PARTIAL,
  "f",	FREEVAR,
  "fg", FREE_GROUND,
  "d",	DONTKNOW,
  "e",	EMPTY
  };

/* local defines */

/* TYPE_COUNT is the number of types currently defined */
#define TYPE_COUNT sizeof(typetable) / sizeof(TYPEENTRY)
#define NONE -1	/* Non existant member of type system used for null links */

/*
 *	WARNING ON LATTICE DEFINITIONS:
 *		Although the type system is designed to be easily
 *		expandable and independent of the value bound to
 *		each type, the lattice definition is bound to the
 *		particular value defined to each of the types.
 *		The least upper bound functions use a simple table
 *		lookup.  Changing the definition values will result
 *		in the wrong entry being returned.
 */

/* call_pat_lat is the least upper bound data structure.  It is used to
 * compute the least upper bound of two data types
 *
 * Current Least Upper Bound lattice:
 *			DONTKNOW
 *		       /      | \
 *		      /	      |  \
 *		     /	      |	  \
 *		  PARTIAL_    |	   \
 *		  GROUND    GROUND_ \
 *		  /\	    FREE     \
 *		 /  \        /\	      \
 *		/    \      /  \_______\
 *	       /      \    /    \	\
 *	   PARTIAL    GROUND     SCALAR_ |
 *	      \		 |	 FREE    |
 *	       \         |   ____/   \	 |
 *		\      SCALAR	      FREE
 *		 \       |	       /
 *		  \	 |	      /
 *		   \     |	     /
 *		    \	 |  	    /
 *		     \   |   ______/
 *		      \  |  /
 *		       EMPTY
 */
			/* param 1 */
			  /* param 2 */	/* call_pat_lat */
static SHORT call_pat_lat[][9]	= {
  			/* EMPTY */	{
			  /* EMPTY */		EMPTY,
			  /* SCALAR */		SCALAR,
			  /* GROUND */		GROUND,
			  /* PARTIAL */		PARTIAL,
			  /* FREEVAR */		FREEVAR,
			  /* FREE_SCALAR */	FREE_SCALAR,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* SCALAR */ {
			  /* EMPTY */		SCALAR,
			  /* SCALAR */		SCALAR,
			  /* GROUND */		GROUND,
			  /* PARTIAL */		PARTIAL_GROUND,
			  /* FREEVAR */		FREE_SCALAR,
			  /* FREE_SCALAR */	FREE_SCALAR,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* GROUND */ {
			  /* EMPTY */		GROUND,
			  /* SCALAR */		GROUND,
			  /* GROUND */		GROUND,
			  /* PARTIAL */		PARTIAL_GROUND,
			  /* FREEVAR */		FREE_GROUND,
			  /* FREE_SCALAR */	FREE_GROUND,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* PARTIAL */ {
			  /* EMPTY */		PARTIAL,
			  /* SCALAR */		PARTIAL_GROUND,
			  /* GROUND */		PARTIAL_GROUND,
			  /* PARTIAL */		PARTIAL,
			  /* FREEVAR */		DONTKNOW,
			  /* FREE_SCALAR */	DONTKNOW,
			  /* FREE_GROUND */	DONTKNOW,
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* FREEVAR */ {
			  /* EMPTY */		FREEVAR,
			  /* SCALAR */		FREE_SCALAR,
			  /* GROUND */		FREE_GROUND,
			  /* PARTIAL */		DONTKNOW,
			  /* FREEVAR */		FREEVAR,
			  /* FREE_SCALAR */	FREE_SCALAR,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	DONTKNOW,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* FREE_SCALAR */ {
			  /* EMPTY */		FREE_SCALAR,
			  /* SCALAR */		FREE_SCALAR,
			  /* GROUND */		FREE_GROUND,
			  /* PARTIAL */		DONTKNOW,
			  /* FREEVAR */		FREE_SCALAR,
			  /* FREE_SCALAR */	FREE_SCALAR,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	DONTKNOW,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* FREE_GROUND */ {
			  /* EMPTY */		FREE_GROUND,
			  /* SCALAR */		FREE_GROUND,
			  /* GROUND */		FREE_GROUND,
			  /* PARTIAL */		DONTKNOW,
			  /* FREEVAR */		FREE_GROUND,
			  /* FREE_SCALAR */	FREE_GROUND,
			  /* FREE_GROUND */	FREE_GROUND,
			  /* PARTIAL_GROUND */	DONTKNOW,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* PARTIAL_GROUND */ {
			  /* EMPTY */		PARTIAL_GROUND,
			  /* SCALAR */		PARTIAL_GROUND,
			  /* GROUND */ 		PARTIAL_GROUND,
			  /* PARTIAL */		PARTIAL_GROUND,
			  /* FREEVAR */		DONTKNOW,
			  /* FREE_SCALAR */	DONTKNOW,
			  /* FREE_GROUND */	DONTKNOW,
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  },
			/* DONTKNOW */ {
			  /* EMPTY */		DONTKNOW,
			  /* SCALAR */		DONTKNOW,
			  /* GROUND */		DONTKNOW,
			  /* PARTIAL */		DONTKNOW,
			  /* FREEVAR */		DONTKNOW,
			  /* FREE_SCALAR */	DONTKNOW,
			  /* FREE_GROUND */	DONTKNOW,
			  /* PARTIAL_GROUND */	DONTKNOW,
			  /* DONTKNOW */	DONTKNOW
			  }
};	/* end SHORT call_pat_lat[][] */


/*
 * more instantiated lattice
 * in the following lattice, the higher in the lattice,
 * the more instantiated the type is:
 *			EMPTY
 *			  |
 *			GROUND
 *			/    \
 *		   PARTIAL  DONTKNOW
 *		        \    /
 *		       FREEVAR
 */
static SHORT mil[][9] = {
  			/* EMPTY */	{
			  /* EMPTY */		EMPTY,
			  /* SCALAR */		EMPTY,        
			  /* GROUND */		EMPTY, 
			  /* PARTIAL */		EMPTY, 
			  /* FREEVAR */		EMPTY, 
			  /* FREE_SCALAR */	EMPTY, 
			  /* FREE_GROUND */	EMPTY, 
			  /* PARTIAL_GROUND */	EMPTY, 
			  /* DONTKNOW */	EMPTY 
			  },		   
			/* SCALAR */ {		
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		SCALAR, 
			  /* PARTIAL */		SCALAR, 
			  /* FREEVAR */		SCALAR, 
			  /* FREE_SCALAR */	SCALAR, 
			  /* FREE_GROUND */	SCALAR, 
			  /* PARTIAL_GROUND */	SCALAR, 
			  /* DONTKNOW */	SCALAR
			  },		   
			/* GROUND */ {		
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		GROUND, 
			  /* FREEVAR */		GROUND, 
			  /* FREE_SCALAR */	GROUND, 
			  /* FREE_GROUND */	GROUND, 
			  /* PARTIAL_GROUND */	GROUND, 
			  /* DONTKNOW */	GROUND
			  },		   
			/* PARTIAL */ {		
			  /* EMPTY */		EMPTY,
			  /* SCALAR */		SCALAR,
			  /* GROUND */		GROUND,
			  /* PARTIAL */		PARTIAL, 
			  /* FREEVAR */		PARTIAL, 
			  /* FREE_SCALAR */	PARTIAL, 
			  /* FREE_GROUND */	PARTIAL_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND, 
			  /* DONTKNOW */	PARTIAL
			  },		   
			/* FREEVAR */ {		
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		PARTIAL, 
			  /* FREEVAR */		FREEVAR, 
			  /* FREE_SCALAR */	FREE_SCALAR, 
			  /* FREE_GROUND */	FREE_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND, 
			  /* DONTKNOW */	DONTKNOW
			  },		   
			/* FREE_SCALAR */ {	
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		PARTIAL, 
			  /* FREEVAR */		FREE_SCALAR, 
			  /* FREE_SCALAR */	FREE_SCALAR, 
			  /* FREE_GROUND */	FREE_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND, 
			  /* DONTKNOW */	FREE_SCALAR
			  },		   
			/* FREE_GROUND */ {	
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		PARTIAL_GROUND, 
			  /* FREEVAR */		FREE_GROUND, 
			  /* FREE_SCALAR */	FREE_GROUND, 
			  /* FREE_GROUND */	FREE_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND, 
			  /* DONTKNOW */	FREE_GROUND
			  },		   
			/* PARTIAL_GROUND */ {	
			  /* EMPTY */		EMPTY, 
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		PARTIAL_GROUND, 
			  /* FREEVAR */		PARTIAL_GROUND, 
			  /* FREE_SCALAR */	PARTIAL_GROUND, 
			  /* FREE_GROUND */	PARTIAL_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	PARTIAL_GROUND
			  },		   
			/* DONTKNOW */ {	
			  /* EMPTY */		EMPTY,
			  /* SCALAR */		SCALAR, 
			  /* GROUND */		GROUND, 
			  /* PARTIAL */		PARTIAL, 
			  /* FREEVAR */		DONTKNOW, 
			  /* FREE_SCALAR */	FREE_SCALAR,
			  /* FREE_GROUND */	FREE_GROUND, 
			  /* PARTIAL_GROUND */	PARTIAL_GROUND,
			  /* DONTKNOW */	DONTKNOW
			  }
};	/* end SHORT mil_lat[][] */




/*
 * SHORT lub_callpat(SHORT type1, type2)
 * Given two elements of the type domain, determines the least
 * least upper bound of the two in the call pattern lattice.
 *
 * Currently uses a simple table lookup.
 * When the type domain grows, a more elegant solution will
 * need to be used in order to prevent exponential explosion.
 */
SHORT lub_callpat(type1, type2)
     SHORT	type1, type2;
{
  return call_pat_lat[type1][type2];
}

/*
 * SHORT lub_instanstiation(SHORT type1, SHORT type2)
 * Given two elements of the type domain, returns the
 * least upper bound of the instantiation lattice.
 */
SHORT lub_instantiation(type1, type2)
     SHORT	type1, type2;
{
  return mil[type1][type2];
} /* end lub_instantiation */

/* SHORT find_type(SHORT *name)
 * Finds the type value associated with a string.
 * If the name is not in the type system, type
 * DONTKNOW is returned and a warning message is issued.
 */
SHORT find_type(name)
     SHORT *name;
{
  SHORT	i;

  for(i=0; i < TYPE_COUNT; i++)
    if (! strcmp(name, typetable[i].name))
      return typetable[i].entry;
  sprintf(cmperrmsg, "Warning:  Bad type name: %s treating as DONTKNOW", name);
  cmperror(0, cmperrmsg);
  return DONTKNOW;
} /* end find_type */
