/**************************************************************************
 * entry.c
 * 
 * Abstract:	Generates the connection graph for a Prolog program
 *		that has already been parsed.
 *
 * Author:  	David Roch
 * Histroy:	06/13/88	created rev version after discussion w/Sanjay
 *
 **************************************************************************/

/* header files */
#include "typedefs.h"
#include "pgm_typedefs.h"
#include "entry.h"

#include "pgm_macros.c"
#include "gpmacros.c"
#include "entry_macros.c"
#include "macros.c"


extern NameEntry	*NmTable[NmHashVal];
extern Procedure	procedures[MaxProcs];
extern SLOT		ListFunc, NilAtom;

ENTRYPT	*entry_list;		/* list of entry points */
char	cmperrmsg[256];		/* string for cmperror users to write to */

/*
 * cmperror(level, message)
 * reports a compiler error to standard error.
 * message is printed on the stderr stream
 * if level > 0 then the program is exited using that value
 */

cmperror(level, mesg)
     SHORT level;
     char *mesg;
{
  fprintf(stderr, "%s\n", mesg);
  if (level > 0)
    exit(level);
}

/*
 * show_entries()
 * Writes entry list to stdout
 * Primarily for debugging purposes
 */
void show_entries()
{
  ENTRYPT *element;
  
  printf("Entry points:\n");
  element = entry_list;
  while (element != NULL) {
    printf("   %s/%d flags (octal) %o\n",
	   procedures[element->proc].head,
	   procedures[element->proc].arity,
	   procedures[element->proc].flags);
    element = element->next;
  }
}


/*
 * init_entry_list
 * Initialize the entry list
 */
void init_entry_list()
{
  entry_list = NULL;
}

/*
 * entry_dec(SLOT term)
 * Parse an entry declaration.
 * May either be a single entry or a list of entries
 */
void entry_dec(term) 
     SLOT term;
{
  SHORT		arity, functor;
  SLOT		head, tail, proc;
  ENTRYPT	*new_entry;
  char		go;
  
  arity = ARITY((SLOT *) term);
  if (arity != 1) {
    sprintf(cmperrmsg, ":- entry/%d - arity must be 1\n", arity);
    cmperror(0, cmperrmsg);
  } else {
    tail = term += sizeof(SLOT); /* move to argument of callpat */
    DEREF(tail);
    go = 1;
    while (go) {
      functor = GET_FUNCTOR((SLOT *) tail);
      arity = ARITY((SLOT *) tail);
      if (CONS_CELL(tail)) {
	head = tail + sizeof(SLOT);	/* head of list */
	DEREF(head);
	NEXT_CELL(tail);		/* tail of list */
	DEREF(tail);
      } else {
	head = tail;	/* no need to DEREF, done last iteration */
	go = 0; /* end of list or no list */
      }
      /* head points to the predicate to be processed */
      functor = GET_FUNCTOR((SLOT *) head);
      if (functor != GET_FUNCTOR(&NilAtom)) {
	arity = ARITY((SLOT *) head);
	if (! strcmp("/",procedures[functor].head) &&
	    procedures[functor].arity == 2) {	/* process / */
	  proc = head += sizeof(SLOT);
	  DEREF(proc);
	  functor = GET_FUNCTOR((SLOT *) proc);
#	  ifdef DARDEBUG
	     printf("Functor being processed %s/%d\n",
		    procedures[functor].head, procedures[functor].arity);
#	  endif
	  head += sizeof(SLOT);
	  DEREF(head);
	  if (TAG_IS_INTEGER((SLOT *) head)) {
	    arity = INTVALUE((SLOT *) head);
	    /* Find atom associated with name of procedure, then find
	     * the procedure of atom and arity
	     */
	    functor = ProcFor(PgmLookupAtom(procedures[functor].head), arity);
	    Malloc_Entrypt(new_entry);	/* allocate entry */
	    new_entry->proc = functor;
	    if (entry_list == NULL) {
	      entry_list = new_entry;
	      new_entry->next = NULL;
	    } else {
	      new_entry->next = entry_list;
	      entry_list = new_entry;
	    } /* end if (entry_list == NULL) */
	    /* set flags to indicate entry point & not dead code */
	    SET_FLAG(procedures[functor].flags, VISITED);
	    SET_FLAG(procedures[functor].flags, ENTRY);
	  } else {
	    sprintf(cmperrmsg, "Warning: entry(%s/NON_INTEGER).",
		 procedures[functor].head);
	    cmperror(0, cmperrmsg);
	  } /* end if TAG_IS_INTEGER */
	}
	else {
	  cmperror(0, "Warning:  bad entry declaration");
	} /* end if '/'/2 */
      } /* end if (functor != GET_FUNCTOR(&NilAtom)) */
    } /* while (go) */
  } /* end if (arity != 1) else { */
} /* end entry_dec */

