/*
 * directive.c
 * Parses compiler directives
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *	KAH - Keith Hamburg
 *
 * HISTORY:
 *	6/25/88 - Created DAR
 *	7/04/88 - Added conjunctive directives :- a, b, ..., z. DAR
 *	7/30/88 - Merged in KAH's routines for predicate compiler directives
 */

/* header files */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "directive.h"

#include "macros.c"
#include "pgm_macros.c"
#include "gpmacros.c"

/* externals */
extern	Procedure	procedures[];
extern	char		cmperrmsg[];
extern	SLOT		CommaFunc, ListFunc, NilAtom;
extern 	BOOLEAN		DEBUG2;

/* local defines */
#define MaxDet 1024

/*-------------------- Start of KAH's routines -------------------*/
/* * * * * * * * * * * * * * * * * * */
/*                                   */
/* Process Compiler Directives:      */
/* 	:- det()                     */
/* 	:- nofail()                  */
/* 	:- mutex()                   */
/* 	:- seqex()                   */
/*                                   */
/* * * * * * * * * * * * * * * * * * */

static struct 
	{
	char	name[30];
	char	type[7];
	SHORT	arity;
	}	detstore[MaxDet];
static SHORT det_num = 0;


/* SetDet()
 * 
 * is actually mis-named.  It follows the parse-tree for a compiler 
 *   directive and stores the name of the directive and its type for use
 *   in UserSetDet(). (see below)
 * The only types of compiler directives that this will ever be called
 *   with, unless we add more, are:    (see directive.c)
 *
 * 	:- det()         <-- Hence, the name SetDet.
 * 	:- nofail()
 *	:- mutex()
 *	:- seqex()
 */
void SetDet(term,type)
     SLOT term;
     char *type;
{
  SHORT	arity, functor, i, call, succ, call_typ, succ_typ;
  SLOT	arityterm, slot, head, tail, arg, args;
  char	go;
  
  arity = ARITY((SLOT *) term);
  if (arity != 1) {
    fprintf(stderr, ":- callpat/%d - arity must be 1\n", arity);
  } else {
    tail = term += sizeof(SLOT); /* move to argument of callpat */
    DEREF(tail);
    go = 1;
    while (go) {
      functor = GET_FUNCTOR((SLOT *) tail);
      arity = ARITY((SLOT *) tail);
      if (CONS_CELL(tail)) {
	head = tail + sizeof(SLOT);	/* head of list */
	DEREF(head);
	NEXT_CELL(tail);		/* tail of list */
	DEREF(tail);
      } else {
	head = tail;	/* no need to DEREF, done last iteration */
	go = 0; /* end of list or no list */
      }
      /* head points to the predicate to be processed */
      functor = GET_FUNCTOR((SLOT *) head);
      if (functor != GET_FUNCTOR(&NilAtom)) {
        head += sizeof(SLOT);
        arityterm = head + sizeof(SLOT);
        DEREF(head);
        functor = GET_FUNCTOR((SLOT *) head);
        DEREF(arityterm);
        if (TAG_IS_INTEGER((SLOT *)arityterm))  {
          /* Store the name, type, and arity of this 
             directive for later use in UserSetDet() */
          strcpy(detstore[det_num].name,procedures[functor].head);
          strcpy(detstore[det_num].type,type);
          detstore[det_num].arity = INTVALUE((SLOT *) arityterm);
          det_num++;
          } /* end if (TAG_IS_INTEGER ... */
      else {
        printf("Warning: %s(%s/NON_INTEGER),\n",type,procedures[functor].head);
        }
      } /* end if (functor != GET_FUNCTOR(&NilAtom)) */
    } /* while (go) */
  } /* end if (arity != 1) else { */
} /* end callpat_dec */

/* UserSetDet()
 * 
 * Set the following bits in the FLAGS field for a procedure:
 *
 * 	DETERMINISTIC
 *	CANNOT_FAIL
 * 	CLAUSES_MUT_EX
 * 	CLAUSES_SEQ_EX
 *
 * This is done after all parsing has been completed. This assures 
 *   that all of the procedure entries have been created before 
 *   attempting to modify them.
 */

UserSetDet()
   {
   SHORT i;
   SHORT functor;

   for (i=0; i<det_num; i++)
      {
      functor = niProcFor(PgmLookupAtom(detstore[i].name),detstore[i].arity);
      if (functor != -1)
         {
         if (strcmp("det",detstore[i].type) == 0)
            {
            SET_FLAG(procedures[functor].flags,DETERMINISTIC);
	    TRACE2(printf("SetDet: "));
	    TRACE2(PrVector1W(procedures[functor].flags));
	    TRACE2(printf("\n"));
            }
         else if (strcmp("nofail",detstore[i].type) == 0)
            {
            SET_FLAG(procedures[functor].flags,CANNOT_FAIL);
	    TRACE2(printf("SetDet: "));
	    TRACE2(PrVector1W(procedures[functor].flags));
	    TRACE2(printf("\n"));
            }
         else if (strcmp("mutex",detstore[i].type) == 0)
            {
            SET_FLAG(procedures[functor].flags,CLAUSES_MUT_EX);
	    TRACE2(printf("SetDet: "));
	    TRACE2(PrVector1W(procedures[functor].flags));
	    TRACE2(printf("\n"));
            }
         else if (strcmp("seqex",detstore[i].type) == 0)
            {
            SET_FLAG(procedures[functor].flags,CLAUSES_SEQ_EX);
	    TRACE2(printf("SetDet: "));
	    TRACE2(PrVector1W(procedures[functor].flags));
	    TRACE2(printf("\n"));
            }
         else 
            {
            fprintf(stderr,"Warning: directive %s not allowed.\n",detstore[i].type);
            }
         TRACE2(show_proc_and_flags(functor));
         } /* end if (functor != -1) */
       else 
         {
         fprintf(stderr,"Warning: directive %s(%s/%d) -- %s/%d does not exist\n"
                       ,detstore[i].type,detstore[i].name,detstore[i].arity
                       ,detstore[i].name,detstore[i].arity);
         }
      } /* end for (i ... */
   }

show_proc_and_flags(functor)
SLOT functor;
   {
   printf("Procedure: %s/%d -- flags = %d\n",procedures[functor].head,
           procedures[functor].arity, procedures[functor].flags);
   }




/*------------------------end of KAH's code-------------------------*/
void ParseDirective(slot)
     SLOT *slot;
{
  register char	i;
  SHORT		functor, arity, proc;
  SLOT		term, next;
  char		go;

  slot++;	/* move over :- to directive */
  term = *slot;
  DEREF(term);
  go = 1;
  while (go) {
    if (COMMA(term)) {
      next = term + sizeof(SLOT);
      DEREF(next);
      term = term + (sizeof(SLOT) << 1) ; /* move term to next functor */
      DEREF(term);
    } else {
      next = term;
      go = 0;
    } /* end if COMMA */
    functor = GET_FUNCTOR((SLOT *) next);
    TRACE2(printf("%s/%d\n", procedures[functor].head, procedures[functor].arity));
    for (i=0; i < DIR_COUNT; i++) {
      if (! strcmp(procedures[functor].head, directives[i]))
	break;
    }
    if (i == DIR_COUNT) {
      sprintf(cmperrmsg, "Warning:  Unknown compiler directive - %s/%d",
	      procedures[functor].head, procedures[functor].arity);
      cmperror(0, cmperrmsg);
    }
    else {
      /* When each directive is called, it is given a slot which
       * contains the address of the functor associated with the
       * directive.  It has already been dereferenced, so there
       * is no need for the called routine to perform a DEREF.
       */
      functor = GET_FUNCTOR((SLOT *) next);
      switch (directives[i].id) {
      case DIR_ENTRY:
	entry_dec(next);
	break;
      case DIR_CALLPAT:
	callpat_dec(next);
	break;
      case DIR_DET:
        SetDet(next, "det");
        break;
      case DIR_NOFAIL:
        SetDet(next, "nofail");
        break;
      case DIR_MUTEX:
        SetDet(next, "mutex");
        break;
      case DIR_SEQEX:
        SetDet(next, "seqex");
        break;
      } /* end switch (directives[i].id) */
    } /* end else */
  } /* end while (go) */
} /* end ParseDirective */
