/*----------------------------------------------------------------------
 *		cg.c - code generator module
 *
 *		write pseudo-code from pgm data structure
 *--------------------------------------------------------------------*/

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "pgm.h"
#include "macros.c"
#include "pgm_macros.c"
#include "be_macros.c"
#include "cg.h"
#include "instruct.h"
#include <varargs.h>
#include "bit_macros.c"

#define	FUNCTOR_NUM(x) ((LONG)(x) >> SYM_TAB_SHIFT)
CLAUSE *firstClause();

extern char OBJFILE[];
extern char PUREFILE[];
extern LONG CONS;
extern SHORT MAXLINKSINPROGRAM;

/*=====  CodeGen - entry point for code generator  =====*/

CodeGen()
{
	cgInit();
	EmitCode();
	DumpPureCode(term_space,next_term);
}


/*=====	cgInit - initializations for code generator =====*/

cgInit()
{
	LONG i;

	InitReg();    /* mark all pseudo-regs free */
	Instr = (Instruction *)calloc( sizeof(Instruction), MAXINSTR );
	Symtab = (Label *)calloc( sizeof(Label), MAXSYM );
	Entrytab = (LONG *)calloc( sizeof(Entry), MaxProcs );
	/*=====  initialize instruction argument area */
	ArgSpace = argPtr = (LONG *)calloc( sizeof(LONG), ARGSPACES );
	for( i = 0; i < ARGSPACES; i++ )
		ArgSpace[ i ] = NULLPTR;
	/*=====  clear generated instructions buffer */
	for( i = 0; i < MAXINSTR; i++ ) {
		Instr[i].opcode = -1;
		Instr[i].arg = NULL;
	}
	Loc = 0;    /* location counter */
	LabelCount = 0;     /* number of labels */
}


/*----------------------------------------------------------------------
 *		EmitCode - generate pseudo code
 *
 *		traverse data structures for parsed source and drive clause-by-clause
 *		code generation routines.
 *---------------------------------------------------------------------*/


EmitCode()
{
	Procedure *p;
	CLAUSE *c, *cstk[100];
	register LONG i, j;
	LONG cstkPtr, arcsFired;
	BOOLEAN allbuiltin, ismain = FALSE;
	BOOLEAN HeadHasTerms();
	BOOLEAN needswitch, AllTermsSameType();
	LONG nextE;
	LONG statusbits;

	emit( nop );
	/*===== process each clause of each procedure.
	 *===== do this in reverse order of clause list (i.e. source code order) */
	for(p = procedures; p < procs; p++) 
	{
	    SetEntryPoint( p-procedures, Loc );
	    for(cstkPtr = 0, c = p->clauses; c != NULL; c = c->next )
	        cstk[ cstkPtr++ ] = c;
	    if (cstkPtr > 0)
	    {
		ismain = (strcmp(p->head, "main$") == 0);
		needswitch = (HeadHasTerms(cstk[0]) && 
			      !AllTermsSameType(cstk, cstkPtr-1));
		statusbits = 0;
		if (IN_FLAG(p->flags, DETERMINISTIC))
		    statusbits |= (STAT_DET | STAT_PARENT_DET); 
		if (IN_FLAG(p->flags, CANNOT_FAIL))
		    statusbits |= STAT_ALWAYS_SUC;
		if (IN_FLAG(p->flags, CLAUSES_SEQ_EX))
		{
		    /* generate sequential code for predicate */
		    SetLabel( cstk[cstkPtr-1],HEAD );   /* entry pt from nonseq */
		    emit( EnterSeqCode );
		    SetLabel( cstk[cstkPtr-1],SEQHEAD ); /* entrypt from seq */
		    if (needswitch)
		        Emit( SwitchOnTerm, "llll", cstk[cstkPtr-1], ADDRCONST,
			     cstk[cstkPtr-1], ADDRATOM, cstk[cstkPtr-1], 
			     ADDRLIST, cstk[cstkPtr-1], ADDRSTRUCT);  
		    for( i = cstkPtr-1; i >= 0; i-- , nextE = -3)
		    {
			if (needswitch)
			    for(j = cstkPtr - 1; j >= i; j--)
			        SetSwitchLabels(cstk[j], cstk[i]);
			EmitSeqHead( cstk[i], ismain, &nextE);
			EmitSeqBody( cstk[i], ismain, nextE );
			if (i > 0)
			{
			    SetLabel( cstk[i], nextE); 
			    if (needswitch)
			    Emit( SwitchOnTerm, "llll", cstk[i-1], ADDRCONST,
				 cstk[i-1], ADDRATOM, cstk[i-1], ADDRLIST, 
				 cstk[i-1], ADDRSTRUCT);  
/*			    SetLabel( cstk[i], nextE); */
			}
		    }
		}
		else  
		{
		    SetLabel(cstk[cstkPtr-1], HEAD);
		    if (needswitch)
		    Emit( SwitchOnTerm, "llll", cstk[cstkPtr-1], ADDRCONST,
			 cstk[cstkPtr-1], ADDRATOM, cstk[cstkPtr-1], 
			 ADDRLIST, cstk[cstkPtr-1], ADDRSTRUCT); 
		    for( i = cstkPtr-1; i >= 0; i-- )
		    {
			if (needswitch)
			for(j = cstkPtr - 1; j >= i; j--)
			    SetSwitchLabels(cstk[j], cstk[i]);
			nextE = GenArcLabel();
			EmitHead(cstk[i], ismain, &arcsFired, &allbuiltin, 
				 nextE, MUTEXJUMP, statusbits);

			SetLabel( cstk[i], nextE );  
			if (!IN_FLAG(p->flags, CLAUSES_MUT_EX)) 
			    SetLabel( cstk[i], MUTEXJUMP );  
			if (needswitch && i > 0)
			Emit( TryMeElse, "llll", cstk[i-1], ADDRCONST,
			     cstk[i-1], ADDRATOM, cstk[i-1], ADDRLIST,
			     cstk[i-1], ADDRSTRUCT);  
			if (i == 0)
			{
			    if (needswitch)
			    for( j = cstkPtr-1; j >= 0; j-- )
			        ResolveSwitchLabels(cstk[j]);
			    if (!ismain)
			    {
				if (IN_FLAG(p->flags, CLAUSES_MUT_EX)) 
				   for(j = cstkPtr-1; j >= 0; j-- )
				       SetLabel( cstk[j], MUTEXJUMP );  
				if ( !(statusbits & STAT_DET) ||
				     !(statusbits &  STAT_ALWAYS_SUC))
				    emit( SendControlMsg );
				emit (Suspend );
			    }
			    else if (allbuiltin)
			         emit( Exit );
 			    else emit( Suspend );
			}
		    }
		    for( i = cstkPtr-1; i >= 0; i-- )
		        if (!allbuiltin)
		            EmitBody(cstk[i], ismain, arcsFired, statusbits);
		}
	    }
	}
	/*=====  set startup address */
	SetStartAddress();
	/*=====  write instruction buffer to output file */
	cgDumpCode();
}


/*======================================================================
 *	EmitBody - generate code for body of clause c
 *
 *	for each arc of the djg for clause c,
 *	emit code to process a response on the arc.  join with other arcs
 *	feeding into the destination node, then fire that node on tuples
 *	successfully joined.
 *=====================================================================*/

EmitBody( c, ismain, arcsFired, statusbits)
CLAUSE *c;
BOOLEAN ismain;
LONG arcsFired;
LONG statusbits;
{
    SHORT lit, j, succNode;
    LONG tmparcsFired;
    BOOLEAN isbuiltin, solnSent;
    SHORT predcount;  
    LONG  SetStatusBits();
    
    TRACE(OsPrint("EmitBody: %d arcs\n", c->num_arcs));
    if (c->num_arcs == 0) return;

    isbuiltin = 0;	
    for( lit=1; lit <= c->num_arcs; lit++ ) 
    {
	if(arcsFired & (1<< ARC(lit)) && (c->builtin_index[lit] != 0))
	   continue;
	solnSent = FALSE;
	/*==	define entry point for this arc =====*/
	SetLabel( c, lit);
	succNode = c->succ_node[ ARC(lit) ];
	predcount = c->pred_count[ succNode ];
	statusbits |= SetStatusBits(c, lit);
	if (!isbuiltin)
	    Emit( CheckResponse, "dd", succNode, statusbits );
	if( predcount > 1 ) 
	{
	    /* branches should either send solutions or fire the same
	       arcs (emanating from succNode). After that has been done,
	       code for the responses for those arcs needs to be generated */
	    tmparcsFired = Join(c,succNode,ARC(lit), ismain, statusbits);
/*	    printf("arcs fired after join are %x, %x\n",tmparcsFired, c->pred_arcs[1]);  */
	}
	else tmparcsFired = Fire(c, succNode, -3, FALSE, FALSE, ismain, statusbits);
	if (solnSent = (succNode==1 || (tmparcsFired & c->pred_arcs[1]) != 0)) 
	     emit( Suspend );  /* needs cleaning up */
	else emit( Suspend );
	arcsFired |= tmparcsFired;
    }

    /*==	emit code for last arc  ==*/
    if(!solnSent) 
    {
	if (!ismain)
	    Emit( SendClauseResp, "d", MAXLINKSINPROGRAM);
	emit( Suspend );
   }
}


/*======================================================================
 *		Join - join response on arc and insert extended tuples in node
 *=====================================================================*/

Join( c, node, arc, ismain, statusbits)
CLAUSE *c;		  /* clause whose djg we are processing */
LONG node;		  /* node that response arc feeds into */
LONG arc;		  /* arc that is currently being processed */
BOOLEAN ismain;		  /* is this the top level query? */
LONG statusbits; 	  /* determinacy and nofail info */
{
    SHORT i, arcindex;
    LONG  jumplabel1, jumplabel2, jumplabel3;
    SHORT otherarc;
    EXTEND *backptrs = &(c->NodeIndices[node]);
    FWDBACK *arcinfo = &(c->ArcIndices[arc]);
    SHORT chainindex;
    SHORT Pbptr_idx, index;
    LONG arcsFired;
    BOOLEAN allbuiltin; 

    arcindex = JoinArcIndex(c, arc, node);
    jumplabel3 = GenArcLabel();
    if ((Pbptr_idx = arcinfo->Pbptr_idx) <= 0)
         Pbptr_idx -= c->maxlinks;   /* displace by max links */

    Emit( SetUpJoin, "dddddl",  arcindex, Pbptr_idx, 
	 			arcinfo->fptr_other, arcinfo->fptr_me, 
	 			arcinfo->arclinkindex, c, jumplabel3);
    jumplabel1 = GenArcLabel();
    SetLabel(c, jumplabel1);
    if (arcinfo->MCA_num > 0)
    {
	otherarc = 0;
	while(!BitInVector1W(c->pred_arcs[node], otherarc) || otherarc == arc)
	    otherarc++;
	for(i=0; i < arcinfo->MCA_num; i++)
	    Emit(MCACheck,"ddddl",arcindex, arcinfo->MCA_me[i] - c->maxlinks,
		 arcinfo->MCA_other[i] - c->maxlinks, 
		 c->ArcIndices[otherarc].arclinkindex, c, jumplabel3);
    }
    Emit( JoinTuples,"dddd", node, c->maxlinks,
	 			   backptrs->count, c->fwd_ptr[node]);
    for(i=0; i< backptrs->count; i++)
    {
	if ((index = backptrs->index[i]) <= 0)
	     index -= c->maxlinks;   /* displace by max links */

	if ( backptrs->nodecode[i] > 0)  /* points to a pred node */
	     Emit(CopyBackPtr, "dddd", -i-c->maxlinks,	/* -ve for access */
		 CopyPtrIndex(c,backptrs->nodecode[i],node), index, 1);
	else Emit(CopyBackPtr, "dddd", -i-c->maxlinks,	/* -ve for access */
		  backptrs->nodecode[i], index, 1);
    }
    jumplabel2 = GenArcLabel();
    arcsFired = Fire(c, node, jumplabel2, FALSE, TRUE, ismain, statusbits); 
    SetLabel(c, jumplabel2);
    Emit( NextTuple, "ddl" , arcindex , arcinfo->arclinkindex, c, jumplabel1 );
    SetLabel(c, jumplabel3);
    return(arcsFired);
}



/*========================================================================
 *		Fire - emit code to start subgoal for each arc leaving a djg node
 *
 *		FireArc	arcNum,goalEntry,respEntry,pureCode
 *		is generated for each arc leaving *node*:
 *
 *		respEntry is the object code address of the code to process a response
 *		for the subgoal.  a response on an arc will fire the next node on the
 *		djg, after successful join if there are multiple incoming arcs.
 *
 *		pureCode is the pure code index of the literal labeling the arc.
 *========================================================================*/

LONG Fire( clause, node, nextEntry, ishead, isjoin, ismain, statusbits)
CLAUSE *clause;
SHORT node;
LONG nextEntry;
BOOLEAN ishead;           /* is the call made by EmitHead? */
BOOLEAN isjoin;           /* is the call made by Join? */
BOOLEAN ismain;           /* is the call made by main clause? */
LONG    statusbits; 	  /* determinacy and nofail info */
{
    register LONG i;
    register LONG arcVector;
    LONG lit, j, arcsfired = 0;
    SHORT  numforks, NumForksFromNode();
    SHORT  numbackptrs =  clause->NodeIndices[node].count,
           numfwdptrs  =  clause->fwd_ptr[node];
    EXTEND *backptrs;
    BOOLEAN BuiltinsinFork(), savednodetup, foundnonbuiltin; 
    BOOLEAN needarctuple = FALSE;
    LONG    SetStatusBits();
    SHORT   index;
    
    arcVector = clause->succ_arcs[ node ];
    arcsfired |= arcVector;
    if ((numforks = NumForksFromNode( arcVector )) > MAXFORKS)
    {
	OsPrint("Current implementation restricts outgoing arcs from a node to %d. Aborting compilation\n", MAXFORKS);
	exit();
    }
    if (!isjoin & !clause->sequential)
    {
	if (numforks > 1 && (ishead || (numbackptrs + numfwdptrs > 0)))
	{
	    Emit( SetUpNodeTuple, "dddd",node, clause->maxlinks,
		 			 numbackptrs, numfwdptrs);
	    if (ishead)
	    {
#ifdef PRIORITY
		Emit( CreateNewQuery, "ddddd", clause->num_nodes, 
		     clause->num_arcs, RopmGetQuerySize(clause),
		     clause->numClauseBits, clause->clauseNum);   
#else
		Emit( CreateNewQuery, "ddd", clause->num_nodes, 
		     clause->num_arcs, RopmGetQuerySize(clause) );   
#endif
		ishead = FALSE;
	    }
	}
	backptrs = &(clause->NodeIndices[node]);
	for(i=0; i< backptrs->count; i++)
	{
	    if ((index = backptrs->index[i]) <= 0)
	         index -= clause->maxlinks;
            Emit(CopyBackPtr, "dddd", -i - clause->maxlinks, 1, index, 0);
	}
    }				/* Note: "-i" -ve indices for access */
    /*--  find arcs leading out of *node*.  succ_arcs[node] is a LONG with
     *--  bit i set for each arc i emanating from *node*   */
    if (savednodetup = (numforks > 1 && BuiltinsinFork(clause, arcVector)))
        Emit( PushStack, "d", NODETUPSTACK);
    for( lit=1; lit <= clause->num_arcs; lit++ )
         if( SUCCESSOR( arcVector, ARC(lit) ) ) 
	 {
	     TRACE(OsPrint("Fire node %d: arcs %x\n", node, arcVector));
	     if( foundnonbuiltin = (clause->builtin_index[ lit ] == 0))   
	     {					
		 SHORT temp;
		 CLAUSE *c;
		 
		 if (ishead)
		 {
		     Emit( SetUpNodeTuple, "dddd",node, clause->maxlinks,
			   numbackptrs, numfwdptrs);
#ifdef PRIORITY
		     Emit( CreateNewQuery, "ddddd", clause->num_nodes, 
			  clause->num_arcs, RopmGetQuerySize(clause),
			  clause->numClauseBits, clause->clauseNum );
#else
		     Emit( CreateNewQuery, "ddd", clause->num_nodes, 
			  clause->num_arcs, RopmGetQuerySize(clause) );   
#endif
		     ishead = FALSE;
		 }
		 temp = GET_FUNCTOR( ((SLOT *)(clause->lit_position[lit])) );
		 c = procedures[temp].clauses;
		 if (c == NULL) 
		 {
		     OsPrint("clause definition ");
		     OsPrint("``%s/%d''", PgmGetAtom(GET_FUNCTOR(clause->lit_position[lit])), ARITY(clause->lit_position[lit]));
		     OsPrint(" not found - aborting compilation!\n");
		     exit(1);
		 }
		 while( c->next != NULL ) c = c->next;
		 statusbits |= SetStatusBits(clause, lit);
#ifdef PRIORITY
		 Emit( FireArc, "ddllpddd", ARC(lit), node, c, HEAD, clause,lit,
		       clause->lit_position[lit], statusbits,
		       clause->node_distance[node], clause->numSeqLitBits);
#else
		 Emit( FireArc, "ddllpd", ARC(lit), node, c, HEAD, clause, lit,
		       clause->lit_position[lit], statusbits );
#endif
	     }
	     else 
	     {
		 SHORT nextnode;

		 GenBuiltinCode(clause, clause->lit_position[lit], nextEntry,
				clause->builtin_index[lit]);
		 TRACE(OsPrint("builtin functor\n"));
		 nextnode = clause->succ_node[ARC(lit)];
		 if (clause->pred_count[nextnode] > 1)
		 {
		     if (isjoin)
		         Emit( PushStack, "d", JOINSTACK );
		     arcsfired |= Join(clause, nextnode, ARC(lit), ismain,
				       statusbits, needarctuple);
		     needarctuple = TRUE;
		     if (isjoin)
		         Emit( PopStack, "d", JOINSTACK );
		 }
		 else arcsfired |= Fire(clause, nextnode, nextEntry, ishead,
					isjoin, ismain, statusbits);
	     }
	     if (savednodetup && !foundnonbuiltin && --numforks > 0)
	         Emit( TopStack, "d", NODETUPSTACK);
	 }
    if (node == 1 && !ismain)
        Emit( SendClauseResp, "d", MAXLINKSINPROGRAM );
    if (savednodetup)
        Emit( PopStack, "d", NODETUPSTACK);

    return arcsfired;
}



LONG SetStatusBits(c, lit)
CLAUSE *c;
SHORT lit;
{
    LONG   statusbits = 0;
    SHORT  litpos;
    Procedure *p;

    if (PgmLitIsDet(c, ARC(lit)))
        statusbits |= STAT_DET; 
    if (PgmLitIsNoFail(c, ARC(lit)))
        statusbits |= STAT_ALWAYS_SUC;

    litpos = GET_FUNCTOR(c->lit_position[lit]);
    p = &procedures[litpos];
    if (IN_FLAG(p->flags, DETERMINISTIC))
        statusbits |= (STAT_DET | STAT_PARENT_DET); 
    if (IN_FLAG(p->flags, CANNOT_FAIL))
        statusbits |= STAT_ALWAYS_SUC;
    return statusbits;
}



CLAUSE *firstClause( term )
SLOT *term;
{
	CLAUSE *PgmClauses();

	if( SLOT_TAG( term ) != FUNCTOR && SLOT_TAG( term ) != CLOSED_TERM ) {
		TRACE(OsPrint("firstClause: term is not functor!\n"));
		exit(1);
	}
	TRACE(OsPrint("firstClause: got functor,its name is "));
	TRACE(OsPrintf("%s",PgmGetAtom(GET_FUNCTOR(term))));
	TRACE(OsPrint("\n"));
	return( PgmClauses( term ) );
}



/*======================================================================
 *	symbol table routines
 *
 *	a symbol is a pair (clause,arc).  symbols are bound to LONG ints
 *	usually representing pcode addresses of entry points.  a symbol
 *	can be defined at the current location by calling SetLabel.  a
 *	symbol can be referenced by calling RefLabel whether or not the
 *	symbol has been defined.  if it is not defined, the reference is
 *	placed on a deferred list and resolved by a SetLabel call which
 *	defines the address.
 *====================================================================*/

/*
 *	lookup - find or add entry to entry point symbol table
 *
 *	the "name" of a symbol is the pair (clause,arc) corresponding to
 *	the location of that label.  (labels are only generated at the
 *	start of clause code and the start of response code).  if the
 *	specified name does not exist it is added.
 *
 *	entry		"symbol name"
 *	exit		name added to symbol table if not already there
 *	returns	pointer to symbol table entry
 */

Label *Lookup( clause, arc )
LONG clause;
SHORT arc;
{
	register LONG i;

	for( i=0; i<LabelCount; i++ )
		if( Symtab[ i ].idClause == clause && Symtab[i].idArc == arc ) 
			return &Symtab[i];
	Symtab[LabelCount].idClause = clause;
	Symtab[LabelCount].idArc = arc;
	Symtab[LabelCount].address = -1;
	Symtab[LabelCount].refChain = NULL;
	return &Symtab[LabelCount++];
}

/*
 *		setlabel - define entry point at current location counter
 *
 *		the label is enterd in the symbol table if it is not already
 *		there, and its address is set to Loc.
 *		if there are pending references, these are resolved at this point.
 */

SetLabel( clause, arc )
CLAUSE *clause;
SHORT arc;
{
        BOOLEAN LabelIsSet();
	Label *sp;
	IChain *ic;

	if(LabelIsSet(clause, arc)) {
	    TRACE(OsPrint("SetLabel: attempted to redefine entry for clause %d arc %d\n",
			clause, arc ));
		TRACE(OsPrint("address is already %ld\n", sp->address ));
	}
	sp = Lookup( (LONG)clause, arc );
	sp->address = Loc;
	TRACE(OsPrint("SetLabel: set <" ));
	TRACE(OsPrintf("%s",PgmGetAtom(GET_FUNCTOR( (SLOT *)(clause->lit_position[0]) ) )));
	if( arc == HEAD )
		TRACE(OsPrint(",HEAD"));
	else
		TRACE(OsPrint(",%d", arc ));
	TRACE(OsPrint("> to %d\n", Loc ));

	/*--- resolve any pending references to this symbol */
	ic = sp->refChain;
	while( ic != NULL ) {
		ic->instr->arg[ ic->arg ] = Loc;
		ic = ic->next;
	}
	sp->refChain = NULL;
}


BOOLEAN LabelIsSet(clause, entry)
CLAUSE *clause;
LONG entry;
{
    Label *sp;

    sp = Lookup( (LONG) clause, entry);
    return(sp->address != -1);
}

/*
 *		reflabel - reference an arbitrary label
 *
 *		if the label has not been defined yet, this reference is added
 *		to the unresolved reference chain for the label
 */

RefLabel( clause, arc, arg )
CLAUSE *clause;
SHORT arc, arg;
{
	Label *sp;
	IChain *ic;

	sp = Lookup( (LONG)clause, arc );
	if( sp->address != -1 )
		return( sp->address );
	else {
		ic				= (IChain *)malloc( sizeof( IChain ) );
		ic->instr	= &Instr[ Loc ];
		ic->arg		= arg;
		ic->next		= sp->refChain;
		sp->refChain= ic;
		return( -2 );	/* so we will notice unresolved references */
		/*Instr[Loc].arg[arg] = -2;    /* so we will notice unresolved refs */
	}
}


/*
 *		GenArcLabel - return fake arc number
 */

GenArcLabel()
{
	static LONG nextLabel = ARCFREE;

	return nextLabel++;
}


/*======================================================================
 *		SetEntryPoint - add to entry point table
 *
 *		arguments are the functor index and the byte code address
 *======================================================================*/

SetEntryPoint( proc, address )
LONG proc, address;
{
	Entrytab[ proc ] = address;
}


/*======================================================================
 *
 *		"register" allocation routines
 *
 *====================================================================*/

AllocReg()
{
	register LONG count;

	for( count = 0; count < MAXREG; count++ )
		if( regMap[ count ] == 0 ) {
			regMap[ count ] = 1;
			return( count );
		}
	OsPrint("AllocReg - out of registers\n");
	exit(1);
}


FreeReg( reg )
LONG reg;
{
	regMap[ reg ] = 0;
}


InitReg()
{
	register LONG count;

	regMap = (LONG *)malloc( MAXREG * sizeof(*regMap) );
	for( count = 0; count < MAXREG; count++ )
		regMap[ count ] = 0;
}


/*======================================================================
 *
 *		code emitter routines
 *
 *====================================================================*/

cgDumpCode()
{
	LONG i, pi;
	register LONG j;
	FILE *objfd;

	/*=====  open object code file for output.  fd is objfd */
	objfd = fopen(OBJFILE, "w");

	if( objfd == NULL ) {
		OsPrint("can't open %s for output\n", OBJFILE);
		exit(1);
	}

	/*=====  write instructions */
	for( i=0; i<Loc; i++ ) {
		fprintf( objfd,"%d,%d,%s\t",Instr[i].opcode,Instr[i].argCount,Instr[i].argpat);
		for( j=0; j<Instr[i].argCount-1; j++ )
			fprintf( objfd,"%d ", Instr[i].arg[j] );
		if( j < Instr[i].argCount )
			fprintf( objfd, "%d", Instr[i].arg[j] );
		fprintf( objfd, "\n");
	}
	fprintf( objfd, "%%%%\n");

	/*=====  write ProcNames (procedure names, one per line) */
	for( pi = 0; pi < procs-procedures; pi++ )
		fprintf( objfd, "%s %d\n", procedures[pi].head, procedures[pi].arity );
	fprintf( objfd, "%%%%\n");

	/*=====  write TopNames (names of variables in query, one per line) */
	for( pi = 1; TopNames[pi] != 0; pi++ )
		fprintf( objfd, "%s\n", TopNames[pi] );
	fprintf( objfd, "%%%%\n");

	/*--- clean up */
	fclose( objfd );
}


/*----------------------------------------------------------------------
 *		emit an instruction
 *
 *		Emit( opcode, ctrl, operand1, operand2, ... );
 *		each char of "ctrl" describes an argument, as follows:
 *			d	long integer
 *			l	label, expect clause and arc consecutively
 *			p	pure code index.  unlike d, will be "relocated".
 *
 *		NOTE that labels (l) require two operands.
 *---------------------------------------------------------------------*/

Emit( va_alist )
va_dcl
{
	va_list arg;
	char *ctrl, c;
	LONG operand, opnum, opcode;
	CLAUSE *clause;

	va_start(arg);
	opcode = va_arg( arg, LONG );    /* first Emit argument */
	ctrl = va_arg( arg, char * );    /* second Emit argument */
	Instr[ Loc ].arg = argPtr;       /* set pointer into arg space */
	opnum = 0;
	while( c = ctrl[opnum] ) {
		switch( c ) {
		case 'd':
			*argPtr++ = va_arg( arg, LONG );
			break;
		case 'l':
			clause = va_arg( arg, CLAUSE * );
			operand = va_arg( arg, LONG );
			*argPtr++ = RefLabel( clause, operand, opnum );
			break;
		case 'p':
			*argPtr = PCINDEX( va_arg( arg, SLOT * ));
			*argPtr++;
			break;
		default:
			break;
		}
		opnum++;
	}
	Instr[ Loc ].opcode = opcode;
	Instr[ Loc ].argCount = opnum;
	Instr[ Loc ].argpat = ctrl;
	Loc++;
	va_end(arg);
}


/*
 *		set the opcode field of instruction 0 to the start address of the
 *		program, i.e. the generated code for the query
 */

SetStartAddress()
{
	LONG start;
	SLOT topterm;
	CLAUSE *c, *PgmClauses();

	start = Entrytab[ PgmMainFunctor() ];
	MainEntryPt() = ( start==0 ? -1 : start );
	INSERT_CLOSED_TERM(&topterm, PgmMainFunctor(), 0);
	c = PgmClauses(&topterm);
	InsertTopQueryData(c);
}



/* JoinArcIndex returns 1 if  a given join arc has a lower arc number 
 * than the "other" join arc, 0 otherwise.
 * Caveat: Obviously works only for binary joins 			*/

JoinArcIndex(c, joinarc, joinnode)
CLAUSE *c;
SHORT joinarc,joinnode;
{
    SHORT i;
    
    for(i=0; i<c->num_arcs; i++)
	if (BitInVector1W(c->pred_arcs[joinnode], i))
	{
	    if (i > joinarc) 
	        return 0;
	    if (i < joinarc) 
	        return 1;
	}
    return -1;  /* error */
}



/* CopyPtrIndex tells us which context tuple to use to copy pointers into a
   newly created node tuple - given the pred. index it is to come from */

CopyPtrIndex(c,prednode, joinnode)
CLAUSE *c;
SHORT prednode, joinnode;
{
    SHORT i;
    
    for(i=0; i<c->num_arcs; i++)
	if (BitInVector1W(c->pred_arcs[joinnode], i))
	{
	    if (c->pred_node[i] == prednode)
		 return 1;
	    else return 2;
	}
    return -1;  /* error */
}


HeadTupSize(c)
CLAUSE *c;
{
    if (c->sequential || c->num_nodes == 1)
         return(c->num_vars);
    else return(c->num_vars + c->fwd_ptr[0]);
}




GenBuiltinCode(c, term, nextEntry, bindex)
CLAUSE *c;
SLOT  *term;
LONG nextEntry;
SHORT bindex;
{
    SLOT *subterm1, *subterm2, *subterm3;

    switch (bindex)
    {
    case BUILTIN_WRITE:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    Emit(Write ,"p", subterm1);
	    break;
    case BUILTIN_NL:
	    emit( Newline);
	    break;
    case BUILTIN_IS:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2) &&
		    INDEX(subterm1) != INDEX(subterm2))
		    Emit( ISVarVar, "ddl", INDEX(subterm1), INDEX(subterm2), 
			 c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		        Emit(ISVarInt,"ddl",INDEX(subterm1),INTVALUE(subterm2),
			     c, nextEntry);
		else
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( ISVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		    Emit( ISIntVar,"ddl",INTVALUE(subterm1), INDEX(subterm2), 
			 c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) != INTVALUE(subterm2))
		        Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
	        else
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( ISIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else 
	    {
		OsPrint("illegal slot tag on left side of IS\n Compilation aborted.\n");
		exit(1);
	    }
	    break;
    case BUILTIN_NE:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    if (INDEX(subterm1) == INDEX(subterm2))
		         Emit( Fail, "l", c, nextEntry);
		    else Emit( NEVarVar,"ddl",INDEX(subterm1), INDEX(subterm2),
			      c, nextEntry);
		}
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( NEVarInt, "ddl", INDEX(subterm1),INTVALUE(subterm2),
			  c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( NEVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( NEVarInt,"ddl", INDEX(subterm2),INTVALUE(subterm1),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) == INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( NEIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( NEVarAcc, "dl", INDEX(subterm2), c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( NEIntAcc, "ddl", INTVALUE(subterm2), 0, c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( NEAccAcc, "ddl", 0, 1, c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_LT:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    if (INDEX(subterm1) == INDEX(subterm2))
		         Emit(Fail, "l", c, nextEntry);
		    else Emit(LTVarVar,"ddl", INDEX(subterm1), INDEX(subterm2),
			      c, nextEntry);
		}
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( LTVarInt, "ddl", INDEX(subterm1),INTVALUE(subterm2),
			  c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( LTVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( LTIntVar,"ddl", INTVALUE(subterm1), INDEX(subterm2),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) >= INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( LTIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( LTAccVar, "ddl", 0, INDEX(subterm2), c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( LTAccInt, "ddl", 0, INTVALUE(subterm2), c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( LTAccAcc, "ddl", 0, 1, c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_GT:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    if (INDEX(subterm1) == INDEX(subterm2))
		         Emit(Fail, "l", c, nextEntry);
		    else Emit(GTVarVar,"ddl", INDEX(subterm1), INDEX(subterm2),
			  c, nextEntry);
		}
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( GTVarInt, "ddl", INDEX(subterm1),INTVALUE(subterm2), c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( GTVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( GTIntVar,"ddl", INTVALUE(subterm1), INDEX(subterm2),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) <= INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( GTIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( GTAccVar, "ddl", 0, INDEX(subterm2), c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( GTAccInt, "ddl", 0, INTVALUE(subterm2), c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( GTAccAcc, "ddl", 0, 1, c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_EQ:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2) &&
		    INDEX(subterm1) != INDEX(subterm2))
		     Emit( EQVarVar, "ddl", INDEX(subterm1), INDEX(subterm2), 
			  c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQVarInt, "ddl", INDEX(subterm1),INTVALUE(subterm2),
			  c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQVarInt,"ddl", INDEX(subterm2), INTVALUE(subterm1),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) != INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQVarAcc, "ddl", INDEX(subterm2), 0, c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQIntAcc, "ddl", INTVALUE(subterm2), 0, c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( EQAccAcc, "ddl", c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_DIV:
	    break;
    case BUILTIN_EQLT:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2) &&
		    INDEX(subterm1) != INDEX(subterm2))
		     Emit( EQLTVarVar,"ddl", INDEX(subterm1), INDEX(subterm2), 
			  c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQLTVarInt,"ddl",INDEX(subterm1),INTVALUE(subterm2),
			  c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQLTVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQLTIntVar,"ddl",INTVALUE(subterm1),INDEX(subterm2),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) > INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQLTIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQLTAccVar, "ddl", 0, INDEX(subterm2), c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQLTIntAcc, "ddl", INTVALUE(subterm2), 0, c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( EQLTAccAcc, "ddl", 0, 1, c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_EQGT:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2) &&
		    INDEX(subterm1) != INDEX(subterm2))
		     Emit( EQGTVarVar,"ddl", INDEX(subterm1), INDEX(subterm2),
			  c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQGTVarInt,"ddl",INDEX(subterm1),INTVALUE(subterm2),
			  c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQGTVarAcc, "ddl", INDEX(subterm1), 0, c, nextEntry);
		}
	    }
	    else if (TAG_IS_INTEGER(&term[1]))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQGTIntVar,"ddl",INTVALUE(subterm1),INDEX(subterm2),
			  c,nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		{
		    if (INTVALUE(subterm1) < INTVALUE(subterm2))
		       Emit(Fail, "l", c, nextEntry);
		 /* else you succeed - do nothing */
		}
		else    
		{
		    Evaluate(subterm2, 0, 0);
		    Emit( EQGTIntAcc, "ddl", INTVALUE(subterm1), 0, c, nextEntry);
		}
	    }
	    else
	    {
		Evaluate(subterm1, 0, 0);
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( EQGTAccVar, "ddl", 0, INDEX(subterm2), c, nextEntry);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( EQGTAccInt, "ddl", 0, INTVALUE(subterm2), c, nextEntry);
	        else 
		{
		    Evaluate(subterm2, 1, 0);
		    Emit( EQGTAccAcc, "ddl", 0, 1, c, nextEntry);
		}
	    }
	    break;
    case BUILTIN_VAR:
	    if (TAG_IS_TUPLE_INDEX_1(&term[1]))
	         Emit(Var, "dl", INDEX(&term[1]), c, nextEntry );
	    else Emit(Fail, "l", c, nextEntry );
	    break;
    case BUILTIN_NONVAR:
	    if (TAG_IS_TUPLE_INDEX_1(&term[1]))
	         Emit(NonVar, "dl", INDEX(&term[1]), c, nextEntry );
	    break;
    case BUILTIN_FUNCTOR:
	    Emit(BuiltinFunc, "pdl", term, c->maxlinks, c, nextEntry);
	    break;
    case BUILTIN_ATOM:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    if (TAG_IS_FUNCTOR(subterm1))   
	    {
	        if (!ARITY_IS_ZERO(subterm1))
		      Emit(Fail, "l",  c, nextEntry );
	    }
	    else if (TAG_IS_TUPLE_INDEX_1(subterm1))
	              Emit(Atom, "pl",subterm1,  c, nextEntry );
	    else Emit(Fail, "l",  c, nextEntry );
	    break;
    case BUILTIN_ARG:
	    Emit(BuiltinArg, "pl", term, c, nextEntry);
	    break;
    case BUILTIN_INT:
	    if (!TAG_IS_INTEGER(&term[1]))
	        if (TAG_IS_TUPLE_INDEX_1(&term[1]))
	              Emit(Int, "dl",INDEX(&term[1]),  c, nextEntry );
	         else Emit(Fail, "l",  c, nextEntry );
	    break;
    case BUILTIN_EQUAL:
	    GenBuiltinEqual(c, nextEntry, &term[1], &term[2]);
	    break;
    case BUILTIN_NEQUAL:
	    GenBuiltinNEqual(c, nextEntry, &term[1], &term[2]);
	    break;
    case BUILTIN_UNIV:
	    Emit(BuiltinUniv, "pl", term, c, nextEntry);
	    break;
    case BUILTIN_PRVARS:
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    Emit(PrintVars ,"pl", subterm1, c, nextEntry );
	    break;
    case HALT:
	    emit( Halt);
	    break;
    case NULLFACT:
	    emit( nop );
	    break;
    case CUT:
	    emit( Cut );
	    break;
    default:  
        OsPrint("error illegal builtin index found in program\n");
	break;
    }
}



Evaluate(term, regno, tempno)
SLOT *term;
SHORT regno;   /* put result in regno */
SHORT tempno;  /* tempno temporaries in use so far */
{
    extern SLOT FUNCTOR_PLUS, FUNCTOR_MINUS, FUNCTOR_MULTIPLY,
    	        FUNCTOR_DIVIDE, FUNCTOR_MOD, FUNCTOR_DIV;

    SLOT *subterm1, *subterm2;

    switch (SLOT_TAG(term))
    {
    case FUNCTOR:
    case CLOSED_TERM:
	if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_PLUS))
	{
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( CopyVartoAcc, "dd", INDEX(subterm2), regno);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( CopyInttoAcc, "dd", INTVALUE(subterm2), regno);
		else Evaluate(subterm2, regno, tempno);
		Emit( AddVartoAcc, "dd", INDEX(subterm1), regno);
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( CopyVartoAcc, "dd", INDEX(subterm2), regno);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( CopyInttoAcc, "dd", INTVALUE(subterm2), regno);
		else Evaluate(subterm2, regno, tempno);
		Emit( AddInttoAcc, "dd", INTVALUE(subterm1), regno);
	    }
	    else 
	    {
		Evaluate(subterm1, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm2, regno, tempno);
		Emit( AddTemp, "dd", --tempno, regno);
	    }
	}
	else if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_MINUS))
	{
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( SubVarFromAcc, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( SubIntFromAcc, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( SubtractTemp, "dd", --tempno, regno);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( SubVarFromAcc, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( SubIntFromAcc, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( SubtractTemp, "dd", --tempno, regno);
		}
	    }
	    else
	    {
		Evaluate(subterm2, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm1, regno, tempno);
		Emit( SubtractTemp, "dd", --tempno, regno);
	    }
	}
	else if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_MULTIPLY))
	{
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( CopyVartoAcc, "dd", INDEX(subterm2), regno);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( CopyInttoAcc, "dd", INTVALUE(subterm2), regno);
		else Evaluate(subterm2, regno, tempno);
		Emit( MultVartoAcc, "dd", INDEX(subterm1), regno);
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( CopyVartoAcc, "dd", INDEX(subterm2), regno);
		else if (TAG_IS_INTEGER(subterm2))
		     Emit( CopyInttoAcc, "dd", INTVALUE(subterm2), regno);
		else Evaluate(subterm2, regno, tempno);
		Emit( MultInttoAcc, "dd", INTVALUE(subterm1), regno);
	    }
	    else 
	    {
		Evaluate(subterm1, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm2, regno, tempno);
		Emit( MultiplyTemp, "dd", --tempno, regno);
	    }
	}
	else if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_DIVIDE))
	{
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivideTemp, "dd", --tempno, regno);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivideTemp, "dd", --tempno, regno);
		}
	    }
	    else
	    {
		Evaluate(subterm2, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm1, regno, tempno);
		Emit( DivideTemp, "dd", --tempno, regno);
	    }
	}
	else if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_MOD))
	{
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( ModAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( ModAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( ModTemp, "dd", --tempno, regno);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( ModAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit(ModAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( ModTemp, "dd", --tempno, regno);
		}
	    }
	    else
	    {
		Evaluate(subterm2, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm1, regno, tempno);
		Emit( ModTemp, "dd", --tempno, regno);
	    }
	}
	else if (GET_FUNCTOR(term) == GET_FUNCTOR(&FUNCTOR_DIV))
	{   /* for now it is the same as divide */
	    subterm1 = &term[1];
	    Chase_Term(subterm1, NULL);
	    subterm2 = &term[2];
	    Chase_Term(subterm2, NULL);
	    if (TAG_IS_TUPLE_INDEX_1(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyVartoAcc, "dd", INDEX(subterm1), regno);
		    Emit( DivideTemp, "dd", --tempno, regno);
		}
	    }
	    else if (TAG_IS_INTEGER(subterm1))
	    {
		if (TAG_IS_TUPLE_INDEX_1(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivAccbyVar, "dd", INDEX(subterm2), regno);
		}
		else if (TAG_IS_INTEGER(subterm2))
		{
		    Emit( CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivAccbyInt, "dd", INTVALUE(subterm2), regno);
		}
		else
		{
		    Evaluate(subterm2, regno, tempno);
		    Emit(StoreinTemp, "dd", tempno++, regno);
		    Emit(CopyInttoAcc, "dd", INTVALUE(subterm1), regno);
		    Emit( DivideTemp, "dd", --tempno, regno);
		}
	    }
	    else
	    {
		Evaluate(subterm2, regno, tempno);
		Emit( StoreinTemp, "dd", tempno++, regno);
		Evaluate(subterm1, regno, tempno);
		Emit( DivideTemp, "dd", --tempno, regno);
	    }
	}
	break;

    case TUPLE_INDEX_1:
	Emit( CopyVartoAcc, "dd", INDEX(term), regno);
	break;
    case INTEGER:
	Emit( CopyInttoAcc, "dd", INTVALUE(term), regno);
	break;

    default: 
	OsPrint("Illegal slot tag found in evaluate\n");
	OsPrint("Compilation aborted.\n");
	exit(1);
    }
}







GenBuiltinEqual(c, nextEntry, subterm1, subterm2)
CLAUSE *c;
LONG nextEntry;
SLOT *subterm1, *subterm2;
{
    SHORT i;

    Chase_Term(subterm1, NULL);
    Chase_Term(subterm2, NULL);
    if (TAG_IS_FUNCTOR(subterm1))
    {
	if (ARITY_IS_ZERO(subterm1))
	{
	    if (TAG_IS_FUNCTOR(subterm2))
	    {
		if (!ARITY_IS_ZERO(subterm2))
		     Emit( Fail, "l", c, nextEntry);
	    }
	    else if (!TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( Fail, "l", c, nextEntry);
	    else Emit( UnifyVarAtom, "dpl", INDEX(subterm2),subterm1,
		       c, nextEntry);
	}	
	else
	{
	    if (TAG_IS_FUNCTOR(subterm2))
	    {
		if (ARITY(subterm1) != ARITY(subterm2))
		     Emit( Fail, "l", c, nextEntry);
		for(i=1; i <= ARITY(subterm1); i++)
		    GenBuiltinEqual(c, nextEntry, &subterm1[i], &subterm2[i]);
	    }
	    else if (!TAG_IS_TUPLE_INDEX_1(subterm2))
		     Emit( Fail, "l", c, nextEntry);
	    else Emit(UnifyVarTerm, "dpl", INDEX(subterm2), subterm1,
		       c, nextEntry);
	}
    }
    else if (TAG_IS_INTEGER(subterm1))
    {
	if (TAG_IS_FUNCTOR(subterm2))
	     Emit( Fail, "l", c, nextEntry);
	else if (TAG_IS_INTEGER(subterm2))
	{
	    if (INTVALUE(subterm1) != INTVALUE(subterm2))
	         Emit( Fail, "l", c, nextEntry);
	}
	else if (!TAG_IS_TUPLE_INDEX_1(subterm2))
	     Emit( Fail, "l", c, nextEntry);
	else Emit( UnifyVarInt, "pdl", INDEX(subterm2),subterm1,
		   c, nextEntry);
    }
    else if (!TAG_IS_TUPLE_INDEX_1(subterm1))
    {
	OsPrint("error! illegal slot tag in lhs of BuiltinEqual\n");
	OsPrint("Compilation aborted\n");
	exit(1);
    }
    else
    {
	if (TAG_IS_FUNCTOR(subterm2))
	{
	    if (ARITY_IS_ZERO(subterm2))
	         Emit( UnifyVarAtom, "dpl", INDEX(subterm1), subterm2, 
		       c, nextEntry);
	    else Emit( UnifyVarTerm, "dpl", INDEX(subterm1), subterm2, 
		       c, nextEntry);
	}
	else if (TAG_IS_INTEGER(subterm2)) 
	     Emit( UnifyVarInt, "ddl", INDEX(subterm1), 
		   INTVALUE(subterm2), c, nextEntry);
	else if (!TAG_IS_TUPLE_INDEX_1(subterm2))
	     Emit( Fail, "l", c, nextEntry);
	else if (INDEX(subterm1) != INDEX(subterm2))
	     Emit( UnifyVarVar, "ddl", INDEX(subterm1), INDEX(subterm2),
		  c, nextEntry);
    }
}



GenBuiltinNEqual(c, nextEntry, subterm1, subterm2)
CLAUSE *c;
LONG nextEntry;
SLOT *subterm1, *subterm2;
{
    SHORT i;

    Chase_Term(subterm1, NULL);
    Chase_Term(subterm2, NULL);
    if (TAG_IS_FUNCTOR(subterm1))
    {
	if (ARITY_IS_ZERO(subterm1))
	{
	    if (TAG_IS_FUNCTOR(subterm2))
	    {
		if (ARITY_IS_ZERO(subterm2) ||
		    GET_FUNCTOR(subterm1) == GET_FUNCTOR(subterm2))
		     Emit( Fail, "l", c, nextEntry);
	    }
	    else if (TAG_IS_TUPLE_INDEX_1(subterm2))
	         Emit( NEqualVarAtom, "dpl", INDEX(subterm2), subterm1,
		       c, nextEntry);
	}
	else
	{
	    if (TAG_IS_FUNCTOR(subterm2))
	    {
		if (GET_FUNCTOR(subterm1) == GET_FUNCTOR(subterm2) &&
		    ARITY(subterm1) == ARITY(subterm2))
		for(i=1; i <= ARITY(subterm1); i++)
		    GenBuiltinNEqual(c, nextEntry, &subterm1[i], &subterm2[i]);
	    }
	    else if (TAG_IS_TUPLE_INDEX_1(subterm2))
	         Emit( NEqualVarTerm, "dpl", INDEX(subterm2), subterm1,
		       c, nextEntry);
	}
    }
    else if (TAG_IS_INTEGER(subterm1))
    {
	if (TAG_IS_INTEGER(subterm2))
	{
	    if (INTVALUE(subterm1) == INTVALUE(subterm2))
	         Emit( Fail, "l", c, nextEntry);
	}
	else if (TAG_IS_TUPLE_INDEX_1(subterm2))
	     Emit( NEqualVarInt, "ddl", INDEX(subterm2), INTVALUE(subterm1),
		   c, nextEntry);
    }
    else if (!TAG_IS_TUPLE_INDEX_1(subterm1))
    {
	OsPrint("error! illegal slot tag in lhs of BuiltinNEqual\n");
	OsPrint("Compilation aborted\n");
	exit(1);
    }
    else
    {
	if (TAG_IS_FUNCTOR(subterm2))
	{
	    if (ARITY_IS_ZERO(subterm2))
	         Emit( NEqualVarAtom, "dpl", INDEX(subterm1), subterm2, 
		       c, nextEntry);
	    else Emit( NEqualVarTerm, "dpl", INDEX(subterm1), subterm2, 
		       c, nextEntry);
	}
	else if (TAG_IS_INTEGER(subterm2)) 
	     Emit( NEqualVarInt, "ddl", INDEX(subterm1), 
		   INTVALUE(subterm2), c, nextEntry);
	else if (TAG_IS_TUPLE_INDEX_1(subterm2))
	{
	    if (INDEX(subterm1) != INDEX(subterm2))
	         Emit( NEqualVarVar, "ddl", INDEX(subterm1), INDEX(subterm2),
		      c, nextEntry);
	    else Emit( Fail, "l", c, nextEntry);
	}
    }
}




SetSwitchLabels(refclause, clause)
CLAUSE *refclause;
CLAUSE *clause;
{
    SLOT *term;
    BOOLEAN LabelIsSet();

    term = clause->lit_position[0];    /* head of clause */
    term = &term[1];   /* first term in head of clause */
    switch (SLOT_TAG(term))
    {
    case TUPLE_INDEX_1:
	if (!LabelIsSet(refclause, ADDRCONST))
	    SetLabel(refclause, ADDRCONST);
	if (!LabelIsSet(refclause, ADDRATOM))
	    SetLabel(refclause, ADDRATOM);
	if (!LabelIsSet(refclause, ADDRLIST))
	    SetLabel(refclause, ADDRLIST);
	if (!LabelIsSet(refclause, ADDRSTRUCT))
	    SetLabel(refclause, ADDRSTRUCT);
	break;
    case ABSOLUTE_ADDRESS:
	term = ABS_ADDRESS(term);
	if (!TAG_IS_FUNCTOR(term))
	{
	    OsPrint("absolute address in pure code points to non-functor\n");
	    break;
	}
	/* else no break: drop into next term */

    case TERM_SPC_INDEX_1:
    case FUNCTOR:
    case CLOSED_TERM:
	if (TAG_IS_TERM_INDEX_1(term))
	    term = &term_space[INDEX(term)];

	if (ARITY_IS_ZERO(term))
	{
	    if (!LabelIsSet(refclause, ADDRATOM))
	         SetLabel(refclause, ADDRATOM);
	}
	else if (GET_FUNCTOR(term) == CONS)
	{
	    if (!LabelIsSet(refclause, ADDRLIST))
	         SetLabel(refclause, ADDRLIST);
	}
	else if (!LabelIsSet(refclause, ADDRSTRUCT))
	     SetLabel(refclause, ADDRSTRUCT);
	break;

    case INTEGER:
    case REAL:
	if (!LabelIsSet(refclause, ADDRCONST))
	     SetLabel(refclause, ADDRCONST);
	break;

    default:
	OsPrint("illegal slot tag in pure code, compilation aborted\n");
    }
}



ResolveSwitchLabels(clause)
CLAUSE *clause;
{
    BOOLEAN LabelIsSet();

    if (!LabelIsSet(clause, ADDRCONST))
         SetLabel(clause, ADDRCONST);
    if (!LabelIsSet(clause, ADDRATOM))
         SetLabel(clause, ADDRATOM);
    if (!LabelIsSet(clause, ADDRLIST))
         SetLabel(clause, ADDRLIST);
    if (!LabelIsSet(clause, ADDRSTRUCT))
         SetLabel(clause, ADDRSTRUCT);
}


BOOLEAN HeadHasTerms(clause)
CLAUSE *clause;
{
   return (ARITY(clause->lit_position[0]));
}


BOOLEAN AllTermsSameType(clauseptr, index)
CLAUSE **clauseptr;
SHORT index;
{
    SLOT *term;
    CLAUSE *clause;
    SHORT i;
    BOOLEAN allconst = FALSE, 
	    allatom = FALSE, 
    	    alllist = FALSE, 
    	    allstruct = FALSE;

    for (i = index; i >= 0; i--)
    {
	clause = clauseptr[i];
	term = clause->lit_position[0];    /* head of clause */
	term = &term[1];   /* first term in head of clause */
	switch (SLOT_TAG(term))
	{
	case TUPLE_INDEX_1:
	    break;
	case ABSOLUTE_ADDRESS:
	    term = ABS_ADDRESS(term);
	    if (!TAG_IS_FUNCTOR(term))
	    {
		OsPrint("absolute address in pure code points to non-functor\n");
		break;
	    }
	    /* else no break: drop into next term */

	case TERM_SPC_INDEX_1:
	case FUNCTOR:
	case CLOSED_TERM:
	    if (allconst) 
	        return FALSE;
	    if (TAG_IS_TERM_INDEX_1(term))
	        term = &term_space[INDEX(term)];
	    if (ARITY_IS_ZERO(term))
	    {
		if (alllist || allstruct)
		   return FALSE;
	        allatom = TRUE;
	    }
	    else if (GET_FUNCTOR(term) == CONS)
	    {
		if (allatom || allstruct)
	            return FALSE;
		alllist = TRUE;
	    }
	    else 
	    {
		if (allatom || alllist)
		    return FALSE;
		allstruct = TRUE;
	    }
	    break;

	case INTEGER:
	case REAL:
	    if (allatom || alllist || allstruct)
	        return FALSE;
	    allconst = TRUE;
	    break;

	default:
	    OsPrint("illegal slot tag in pure code, compilation aborted\n");
	}
    }
    return TRUE;
}



SHORT NumForksFromNode( arcvector)
LONG arcvector;
{
    SHORT numforks = 0;

    while (arcvector != 0)
    {
	numforks += (arcvector & 0x01);
	arcvector = arcvector >> 1;
    }
    return numforks;
}


BOOLEAN BuiltinsinFork(c, arcvector)
CLAUSE *c;
LONG arcvector;
{
    SHORT lit;

    for( lit=1; lit <= c->num_arcs; lit++ )
         if(SUCCESSOR(arcvector, ARC(lit)) && 
	    (c->builtin_index[lit] != 0))
	 return TRUE;
    return FALSE;
}
