/*
 * bit_vectors.c
 * Yet more routines to go with Keith's nifty bit vector
 * macros.
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:
 *	08/06/88 - created DAR
 *	08/17/88 - added VectorCardinality
 */

/* includes */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "bit_macros.c"

/*
 * SHORT IsVectorEmpty(VECTOR *bit_vector)
 * returns non zero if no bits are set
 *
 * WARNING:  When you allocate a bit vector, the allocation
 * routine request the smallest number of words large enough
 * to store the number of bits needed.  Consequently, your bit
 * vector often has an extra number of bits.  You may inadvertently
 * change these (e.g. Invert).  Keep this in mind when testing
 * for an empty vector.
 */
SHORT IsVectorEmpty(BV)
     VECTOR *BV;
{
  SHORT	i, Len;

  Len = VectorLength(BV);
  *(BV) = *(BV) & ~0xFF;	/* empty out tag */
  for (i = 0; i < Len; i++) {
    if ( *(BV + i) != 0) {
      ReplaceLength(BV, Len);
      return 0;
    }
  }
  ReplaceLength(BV, Len);
  return 1;
} /* end IsVectorEmpty() */

/* void WriteVector(VECTOR *bit_vector)
 * Displays a bit vector on stdout.
 * Format:  1st line - index
 *	    2nd line - actual bits
 * Intended as a debugging aid
 */

void WriteVector(BV)
     VECTOR *BV;
{
  SHORT	i, Len, Bits;
  
  Len = VectorLength(BV);
  Bits = Len * sizeof(LONG)*8 - 8;
  for (i=0; i < Bits; i++)
    if (i % 10 == 0 && i) {
      putchar('+');
    } else {
      putchar('0' + i % 10);
    }
  printf("\n");
  for (i=0; i < Bits; i++)
    putchar('0' + BitInVector(BV, i)); 
  printf("\n");
} /* end WriteVector */

/* void WriteBitsVector(VECTOR *bit_vector)
 * Tells which bits are set in a bit bit vector.
 * Outputs to stdout.
 * Format:  bits:  1st, 2nd, ... nth
 * Intended as a debugging aid
 */

void WriteBitsVector(BV)
     VECTOR *BV;
{
  SHORT	i, Len, Bits, first = 1;
  
  Len = VectorLength(BV);
  Bits = Len * sizeof(LONG)*8 - 8;
  printf("Bits:  [");
  for (i=0; i < Bits; i++) {
    if (BitInVector(BV, i)) {
      if (first)
	first = 0;
      else
	putchar(',');
      printf("%d", i);
    } /* end if (BitInVector(BV, i)... */
  } /* end for (i=0... */
  printf("]\n");
} /* end WriteBitsVector */

/* SHORT VectorCardinality(VECTOR *BV)
 * Return the number of set bits in bit vector
 */
SHORT VectorCardinality(BV)
     VECTOR *BV;
{
  SHORT i, len, count = 0;
  LONG bits;

  len = VectorLength(BV);
  *BV &= ~(BVLENMASK);	/* clear length mask */
  for (i=0; i<len; i++) {
    bits = *(BV+i);
    while (bits) {
      if (bits & 1)
	count++;
      bits = bits >> 1;	/* right shift 1 */
    } /* end while (bits) */
  } /* end for i */
  ReplaceLength(BV, len);
  return count;
} /* end VectorCardinality() */
