/*--------------------------------------------------------------------------
 *		load.c - load byte code and pure code files
 *
 *		Entry points --
 *			LoadFiles()		read PUREFILE into term_space and OBJFILE into Instr
 *
 *		term_space is allocated MaxTerm slots.  Pure code from file PURECODE
 *		is read into term_space and relocated.
 *
 *		Instr is allocated and loaded with byte code from file OBJFILE.
 *		Loc is set to the number of instructions in Instr[].
 *
 *		Modifications --
 *		krn 88/11/2  read PgmEntryPoint from first instruction (addr 0)
 *------------------------------------------------------------------------*/

#include "typedefs.h"
#include "macros.c"
#include "gen.h"

#define	LINEMAX 2000
#define	ISDELIM(c) (c==' ' || c=='\t' || c=='\n' || c==',')

/*===  Instr and term_space are loaded with the byte code instructions
 *===  and the pure code for the source program, respectively.  the caller
 *===  should declare these both. */

extern Instruction *Instr;

/*===  procs will be loaded with functor names, both builtins and
 *===  user-defined functors.  it consists of the procedure.name entries
 *===  from pgm.  TopNames is an array of strings containing the names of
 *===  variables in the query, if any.  It is the same as in pgm. */

Proc procs[512];				/* exported and set */
SHORT procNum;					/* exported and set */
extern LONG PURE_CODE_MARKER;
extern LONG maxTerm;
extern char *TopNames[];		/* imported and set */

/*===  helper variables for setting up the proc and TopNames tables.
 *===  these are only used locally.  */

/*LOCAL char *litPtr;*/
LOCAL LONG topNum;
char *index();


/*=====  entry point to load pure code and byte code files  =====*/

LoadFiles(argc, argv)
int argc;
char **argv;
{
	LONG i, temp;
	char OBJFILE[100], tempstr[25];
	char PUREFILE[100];
	extern int Loc;

	/* search for first command line argument that is not preceded by + or -.
	 * take this to be the file name.
	 */
	strcpy(OBJFILE,"Bytecode");
	strcpy(PUREFILE,"Bytecode.pure");
	for (i=1; i<argc; i++) {
	        /* the switches */
		if (sscanf(argv[i], "%d", &temp) ||
		    sscanf(argv[i], "-%s", tempstr) ||
		    sscanf(argv[i], "+%s", tempstr))
		continue;

	    strcpy(OBJFILE, argv[i]);
	    strcat(OBJFILE, ".bc");
	    strcpy(PUREFILE, argv[i]);
	    strcat(PUREFILE, ".pure");
		break;
	}
	if( (PURE_CODE_MARKER =LoadPurecode( PUREFILE )) == -1 ) exit(1);
	TRACE(OsPrintf("%d pure code slots read\n", PURE_CODE_MARKER));
	if( (i=LoadBytecode( OBJFILE )) == -1 ) exit(1);
	OsPrint("%d byte code instructions read\n\n", i);
	Loc = i;
}


LOCAL LoadBytecode( file )
char *file;
{
	FILE *fd;
	char line[LINEMAX], argbuf[LINEMAX], argpat[LINEMAX];
	char *s, *p, c;
	LONG icount, *argPtr, argptr[8000];
	int opcode, argcount, temp, i;
	Instruction *instrBuf;

	argPtr = argptr;
	instrBuf = Instr = (Instruction *) OsAllocInit(sizeof(Instruction) * MAXINSTR); 
	if (Instr == NULL)
	{
	    OsPrint("Unable to allocate Instruction array\n");
	    exit(1);
	}   
	if( (fd = fopen( file, "r" )) == NULL ) return( -1 );
	TRACE(OsPrintf("term_space = 0x%x\n", term_space ));
	icount = 0;

	while( fgets(line,LINEMAX,fd) != NULL ) {
		if( line[0] == '%' && line[1] == '%' )
			break;
		sscanf( line, "%d,%d,%s", &opcode, &argcount, argpat );
		if( icount == 0 ) {
			/*===  interpret all arguments as integers
			 *===  krn 88/11/2  */
			strcpy( argpat, "ddddddd");
		}
		s = index(line,'\t');
		if( s == 0 ) {
			OsPrint("LoadBytecode: missing '\t' after arg count, line %s\n", line );
			return(-1);
		}
		s++;
		instrBuf[icount].opcode = opcode;
		instrBuf[icount].argCount = argcount;
		instrBuf[icount].arg = argPtr;

		for( i = 0; i < argcount; i++ ) {
			while( c = *s ) if( !ISDELIM(c)) break; else s++;
			for (p = argbuf; c = *s++; *p++ = c )
				if( ISDELIM(c) ) break;
			*p = 0;

			if( sscanf( argbuf, "%d", &temp ) != 1 ) {
				OsPrint("LoadBytecode: conversion error on line %s\n", line );
				return(-1);
			}
			if( argpat[i] == 'p' ) {    /* if pure code pointer */
/*				OsPrint("Relocate %d to %d\n", temp, (LONG)(term_space+temp) );*/
				temp = (LONG)( term_space+temp );
			}
			else if (argpat[i] == '\0' && icount > 0) /*mismatch*/
				OsPrint("Not enough argpats, line = %s\n", line );
			*argPtr++ = temp;
		}
		icount++;
	}
	ReadMiscStuff( fd );
	fclose(fd);
	return icount;
}


ReadMiscStuff( fd )
FILE *fd;
{
        extern SLOT FUNCTOR_PLUS, FUNCTOR_MINUS, FUNCTOR_MULTIPLY, 
		    FUNCTOR_DIVIDE, FUNCTOR_MOD, FUNCTOR_DIV;
	extern LONG CONS, NILATOMINDEX;
	char buf[100], litptr[8000] ;
	register char *litPtr = litptr;

	/*=====  read procedure names  =====*/
	procNum = 0;
	while( fgets(buf,LINEMAX,fd) != NULL ) {
		if( buf[0] == '%' && buf[1] == '%' )
			break;
		sscanf( buf, "%s %d", litPtr, &procs[procNum].arity );
		procs[procNum].name = litPtr;
		while( *litPtr++ );
		if (strcmp(procs[procNum].name,"+") == 0)
		     INSERT_FUNCTOR(&FUNCTOR_PLUS, procNum, 2);
		else if (strcmp(procs[procNum].name,"-") == 0)
		     INSERT_FUNCTOR(&FUNCTOR_MINUS, procNum, 2);
		else if (strcmp(procs[procNum].name,"*") == 0)
		     INSERT_FUNCTOR(&FUNCTOR_MULTIPLY, procNum, 2);
		else if (strcmp(procs[procNum].name,"/") == 0)
		     INSERT_FUNCTOR(&FUNCTOR_DIVIDE, procNum, 2);
		else if (strcmp(procs[procNum].name,"mod") == 0)
		     INSERT_FUNCTOR(&FUNCTOR_MOD, procNum, 2);
		else if (strcmp(procs[procNum].name,".") == 0)
		     CONS = procNum;
		else if (strcmp(procs[procNum].name,"[]") == 0)
		     NILATOMINDEX = procNum;
		procNum++;
	}
	/*=====  read topnames (query variable names) 
	 *=====  note that the names start at 1 */
	topNum = 1;
	while( fgets(litPtr,LINEMAX,fd) != NULL ) {
		if( litPtr[0] == '%' && litPtr[1] == '%' )
			break;
		TopNames[ topNum++ ] = litPtr;
		while( *litPtr != '\n' )
			litPtr++;
		*litPtr++ = '\0';
	}
}


LOCAL LoadPurecode( file )
char *file;
{
	FILE *fd;
	LONG count;
	SLOT temp_space[8000];   /*  needed temporarily for relocation */

	fd = fopen( file, "r" );
	if( fd == NULL ) return(-1);
	count = fread( temp_space, sizeof(SLOT), maxTerm, fd );
	if( count == 0 ) {
		OsPrint("LoadPurecode: error reading file\n");
		exit(1);
	}
	term_space = (SLOT *) OsAllocInit(sizeof(SLOT) * maxTerm); 
	if (term_space == NULL)
	{
	    OsPrint("Unable to allocate term space\n");
	    exit(1);
	}   
	Relocate( temp_space, term_space, count, term_space );
	next_term = count;    /* first available slot in term_space */
	fclose(fd);
	return( count );
}



LOCAL Relocate( oldPure, newPure, cnt, base )
SLOT *oldPure, *newPure;
LONG cnt;
SLOT *base;
{
	LONG i;
	register SLOT oldSlotVal;

	for( i=0; i<cnt; i++ ) {
		/*=====  new slot value defaults to old slot value  =====*/
		newPure[i] = oldSlotVal = oldPure[i];
		switch( SLOT_TAG( (oldPure+i) )) {
		case ABSOLUTE_ADDRESS:
			*(newPure + i) = ((LONG)oldSlotVal) + (LONG)base;
			if( SLOT_TAG( (newPure+i) ) != ABSOLUTE_ADDRESS )
				OsPrint("ABS: error, no longer absolute address\n");
			break;
		case MOLECULE:
			break;
		case FUNCTOR:
		case CLOSED_TERM:
			break;
		case TUPLE_INDEX_1:
			break;
		case TUPLE_INDEX_2:
			break;
		case RELATIVE_INDEX:
			break;
		case TERM_SPC_INDEX_1 :
			INSERT_INDEX( (newPure+i), (SLOT)( base+(LONG)oldSlotVal ), TERM_SPC_INDEX_1 );
			break;
		case TERM_SPC_INDEX_2:
			break;
		case INTEGER:
			break;
		case REAL:
			break;
		case UNBOUND_VAR:
			INSERT_INTEGER( (newPure+i), 123 );
			break;
		case BLOCKED_VAR:
			break;
		case TUPLE_SIZE:
			break;
		case FREE_SLOT:
			break;
		default:
			break;
		}
	}
}



char *index(str, c)
char *str;
char c;
{
    for(; *str != '\0'; str++)
        if (*str == c)
            return(str);
    return(NULL);
}

