/****************************************************************
Copyright (C) The University of Melbourne 1993
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name of The University of Melbourne 
or any of its entities not be used in advertising or publicity
pertaining to distribution of the software without specific,
written prior permission.

THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.

AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
	  Andrew Davison (ad@cs.mu.oz.au)

COMMENTS : This file contains symbol table stuff
****************************************************************/

/* Include all the header files needed */
#include "bebop.h"
#include "y.tab.h"

/* Prototypes */
void dump_tree(TREEPTR root);
void free_tree(TREEPTR root);

/* The symbol tables */
VARPTR variable_table[HASHSIZE];
FAPTR  predicate_table[HASHSIZE];
static CONSTPTR constant_table[HASHSIZE];

/* NOTE : the order here of tokens is the same order as in yacc
   do not change unless you change the TOK_??? in yacc as well
   The order is also important, for the first string token[0]
   so we can use the #defines in header to point to exact symbol.
*/
const char *token_table[] = { "!(),.:;[]|",
			      "i",
			      "o",
			      "if",
			      "then",
			      "else",
			      "lazyDet",
			      "eagerDet",
			      "when",
			      "ever",
			      "ground",
			      "and",
			      "or",
			      "\\+",
			      "is",
			      "?-",
			      ":-",
			      "->",
			      "~=",
			      "=..",
			      "=",
			      "\\=",
			      "==",
			      "\\==",
			      "@<",
			      "@>",
			      "@=<",
			      "@>=",
			      "mod",
			      "+",
			      "-",
			      "*",
			      "/",
			      "//",
			      "**",
			      "/\\",
			      "\\/",
			      "^",
			      "<<",
			      ">>",
			      "\\",
			      "<",
			      "=<",
			      ">",
			      ">=",
			      "=:=",
			      "=\\=" };

/* Hash : simple hash function but quick */
static unsigned int hash(const char *s)
{
	unsigned int hashval;

	for(hashval = 0; *s != '\0'; s++)
		hashval = *s + 31 * hashval;
	return (hashval % HASHSIZE);
}


/* Real_var_lookup : look up variable in variable symbol table
   return ptr to it or null if not there
*/
static VARPTR real_var_lookup(const char *s, unsigned int hashval )
{
	VARPTR ptr;

	for(ptr = variable_table[hashval]; ptr != NULL; ptr = ptr->next)
		if (strcmp(s, ptr->varname) == 0)
			return (ptr);
	
	return (NULL);
}

/* Var_lookup : looks up string passed to it */
VARPTR var_lookup(const char *s)
{
	unsigned int hashval = hash(s);

	return (real_var_lookup(s, hashval));

}

/* Pred_lookup : look up predicate in variable symbol table
   return ptr to it or null if not there
*/
static FAPTR pred_lookup(const char *s, unsigned int hashval)
{
	FAPTR ptr;

	for(ptr = predicate_table[hashval]; ptr != NULL; ptr = ptr->next)
		if (strcmp(s, ptr->faname) == 0)
			return (ptr);
	
	return (NULL);
}

/* Var_insert : insert variable into variable table 
   return pointer into hashtable for it 
*/
VARPTR var_insert(const char *var, int length)
{
	VARPTR ptr;
	unsigned int hashval = hash(var);

	if (!(ptr = real_var_lookup(var, hashval)))
	{
		ptr = NEW(VENTRY);
		ptr->varname = copy_string(var, length);
		ptr->next = variable_table[hashval];
		ptr->mode = UNKNOWN;
		ptr->defined = TRUE;
		ptr->dbs = FALSE;
		ptr->value = NULL;
		ptr->type = UNKNOWN;
		ptr->sys = FALSE;
		variable_table[hashval] = ptr;
	}
	else
	{
		ptr->defined = FALSE;	/* Signal defined more than once */
	}

	return(ptr);
}

/* Pred_insert : insert predicate into predicate table 
   return pointer into hashtable for it 
*/
FAPTR pred_insert(const char *name, int length)
{
	FAPTR ptr;
	unsigned int hashval = hash(name);

	if (!(ptr = pred_lookup(name, hashval)))
	{
		ptr = NEW(FAENTRY);
		ptr->faname = copy_string(name, length);
		ptr->next = predicate_table[hashval];
		ptr->arity = NO_ARITY;	/* i.e -1 to indicate we don't know */
		ptr->defined = FALSE;
		ptr->called = FALSE;
		ptr->decl = FALSE;
		ptr->sys = FALSE;
		predicate_table[hashval] = ptr;
	}

	return(ptr);
}

/* Pred2_insert : inserts another predicate with the same name but
   different arity into table
*/
FAPTR pred2_insert(FAPTR first, const int arity)
{
	FAPTR ptr;

	ptr = first;

	while (ptr && ptr->arity != arity)
	{
		ptr = ptr->next;
	}

	if (!ptr)
	{
		ptr = NEW(FAENTRY);
		ptr->faname = copy_string(first->faname , strlen(first->faname));
		ptr->next = first->next;
		ptr->arity = arity;
		ptr->defined = FALSE;
		ptr->called = FALSE;
		ptr->decl = FALSE;
		ptr->sys = FALSE;
		first->next = ptr;
	}

	return ptr;
}


/* Const_insert : insert constant into constant table
   return pointer to it.
*/
CONSTPTR const_insert(const char *str, int length, int lineno, const char *error_line, int error_len)
{
	CONSTPTR cptr = NEW(CONSTENTRY), cptr2;
	unsigned int hashval = hash(str);

	cptr2 = constant_table[hashval];
	constant_table[hashval] = cptr;
	cptr->next = cptr2;
	cptr->constant = copy_string(str, length);
	cptr->lineno = lineno;
	cptr->called = FALSE;
	cptr->possible_error_line = copy_string(error_line, error_len);

	return(cptr);
}


/* Init_symbol : initialise the symbol tables to null and insert
   known system variables or predicates
*/
void init_symbol(void)
{
	int i;
	FAPTR fptr;
	VARPTR vptr;
	static char been_here = FALSE;

	/* List of system variables, their name, length of name, 
	   and type
	*/
	static struct
		{
		   const char *name;
		   int length;
		   char type;
		}sys_var[] = { ID, 4, VISIBLE,
			       INPUTSTR, 5, INVISIBLE,
			       NULL, 0, 0 
			      };
	/* List of know (should be more) that is all of the Prolog system
	   predicates but for now just basic ones, their name, length of name
	   and arity of predicate
	*/
	static struct
		{
		   const char *name;
		   int length;
		   int arity;
		}sys_pred[] = { "demo", 4, 3,
				"fail", 4, 0,
				"true", 4, 0,
				CALLCLASS, LEN_CCLASS, 1,
				"bound", 5, 1,
				/*"flush_write", 11, 1,*/
				/*"flush_writeln", 13, 1,*/
				  NULL, 0, 0 
			       };

	/* If not already done so then load in system stuff and init */
	if (!been_here)
	{
		/* Init tables to null */
		for(i = 0; i < HASHSIZE; ++i)
		{
			variable_table[i] = NULL;
			predicate_table[i] = NULL;
			constant_table[i] = NULL;
		}
		/* Now insert system stuff */
		for(i = 0; sys_var[i].name; i++)
		{
			vptr = var_insert(sys_var[i].name, sys_var[i].length);
			vptr->sys = TRUE;
			vptr->type = sys_var[i].type;
	
		}
		for(i = 0; sys_pred[i].name; i++)
		{
			fptr = pred_insert(sys_pred[i].name, sys_pred[i].length);
			fptr->sys = TRUE;
			fptr->arity = sys_pred[i].arity;
			fptr->defined = TRUE;
			fptr->decl = TRUE;
		}
		been_here = TRUE;
	}
}

/* Check_const : checks the constant table to make sure a constant
   is not being called as goal. This should be done during parsing
   but is it very hard to do without causing some major conflicts
   in the grammer, thus it is done after the parsing is finished
*/
void check_const(void)
{
	extern int nerrors;
	CONSTPTR cptr;
	int i;

	for(i = 0; i < HASHSIZE; i++)
	{
		cptr = constant_table[i];

		while(cptr)
		{
			if (cptr->called)
			{
				++nerrors;
				fprintf(stderr,"\n\nLine %d : ", cptr->lineno);
				fprintf(stderr,"%s", cptr->possible_error_line);
				fprintf(stderr,"\tconstant %s called as a goal\n", cptr->constant);

			}
			cptr = cptr->next;
		}
	}
}

/* Dump_tables : just dumps the internal has tables if the user desires
*/
void dump_tables(void)
{
	VARPTR var;
	FAPTR pred;
	CONSTPTR cptr;
	int i;


	for(i = 0; i < HASHSIZE; i++)
	{
		var = variable_table[i];

		if (var)
		{
			while (var)
			{
				fprintf(stderr, "Var = %s, Mode = %d, Def = %d, Dbs = %d, Type = %d, Sys = %d\n", var->varname, var->mode, var->defined, var->dbs, var->type, var->sys);
				if (var->value)
				{
					fprintf(stderr,"Var's value = \n");
					dump_tree(var->value);
					fprintf(stderr,"\n");
				}
				var = var->next;
			}
		}
	}

	for(i = 0; i < HASHSIZE; i++)
	{
		pred = predicate_table[i];

		if (pred)
		{
			while (pred)
			{
				fprintf(stderr, "Functor = %s, Arity = %d, Def = %d, Called = %d, Decl = %d, Sys = %d\n", pred->faname, pred->arity, pred->defined, pred->called, pred->decl, pred->sys);
				pred = pred->next;
			}
		}
	}

	for(i = 0 ; i < HASHSIZE; i++)
	{
		cptr = constant_table[i];

		if(cptr)
		{
			while(cptr)
			{
				fprintf(stderr,"Constant = %s, Lineno = %d, Line = %s, Called = %d\n", cptr->constant, cptr->lineno, cptr->possible_error_line, cptr->called);
				cptr = cptr->next;
			}
		}
	}

}

/* Free_tables : free's the tables of non-system entries ready for a new
   file to parse and compile
*/
void free_tables(void)
{
	VARPTR varptr, varptr2, var_sys, var_sys_hd;
	FAPTR faptr, faptr2, pred_sys, pred_sys_hd; 
	CONSTPTR cptr, cptr2;
	int i;

	/* Make dummy nodes */
	var_sys = var_sys_hd = NEW(VENTRY);
	pred_sys = pred_sys_hd = NEW(FAENTRY);
	pred_sys->next = NULL;
	var_sys->next = NULL;

	for(i = 0; i < HASHSIZE; ++i)
	{
		faptr = predicate_table[i];

		while(faptr)
		{
			if (faptr->sys)
			{
				pred_sys->next = faptr;
				pred_sys = pred_sys->next;
				faptr = faptr->next;
				pred_sys->next = NULL;
			}
			else
			{
				faptr2 = faptr;
				faptr = faptr->next;
				free(faptr2->faname);
				free(faptr2);
			}
		}
		predicate_table[i] = pred_sys_hd->next;
		pred_sys = pred_sys_hd;
		pred_sys->next = NULL;
	}
			
	for(i = 0; i < HASHSIZE; ++i)
	{
		varptr = variable_table[i];

		while(varptr)
		{
			if (varptr->sys)
			{
				var_sys->next = varptr;
				var_sys = var_sys->next;
				varptr = varptr->next;
				var_sys->next = NULL;
			}
			else
			{
				varptr2 = varptr;
				varptr = varptr->next;
				free(varptr2->varname);
				free_tree(varptr2->value);
				free(varptr2);
			}
		}
		variable_table[i] = var_sys_hd->next;
		var_sys = var_sys_hd;
		var_sys->next = NULL;
	}

	for(i = 0; i < HASHSIZE; ++i)
	{
		cptr = constant_table[i];

		while(cptr)
		{
			cptr2 = cptr;
			cptr = cptr->next;
			free(cptr2->constant);
			free(cptr2->possible_error_line);
			free(cptr2);
		}
		constant_table[i] = NULL;
	}

	free(pred_sys);
	free(var_sys);
}

