#include <errno.h>
#include <string.h>
#include "include.h"
#include "term.h"
#include "tree.h"
#include "predicate.h"
#include "exstate.h"
#include "engine.h"
#include "unify.h"
#include "initial.h"
#include "storage.h"
#include "config.h"
#include "names.h"
#include "error.h"
#include "builtin.h"
#include "display.h"

void ftoa PROTO((double,char *));
static void itoa PROTO((long,int,char *));
static bool list_to_cstring PROTO((Argproto,Term,char *,int));

extern char *ecvt PROTO((double, int, int *, int *));
extern char *gcvt PROTO((double, int, char *));

Term term_list;

/************************************************************************/
/***************    					*****************/
/***************    	     TERM TESTS			*****************/
/***************    					*****************/
/************************************************************************/


bool akl_var_descriptor(Arg)
     Argdecl;
{
  register Term v, t, d;

  Deref(v, A(0));
  MakeIntegerTerm(t, variable_descriptor(v)); /* Loses bits... */

  Deref(d, A(1));
  return unify(d, t, exs->andb, exs);
}

/* -------------------------------------------------------------------- */

bool akl_var(Arg)
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  if(IsVar(X0)) return TRUE;
  else return FALSE;
}

/* -------------------------------------------------------------------- */

bool akl_eq(Arg)
    Argdecl;
{
  register Term X0, X1;

  Deref(X0, A(0));
  Deref(X1, A(1));
  if(X0 == X1) return TRUE;
  else return FALSE;
}

/* -------------------------------------------------------------------- */

bool akl_data(Arg)		/* MAKE WAM INSTRUCTION /KB */
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return TRUE;
}

/* -------------------------------------------------------------------- */

bool akl_atom(Arg)		/* MAKE WAM INSTRUCTION /KB */
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsATM(X0);
}

/* -------------------------------------------------------------------- */

bool akl_integer(Arg)		/* MAKE WAM INSTRUCTION /KB */
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsINT(X0);
}

/* -------------------------------------------------------------------- */

bool akl_float(Arg)		/* MAKE WAM INSTRUCTION /KB */
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsFLT(X0);
}

/* -------------------------------------------------------------------- */

bool akl_number(Arg)
    Argdecl;
{
  register Term X0;
    
  Deref(X0, A(0));    
  IfVarSuspend(X0);
  return (IsINT(X0) || IsFLT(X0));
}

/* -------------------------------------------------------------------- */

bool akl_atomic(Arg)		/* MAKE WAM INSTRUCTION /KB */
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsAtomic(X0);
}

/* -------------------------------------------------------------------- */

bool akl_compound(Arg)		/* MAKE WAM INSTRUCTION /KB */
     Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsCompound(X0);
}

/* -------------------------------------------------------------------- */

bool akl_sys_generic(Arg)
    Argdecl;
{
  register Term X0;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  return IsGEN(X0);
}

/* ------------------------------------------------------- */

andbox tempa;

bool akl_dif(Arg)
    Argdecl;
{
  Term X0, X1;
  bool res;
  int no_entries;
  trailentry *tr, *stop;

  Deref(X0, A(0));
  Deref(X1, A(1));

  no_entries = (exs->trail.current - exs->context.current->trail);

  res = unify(X0, X1, &tempa, exs);

  stop = (exs->context.current->trail + no_entries);

  if(res == FALSE) {
    tr = exs->trail.current;
    for(; tr != stop ; tr--) {
	Bind(Ref(tr->var),tr->val);
    }
    exs->trail.current = stop;
    return TRUE;
  }

  /* assert(res == TRUE) */

  tr = exs->trail.current;

  if(tr == stop)
    return FALSE;

  for(; tr != stop ; tr--) {
    Term t;
    *(++(exs->suspend.current)) = Ref(tr->var);
    if(exs->suspend.current == exs->suspend.end) {
      reinit_suspend(exs);
    }
    Deref(t,RefTerm(Ref(tr->var)));
    if(IsVar(t)) {
      *(++(exs->suspend.current)) = Ref(t);
      if(exs->suspend.current == exs->suspend.end) {
	reinit_suspend(exs);
      }
    }
    Bind(Ref(tr->var),tr->val);
  }
  exs->trail.current = stop;
  return SUSPEND;
}


/************************************************************************/
/***************    					*****************/
/***************    	ATOM AND FUNCTOR COMPARE	*****************/
/***************    					*****************/
/************************************************************************/


#define AtomComparison(Operator)\
{\
  register Term X0, X1;\
\
  Deref(X0, A(0));\
  Deref(X1, A(1));\
\
  if (IsATM(X0) && IsATM(X1))\
    return (strcmp(AtmPname(Atm(X0)), AtmPname(Atm(X1))) Operator 0);\
\
  IfVarSuspend(X0);\
  IfVarSuspend(X1);\
  return FALSE;\
}

/* -------------------------------------------------------------------- */

bool akl_atom_equal(Arg)
    Argdecl;
{
  AtomComparison(==);
}

/* -------------------------------------------------------------------- */

bool akl_atom_not_equal(Arg)
    Argdecl;
{
  AtomComparison(!=);
}

/* -------------------------------------------------------------------- */

bool akl_atom_less(Arg)
     Argdecl;
{
  AtomComparison(<);
}

/* -------------------------------------------------------------------- */

bool akl_atom_not_less(Arg)
    Argdecl;
{
  AtomComparison(>=);
}

/* -------------------------------------------------------------------- */

bool akl_atom_greater(Arg)
    Argdecl;
{
  AtomComparison(>);
}

/* -------------------------------------------------------------------- */

bool akl_atom_not_greater(Arg)
     Argdecl;
{
  AtomComparison(<=);
}



#define FunctorComparison(Operator)\
{\
  register Term X0, X1;\
  char *name0, *name1;\
  int arity0, arity1;\
\
  Deref(X0, A(0));\
  IfVarSuspend(X0);\
\
  if (IsLST(X0))\
    {\
      name0 = ".";\
      arity0 = 2;\
    }\
  else\
  if (IsSTR(X0))\
    {\
      name0 = StrPname(Str(X0));\
      arity0 = StrArity(Str(X0));\
    }\
  else return FALSE;\
\
  Deref(X1, A(1));\
  IfVarSuspend(X1);\
\
  if (IsLST(X1))\
    {\
      name1 = ".";\
      arity1 = 2;\
    }\
  else\
  if (IsSTR(X1))\
    {\
      name1 = StrPname(Str(X1));\
      arity1 = StrArity(Str(X1));\
    }\
  else return FALSE;\
\
  if (arity0 != arity1) \
    return (arity0 Operator arity1); \
\
  return (strcmp(name0, name1) Operator 0);\
}

/* -------------------------------------------------------------------- */

bool akl_functor_equal(Arg)
    Argdecl;
{
  FunctorComparison(==);
}

/* -------------------------------------------------------------------- */

bool akl_functor_not_equal(Arg)
    Argdecl;
{
  FunctorComparison(!=);
}

/* -------------------------------------------------------------------- */

bool akl_functor_less(Arg)
     Argdecl;
{
  FunctorComparison(<);
}

/* -------------------------------------------------------------------- */

bool akl_functor_not_less(Arg)
    Argdecl;
{
  FunctorComparison(>=);
}

/* -------------------------------------------------------------------- */

bool akl_functor_greater(Arg)
    Argdecl;
{
  FunctorComparison(>);
}

/* -------------------------------------------------------------------- */

bool akl_functor_not_greater(Arg)
     Argdecl;
{
  FunctorComparison(<=);
}



/************************************************************************/
/***************    					*****************/
/***************    	    TERM MANIPULATION		*****************/
/***************    					*****************/
/************************************************************************/


bool akl_atom_char(Arg)
     Argdecl;
{
  Term X0, X1, X2;

  Deref(X0, A(0));
  IfVarSuspend(X0);

  Deref(X1, A(1));
  IfVarSuspend(X1);

  if(IsINT(X0) && IsATM(X1)) {
    char *name = AtmPname(Atm(X1));
    int len = strlen(name);
    int i = IntVal(Int(X0));
    Term ch;

    if(i < len) {
      MakeIntegerTerm(ch, name[i]);
      Deref(X2, A(2));
      return unify(X2, ch, exs->andb, exs);
    }
  }

  return FALSE;
}

/* -------------------------------------------------------------------- */

bool akl_float_to_integer(Arg)
    Argdecl;
{
  Term X0, X1, f;

  Deref(X0, A(0));

  if (!IsFLT(X0))
    {
      IfVarSuspend(X0);
      return FALSE;
    }

  Deref(X1, A(1));
  MakeIntegerTerm(f, (sword)FltVal(Flt(X0)));

  return unify(f,X1,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

bool akl_integer_to_float(Arg)
    Argdecl;
{
  Term X0, X1, f;

  Deref(X0, A(0));

  if (!IsINT(X0))
    {
      IfVarSuspend(X0);
      return FALSE;
    }

  Deref(X1, A(1));
  MakeFloatTerm(f, (double)IntVal(Int(X0)));

  return unify(f,X1,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

bool akl_atom_to_chars(Arg)
    Argdecl;
{
  Term X0, X1, lst;

  Deref(X0, A(0));
  if(IsATM(X0))
    {
      lst = make_string_list(AtmPname(Atm(X0)));
      Deref(X1, A(1));
      return unify(X1,lst,exs->andb,exs);
    }
  IfVarSuspend(X0);
  MINOR_FAULT("atom_to_chars: invalid first argument");
}

/* -------------------------------------------------------------------- */

bool akl_integer_to_chars(Arg)
    Argdecl;
{
  Term X0, X1, lst;

  Deref(X0, A(0));
  if(IsINT(X0))
     {
       char temp[MAXNUMLEN];

       itoa(IntVal(Int(X0)),10,temp); /* change to right base /KB */
       lst = make_string_list(temp);
       Deref(X1, A(1));
       return unify(X1,lst,exs->andb,exs);
     }
  IfVarSuspend(X0);
  MINOR_FAULT("integer_to_chars: invalid first argument");
}

/* -------------------------------------------------------------------- */

bool akl_float_to_chars(Arg)
    Argdecl;
{
  Term X0, X1, lst;

  Deref(X0, A(0));
  if(IsFLT(X0))
    {
       char temp[MAXNUMLEN];
       ftoa(FltVal(Flt(X0)),temp);
       lst = make_string_list(temp);
       Deref(X1, A(1));
       return unify(X1,lst,exs->andb,exs);
     }
  IfVarSuspend(X0);
  MINOR_FAULT("float_to_chars: invalid first argument");
}

/* -------------------------------------------------------------------- */

bool akl_chars_to_atom(Arg)
     Argdecl;
{
  char temp[MAXATOMLEN];
  Term X0, X1;
  bool res;

  Deref(X0, A(0));

  IfVarSuspend(X0);

  res = list_to_cstring(Arg,X0,temp,MAXATOMLEN);

  if (res == TRUE)
    {
      Deref(X1, A(1));
      return unify(X1,TagAtm(store_atom(temp)),exs->andb,exs);
    }
  if (res == FALSE)
    MINOR_FAULT("chars_to_atom: invalid first argument");
  return res;
}

/* -------------------------------------------------------------------- */

bool akl_chars_to_integer(Arg)
    Argdecl;
{
  char temp[MAXNUMLEN];
  Term X0, X1, num;
  bool res;

  Deref(X0, A(0));

  IfVarSuspend(X0);
  if (Eq(X0, NIL))
    MINOR_FAULT("chars_to_integer: invalid first argument - []");

  res = list_to_cstring(Arg,X0,temp,MAXNUMLEN);

  if (res == TRUE)
    {
      errno = 0;
      MakeIntegerTerm(num, atoi(temp));
      if (errno)
	MINOR_FAULT("chars_to_integer: invalid first argument");

      Deref(X1, A(1));
      return unify(X1,num,exs->andb,exs);
    }
  if (res == FALSE)
    MINOR_FAULT("chars_to_integer: invalid first argument");

  return res;
}

/* -------------------------------------------------------------------- */

bool akl_chars_to_float(Arg)
    Argdecl;
{
  char temp[MAXNUMLEN];
  Term X0, X1, num;
  bool res;

  Deref(X0, A(0));

  IfVarSuspend(X0);
  if (Eq(X0, NIL))
    MINOR_FAULT("chars_to_integer: invalid first argument - []");

  res = list_to_cstring(Arg,X0,temp,MAXNUMLEN);

  if (res == TRUE)
    {
      errno = 0;
      MakeFloatTerm(num, atof(temp));
      if (errno != 0 && errno != ERANGE)
	MINOR_FAULT("chars_to_integer: invalid first argument");

      Deref(X1, A(1));
      return unify(X1,num,exs->andb,exs);
    }
  if (res == FALSE)
    MINOR_FAULT("chars_to_float: invalid first argument");

  return res;
}

/* -------------------------------------------------------------------- */

/* term_to_functor(+Term, -Name, -Arity) */

bool akl_term_to_functor(Arg)
     Argdecl;
{
  register Term X0, X1, X2, name, arity;

  Deref(X0, A(0));
  IfVarSuspend(X0);

  if (IsSTR(X0))
    {
      MakeIntegerTerm(arity, StrArity(Str(X0)));
      name = TagAtm(StrName(Str(X0)));
    }
  if (IsAtomic(X0))
    {
      MakeIntegerTerm(arity, 0);
      name = X0;
    }
  if (IsLST(X0))
    {
      MakeIntegerTerm(arity, 2);
      name = term_list;
    }

  Deref(X1, A(1));
  Deref(X2, A(2));
  return unify(X1,name,exs->andb,exs)
    && unify(X2,arity,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

/* functor_to_term(+Name, +Arity, -Term) */

bool akl_functor_to_term(Arg)
     Argdecl;
{
  register Term X0, X1, X2;
  int arity;

  Deref(X0, A(0));
  Deref(X1, A(1));

  if (!IsAtomic(X0))
    {
      IfVarSuspend(X0);
      MINOR_FAULT("functor_to_term/3: first argument is not atomic");
    }

  if (!IsINT(X1))
    {
      IfVarSuspend(X1);
      MINOR_FAULT("functor_to_term/3: second argument is not an integer");
    }

  arity = IntVal(Int(X1));

  if (arity < 0)
    MINOR_FAULT("functor_to_term/3: second argument is negative");

  if(Eq(X0,term_list) && (arity == 2))
    {
      Term l;

      MakeListTerm(l);

      InitVariable(LstCarRef(Lst(l)),exs->andb);
      InitVariable(LstCdrRef(Lst(l)),exs->andb);
      Deref(X2, A(2));
      return unify(X2,l,exs->andb,exs);
    }


  {
    Term s;
    int j;

    if (arity == 0)
      {
	Deref(X2, A(2));
	return unify(X2,X0,exs->andb,exs);
      }

    MakeStructTerm(s,store_functor(Atm(X0),arity),exs->andb);
    for(j = 0; j < arity ; j++) {
      InitVariable(StrArgRef(Str(s),j),exs->andb);
    }
    Deref(X2, A(2));
    return unify(X2,s,exs->andb,exs);
  }
}

/* -------------------------------------------------------------------- */

/* term_to_list(+Term, -ArgList) */

bool akl_term_to_list(Arg)
    Argdecl;
{
  register Term lst;
  Term term, arglist;
  Term arity;
  int i;

  Deref(term, A(0));
  IfVarSuspend(term);

  if (IsSTR(term))
    {
      Term l;

      lst = NIL;
      MakeIntegerTerm(arity, StrArity(Str(term)));

      for (i = StrArity(Str(term)) - 1; i >= 0 ; i--)
	{
	  MakeListTerm(l);
	  LstCdr(Lst(l)) = lst;
	  lst = l;
	  GetStrArg(LstCar(Lst(lst)),Str(term),i);
	}

      MakeListTerm(l);
      LstCdr(Lst(l)) = lst;
      lst = l;
      LstCar(Lst(lst)) = TagAtm(StrName(Str(term)));
    }
  else
  if (IsAtomic(term))
    {
      MakeListTerm(lst);
      LstCdr(Lst(lst)) = NIL;
      LstCar(Lst(lst)) = term;
    }
  else
  if (IsLST(term))
    {
      Term l;

      MakeIntegerTerm(arity, 2);

      MakeListTerm(lst);
      LstCdr(Lst(lst)) = NIL;
      GetLstCdr(LstCar(Lst(lst)),Lst(term));

      MakeListTerm(l);
      LstCdr(Lst(l)) = lst;
      lst = l;
      GetLstCar(LstCar(Lst(lst)),Lst(term));

      MakeListTerm(l);
      LstCdr(Lst(l)) = lst;
      lst = l;
      LstCar(Lst(lst)) = term_list;
    }

  Deref(arglist, A(1));
  return unify(arglist,lst,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

/* list_to_term(+ArgList, -Term) */

bool akl_list_to_term(Arg)
    Argdecl;
{
  int i, arity;
  Term arglist, name, term, *p, t;

  Deref(arglist, A(0));

  if (!IsLST(arglist))
    {
      IfVarSuspend(arglist);
      MINOR_FAULT("list_to_term/2: first argument is not a list");
    }

  GetLstCar(name,Lst(arglist));	/* read functor name */
  Deref(name,name);

  if (!IsAtomic(name))
    {
      IfVarSuspend(name);
      MINOR_FAULT("list_to_term/2: invalid functor name");
    }

  GetLstCdr(arglist,Lst(arglist)); /* and move on */
  Deref(arglist,arglist);

  /* Find arity and if the list is ready */

  {
    Term l;

    for (arity = 0, l = arglist; IsLST(l); arity++)
    {
      GetLstCdr(l,Lst(l));
      Deref(l,l);
    }

    IfVarSuspend(l);
  }

  if (arity == 0)
    t = name;
  else if(Eq(name, term_list) && (arity == 2))
    {
      List l;

      MakeList(l);
      t = TagLst(l);
      p = &LstCar(l);		/* point to first argument */
    }
  else
    {
      Structure s;

      if (!IsATM(name))
	{
	  IfVarSuspend(name);
	  MINOR_FAULT("list_to_term/2: invalid functor name");
	}

      MakeStruct(s,store_functor(Atm(name),arity),exs->andb);
      t = TagStr(s);
      p = StrArgRef(s,0);	/* point to first argument */
    }

  for (i = 0; i < arity; i++)
    {
      GetLstCar(*(p++),Lst(arglist));
      GetLstCdr(arglist,Lst(arglist));
      Deref(arglist,arglist);
    }

  Deref(term, A(1));
  return unify(term,t,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

/* islist(+List) */

bool akl_islist(Arg)
    Argdecl;
{
  Term lst;

  Deref(lst, A(0));

  while (IsLST(lst))
    {
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);
    }

  if (lst == NIL)
    return TRUE;

  IfVarSuspend(lst);

  return FALSE;
}

/* -------------------------------------------------------------------- */

/* concat(+List1, +List2, -NewList) */

bool akl_concat(Arg)
    Argdecl;
{
  Term lst, newlst, l, ln, car;

  Deref(lst, A(0));

  if (lst == NIL)		/* concat([], X, X) */
    {
      Deref(lst, A(1));
      Deref(newlst, A(2));
      return unify(lst,newlst,exs->andb,exs);
    }

  /* Check that the first list is a list and ready built */

  for (l = lst; IsLST(l); )
    {
      GetLstCdr(l,Lst(l));
      Deref(l,l);
    }

  IfVarSuspend(l);

  if (l != NIL)
    return FALSE;

  GetLstCar(car,Lst(lst));
  GetLstCdr(lst,Lst(lst));
  Deref(lst,lst);

  MakeListTerm(l);
  newlst = l;
  LstCar(Lst(l)) = car;

  while (IsLST(lst))
    {
      GetLstCar(car,Lst(lst));
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);

      MakeListTerm(ln);
      LstCar(Lst(ln)) = car;
      LstCdr(Lst(l)) = ln;
      l = ln;
    }

  Deref(ln, A(1));
  LstCdr(Lst(l)) = ln;

  Deref(lst, A(2));
  return unify(lst,newlst,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

/* memberchk(+Element, +List) */

bool akl_memberchk(Arg)
    Argdecl;
{
  Term lst, car, el;

  Deref(lst, A(1));
  Deref(el, A(0));

  while (IsLST(lst))
    {
      GetLstCar(car,Lst(lst));
      if (unify(el,car,exs->andb,exs))
	return TRUE;
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);
    }

  IfVarSuspend(lst);

  return FALSE;
}

/* -------------------------------------------------------------------- */

bool akl_list_to_length(Arg)
     Argdecl;
{
  Term lst, length, X1;
  int len;

  Deref(lst, A(0));

  for (len = 0; IsLST(lst); len++)
    {
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);
    }

  IfVarSuspend(lst);

  if (lst != NIL)
    return FALSE;

  MakeIntegerTerm(length, len);
  Deref(X1, A(1));

  return unify(X1,length,exs->andb,exs);
}

/* -------------------------------------------------------------------- */

bool akl_length_to_list(Arg)
     Argdecl;
{
  Term lst, l, ll, length;
  int len;

  Deref(length, A(0));
  if (!IsINT(length))
    {
      IfVarSuspend(length);
      return FALSE;
    }

  lst = NIL;

  for (len = IntVal(Int(length)); len > 0; len--)
    { 
      MakeListTerm(l);
      LstCdr(Lst(l)) = lst;
      lst = l;

      InitVariable(LstCarRef(Lst(lst)),exs->andb);
    }

  Deref(ll, A(1));
  return unify(lst,ll,exs->andb,exs);
}

/* -------------------------------------------------------------------- */
#ifdef COMPAT08

/* compound_extract(+ListOrStruct, -Name, -Arity) */

bool akl_compound_extract(Arg)
     Argdecl;
{
  register Term X0, X1, X2;

  Deref(X0, A(0));

  if (IsSTR(X0))
    {
      Term arity;
      MakeIntegerTerm(arity, StrArity(Str(X0)));
      Deref(X1, A(1));
      Deref(X2, A(2));
      return unify(X1,TagAtm(StrName(Str(X0))),exs->andb,exs)
	&& unify(X2,arity,exs->andb,exs);
    }
  if (IsLST(X0))
    {
      Term two;
      MakeIntegerTerm(two, 2);
      Deref(X1, A(1));
      Deref(X2, A(2));
      return unify(X1,term_list,exs->andb,exs)
	&& unify(X2,two,exs->andb,exs);
    }
  IfVarSuspend(X0);
  MINOR_FAULT("compound_extract/3: invalid first argument - not a compound");
}

/* -------------------------------------------------------------------- */

/* compound_extract(+ListOrStruct, -Name, -Arity, -ArgList) */

bool akl_compound_extract2(Arg)
    Argdecl;
{
  register Term lst;
  Term X0, X1, X2, X3;
  Term arity;
  int i;

  Deref(X0, A(0));

  if (IsLST(X0))
    {
      Term ln,car,cdr;

      MakeIntegerTerm(arity, 2);
      MakeListTerm(lst);
      GetLstCar(car,Lst(X0));
      LstCar(Lst(lst)) = car;
      MakeListTerm(ln);
      LstCdr(Lst(lst)) = ln;
      GetLstCdr(cdr,Lst(X0));
      LstCar(Lst(ln)) = cdr;
      LstCdr(Lst(ln)) = NIL;
      Deref(X1, A(1));
      Deref(X2, A(2));
      Deref(X3, A(3));
      return unify(X1,term_list,exs->andb,exs)
	&& unify(X2,arity,exs->andb,exs)
	&& unify(X3,lst,exs->andb,exs);
    }
  if (IsSTR(X0))
    {
      Term l, ln;

      MakeIntegerTerm(arity, StrArity(Str(X0)));
      MakeListTerm(ln);
      lst = l = ln;

      GetStrArg(LstCar(Lst(l)),Str(X0),0);

      for (i = 1; i < StrArity(Str(X0)); i++, l = ln)
	{
	  MakeListTerm(ln);
	  LstCdr(Lst(l)) = ln;
	  GetStrArg(LstCar(Lst(ln)),Str(X0),i);
	}
      LstCdr(Lst(ln)) = NIL;

      Deref(X1, A(1));
      Deref(X2, A(2));
      Deref(X3, A(3));
      return unify(X1,TagAtm(StrName(Str(X0))),exs->andb,exs)
	&& unify(X2,arity,exs->andb,exs)
	&& unify(X3,lst,exs->andb,exs);
    }
  IfVarSuspend(X0);
  MINOR_FAULT("compound_extract/4: invalid first argument - not a compound");
}

bool akl_compound_construct(Arg)
     Argdecl;
{
  register Term X0, X1, X2;
  int arity;

  Deref(X0, A(0));
  Deref(X1, A(1));

  if (!IsATM(X0))
    {
      IfVarSuspend(X0);
      MINOR_FAULT("compound_construct/3: first argument is not an atom");
    }

  if (!IsINT(X1))
    {
      IfVarSuspend(X1);
      MINOR_FAULT("compound_construct/3: second argument is not an integer");
    }

  arity = IntVal(Int(X1));

  if (arity <= 0)
    MINOR_FAULT("compound_construct/3: second argument is zero or negative");

  if(Eq(X0, term_list) && (arity == 2))
    {
      Term l;

      MakeListTerm(l);

      InitVariable(LstCarRef(Lst(l)),exs->andb);
      InitVariable(LstCdrRef(Lst(l)),exs->andb);
      Deref(X2, A(2));
      return unify(X2,l,exs->andb,exs);
    }
  else
    {
      Term s;
      int j;

      MakeStructTerm(s,store_functor(Atm(X0),arity),exs->andb);
      for(j = 0; j < arity ; j++) {
	InitVariable(StrArgRef(Str(s),j),exs->andb);
      }
      Deref(X2, A(2));
      return unify(X2,s,exs->andb,exs);
    }
}

/* -------------------------------------------------------------------- */

/* compound_construct(+Name, +Arity, +ArgList, -ListOrStruct) */

bool akl_compound_construct2(Arg)
    Argdecl;
{
  int i, arity;
  Term X0, X1, X2, X3, lst, *p, t;

  Deref(X0, A(0));
  Deref(X1, A(1));

  if (!IsATM(X0))
    {
      IfVarSuspend(X0);
      MINOR_FAULT("compound_construct/4: first argument is not an atom");
    }

  if (!IsINT(X1))
    {
      IfVarSuspend(X1);
      MINOR_FAULT("compound_construct/4: second argument is not an integer");
    }

  arity = IntVal(Int(X1));

  if (arity <= 0)
    MINOR_FAULT("compound_construct/4: second argument is zero or negative");

  Deref(X2, A(2));
  for (i = 0, lst = X2; IsLST(lst); i++)
    {
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);
    }
  IfVarSuspend(lst);

  if ((i != arity) || !Eq(lst, NIL))
    MINOR_FAULT("compound_construct/4: invalid second argument");

  if(Eq(X0, term_list) && (arity == 2))
    {
      List l;

      MakeList(l);
      t = TagLst(l);
      p = &LstCar(l);		/* point to first argument */
    }
  else
    {
      Structure s;

      MakeStruct(s,store_functor(Atm(X0),arity),exs->andb);
      t = TagStr(s);
      p = StrArgRef(s,0);	/* point to first argument */
    }

  for (i = 0, lst = X2; i < arity; i++)
    {
      GetLstCar(*(p++),Lst(lst));
      GetLstCdr(lst,Lst(lst));
      Deref(lst,lst);
    }

  Deref(X3, A(3));
  return unify(X3,t,exs->andb,exs);
}
#endif
/* -------------------------------------------------------------------- */

bool akl_arg(Arg)
    Argdecl;
{
  register Term X0, X1, X2, tmp;
  int i;

  Deref(X0, A(0));

  if (!IsINT(X0))
    {
      IfVarSuspend(X0);
      MINOR_FAULT("arg/3: invalid first argument - not a number");
    }

  Deref(X1, A(1));

  if (IsLST(X1))
    {
      i = IntVal(Int(X0));
      if(i == 1) {
	GetLstCar(tmp, Lst(X1));
	Deref(tmp, tmp);
	Deref(X2, A(2));
	return unify(X2,tmp,exs->andb,exs);
      }
      if(i == 2) {
	GetLstCdr(tmp, Lst(X1));
	Deref(tmp, tmp);
	Deref(X2, A(2));
	return unify(X2,tmp,exs->andb,exs);
      }
      MINOR_FAULT("arg/3: first argument is out of range");
    }

  if (IsSTR(X1))
    {
      i = IntVal(Int(X0));
      if((i > StrArity(Str(X1))) || (i <= 0))
	MINOR_FAULT("arg/3: first argument is out of range");
      i--;
      GetStrArg(tmp,Str(X1),i);
      Deref(tmp, tmp);
      Deref(X2, A(2));
      return unify(X2,tmp,exs->andb,exs);
    }

  IfVarSuspend(X1);
  MINOR_FAULT("arg/3: invalid arguments");
}

/* -------------------------------------------------------------------- */

/* sys:setarg/3,  destructive update of a structure argument */

bool akl_sys_setarg(Arg)
     Argdecl;
{
  register Term X0, X1, X2;

  Deref(X0,A(0));
  Deref(X1,A(1));
  Deref(X2,A(2));

  if(IsINT(X0) && IsSTR(X1) ) {
    int i;
    i = IntVal(Int(X0));
    if(i > 0 && i <= StrArity(Str(X1))) {
        i--;
    	StrArgument(Str(X1),i) = X2;
    	return TRUE;
      } else
	MINOR_FAULT("sys:setarg segmentation fault");
  }
  IfVarSuspend(X0);
  IfVarSuspend(X1);
  return FALSE;
}



/************************************************************************/
/***************    					*****************/
/***************    	    INTEGER ARITHMETIC		*****************/
/***************    					*****************/
/************************************************************************/

#define UnaryArith(Operation)\
{\
  Term X0, X1;\
\
  Deref(X0, A(0));\
  \
  if (IsINT(X0)) {\
    Term ii;\
    int i;\
\
    i = IntVal(Int(X0));\
    Operation ; \
    MakeIntegerTerm(ii, i);\
\
    Deref(X1, A(1));\
    return unify(X1, ii, exs->andb,exs);\
  }\
  IfVarSuspend(X0);\
  return FALSE;\
}


/* -------------------------------------------------------------------- */

bool akl_neg(Arg)
     Argdecl;
{
  UnaryArith(i = -i);
}

/* -------------------------------------------------------------------- */

bool akl_com(Arg)
     Argdecl;
{
  UnaryArith(i = ~i);
}

/* -------------------------------------------------------------------- */

bool akl_inc(Arg)
     Argdecl;
{
  UnaryArith(i++);
}

/* -------------------------------------------------------------------- */

bool akl_dec(Arg)
     Argdecl;
{
  UnaryArith(i--);
}

/* -------------------------------------------------------------------- */

bool akl_abs(Arg)
     Argdecl;
{
  UnaryArith(i = (i < 0) ? -i : i);
}

/* -------------------------------------------------------------------- */



#define Arithmetic(Operation)\
{\
  register Term X0, X1, X2;\
\
  Deref(X0, A(0));\
  Deref(X1, A(1));\
  \
  if((IsINT(X0) && IsINT(X1))) {\
    register Term ii;\
    int i, a, b;\
\
    a = IntVal(Int(X0));\
    b = IntVal(Int(X1));\
\
    Operation ;\
\
    MakeIntegerTerm(ii, i);\
\
    Deref(X2, A(2));\
    return unify(X2, (Term)ii, exs->andb,exs);\
  }\
  IfVarSuspend(X0);\
  IfVarSuspend(X1);\
  return FALSE;\
}

/* -------------------------------------------------------------------- */

bool akl_add(Arg)
     Argdecl;
{
  Arithmetic(i = a + b);
}

/* -------------------------------------------------------------------- */

bool akl_sub(Arg)
    Argdecl;
{
  Arithmetic(i = a - b);
}

/* -------------------------------------------------------------------- */

bool akl_div(Arg)
    Argdecl;
{
  Arithmetic(i = a / b);
}

/* -------------------------------------------------------------------- */

bool akl_mul(Arg)
    Argdecl;
{
  Arithmetic(i = a * b);
}

/* -------------------------------------------------------------------- */

bool akl_mod(Arg)
    Argdecl;
{
  Arithmetic(i = a % b);
}

/* -------------------------------------------------------------------- */

bool akl_min(Arg)
     Argdecl;
{
  Arithmetic(i = (a < b) ? a : b);
}

/* -------------------------------------------------------------------- */

bool akl_max(Arg)
     Argdecl;
{
  Arithmetic(i = (a > b) ? a : b);
}

/* -------------------------------------------------------------------- */

bool akl_and(Arg)
     Argdecl;
{
  Arithmetic(i = a & b);
}

/* -------------------------------------------------------------------- */

bool akl_or(Arg)
     Argdecl;
{
  Arithmetic(i = a | b);
}

/* -------------------------------------------------------------------- */

bool akl_xor(Arg)
     Argdecl;
{
  Arithmetic(i = a ^ b);
}

/* -------------------------------------------------------------------- */

bool akl_shl(Arg)
    Argdecl;
{
  Arithmetic(i = a << b);
}

/* -------------------------------------------------------------------- */

bool akl_shr(Arg)
    Argdecl;
{
  Arithmetic(i = a >> b);
}


bool akl_div4(Arg)
     Argdecl;
{
  Term X0, X1, X2, X3;
  Term i, j;
  int a, b;

  Deref(X0, A(0));
  IfVarSuspend(X0);
  if (!IsINT(X0))
    MINOR_FAULT("div/4: Invalid first argument");

  Deref(X1, A(1));
  IfVarSuspend(X1);
  if (!IsINT(X1))
    MINOR_FAULT("div/4: Invalid second argument");

  a = IntVal(Int(X0));
  b = IntVal(Int(X1));

  MakeIntegerTerm(i, a / b);
  MakeIntegerTerm(j, a % b);

  Deref(X2, A(2));
  Deref(X3, A(3));
  return unify(X2, i, exs->andb,exs) && unify(X3, j, exs->andb,exs);
}



/************************************************************************/
/***************    					*****************/
/***************    	   INTEGER COMPARISION		*****************/
/***************    					*****************/
/************************************************************************/


#define Comparison(Operator)\
{\
  register Term X0, X1;\
    \
  Deref(X0, A(0));        \
  Deref(X1, A(1));        \
\
  if (IsINT(X0) && IsINT(X1))\
    return IntVal(Int(X0)) Operator IntVal(Int(X1));\
\
  IfVarSuspend(X0);\
  IfVarSuspend(X1);\
  return FALSE;\
}

/* -------------------------------------------------------------------- */

/* ==/2, suspend, fail or succeed */
bool akl_equal(Arg)
    Argdecl;
{
  Comparison(==);
}

/* -------------------------------------------------------------------- */

/* =\=/2, suspend, fail or succeed */
bool akl_not_equal(Arg)
    Argdecl;
{
  Comparison(!=);
}

/* -------------------------------------------------------------------- */

/* </2, suspend, fail or succeed */
bool akl_less(Arg)
     Argdecl;
{
  Comparison(<);
}

/* -------------------------------------------------------------------- */

/* >=/2, suspend, fail or succeed */
bool akl_not_less(Arg)
    Argdecl;
{
  Comparison(>=);
}

/* -------------------------------------------------------------------- */

/* =</2, suspend, fail or succeed */
bool akl_not_greater(Arg)
     Argdecl;
{
  Comparison(<=);
}

/* -------------------------------------------------------------------- */

/* >/2, suspend, fail or succeed */
bool akl_greater(Arg)
    Argdecl;
{
  Comparison(>);
}


/************************************************************************/
/***************    					*****************/
/***************    	FLOATING POINT ARITHMETIC	*****************/
/***************    					*****************/
/************************************************************************/

/* Note that the C corresponding type is "double float" */


#define UnaryFloatArith(Operation)\
{\
  Term X0, X1;\
\
  Deref(X0, A(0));\
  \
  if (IsFLT(X0)) {\
    Term ii;\
    double i;\
\
    i = FltVal(Flt(X0));\
    Operation ; \
    MakeFloatTerm(ii, i);\
\
    Deref(X1, A(1));\
    return unify(X1, ii, exs->andb,exs);\
  }\
  IfVarSuspend(X0);\
  return FALSE;\
}


/* -------------------------------------------------------------------- */

bool akl_fp_neg(Arg)
     Argdecl;
{
  UnaryFloatArith(i = -i);
}

/* -------------------------------------------------------------------- */

bool akl_fp_inc(Arg)
     Argdecl;
{
  UnaryFloatArith(i++);
}

/* -------------------------------------------------------------------- */

bool akl_fp_dec(Arg)
     Argdecl;
{
  UnaryFloatArith(i--);
}

/* -------------------------------------------------------------------- */

bool akl_fp_abs(Arg)
     Argdecl;
{
  UnaryFloatArith(i = (i < 0.0) ? -i : i);
}

/* -------------------------------------------------------------------- */



#define FloatArithmetic(Operation)\
{\
  register Term X0, X1, X2;\
\
  Deref(X0, A(0));\
  Deref(X1, A(1));\
  \
  if((IsFLT(X0) && IsFLT(X1))) {\
    Term ii;\
    double i, a, b;\
\
    a = FltVal(Flt(X0));\
    b = FltVal(Flt(X1));\
\
    Operation ;\
\
    MakeFloatTerm(ii, i);\
\
    Deref(X2, A(2));\
    return unify(X2, ii, exs->andb,exs);\
  }\
  IfVarSuspend(X0);\
  IfVarSuspend(X1);\
  return FALSE;\
}

/* -------------------------------------------------------------------- */

bool akl_fp_min(Arg)
     Argdecl;
{
  FloatArithmetic(i = (a < b) ? a : b);
}

/* -------------------------------------------------------------------- */

bool akl_fp_max(Arg)
     Argdecl;
{
  FloatArithmetic(i = (a > b) ? a : b);
}
/* -------------------------------------------------------------------- */

bool akl_fp_add(Arg)
     Argdecl;
{
  FloatArithmetic(i = a + b);
}

/* -------------------------------------------------------------------- */

bool akl_fp_sub(Arg)
    Argdecl;
{
  FloatArithmetic(i = a - b);
}

/* -------------------------------------------------------------------- */

bool akl_fp_div(Arg)
    Argdecl;
{
  FloatArithmetic(i = a / b);
}

/* -------------------------------------------------------------------- */

bool akl_fp_mul(Arg)
    Argdecl;
{
  FloatArithmetic(i = a * b);
}

/* -------------------------------------------------------------------- */



/************************************************************************/
/***************    					*****************/
/***************    	FLOATING POINT COMPARE		*****************/
/***************    					*****************/
/************************************************************************/


#define FloatComparison(Operator)\
{\
  register Term X0, X1;\
    \
  Deref(X0, A(0));        \
  Deref(X1, A(1));        \
\
  if (IsFLT(X0) && IsFLT(X1))\
    return (FltVal(Flt(X0)) Operator FltVal(Flt(X1)));\
\
  IfVarSuspend(X0);\
  IfVarSuspend(X1);\
  return FALSE;\
}

/* -------------------------------------------------------------------- */

bool akl_fp_equal(Arg)
    Argdecl;
{
  FloatComparison(==);
}

/* -------------------------------------------------------------------- */

bool akl_fp_not_equal(Arg)
    Argdecl;
{
  FloatComparison(!=);
}

/* -------------------------------------------------------------------- */

bool akl_fp_less(Arg)
     Argdecl;
{
  FloatComparison(<);
}

/* -------------------------------------------------------------------- */

bool akl_fp_not_less(Arg)
    Argdecl;
{
  FloatComparison(>=);
}

/* -------------------------------------------------------------------- */

bool akl_fp_greater(Arg)
    Argdecl;
{
  FloatComparison(>);
}

/* -------------------------------------------------------------------- */

bool akl_fp_not_greater(Arg)
     Argdecl;
{
  FloatComparison(<=);
}


/* -------------------------------------------------------------------- */

Term make_string_list(str)
    char *str;
{
    register Term res;
    register Term val;

    if(*str == '\0')
	return NIL;

    MakeListTerm(res);
    MakeIntegerTerm(val, ((uword) *(str++)));
    LstCar(Lst(res)) = val;
    LstCdr(Lst(res)) = make_string_list(str);

    return res;
}

/* -------------------------------------------------------------------- */

static void itoa(l, base, cbuf)
     long l;
     int base;
     char *cbuf;
{
  char hibase = 'a'-10;
  int sx = (l<0);
  long digit;
  register char *c0, *c, d;

  c = cbuf;
  if (base<0)
    hibase = 'A'-10,
    base = -base;
  if (sx)
    *c++ = '-',
    l = -l;

  do
    digit = l % base,
    l /= base,
    *c++ = (digit<10 ? '0'+digit : hibase+digit);
  while (l>0);

  *c++ = 0;
  for (c0=cbuf+sx, c-=2; c0<c; c0++, c--)
    d = *c0, *c0 = *c, *c = d;
}

/* -------------------------------------------------------------------- */

/* Float to chars using just enough digits to ensure that distinct numbers
   come out differently.  Reasonable machines use between 52 and 56 mantissa
   bits, yielding 15-17 significant digits.  Check first if 15 digits
   suffice; if not, use 16 or 17 digits.
*/
void ftoa(f, cbuf)
     double f;
     char *cbuf;
{
  double ff;
  register char *str, *str2;
  register int i, j;

#ifndef HAS_NO_GCVT
  gcvt(f, 15, str=cbuf);
#else
  sprintf(str=cbuf, "%.15g", f); /* gcvt broken */
#endif
  if (str[0]=='-') str++;
  switch (str[0])
    {
    case '+':			/* hpux */
    case '-':			/* hpux */
    case 'I':
      strcpy(str, "inf");
      return;
    case '?':			/* hpux */
    case 'N':
      strcpy(str, "nan");
      return;
    }
  ff = atof(cbuf);
  if (f != ff)
#ifndef HAS_NO_GCVT
    gcvt(f, 16, str=cbuf);
#else
    sprintf(str=cbuf, "%.16g", f);	/* gcvt broken */
#endif
  ff = atof(cbuf);
  if (f != ff)
#ifndef HAS_NO_GCVT
    gcvt(f, 17, str=cbuf);
#else
    sprintf(str=cbuf, "%.17g", f);	/* gcvt broken */
#endif

				/* add ".0" if need be */
  while (TRUE)
    {
      switch (*str++)
	{
	case 0:
	case 'e':
	case 'E':
	  i = str-cbuf-1;
	  j = strlen(cbuf);
	  str = cbuf;
	  while (j>=i)
	    str[j+2] = str[j], --j;
	  str[i++] = '.';
	  str[i++] = '0';
	  goto strip;
	case '.':
	  switch (*str)
	    {
	    case 0:
	    case 'e':
	    case 'E':
	      *str++ = '0',
	      *str++ = 0;
	    }
	  goto strip;
	default:
	  ;
	}
    }

 strip:			/* strip trailing zeros for some machines */
  str = cbuf;
  while (TRUE)
    switch (*str++)
      {
      case '.':
	i = -1;
	break;
      case '0':
	i++;
	break;
      case 0:
      case 'e':
      case 'E':
	goto move;
      default:
	i = 0;
      }
 move:
  if (i>0)
    {
      --str;
      str2 = str-i;
      while ((*str2++ = *str++))
	;
    }
}

/* OLD
void ftoa(f, cbuf)
     double f;
     register char *cbuf;
{
  int prec=SP_FLOAT_SIGNIF;
  int decpt, sign, exp;
  char *str = ecvt(f, prec, &decpt, &sign);
  register char *c1, *c2;

  if (sign)
    *cbuf++ = '-';
  if (str[0]=='I')
    {
      strcpy(cbuf, "inf");
      return;
    }
  else if (str[0]=='N')
    {
      strcpy(cbuf, "nan");
      return;
    }
  while (prec>decpt && str[prec-1]=='0')
    --prec;
  str[prec] = 0;
  strcpy(cbuf, str);
  exp = decpt-1;
  if (exp<-4 || exp>=prec)
    {
      if (prec==1)
	cbuf[prec++] = '0';
      c1=cbuf+prec, c2=c1+1;
      *c2 = 'E';
      itoa(exp, 10, c2+1);
      while (c1>cbuf+1)
	*(--c2) = *(--c1);
      cbuf[1] = '.';
    }
  else if (exp<0)
    {
      c1=cbuf+prec+1, c2=c1+1-exp;
      while (c1>cbuf)
	*(--c2) = *(--c1);
      while (c1<=cbuf-exp)
	*c1++ = '0';
      cbuf[1] = '.';
    }
  else if (exp==prec-1)
    strcpy(cbuf+prec, ".0");
  else
    {
      c1=cbuf+prec+1, c2=c1+1;
      while (c1>cbuf+exp+1)
	*(--c2) = *(--c1);
      cbuf[exp+1] = '.';
    }
}
*/
static bool list_to_cstring(Arg,lst,string,maxlen)
     Argdecl;
     Term lst;
     char *string;
     int maxlen;
{
  int i;
  Term next;

  for (i = 0; IsLST(lst); )
    {
      GetLstCar(next,Lst(lst));
      Deref(next,next);
      if(IsINT(next))
	{
	  string[i++] = (char) IntVal(Int(next));
	  if(i == (maxlen -1) )
	    {
	      return FALSE;
	    }
	  GetLstCdr(lst,Lst(lst));	/* move to next element */
	  Deref(lst,lst);
	}
      else
	{
	  IfVarSuspend(next);	/* not bound yet */
	  return FALSE;		/* not a valid list element */
	}
    }

  IfVarSuspend(lst);		/* not bound yet */
  if(!Eq(lst, NIL))
    return FALSE;
  string[i] = '\0';
  return TRUE;
}


void initialize_builtin() {

  term_list = TagAtm(store_atom("."));

  define("$var_descriptor",akl_var_descriptor,2);
  define("$var",akl_var,1);
  define("$eq",akl_eq,2);

  define("data",akl_data,1);
  define("atom",akl_atom,1);
  define("integer",akl_integer,1);
  define("float",akl_float,1);
  define("atomic",akl_atomic,1);
  define("compound",akl_compound,1);
  define("sys_generic",akl_sys_generic,1);

  define("atom_equal", akl_atom_equal,2);
  define("atom_not_equal", akl_atom_not_equal,2);
  define("atom_less", akl_atom_less,2);
  define("atom_not_less", akl_atom_not_less,2);
  define("atom_greater", akl_atom_greater,2);
  define("atom_not_greater", akl_atom_not_greater,2);

  define("functor_equal", akl_functor_equal,2);
  define("functor_not_equal", akl_functor_not_equal,2);
  define("functor_less", akl_functor_less,2);
  define("functor_not_less", akl_functor_not_less,2);
  define("functor_greater", akl_functor_greater,2);
  define("functor_not_greater", akl_functor_not_greater,2);

  define("atom_char",akl_atom_char,3);
  define("atom_to_chars",akl_atom_to_chars,2);
  define("integer_to_chars",akl_integer_to_chars,2);
  define("float_to_chars",akl_float_to_chars,2);
  define("chars_to_atom",akl_chars_to_atom,2);
  define("chars_to_integer",akl_chars_to_integer,2);
  define("chars_to_float",akl_chars_to_float,2);
  define("term_to_list",akl_term_to_list,2);
  define("list_to_term",akl_list_to_term,2);
  define("term_to_functor",akl_term_to_functor,3);
  define("functor_to_term",akl_functor_to_term,3);
  define("list_to_length",akl_list_to_length,2);
  define("length_to_list",akl_length_to_list,2);
  define("islist",akl_islist,1);
  define("concat",akl_concat,3);
  define("memberchk",akl_memberchk,2);

  define("integer_to_float",akl_integer_to_float,2);
  define("float_to_integer",akl_float_to_integer,2);

#ifdef COMPAT08
  define("atom_to_list",akl_atom_to_chars,2);
  define("integer_to_list",akl_integer_to_chars,2);
  define("float_to_list",akl_float_to_chars,2);
  define("list_to_atom",akl_chars_to_atom,2);
  define("list_to_integer",akl_chars_to_integer,2);
  define("list_to_float",akl_chars_to_float,2);
  define("compound_extract",akl_compound_extract,3);
  define("compound_extract",akl_compound_extract2,4);
  define("compound_construct",akl_compound_construct,3);
  define("compound_construct",akl_compound_construct2,4);
#endif

  define("arg",akl_arg,3);
  define("sys_setarg",akl_sys_setarg,3);

  define("neg",akl_neg,2);
  define("com",akl_com,2);
  define("inc",akl_inc,2);
  define("dec",akl_dec,2);
  define("abs",akl_abs,2);
  define("min",akl_min,3);
  define("max",akl_max,3);
  define("add",akl_add,3);
  define("sub",akl_sub,3);
  define("mul",akl_mul,3);
  define("div",akl_div,3);
  define("div",akl_div4,4);
  define("mod",akl_mod,3);
  define("and",akl_and,3);
  define("or",akl_or,3);
  define("xor",akl_xor,3);
  define("shl",akl_shl,3);
  define("shr",akl_shr,3);

  define("int_equal", akl_equal,2);
  define("int_not_equal", akl_not_equal,2);
  define("int_less", akl_less,2);
  define("int_not_less", akl_not_less,2);
  define("int_greater", akl_greater,2);
  define("int_not_greater", akl_not_greater,2);

  define("fp_neg",akl_fp_neg,2);
  define("fp_inc",akl_fp_inc,2);
  define("fp_dec",akl_fp_dec,2);
  define("fp_abs",akl_fp_abs,2);
  define("fp_min",akl_fp_min,3);
  define("fp_max",akl_fp_max,3);
  define("fp_add",akl_fp_add,3);
  define("fp_sub",akl_fp_sub,3);
  define("fp_mul",akl_fp_mul,3);
  define("fp_div",akl_fp_div,3);

  define("fp_equal", akl_fp_equal,2);
  define("fp_not_equal", akl_fp_not_equal,2);
  define("fp_less", akl_fp_less,2);
  define("fp_not_less", akl_fp_not_less,2);
  define("fp_greater", akl_fp_greater,2);
  define("fp_not_greater", akl_fp_not_greater,2);

  /* somewhat experimental */

  define("dif",akl_dif,2);

  tempa.env = NULL;
  tempa.status = STABLE;
  tempa.tried = NULL;
  tempa.cont = NULL;
  tempa.constr = NULL;
  tempa.unify = NULL;
  tempa.father = NULL;
  tempa.previous = NULL;
  tempa.next = NULL;
#ifdef TRACE
  tempa.id = 0;
  tempa.trace = 0;
#endif

}
