/*
 * Copyright (C) 1990, 1993 by Ted Kim.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * res.c - X toolkit type resource management
 *
 * pxResourceDatabases
 * pxOpenResourceDatabase
 * pxCloseResourceDatabase
 *
 * pxQueryResourceDatabase
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xresource.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"
#include "attrib.h"

extern char *getenv();

typedef struct _PxRDB {		/* X resource database */
  struct _PxRDB *next;
  int des;
  XrmDatabase xRDB;
} PxRDB;

DL pxRDBList = {NULL, sizeof(PxRDB), 0};

/*
 * pxResourceDatabases(-Next)
 *	integer: Next
 */

void
pxResourceDatabases(pn)
     FFInteger *pn;
{
  if ((pxList = pxRDBList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

#ifndef XLIBDIR
#define XLIBDIR "/usr/lib/X11"
#endif

char libdir[] = XLIBDIR;

typedef struct {		/* pathname substitutions */
  char *full;			/* full language spec */
  int lang;			/* language spec seen flag */
  char *class;
  char *custom;
  char *homedir;
  char *hostname;
  char *appresdir;
  char *libdir;
} PxSubstitutions;

/*
 * char *
 * pxNextFilename(src, dst, sp)
 *
 * copies path element in src to dst, making substitutions along the way
 * returns start of next path element
 */

#define SLASH '/'

#define SUBSTITUTE(FIELD) \
  if (FIELD != NULL) { \
    strcpy(dst, FIELD); \
    dst += strlen(FIELD); \
    break; \
  } 

#define COMPRESS(FIELD) \
  if (FIELD == NULL && *(src + 1) == SLASH) \
    src++

char *			
pxNextFilename(src, dst, sp)
     char *src, *dst;
     PxSubstitutions *sp;
{
  if (src == NULL || *src == '\0') /* empty string */
    return NULL;
  for (;; src++) {
    switch (*src) {
    case '\0':			/* end of string */
      *dst = '\0'; return NULL;
    case ':':			/* end of path */
      *dst = '\0'; return ++src;
    case '%':			/* % escape */
      src++;
      switch (*src) {
      case '\0':		/* end of string */
	*dst = '\0'; return NULL;
      case ':':			/* end of path */
	*dst = '\0'; return ++src;
      case '%':			/* %% = % */
	*dst = '%'; dst++; break;			
      case 'A':			/* %A = $XAPPLRESDIR */
	SUBSTITUTE(sp->appresdir) COMPRESS(sp->appresdir); break;
      case 'N':			/* %N = class */
	SUBSTITUTE(sp->class) break;
      case 'C':			/* %C = customization */
	SUBSTITUTE(sp->custom) COMPRESS(sp->custom); break;
      case 'D':			/* %D = $XLIBDIR */
	SUBSTITUTE(sp->libdir) COMPRESS(sp->libdir); break;
      case 'H':			/* %H = hostname */
	SUBSTITUTE(sp->hostname) break;
      case 'L':			/* %L = full language spec */
	SUBSTITUTE(sp->full) COMPRESS(sp->full); break;
      case 'l':			/* %l = just language part */
	if (sp->lang) { 
	  char *ptr;
	  for (ptr = sp->full;
	       *ptr != '\0' && *ptr != '_' && *ptr != '.';
	       ptr++, dst++)
	    *dst = *ptr;
	} 
        COMPRESS(sp->lang);	/* this works! */
	break;
      default:			/* % followed by something else */
	*dst = '%'; dst++; *dst = *src; 
      }
      break;
    case '~':			/* ~ = homedir */
      SUBSTITUTE(sp->homedir) 
      if (strcmp(sp->homedir, "/") == 0) src++;	/* superuser home */
      break;
    default:
      *dst = *src; dst++;
    }
  }
}

bool
pxTestFile(fn)
     char *fn;
{
  struct stat status;

  if (access(fn, R_OK) == 0 &&
      stat(fn, &status) == 0 &&
      (status.st_mode & S_IFDIR) == 0) /* not a directory */
    return true;
  return false;
}

bool
pxSearchPath(path, sp, fn)
  char *path, *fn;
  PxSubstitutions *sp;
{
  char *next;

  if (path == NULL || *path == '\0')
    return false;
  for (next = path; next != NULL;) {
    next = pxNextFilename(next, fn, sp);
    if (pxTestFile(fn)) 
      return true;
  }
  return false;
}

/*
 * Bug in R5 pl22: XrmGetStringDatabase bombs with NULL arg!
 */
#define SafeGetStringDB(X) ( (X) ? XrmGetStringDatabase(X) : NULL )

/*
 * Resource Search Paths
 *	
 * XAPPLRESDIR search path
 * default XUSERFILESEARCHPATH
 * default XFILESEARCHPATH
 */

#define XARD_SEARCHPATH \
"%A/%L/%N%C:%A/%l/%N%C:%A/%N%C:%A/%L/%N:%A/%l/%N:%A/%N:~/%N"
#define DEFAULT_XUFSP \
"~/%L/%N%C:~/%l/%N%C:~/%N%C:~/%L/%N:~/%l/%N:~/%N"
#define DEFAULT_XFSP \
"%D/%L/app-defaults/%N%C:%D/%l/app-defaults/%N%C:%D/app-defaults/%N%C:%D/%L/app-defaults/%N:%D/%l/app-defaults/%N:%D/app-defaults/%N"

XrmDatabase
pxUserEnvironmentResources(sp)
     PxSubstitutions *sp;	/* path substitutions */
{
  char fn[MAXPATHLEN];		/* file name */

  if (pxSearchPath(getenv("XENVIRONMENT"), sp, fn))
    return XrmGetFileDatabase(fn);
  if (pxSearchPath("~/.Xdefaults-%H", sp, fn))
    return XrmGetFileDatabase(fn);
  return NULL;
}

XrmDatabase
pxDisplayResources(xdp, sp)
     Display *xdp;
     PxSubstitutions *sp;
{
  char fn[MAXPATHLEN];

  if (xdp->xdefaults) {
    return SafeGetStringDB(xdp->xdefaults);
  }
  if (pxSearchPath("~/.Xdefaults", sp, fn))
    return XrmGetFileDatabase(fn);
}

XrmDatabase
pxApplicationUserResources(sp)
     PxSubstitutions *sp;
{
  char fn[MAXPATHLEN];
  char *path;

  if (pxSearchPath(getenv("XUSERFILESEARCHPATH"), sp, fn))
    return XrmGetFileDatabase(fn);
  path = (sp->appresdir) ? XARD_SEARCHPATH : DEFAULT_XUFSP;
  if (pxSearchPath(path, sp, fn))
    return XrmGetFileDatabase(fn);
  return NULL;
}

XrmDatabase
pxApplicationClassResources(fr, sp)
     char *fr;			/* fallback resources */
     PxSubstitutions *sp;
{
  char fn[MAXPATHLEN];

  if (pxSearchPath(getenv("XFILESEARCHPATH"), sp, fn))
    return XrmGetFileDatabase(fn);
  if (pxSearchPath(DEFAULT_XFSP, sp, fn))
    return XrmGetFileDatabase(fn);
  else 
    return SafeGetStringDB(fr);
}

/*
 * pxOpenResourceDatabase(+Connection, +Name, +Class, +OLen, -RDB, 0)
 *	integer: Connection, OLen, RDB
 *	string: Name, Class
 */

FFInteger
pxOpenResourceDatabase(pc, pn, pcl, pol, prdb) 
     FFInteger pc, pol, *prdb;
     FFString pn, pcl;
{
  PxConnection *pcp;
  PxRDB *prdbp;
  PxSubstitutions subs;

  char *full, *homedir, *hostname;
  char hn[MAXHOSTNAMELEN + 1];	/* for hostname */

  XrmName name_list[3];
  XrmClass class_list[3];
  XrmRepresentation rep;
  XrmValue val;
  XrmDatabase langdb, src, dst, final;

  PX_ErrorInit("xOpenResourceDatabase/7");
  *prdb = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();

  full = NULL;
  homedir = getenv("HOME");
  if (gethostname(hn, MAXHOSTNAMELEN + 1) == 0) 
    hostname = hn;

  subs.full = NULL;		/* initialize substitutions */
  subs.lang = false;
  subs.class = NULL;
  subs.custom = NULL;
  subs.homedir = homedir;
  subs.hostname = NULL;
  subs.appresdir = NULL;
  subs.libdir = NULL;
				/* natural language */
  src = SafeGetStringDB(pxMallocBlock); /* opts */
  langdb = pxDisplayResources(pcp->xDisplay, &subs); /* display res/pref file */
  XrmMergeDatabases(src, &langdb);
  name_list[0] = XrmStringToName(pn); 
  name_list[1] = XrmStringToName("xnlLanguage");
  name_list[2] = NULLQUARK;
  class_list[0] = XrmStringToClass(pcl);
  class_list[1] = XrmStringToClass("XnlLanguge");
  class_list[2] = NULLQUARK;
  if (XrmQGetResource(langdb, name_list, class_list, &rep, &val))
    full = val.addr;
  else
    full = getenv("LANG");

  /* display resources/preferences file, requires: ~ */
  /* do this in inverted order to save copying */
  dst = pxDisplayResources(pcp->xDisplay, &subs); 

				/* screen resources */
  src = SafeGetStringDB(XScreenResourceString(
                        XDefaultScreenOfDisplay(pcp->xDisplay)));
  XrmMergeDatabases(src, &dst);

  subs.hostname = hostname;	/* user environment, requires subs: ~, %H */
  src = pxUserEnvironmentResources(&subs); 
  XrmMergeDatabases(src, &dst);

  src = SafeGetStringDB(pxMallocBlock); /* options */
  XrmMergeDatabases(src, &dst);

				/* customization resource  */
  name_list[1] = XrmStringToName("customization");
  class_list[1] = XrmStringToClass("Customization");
  if (XrmQGetResource(dst, name_list, class_list, &rep, &val))
    subs.custom = val.addr;
  
  /* application class resources, requires: %D, %C, %L, %N, %l */
  /* do this first to save copying and early PXVectorFree */
  subs.full = full;
  if (subs.full)
    subs.lang = true;
  subs.class = pcl;
  subs.homedir = NULL;		/* turn off */
  subs.hostname = NULL;
  subs.libdir = libdir;
  final = pxApplicationClassResources(pxMallocBlock + pol, &subs);
  (void)PXVectorFree();

  /* application user resources, requires: %A, %C, %L, %N, %l, ~ */
  subs.homedir = homedir;
  subs.hostname = hostname;
  subs.appresdir = getenv("XAPPLRESDIR");
  subs.libdir = NULL;		/* turn off */
  src = pxApplicationUserResources(&subs);
  XrmDestroyDatabase(langdb);	/* need to keep until %L no longer needed */
  XrmMergeDatabases(src, &final);
  XrmMergeDatabases(dst, &final);

  if (DLAdd(&pxRDBList, (int *)prdb, (DLHeader **)&prdbp))
    PX_Error(no_memory);
  prdbp->xRDB = final;
  return FF_Ok;
}

/*
 * pxCloseResourceDatabase(+RDB, 0)
 *	integer: RDB
 */
FFInteger
pxCloseResourceDatabase(prdb)
     FFInteger prdb;
{
  PxRDB *prdbp;

  PX_ErrorInit("xCloseResourceDatabase/1");

  if (DLLookup(pxRDBList, (int)prdb, (DLHeader **)&prdbp))
    PX_Error(no_such_rdb);

  XrmDestroyDatabase(prdbp->xRDB);

  DLRemove(&pxRDBList, (int)prdb);
  return FF_Ok;
}


/*
 * pxQueryResourceDatabase(+RDB, +Elements, -Value, 0)
 *	integer: RDB, Elements
 *	string: Value
 */
FFInteger
pxQueryResourceDatabase(prdb, pe, pv)
     FFInteger prdb, pe;
     FFString *pv;
{
  PxRDB *prdbp;

  char **src;
  int i;

  XrmQuark *dst;
  XrmName *names;
  XrmClass *classes;
  XrmRepresentation rep;
  XrmValue value;

  PX_ErrorInit("xQueryResourceDatabase/4");
  *pv = pxNil;

  if (DLLookup(pxRDBList, (int)prdb, (DLHeader **)&prdbp)) {
    (void)PXVectorFree();
    PX_Error(no_such_rdb);
  }

  if ((names = (XrmName *)malloc(sizeof(XrmQuark) * 2 * (pe + 1))) == NULL) {
    (void)PXVectorFree();
    PX_Error(no_memory);
  }
  classes = names + pe + 1;		
				/* do names and reverse order */
  dst = classes - 1;
  *dst = NULLQUARK;		/* put in null */
  for (i = 0, src = (char **)pxMallocBlock, dst--; /* dst counts down */
       i < pe; i++, src++, dst--) 
    *dst = XrmStringToName(*src);
				/* do classes and reverse order */
  dst = classes + pe;
  *dst = NULLQUARK;		/* put in null */
  for (i = 0, dst--; i < pe; i++, src++, dst--) /* dst counts down */
    *dst = XrmStringToClass(*src);

  (void)PXVectorFree();

  if (!XrmQGetResource(prdbp->xRDB, names, classes, &rep, &value))
    return FF_Fail;
  *pv = value.addr;
  return FF_Ok;
}

/*
 * eof
 */
