%
% Copyright (C) 1990 by Ted Kim.
%
% This software work was developed at UCLA with support in part from
% DARPA Contract F29601-87-C-0072.
%

%
% pred.pl
% X Window Interface for Prolog (XWIP)
% Predicates
%

% connections
xOpenConnection(Name, Connection) :- pxOpenConnection(Name, Connection, 0).

xCloseConnection(Connection) :- pxCloseConnection(Connection, 0).

xConnections(ConnectionList) :- pxConnections(N), pxListGet(N, ConnectionList).

xQueryConnection(Connection, QueryList) :-
	pxQueryList(QueryList, Connection, pxcConnectionAttribute).

% formats
xFormats(FormatList) :- pxFormats(N), pxListGet(N, FormatList).

xQueryFormat(Format, QueryList) :-
	pxQueryList(QueryList, Format, pxcFormatAttribute).

% screens
xScreens(ScreenList) :- pxScreens(N), pxListGet(N, ScreenList).

xQueryScreen(Screen, QueryList) :-
	pxQueryList(QueryList, Screen, pxcScreenAttribute).

% depths
xDepths(DepthList) :- pxDepths(N), pxListGet(N, DepthList).

xQueryDepth(Depth, QueryList) :-
	pxQueryList(QueryList, Depth, pxcDepthAttribute).

% visuals
xVisuals(VisualList) :- pxVisuals(N), pxListGet(N, VisualList).

xQueryVisual(Visual, QueryList) :-
	pxQueryList(QueryList, Visual, pxcVisualAttribute).

% windows
xCreateWindow(Connection, Parent, X, Y, Width, Height, BorderWidth, Depth,
  Class, Visual, AttributeList, Window) :-
	pxcWindowDepth(RDepth, Depth),
	pxcWindowClass(RClass, Class),
	pxcParent(RVisual, Visual),
	Width >= 0, Height >= 0, RDepth >= 0,
	pxAttributeList(AttributeList, pxcSetWindowAttribute),
	pxCreateWindow(Connection, Parent, X, Y, Width, Height, BorderWidth,
	  RDepth, RClass, RVisual, Window, 0).

xDestroyWindow(Connection, Window) :- pxDestroyWindow(Connection, Window, 0).

xDestroySubwindows(Connection, Window) :-
	pxDestroySubwindows(Connection, Window, 0).

% window information
xQueryWindow(Connection, Window, QueryList) :-
	pxQueryWindow(Connection, Window, 0),
	pxAttributeList(QueryList, pxcGetWindowAttribute).

xSetWindow(Connection, Window, AttributeList) :-
	pxAttributeList(AttributeList, pxcSetWindowAttribute),
	pxSetWindow(Connection, Window, 0).

xQueryTree(Connection, Window, Root, Parent, Children) :-
	pxQueryTree(Connection, Window, Root, Parent, N, 0),
	pxVectorGet(N, Children).

xTranslateCoordinates(Connection, Source, Dest, SrcX, SrcY, DestX, DestY,
  Child) :-
	pxTranslateCoordinates(Connection, Source, Dest, SrcX, SrcY, DestX,
	  DestY, Child, 0).

% other window operations
xMapWindow(Connection, Window) :- pxMapWindow(Connection, Window, 0).

xMapSubwindows(Connection, Window) :- pxMapSubwindows(Connection, Window, 0).

xUnmapWindow(Connection, Window) :- pxUnmapWindow(Connection, Window, 0).

xUnmapSubwindows(Connection, Window) :-
	pxUnmapSubwindows(Connection, Window, 0).

xCirculateSubwindows(Connection, Window, LowerHighest) :-
	pxCirculateSubwindows(Connection, Window, LowerHighest, 0).

% atoms

xAtomExists(_, AtomName, Atom) :- pxAtom(AtomName, Atom), !.
xAtomExists(Connection, AtomName, Atom) :-
	pxInternAtom(Connection, AtomName, 1, Atom, 0).

xAtom(_, AtomName, Atom) :- pxAtom(AtomName, Atom), !.
xAtom(Connection, AtomName, Atom) :-
	nonvar(AtomName), pxInternAtom(Connection, AtomName, 0, Atom, 0).
xAtom(Connection, AtomName, Atom) :-
	var(AtomName),
	pxGetAtomName(Connection, Atom, AtomName, 0),
	pxVectorXFree.

% properties
xWindowProperties(Connection, Window, PropertyList) :-
	pxWindowProperties(Connection, Window, N, 0),
	pxVectorGet(N, PropertyList).

xGetProperty(Connection, Window, Property, Offset, Length, DeleteIfEnd,
  RequestedType, WantAtom, Type, Remaining, Value) :-
	pxcBoolean(RDIE, DeleteIfEnd),
	pxcAny(RRT, RequestedType),
	pxcBoolean(RWA, WantAtom),
	Offset >= 0, Length >= 0,
	pxGetProperty(Connection, Window, Property, Offset, Length, RDIE, RRT,
	  RWA, Type, Remaining, Format, Atom, N, 0),
	pxPropertyCompose(Format, Atom, N, Value).

xSetProperty(Connection, Window, Property, Type, Mode, Data) :-
	atom(Data), !,
	pxcPropertyMode(RMode, Mode),
	pxSetProperty(Connection, Window, Property, Type, RMode, 0, Data, 0).
xSetProperty(Connection, Window, Property, Type, Mode, Data) :-
	pxcPropertyMode(RMode, Mode),
	pxPropertyDecompose(Data, Format, List, Length),
	pxVectorAlloc(Format, Length, 0), pxVectorPut(List),
	pxSetProperty(Connection, Window, Property, Type, RMode, Format, [], 0).

xDeleteProperty(Connection, Window, Property) :-
	pxDeleteProperty(Connection, Window, Property, 0).

xRotateProperties(Connection, Window, Positions, PropertyList) :-
	pxListLength(PropertyList, Len),
	pxVectorAlloc(32, Len, 0), pxVectorPut(PropertyList),
	pxRotateProperties(Connection, Window, Positions, 0).

% selections
xGetSelectionOwner(Connection, Selection, Window) :-
	pxGetSelectionOwner(Connection, Selection, Window, 0).

xSetSelectionOwner(Connection, Selection, Owner, Time) :-
	pxcNone(ROwner, Owner),
	pxcTime(RTime, Time),
	pxUnsignedPut1(RTime),
	pxSetSelectionOwner(Connection, Selection, ROwner, 0) .

xConvertSelection(Connection, Selection, DestType, DestProperty, DestWindow,
  Time) :-
	pxcNone(RDestProperty, DestProperty),
	pxcTime(RTime, Time),
	pxUnsignedPut1(RTime),
	pxConvertSelection(Connection, Selection, DestType, RDestProperty,
	   DestWindow, 0).

% pixmaps
xCreatePixmap(Connection, Screen, Width, Height, Depth, Pixmap) :-
	Width >= 0, Height >= 0, Depth >= 0,
	pxCreatePixmap(Connection, Screen, Width, Height, Depth, Pixmap, 0).

xDestroyPixmap(Connection, Pixmap) :- pxDestroyPixmap(Connection, Pixmap, 0).

xQueryPixmap(Connection, Pixmap, QueryList) :-
	pxQueryPixmap(Connection, Pixmap, 0),
	pxAttributeList(QueryList, pxcPixmapAttribute).

% colormaps
xCreateColormap(Connection, Screen, Visual, AllocAll, Colormap) :-
	pxcBoolean(RAllocAll, AllocAll),
	pxCreateColormap(Connection, Screen, Visual, RAllocAll, Colormap, 0).

xDestroyColormap(Connection, Colormap) :-
	pxDestroyColormap(Connection, Colormap, 0).

xMoveToNewColormap(Connection, Current, New) :-
	pxMoveToNewColormap(Connection, Current, New, 0).

xLoadedColormaps(Connection, Screen, ColormapList) :-
	pxLoadedColormaps(Connection, Screen, N, 0),
	pxVectorGet(N, ColormapList).

xLoadColormap(Connection, Colormap) :-
	pxLoadColormap(Connection, Colormap, 0).

xUnloadColormap(Connection, Colormap) :-
	pxUnloadColormap(Connection, Colormap, 0).

% color
xParseColor(Connection, Colormap, Specification, xColor(ER, EG, EB)) :-
	pxParseColor(Connection, Colormap, Specification, ER, EG, EB, 0).

xQueryColor(Connection, Colormap, Name, xColor(AR, AG, AB), 
  xColor(ER, EG, EB)) :-
	pxQueryColor(Connection, Colormap, Name, AR, AG, AB, ER, EG, EB, 0).

xGetColors(Connection, Colormap, CellList) :-
	pxListLength(CellList, Len), pxVectorAlloc(14, Len, 0),
	pxPixelPut(CellList),
	pxGetColors(Connection, Colormap, 0),
	pxColorGetN(Len, CellList).

xRequestColor(Connection, Colormap, xColor(R, G, B), 
  xCell(Pixel, xColor(AR, AG, AB))) :-
	pxRequestColor(Connection, Colormap, R, G, B, Pixel, AR, AG, AB, 0).

xRequestNamedColor(Connection, Colormap, Name, xCell(Pixel, xColor(AR, AG, AB)),
  xColor(ER, EG, EB)) :-
	pxRequestNamedColor(Connection, Colormap, Name, Pixel, AR, AG, AB,
	  ER, EG, EB, 0).

xFreeColors(Connection, Colormap, PixelList, PlaneMask) :-
	pxListLength(PixelList, Len),
	pxVectorAlloc(32, Len, 0), pxVectorPut(PixelList),
	pxFreeColors(Connection, Colormap, PlaneMask, 0).

xAllocColorCells(Connection, Colormap, Contiguous, Planes, Pixels, 
  PlaneList, PixelList) :-
	pxcBoolean(RContiguous, Contiguous),
	Planes >= 0, Pixels >= 1, Total is Planes + Pixels,
	pxVectorAlloc(32, Total, 0),
	pxAllocColorCells(Connection, Colormap, RContiguous, Planes, Pixels, 0),
	pxVectorGetN(Planes, PlaneList), pxVectorGetN(Pixels, PixelList).

xAllocColorPlanes(Connection, Colormap, Contiguous, Colors, Reds, Greens, Blues,
  PixelList, RedMask, GreenMask, BlueMask) :-
	pxcBoolean(RContiguous, Contiguous),
	Colors >= 1, Reds >= 0, Greens >= 0, Blues >= 0,
	pxVectorAlloc(32, Colors, 0),
	pxAllocColorPlanes(Connection, Colormap, RContiguous, Colors, Reds,
	  Greens, Blues, RedMask, GreenMask, BlueMask, 0),
	pxVectorGetN(Colors, PixelList).

xSetColors(Connection, Colormap, AssignmentList) :-
	pxListLength(AssignmentList, Len), pxVectorAlloc(14, Len, 0),
	pxColorPut(AssignmentList),
	pxSetColors(Connection, Colormap, 0).

xSetNamedColor(Connection, Colormap, Name, xCell(Pixel, xColor(R, G, B))) :-
	pxRedCheck(R, _, RBit),
	pxGreenCheck(G, _, GBit),
	pxBlueCheck(B, _, BBit),
	Flags is RBit \/ GBit \/ BBit,
	pxSetNamedColor(Connection, Colormap, Name, Pixel, Flags, 0).

% GCs
xGCs(GCList) :- pxGCs(N), pxListGet(N, GCList).

xCreateGC(Connection, Drawable, AttributeList, GC) :-
	pxAttributeList(AttributeList, pxcSetGCAttribute),
	pxCreateGC(Connection, Drawable, GC, 0).

xDestroyGC(Connection, GC) :- pxDestroyGC(Connection, GC, 0).

xCopyGC(Connection, Src, BitList, Dest) :-
	pxMaskFromBits(Mask, BitList, pxmGC),
	pxCopyGC(Connection, Src, Mask, Dest, 0).

xQueryGC(GC, QueryList) :-
	pxQueryList(QueryList, GC, pxcGetGCAttribute).

xQueryBestSize(Connection, Mode, Screen, Width, Height, ReturnWidth,
  ReturnHeight) :-
	pxcSizeQuery(RMode, Mode),
	Width >= 0, Height >= 0,
	pxQueryBestSize(Connection, RMode, Screen, Width, Height, ReturnWidth,
	  ReturnHeight, 0).

xSetGC(Connection, GC, AttributeList) :-
	pxAttributeList(AttributeList, pxcSetGCAttribute),
	pxSetGC(Connection, GC, 0).

xSetClips(Connection, GC, XOrigin, YOrigin, RectangleList, Ordering) :-
	pxcClipOrdering(ROrdering, Ordering),
	pxRectanglePut(RectangleList),
	pxSetClips(Connection, GC, XOrigin, YOrigin, ROrdering, 0).

xSetDashes(Connection, GC, DashOffset, DashList) :-
	pxListLength(DashList, L), pxVectorAlloc(8, L, 0),
	pxVectorPut(DashList), pxSetDashes(Connection, GC, DashOffset, 0).
	
% graphics
xClearArea(Connection, Window, X, Y, Width, Height, Exposures) :-
	pxcBoolean(RExposures, Exposures),
	Width >= 0, Height >= 0,
	pxClearArea(Connection, Window, X, Y, Width, Height, RExposures, 0).

xCopyArea(Connection, Src, Dest, GC, SrcX, SrcY, Width, Height, DestX, DestY) :-
	Width >= 0, Height >= 0,
	pxCopyArea(Connection, Src, Dest, GC, SrcX, SrcY, Width, Height, DestX,
	  DestY, 0).

xCopyPlane(Connection, Src, Dest, GC, SrcX, SrcY, Width, Height, DestX, DestY,
  Plane) :-
	Width >= 0, Height >= 0,
	pxCopyPlane(Connection, Src, Dest, GC, SrcX, SrcY, Width, Height, DestX,
	  DestY, Plane, 0).

xDrawPoints(Connection, Drawable, GC, PointList, Relative) :-
	pxcBoolean(RRelative, Relative),
	pxPointPut(PointList),
	pxDrawPoints(Connection, Drawable, GC, RRelative, 0).

xDrawLines(Connection, Drawable, GC, PointList, Relative) :-
	pxcBoolean(RRelative, Relative),
	pxPointPut(PointList),
	pxDrawLines(Connection, Drawable, GC, RRelative, 0). 

xDrawSegments(Connection, Drawable, GC, SegmentList) :-
	pxSegmentPut(SegmentList), 
	pxDrawSegments(Connection, Drawable, GC, 0).

xDrawRectangles(Connection, Drawable, GC, RectangleList) :-
	pxRectanglePut(RectangleList),
	pxDrawRectangles(Connection, Drawable, GC, 0).

xDrawArcs(Connection, Drawable, GC, ArcList) :-
	pxArcPut(ArcList),
	pxDrawArcs(Connection, Drawable, GC, 0).

xFillRectangles(Connection, Drawable, GC, RectangleList) :-
	pxRectanglePut(RectangleList),
	pxFillRectangles(Connection, Drawable, GC, 0).

xFillPolygon(Connection, Drawable, GC, PointList, Shape, Relative) :-
	pxcBoolean(RRelative, Relative),
	pxcPolygonShape(RShape, Shape),
	pxPointPut(PointList),
	pxFillPolygon(Connection, Drawable, GC, RShape, RRelative, 0).

xFillArcs(Connection, Drawable, GC, ArcList) :-
	pxArcPut(ArcList),
	pxFillArcs(Connection, Drawable, GC, 0).

% image.c
xImages(ImageList) :- pxImages(N), pxListGet(N, ImageList).

xCreateImage(Connection, Visual, Depth, Format, Offset, Width, Height,
  BitmapPad, BytesPerLine, Image) :-
	pxcImageFormat(RFormat, Format),
	pxcDefault(RBPL, BytesPerLine),
	pxcDefault(RP, BitmapPad),
	Width >= 0, Height >= 0, Depth >= 0,
	pxCreateImage(Connection, Visual, Depth, RFormat, Offset, Width, Height,
	  RP, RBPL, Image, 0).

xDestroyImage(Image) :- pxDestroyImage(Image, 0).

xQueryImage(Image, QueryList) :-
	pxQueryList(QueryList, Image, pxcImageAttribute).

xGetImage(Connection, Drawable, SrcX, SrcY, Width, Height, PlaneMask, DestImage,
  DestX, DestY) :-
	Width >= 0, Height >= 0,
	pxGetImage(Connection, Drawable, SrcX, SrcY, Width, Height, PlaneMask,
	  DestImage, DestX, DestY, 0).

xPutImage(Connection, Drawable, GC, Image, SrcX, SrcY, DestX, DestY, Width,
  Height) :-
	Width >= 0, Height >= 0,
	pxPutImage(Connection, Drawable, GC, Image, SrcX, SrcY, DestX, DestY,
	  Width, Height, 0).

xSubImage(Image, X, Y, Width, Height, SubImage) :-
	Width >= 0, Height >= 0,
	pxSubImage(Image, X, Y, Width, Height, SubImage, 0).

xAddPixel(Image, Value) :- pxAddPixel(Image, Value, 0).

xGetPixel(Image, X, Y, Pixel) :- pxGetPixel(Image, X, Y, Pixel, 0).

xSetPixel(Image, X, Y, Pixel) :- pxSetPixel(Image, X, Y, Pixel, 0).

% fonts
xOpenFont(Connection, Name, Font) :- pxOpenFont(Connection, Name, Font, 0).

xCloseFont(Connection, Font) :- pxCloseFont(Connection, Font, 0).

xLoadedFonts(LoadedFontList) :- pxLoadedFonts(N), pxListGet(N, LoadedFontList).

xLoadFont(Connection, Font, LoadedFont) :-
	pxLoadFont(Connection, Font, LoadedFont, 0).

xUnloadFont(LoadedFont) :- pxUnloadFont(LoadedFont, 0).

% listing fonts
xGetFontPath(Connection, PathList) :-
	pxGetFontPath(Connection, N, 0), pxStringGet(N, PathList).

xSetFontPath(Connection, PathList) :-
	pxStringLength(PathList, E, Len),
	pxStringAlloc(10, E, Len, 0), pxStringPut(PathList),
	pxSetFontPath(Connection, 0).

xListFonts(Connection, Pattern, MaxNames, FontList) :-
	pxListFonts(Connection, Pattern, MaxNames, N, 0),
	pxStringGet(N, FontList).

% font information
xQueryFont(LoadedFont, QueryList) :-
	pxQueryList(QueryList, LoadedFont, pxcFontAttribute).

xGetFontProperty(LoadedFont, Property, Value) :-
	pxGetFontProperty(LoadedFont, Property, Value, 0).

xGetCharInfo(LoadedFont, Character, LeftBearing, RightBearing, Width, Ascent,
  Descent, Attributes) :-
	pxGetCharInfo(LoadedFont, Character, LeftBearing, RightBearing, Width,
	  Ascent, Descent, Attributes, 0).

xTextExtents(LoadedFont, xText(8, CL), LeftBearing, RightBearing, Width, Ascent,
  Descent) :-  !,
	pxListLength(CL, Len), pxVectorAlloc(8, Len, 0), pxVectorPut(CL),
	pxTextExtents(LoadedFont, 8, [], LeftBearing, RightBearing, Width,
	  Ascent, Descent, 0).
xTextExtents(LoadedFont, xText(16, CL), LeftBearing, RightBearing, Width, 
  Ascent, Descent) :- !,
	pxListLength(CL, Len), pxVectorAlloc(7, Len, 0), pxVectorPut(CL),
	pxTextExtents(LoadedFont, 7, [], LeftBearing, RightBearing, Width,
	  Ascent, Descent, 0).
xTextExtents(LoadedFont, Atom, LeftBearing, RightBearing, Width, Ascent,
  Descent) :-
	pxTextExtents(LoadedFont, 0, Atom, LeftBearing, RightBearing, Width,
	  Ascent, Descent, 0).

% drawing text
xDrawText(Connection, Drawable, GC, X, Y, TextList) :-
	pxTextFormat(TextList, IL, T, I, C),
	pxTextAlloc(T, I, C, 0), pxTextPut(IL),
	pxDrawText(Connection, Drawable, GC, X, Y, 0).

xImageText(Connection, Drawable, GC, X, Y, xText(8, CL)) :- !,
	pxListLength(CL, Len), pxVectorAlloc(8, Len, 0), pxVectorPut(CL),
	pxImageText(Connection, Drawable, GC, X, Y, 8, [], 0).
xImageText(Connection, Drawable, GC, X, Y, xText(16, CL)) :- !,
	pxListLength(CL, Len), pxVectorAlloc(7, Len, 0), pxVectorPut(CL),
	pxImageText(Connection, Drawable, GC, X, Y, 7, [], 0).
xImageText(Connection, Drawable, GC, X, Y, Atom) :-
	pxImageText(Connection, Drawable, GC, X, Y, 0, Atom, 0).

% cursor

xCreateFontCursor(Connection, Shape, Cursor) :-
	pxCursorFont(Shape, RShape),
	pxCreateFontCursor(Connection, RShape, Cursor, 0).

xCreateGlyphCursor(Connection, SrcFont, MaskFont, SrcChar, MaskChar,
  xColor(FR, FG, FB), xColor(BR, BG, BB), Cursor) :-
	pxcNone(RMF, MaskFont),
	pxcNone(RMC, MaskChar),
	pxCreateGlyphCursor(Connection, SrcFont, RMF, SrcChar, RMC,
	  FR, FG, FB, BR, BG, BB, Cursor, 0).

xCreatePixmapCursor(Connection, Source, Mask, xColor(FR, FG, FB), 
  xColor(BR, BG, BB), X, Y, Cursor) :-
	pxcNone(RMask, Mask),
	pxCreatePixmapCursor(Connection, Source, RMask, FR, FG, FB, BR, BG, BB,
	  X, Y, Cursor, 0).

xDestroyCursor(Connection, Cursor) :- pxDestroyCursor(Connection, Cursor, 0).

xRecolorCursor(Connection, Cursor, xColor(FR, FG, FB), xColor(BR, BG, BB)) :-
	pxRecolorCursor(Connection, Cursor, FR, FG, FB, BR, BG, BB, 0).

% pointer 
xPointerState(Connection, Window, Root, Child, RootX, RootY, WinX, WinY,
  State) :- 
	pxPointerState(Connection, Window, Root, Child, RootX, RootY, WinX,
	  WinY, Mask, 0),
	pxMaskToBits(Mask, State, pxmKey).

xWarpPointer(Connection, Source, Dest, SrcX, SrcY, Width, Height, DestX,
  DestY) :-
	pxcNone(RSource, Source),
	pxcNone(RDest, Dest),
	Width >= 0, Height >= 0,
	pxWarpPointer(Connection, RSource, RDest, SrcX, SrcY, Width, Height,
	  DestX, DestY, 0).

xQueryPointer(Connection, QueryList) :-
	pxQueryPointer(Connection, 0),
	pxAttributeList(QueryList, pxcGetPointerAttribute).

xSetPointer(Connection, AttributeList) :-
	pxAttributeList(AttributeList, pxcSetPointerAttribute),
	pxSetPointer(Connection, 0).

xGetPointerMapping(Connection, Length, Map) :-
	pxVectorAlloc(8, Length, 0),
	pxGetPointerMapping(Connection, Length, 0),
	pxStructGetN(xPointerMap, Length, Map).

xSetPointerMapping(Connection, Map) :-
	functor(Map, xPointerMap, L),
	pxVectorAlloc(8, L, 0), pxStructPutN(L, Map),
	pxSetPointerMapping(Connection, 0).

% keyboard
xDownKeymap(Connection, Keymap) :-
	pxDownKeymap(Connection, N, 0),
	pxVectorGet(N, MapList), pxKeymapToCodes(MapList, Keymap).

xQueryKeyboard(Connection, QueryList) :-
	pxQueryKeyboard(Connection, 0),
	pxAttributeList(QueryList, pxcGetKeyboardAttribute).

xSetKeyboard(Connection , AttributeList) :-
	pxAttributeList(AttributeList, pxcSetKeyboardAttribute),
	pxSetKeyboard(Connection, 0).

xGetKeyboardMapping(Connection, First, Count, Map) :-
	Count >= 0,
	pxGetKeyboardMapping(Connection, First, Count, K, N, 0),
	pxMapCompose(N, Count, K, xKeyboardMap, xKeysyms, Map).

xSetKeyboardMapping(Connection, First, Map) :-
	pxMapDecompose(Map, xKeyboardMap, xKeysyms, C, K),
	T is C * K, pxVectorAlloc(32, T, 0),
	E is C + 1, pxMapPut(1, E, K, Map),
	pxSetKeyboardMapping(Connection, First, K, C, 0).

xGetModifierMapping(Connection, Map) :-
	pxGetModifierMapping(Connection, K, 0),
	pxMapCompose(0, 8, K, xModifierMap, xKeyCodes, Map).

xSetModifierMapping(Connection, Map, Status) :-
	pxMapDecompose(Map, xModifierMap, xKeyCodes, 8, K),
	T is 8 * K, pxVectorAlloc(17, T, 0), 
	pxMapPut(1, 9, K, Map),
	pxSetModifierMapping(Connection, K, RStatus, 0),
        pxcBoolean(RStatus, Status).

xBell(Connection, Percent) :- pxBell(Connection, Percent, 0).

% client keyboard processing
xKeycodeToKeysym(Connection, Keycode, Index, Keysym) :-
	pxKeycodeToKeysym(Connection, Keycode, Index, KS, 0),
	pxcNoSymbol(KS, Keysym).

xKeysymToKeycode(Connection, Keysym, Keycode) :-
	pxcNoSymbol(KS, Keysym),
	pxKeysymToKeycode(Connection, KS, Keycode, 0).

xRefreshMapping(Event) :-
	pxEventPut(Event), pxRefreshMapping(0).

% window manager 
xSetSaveSet(Connection, Window, Delete) :- 
	pxcBoolean(RDelete, Delete),
	pxSetSaveSet(Connection, Window, RDelete, 0).

xReparentWindow(Connection, Window, Parent, X, Y) :-
	pxReparentWindow(Connection, Window, Parent, X, Y, 0).

xSetCloseDownMode(Connection, Mode) :-
	pxcCloseDown(RMode, Mode), pxSetCloseDownMode(Connection, RMode, 0).

xKillClient(Connection, Resource) :-
	pxcKillResource(RResource, Resource),
	pxKillClient(Connection, RResource, 0).

xGetInputFocus(Connection, Window, Revert) :-
	pxcFocusWindow(RWindow, Window),
	pxcRevertState(RRevert, Revert),
	pxGetInputFocus(Connection, RWindow, RRevert, 0).

xSetInputFocus(Connection, Window, Revert, Time) :-
	pxcFocusWindow(RWindow, Window),
	pxcRevertState(RRevert, Revert),
	pxcTime(RTime, Time),
	pxUnsignedPut1(RTime),
	pxSetInputFocus(Connection, RWindow, RRevert, 0).
 
% grab pointer
xGrabPointer(Connection, GrabWindow, NormalEvents, BitList, PointerRelease, 
  KeyboardRelease, ConfineWindow, Cursor, Time, Status) :-
	pxcBoolean(RNormalEvents, NormalEvents),
	pxcBoolean(RPointerRelease, PointerRelease),
	pxcBoolean(RKeyboardRelease, KeyboardRelease),
	pxcNone(RConfineWindow, ConfineWindow),
	pxcTime(RTime, Time),
	pxUnsignedPut1(RTime),
	pxMaskFromBits(EventMask, BitList, pxmEvent),
	pxGrabPointer(Connection, GrabWindow, RNormalEvents, EventMask,
	  RPointerRelease, RKeyboardRelease, RConfineWindow, Cursor, RStatus,
	  0),
	pxcGrabStatus(RStatus, Status).

xUngrabPointer(Connection, Time) :-
	pxcTime(RTime, Time), pxUnsignedPut1(RTime),
	pxUngrabPointer(Connection, 0).

xSetActivePointer(Connection, BitList, Cursor, Time) :-
	pxcNone(RCursor, Cursor),
	pxcTime(RTime, Time), pxUnsignedPut1(RTime),
	pxMaskFromBits(EventMask, BitList, pxmEvent),
	pxSetActivePointer(Connection, EventMask, RCursor, 0).

xGrabButton(Connection, Button, ModList, GrabWindow, NormalEvents, EventList,
  PointerRelease, KeyboardRelease, ConfineWindow, Cursor) :-
	pxcAny(RButton, Button),
	pxcBoolean(RNormalEvents, NormalEvents),
	pxcBoolean(RPointerRelease, PointerRelease),
	pxcBoolean(RKeyboardRelease, KeyboardRelease),
	pxcNone(RConfineWindow, ConfineWindow),
	pxMaskFromBits(Modifiers, ModList, pxmKey),
	pxMaskFromBits(EventMask, EventList, pxmEvent),
	pxGrabButton(Connection, RButton, Modifiers, GrabWindow, RNormalEvents,
	  EventMask, RPointerRelease, RKeyboardRelease, RConfineWindow, Cursor,
	  0).

xUngrabButton(Connection, Button, ModList, GrabWindow) :-
	pxcAny(RButton, Button),
	pxMaskFromBits(Modifiers, ModList, pxmKey),
	pxUngrabButton(Connection, RButton, Modifiers, GrabWindow, 0).

% grab keyboard
xGrabKeyboard(Connection, GrabWindowm, NormalEvents, PointerRelease,
  KeyboardRelease, Time, Status) :-
	pxcBoolean(RNormalEvents, NormalEvents),
	pxcBoolean(RPointerRelease, PointerRelease),
	pxcBoolean(RKeyboardRelease, KeyboardRelease),
	pxcTime(RTime, Time), pxUnsignedPut1(RTime),
	pxGrabKeyboard(Connection, GrabWindowm, RNormalEvents, RPointerRelease,
	  RKeyboardRelease, RStatus, 0),
	pxcGrabStatus(RStatus, Status).

xUngrabKeyboard(Connection, Time) :-
	pxcTime(RTime, Time), pxUnsignedPut1(RTime),
	pxUngrabKeyboard(Connection, 0).

xGrabKey(Connection, Keycode, ModList, GrabWindow, NormalEvents,
  PointerRelease, KeyboardRelease) :-
	pxcAny(RKeycode, Keycode),
	pxcBoolean(RNormalEvents, NormalEvents),
	pxcBoolean(RPointerRelease, PointerRelease),
	pxcBoolean(RKeyboardRelease, KeyboardRelease),
	pxMaskFromBits(Modifiers, ModList, pxmKey),
	pxGrabKey(Connection, RKeycode, Modifiers, GrabWindow, RNormalEvents,
	  RPointerRelease, RKeyboardRelease, 0).

xUngrabKey(Connection, Keycode, ModList, GrabWindow) :-
	pxcAny(RKeycode, Keycode),
	pxMaskFromBits(Modifiers, ModList, pxmKey),
	pxUngrabKey(Connection, RKeycode, Modifiers, GrabWindow, 0).

% other grab functions
xAllowEvents(Connection, Mode, Time) :-
	pxcEventMode(RMode, Mode),
	pxcTime(RTime, Time), pxUnsignedPut1(RTime),
	pxAllowEvents(Connection, RMode, 0).

xGrabServer(Connection) :- pxGrabServer(Connection, 0).

xUngrabServer(Connection) :- pxUngrabServer(Connection, 0).

% screen saver
xGetScreenSaver(Connection, Timeout, Interval, Blanking, Exposures) :-
	pxcBlanking(RBlanking, Blanking),
	pxcExposures(RExposures, Exposures),
	pxGetScreenSaver(Connection, Timeout, Interval, RBlanking, RExposures,
	  0).

xSetScreenSaver(Connection, Timeout, Interval, Blanking, Exposures) :-
	pxcTimeout(RTimeout, Timeout),
	pxcInterval(RInterval, Interval),
	pxcBlanking(RBlanking, Blanking),
	pxcExposures(RExposures, Exposures),
	pxSetScreenSaver(Connection, RTimeout, RInterval, RBlanking,
	  RExposures, 0).

xScreenSaver(Connection, Activate) :- 
	pxcBoolean(RActivate, Activate),
	pxScreenSaver(Connection, RActivate, 0).

% access control
xQueryAccess(Connection, Control, HostList) :-
	pxcBoolean(RControl, Control),
	pxQueryAccess(Connection, RControl, N, 0),
	pxStringGet(N, HostList).

xSetAccess(Connection, Enable) :-
	pxcBoolean(REnable, Enable), pxSetAccess(Connection, REnable, 0).

xSetHostAccess(Connection, Insert, Host) :-
	pxcBoolean(RInsert, Insert),
	pxSetHostAccess(Connection, RInsert, Host, 0).

% events
xEventsQueued(Connection, Mode, Count) :-
	pxcEventQueue(RMode, Mode),
	pxEventsQueued(Connection, RMode, Count, 0).

xNextEvent(Connection, Remove, Event) :- 
	pxcBoolean(RRemove, Remove),
	pxNextEvent(Connection, RRemove, L, 0),
	pxEventGet(L, Event).

xGetEvent(Connection, Window, BitList, TypeName, Remove, Block, Event) :-
	pxcAny(RWindow, Window),
	pxMaskFromBits(Mask, BitList, pxmEvent),
	pxcEventType(TypeName, Type, _),
	pxcBoolean(RRemove, Remove),
	pxcBoolean(RBlock, Block), 
	pxGetEvent(Connection, RWindow, Mask, Type, RRemove, RBlock, L, 0),
	pxEventGet(L, Event).

xSelectEvent(Connections, ReadFDs, WriteFDs, ExceptFDs, TimeOut,
  SConnections, SReadFDs, SWriteFDs, SExceptFDs, TimeOutFlag) :-
	pxListLength(Connections, CL),
	pxListLength(ReadFDs, RL),
	pxListLength(WriteFDs, WL),
	pxListLength(ExceptFDs, EL),
	Len is CL + RL + WL + EL, pxVectorAlloc(32, Len, 0),
	pxVectorPut(Connections),
	pxVectorPut(ReadFDs),
	pxVectorPut(WriteFDs),
	pxVectorPut(ExceptFDs),
	pxcNone(RTO, TimeOut),
	pxUnsignedPut1(RTO),
	pxSelectEvent(CL, RL, WL, EL, SC, SR, SW, SE, TO, 0),
	pxVectorGetN(SC, SConnections),
	pxVectorGetN(SR, SReadFDs),
	pxVectorGetN(SW, SWriteFDs),
	pxVectorGetN(SE, SExceptFDs),
	pxcBoolean(TO, TimeOutFlag).

xPutBackEvent(Connection, Event) :-
	pxEventPut(Event), pxPutBackEvent(Connection, 0).

xSendEvent(Connection, Window, Propagate, BitList, Event) :-
	pxcSendWindow(RWindow, Window),
	pxcBoolean(RPropagate, Propagate), 
	pxMaskFromBits(EventMask, BitList, pxmEvent),
	pxEventPut(Event),
	pxSendEvent(Connection, RWindow, RPropagate, EventMask, 0).

xFlush(Connection) :- pxFlush(Connection, 0).

xSync(Connection, Discard) :-
	pxcBoolean(RDiscard, Discard), pxSync(Connection, RDiscard, 0).

xSynchronize(Connection, On) :-
	pxcBoolean(ROn, On), pxSynchronize(Connection, ROn, 0).

xGetMotionEvents(Connection, Window, Start, Stop, MotionEventList) :-
	pxcTime(RStart, Start), pxUnsignedPut1(RStart),
	pxcTime(RStop, Stop), pxUnsignedPut2(RStop),
	pxGetMotionEvents(Connection, Window, N, 0),
	pxMotionEventList(N, MotionEventList).

% ICCCM support

xWMGeometry(Screen, Specification, PX, PY, PW, PH, PXN, PYN, BW, 
  SizeHints, X, Y, W, H, xWindowGravity(Gravity), 
  [XHint, YHint, WHint, HHint | SizeHints]) :- 
	xParseGeometry(Specification, UX, UY, UW, UH, UXN, UYN),
	pxComputeWH(SizeHints, UW, PW, xWidthIncrement, xBaseWidth,
	            xMinWidth, xMaxWidth, W, WUF),
	pxComputeWH(SizeHints, UH, PH, xHeightIncrement, xBaseHeight,
		    xMinHeight, xMaxHeight, H, HUF), 
	pxSizeHints(WUF, HUF, W, H, WHint, HHint),
	pxComputeX(UX, UXN, PX, PXN, Screen, W, BW, X, XUF, XN),
	pxComputeY(UY, UYN, PY, PYN, Screen, H, BW, Y, YUF, YN),
	pxPositionHints(XUF, YUF, X, Y, XHint, YHint),
	pxComputeGravity(XN, YN, Gravity).

xIconifyWindow(Connection, Window) :-
	xQueryWindow(Connection, Window, [xRootWindow(Root)]),
	xAtom(Connection, 'WM_CHANGE_STATE', WM_CHANGE_STATE),
	pxcInitialState(Magic, xIconic),
	xSendEvent(Connection, Root, xFalse,
		   [xSubstructureRedirect, xSubstructureNotify],
		   xEvent(xClientMessage, 0, xTrue, Connection,
		          Window, WM_CHANGE_STATE, 32,
			  Magic, 0, 0, 0, 0)).

xWithdrawWindow(Connection, Window) :-
	xQueryWindow(Connection, Window, [xRootWindow(Root)]),
	xUnmapWindow(Connection, Window),
	xSendEvent(Connection, Root, xFalse,
		   [xSubstructureRedirect, xSubstructureNotify],
		   xEvent(xUnmapNotify, 0, xTrue, Connection,
		          0, Window, xFalse)).


xSetWMWindow(Connection, Window, AttributeList) :-
	xQueryWindow(Connection, Window, [xScreen(Screen)]),
	pxAttributeList(AttributeList, pxcSetWindowAttribute),
	pxSetWMWindow(Connection, Screen, Window, 0).

xSetWMProperties(Connection, Window, Name, IconName, Command, Machine,
  NormalHints, WMHints, Class) :-
	(Name == xNone -> true;
	    xSetWMName(Connection, Window, Name)),
	(IconName == xNone -> true;
	    xSetWMIconName(Connection, Window, IconName)),
	(Command == xNone -> true;
	    xSetWMCommand(Connection, Window, Command)),
	(Machine == xNone -> true;
	    xSetWMClientMachine(Connection, Window, Machine)),
	(NormalHints == xNone -> true;
	    xSetWMNormalHints(Connection, Window, NormalHints)),
	(WMHints == xNone -> true;
	    xSetWMHints(Connection, Window, WMHints)),
	(Class == xNone -> true;
	    xSetWMClass(Connection, Window, Class)).

xSetWMClass(Connection, Window, Class) :-
	pxTextToProperty(Class, xTrue, Property),
	xAtom(Connection, 'WM_CLASS', WM_CLASS),
	xAtom(Connection, 'STRING', STRING),
	xSetProperty(Connection, Window, WM_CLASS, STRING, xReplace, Property).

xSetWMClientMachine(Connection, Window, Machine) :- 
	pxTextToProperty(Machine, xFalse, Property),
	xAtom(Connection, 'WM_CLIENT_MACHINE', WM_CLIENT_MACHINE),
	xAtom(Connection, 'STRING', STRING),
	xSetProperty(Connection, Window, WM_CLIENT_MACHINE, STRING, xReplace, 
	  Property). 

xSetWMColormapWindows(Connection, Window, WindowList) :-
	xAtom(Connection, 'WM_COLORMAP_WINDOWS', WM_COLORMAP_WINDOWS),
	xAtom(Connection, 'WINDOW', WINDOW),
	xSetProperty(Connection, Window, WM_COLORMAP_WINDOWS, WINDOW, xReplace,
		xProperty(32, WindowList)).

xSetWMCommand(Connection, Window, Command) :-
	pxTextToProperty(Command, xTrue, Property),
	xAtom(Connection, 'WM_COMMAND', WM_COMMAND),
	xAtom(Connection, 'STRING', STRING),
	xSetProperty(Connection, Window, WM_COMMAND, STRING, xReplace, 
	  Property).  

xSetWMHints(Connection, Window, HintList) :-
	pxPropertySet(pxp_WM_HINTS, 9, HintList, Property),
	xAtom(Connection, 'WM_HINTS', WM_HINTS),
	xSetProperty(Connection, Window, WM_HINTS, WM_HINTS, xReplace, 
	  Property).

xSetWMIconName(Connection, Window, Name) :-
	pxTextToProperty(Name, xFalse, Property),
	xAtom(Connection, 'WM_ICON_NAME', WM_ICON_NAME),
	xAtom(Connection, 'STRING', STRING),
	xSetProperty(Connection, Window, WM_ICON_NAME, STRING, xReplace, 
	  Property).

xGetWMIconSizes(Screen, Sizes) :-
	xQueryScreen(Screen, [xConnection(Connection), xRootWindow(Root)]),
	xAtom(Connection, 'WM_ICON_SIZE', WM_ICON_SIZE),
	xWindowProperties(Connection, Root, Props),
	(pxMember(Props, WM_ICON_SIZE) 
         -> xGetProperty(Connection, Root, WM_ICON_SIZE, 0, 6, xFalse,
	 		 WM_ICON_SIZE, xFalse, WM_ICON_SIZE, Remain, 
			 xProperty(32, First)),
	    (Remain = 0 
	     -> pxPropertyToSize(First, Sizes, [])
             ;  pxPropertyToSize(First, Sizes, Tail),
                xGetProperty(Connection, Root, WM_ICON_SIZE, 6,
			     Remain, xFalse, WM_ICON_SIZE, xFalse,
			     WM_ICON_SIZE, 0, xProperty(32, Rest)),
	        pxPropertyToSize(Rest, Tail, []))
	 ;  Sizes = []).

xSetWMName(Connection, Window, Name) :-
	pxTextToProperty(Name, xFalse, Property),
	xAtom(Connection, 'WM_NAME', WM_NAME),
	xAtom(Connection, 'STRING', STRING),
	xSetProperty(Connection, Window, WM_NAME, STRING, xReplace, Property).

xSetWMNormalHints(Connection, Window, HintList) :-
	pxPropertySet(pxp_WM_NORMAL_HINTS, 18, HintList, Property),
	xAtom(Connection, 'WM_NORMAL_HINTS', WM_NORMAL_HINTS),
	xAtom(Connection, 'WM_SIZE_HINTS', WM_SIZE_HINTS),
	xSetProperty(Connection, Window, WM_NORMAL_HINTS, WM_SIZE_HINTS,
	  xReplace, Property).

xSetWMProtocols(Connection, Window, Protocols) :-
	pxProtocol(Protocols, Connection, AtomList),
	xAtom(Connection, 'WM_PROTOCOLS', WM_PROTOCOLS),
	xAtom(Connection, 'ATOM', ATOM),
	xSetProperty(Connection, Window, WM_PROTOCOLS, ATOM, xReplace, 
	  xProperty(32, AtomList)).

xSetWMTransient(Connection, Window, Owner) :-
	xAtom(Connection, 'WM_TRANSIENT_FOR', WM_TRANSIENT_FOR),
	xAtom(Connection, 'WINDOW', WINDOW),
	xSetProperty(Connection, Window, WM_TRANSIENT_FOR, WINDOW, xReplace,  
	  xProperty(32, [Owner])).

% miscellaneous
xListExtensions(Connection, ExtensionList) :-
	pxListExtensions(Connection, N, 0), pxStringGet(N, ExtensionList).

xQueryExtension(Connection, Name, Opcode, Event, Error) :-
	pxQueryExtension(Connection, Name, Opcode, Event, Error, 0).

xGetDefault(Connection, Program, Option, Default) :-
	pxGetDefault(Connection, Program, Option, Default, 0).

xParseGeometry(Specification, X, Y, Width, Height, XNegative, YNegative) :-
	pxParseGeometry(Specification, RX, RY, RW, RH, F),
	FX is F /\ 1, (FX > 0 -> X = RX; X = xNone),
	FY is F /\ 2, (FY > 0 -> Y = RY; Y = xNone),
	FW is F /\ 4, (FW > 0 -> Width = RW; Width = xNone),
	FH is F /\ 8, (FH > 0 -> Height = RH; Height = xNone),
	FXN is F /\ 16, (FXN > 0 -> XNegative = xTrue; XNegative = xFalse),
	FYN is F /\ 32, (FYN > 0 -> YNegative = xTrue; YNegative = xFalse).

xGetHostName(Name) :- pxGetHostName(Name, 0).

xNoOp(Connection) :- pxNoOp(Connection, 0).

xOk :- pxOk(0).

xPrecision(Bits) :- var(Bits), pxPrecisionGet(Bits).
xPrecision(Bits) :- nonvar(Bits), pxPrecisionSet(Bits, 0).

% obsolete, compatability functions

xGeometry(Screen, Specification, PX, PY, PW, PH, X, Y, W, H, NormalHints) :-
	xWMGeometry(Screen, Specification, PX, PY, PW, PH, xFalse,
	  xFalse, 0, [], X, Y, W, H, _, NormalHints).

xSetNormalHints(Connection, Window, HintList) :-
	xAtom(Connection, 'WM_NORMAL_HINTS', WM_NORMAL_HINTS),
	xAtom(Connection, 'WM_SIZE_HINTS', WM_SIZE_HINTS),
	pxPropertySet(pxp_WM_NORMAL_HINTS, 15, HintList, P),
	xSetProperty(Connection, Window, WM_NORMAL_HINTS, WM_SIZE_HINTS,
	  xReplace, P).

xSetStandardProperties(Connection, Window, Program, IconName, WMHints, Command,
  NormalHints) :-
	xAtom(Connection, 'STRING', STRING),
	xAtom(Connection, 'WM_NAME', WM_NAME),
	xAtom(Connection, 'WM_ICON_NAME', WM_ICON_NAME),
	xAtom(Connection, 'WM_COMMAND', WM_COMMAND),
	(Program == xNone -> true;
	    xSetProperty(Connection, Window, WM_NAME, STRING, xReplace, 
	      Program)), 
	(IconName == xNone -> true;
	    xSetProperty(Connection, Window, WM_ICON_NAME, STRING, xReplace,
	      IconName)),
	(WMHints == xNone -> true; 	      
	    xSetWMHints(Connection, Window, WMHints)),
	(Command == xNone -> true;
	    pxTermToString(Command, S),
	    xSetProperty(Connection, Window, WM_COMMAND, STRING, xReplace,
	      xProperty(8, S))), 
	(NormalHints == xNone -> true;	
	    xSetNormalHints(Connection, Window, NormalHints)).

%
% eof
%
