/*
 * Copyright (C) 1989 by The Regents of the University of California.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * event.c
 *
 * PXVectorToEvent
 * PXEventToVector
 * PXCheckEvent
 *
 * pxEventsQueued
 * pxNextEvent
 * pxGetEvent
 * pxSelectEvent
 *
 * pxPutBackEvent
 * pxSendEvent
 *
 * pxFlush
 * pxSync
 * pxSynchronize
 *
 * pxNormalMotion
 * pxSplitMotion
 * pxGetMotionEvents
 */

#include <stdio.h>
#include <sys/time.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"
#include "attrib.h"

#define XEV (pxAttribBuf.ev.xev)
#define PEV (pxAttribBuf.ev.pev)

/*
 * PXVectorToEvent()
 *
 * converts from Vector32 (pxMallocBlock) to Event (pxAttribuf.ev.xev)
 * somewhat wasteful considering Xlib copies the stuff again in _XEventToWire
 */
PXVectorToEvent()
{
  long *vector = (long *)pxMallocBlock;
  int i, j;
  PxConnection *pcp;

  if (pxElements < 6)
    PX_Error(bad_event_format);
  if (PXLookupConnection(vector[3], &pcp))
    return FF_Fail;
  
  XEV.xany.type = (int)vector[0];
  XEV.xany.serial = (unsigned long)vector[1];
  XEV.xany.send_event = (Bool)vector[2];
  XEV.xany.display = pcp->xDisplay; 
  XEV.xany.window = (Window)vector[4];

  switch (XEV.type) {
  case KeyPress:
  case KeyRelease:
  case ButtonPress:		/* button event is really same struct */
  case ButtonRelease:
    if (pxElements != 15)
      PX_Error(bad_event_format);
    for (i = 5; i < 15; i++)
      XEV.pad[i] = vector[i];
    break;
  case MotionNotify:
    if (pxElements != 15)
      PX_Error(bad_event_format);
    for (i = 5; i < 13; i++)
      XEV.pad[i] = vector[i];
    XEV.xmotion.is_hint = (char)(0xFF & vector[13]);
    XEV.xmotion.same_screen = (Bool)vector[14];
    break;
  case EnterNotify:
  case LeaveNotify:
    if (pxElements != 17)
      PX_Error(bad_event_format);
    for (i = 5; i < 17; i++)
      XEV.pad[i] = vector[i];
    break;
  case FocusIn:
  case FocusOut:
  case NoExpose:		
  case UnmapNotify:
  case MapNotify:
  case ResizeRequest:		
  case CirculateNotify:
  case CirculateRequest:
  case SelectionClear:
    if (pxElements != 7)
      PX_Error(bad_event_format);
    XEV.pad[5] = vector[5];
    XEV.pad[6] = vector[6];
    break;
  case KeymapNotify:
    if (pxElements != 37)
      PX_Error(bad_event_format);
    for (i = 0; i < 32; i++)
      XEV.xkeymap.key_vector[i] = (char)(0xFF & vector[5 + i]);
    break;
  case Expose:
  case ReparentNotify:
  case SelectionRequest:
    if (pxElements != 10)
      PX_Error(bad_event_format);
    for (i = 5; i < 10; i++) 
      XEV.pad[i] = vector[i];
    break;
  case GraphicsExpose:
  case CreateNotify:
    if (pxElements != 12)
      PX_Error(bad_event_format);
    for (i = 5; i < 12; i++) 
      XEV.pad[i] = vector[i];
    break;
  case VisibilityNotify:
  case DestroyNotify:
  case MapRequest:
    if (pxElements != 6)
      PX_Error(bad_event_format);
    XEV.pad[5] = vector[5];
    break;
  case ConfigureNotify:
    if (pxElements != 13)
      PX_Error(bad_event_format);
    for (i = 5; i < 13; i++) 
      XEV.pad[i] = vector[i];
    break;
  case ConfigureRequest:
    if (pxElements != 14)
      PX_Error(bad_event_format);
    for (i = 5; i < 14; i++) 
      XEV.pad[i] = vector[i];
    break;
  case GravityNotify:
  case ColormapNotify:
  case MappingNotify:
  case PropertyNotify:
    if (pxElements != 8)
      PX_Error(bad_event_format);
    XEV.pad[5] = vector[5];
    XEV.pad[6] = vector[6];
    XEV.pad[7] = vector[7];
    break;
  case SelectionNotify:
    if (pxElements != 9)
      PX_Error(bad_event_format);
    for (i = 5; i < 9; i++) 
      XEV.pad[i] = vector[i];
    break;
  case ClientMessage:
    if (pxElements < 7)
      PX_Error(bad_event_format);
    j = pxElements - 7;
    XEV.xclient.message_type = (Atom)vector[5];
    switch (XEV.xclient.format = (int)vector[6]) {
    case 8:
      if (j > 20)
	PX_Error(bad_event_format);
      for (i = 0; i < j; i++)
	XEV.xclient.data.b[i] = (char)(0xFF & vector[7 + i]);
      break;
    case 16:
      if (j > 10)
	PX_Error(bad_event_format);
      for (i = 0; i < j; i++)
	XEV.xclient.data.s[i] = (short)(0xFFFF & vector[7 + i]);
      break;
    case 32:
      if (j > 5)
	PX_Error(bad_event_format);
      for (i = 0; i < j; i++)
	XEV.xclient.data.l[i] = (int)vector[7 + i];
      break;
    default:
      PX_Error(bad_event_format);
    }
    break;
  default:
    PX_Error(unknown_event_type);
  }
  return FF_Ok;
}

/*
 * PXEventToVector()
 *
 * converts from event (pxAttribBuf.ev.xev) to Vector32 (pxAttribBuf.ev.pev)
 */
PXEventToVector()
{
  DLHeader *dlhp;
  int type = XEV.xany.type;
  int i;
				/* common to all events */
  PEV[0] = type;
  PEV[1] = (long)XEV.xany.serial;
  PEV[2] = XEV.xany.send_event;
  if (DLFind(pxConnectionList, (char *)(XEV.xany.display), &dlhp))
    return FF_Fail;
  PEV[3] = dlhp->des;
  PEV[4] = (long)XEV.xany.window;
				/* type specific fields */

  switch (type) {
  case KeymapNotify:
    for (i = 0; i < 32; i++)
      PEV[5 + i] = (long)(XEV.xkeymap.key_vector[i]) & 0xFF;
    pxElements = 37;
    return FF_Ok;
  }

  PEV[5] = XEV.pad[5];
  switch (type) {
  case VisibilityNotify:
  case DestroyNotify:
  case MapRequest:
    pxElements = 6;
    return FF_Ok;
  }

  PEV[6] = XEV.pad[6];
  switch (type) {
  case FocusIn:
  case FocusOut:
  case NoExpose:		
  case UnmapNotify:
  case MapNotify:
  case ResizeRequest:		
  case CirculateNotify:
  case CirculateRequest:
  case SelectionClear:
    pxElements = 7;
    return FF_Ok;
  case ClientMessage:
    switch (XEV.xclient.format) {
    case 8:
      for (i = 0; i < 20; i++)
	PEV[7 + i] = (long)(XEV.xclient.data.b[i]) & 0xFF;
      pxElements = 27;
      return FF_Ok;
    case 16:
      for (i = 0; i < 10; i++)
	PEV[7 + i] = (long)(XEV.xclient.data.s[i]) & 0xFFFF;
      pxElements = 17;
      return FF_Ok;
    case 32:
      for (i = 0; i < 5; i++)
	PEV[7 + i] =  XEV.xclient.data.l[i];
      pxElements = 12;
      return FF_Ok;
    default:
      PX_Error(bad_event_format);
    }
  }

  PEV[7] = XEV.pad[7];
  switch (type) {
  case GravityNotify:
  case PropertyNotify:
  case ColormapNotify:
  case MappingNotify:
    pxElements = 8;
    return FF_Ok;
  }

  PEV[8] = XEV.pad[8];
  switch (type) {
  case SelectionNotify:    
    pxElements = 9;
    return FF_Ok;
  }

  PEV[9] = XEV.pad[9];
  switch (type) {
  case Expose:
  case ReparentNotify:
  case SelectionRequest:
    pxElements = 10;
    return FF_Ok;
  }

  PEV[10] = XEV.pad[10];
  PEV[11] = XEV.pad[11];
  switch (type) {
  case GraphicsExpose:
  case CreateNotify:
    pxElements = 12;
    return FF_Ok;
  }

  PEV[12] = XEV.pad[12];
  switch (type) {
  case MotionNotify:
    PEV[13] = (long)(XEV.xmotion.is_hint) & 0xFF;
    PEV[14] = XEV.xmotion.same_screen;
    pxElements = 15;
    return FF_Ok;
   case ConfigureNotify:   
    pxElements = 13;
    return FF_Ok;
  }

  PEV[13] = XEV.pad[13];
  switch (type) {
  case ConfigureRequest:
    pxElements = 14;
    return FF_Ok;
  }

  PEV[14] = XEV.pad[14];
  switch (type) {
  case KeyPress:
  case KeyRelease:		
  case ButtonPress:		
  case ButtonRelease:
    pxElements = 15;
    return FF_Ok;
  }

  PEV[15] = XEV.pad[15];
  PEV[16] = XEV.pad[16];
  switch (type) {
  case EnterNotify:
  case LeaveNotify:
    pxElements = 17;
    return FF_Ok;
  default:
    PX_Error(unknown_event_type);
  }
}

/*
 * Copyright (C) 1989 by The Regents of the University of California.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/* Bool
 * PXCheckEvent(Display, Event, Args)
 *
 * predicate for pxGetEvent
 */

#define AllPointers (PointerMotionMask|PointerMotionHintMask|ButtonMotionMask)
#define AllButtons (Button1MotionMask|Button2MotionMask|Button3MotionMask|\
		    Button4MotionMask|Button5MotionMask)

extern long _event_to_mask [];

Window pxCheckWindow = 0;
long pxCheckMask = 0;
pxCheckType = 0;

/*ARGSUSED*/
Bool
PXCheckEvent(display, event, args)
     Display *display;
     XEvent *event;
     char *args;
{
  if (pxCheckWindow && (event->xany.window != pxCheckWindow))
    return False;
  if (pxCheckMask && !((_event_to_mask[event->type] & pxCheckMask) &&
		   ((event->type != MotionNotify) ||
		    (pxCheckMask & AllPointers) ||
		    (pxCheckMask & AllButtons & (long)(event->xmotion.state)))))
    return False;
  if (pxCheckType && (event->type != pxCheckType))
    return False;
  return True;
}

/*
 * pxEventsQueued(+Connection, +Mode, -Count, 0)
 *	integer: Connection, Mode, Count
 */
FFInteger
pxEventsQueued(pc, pm, pcnt)
	FFInteger pc, pm, *pcnt;
{
  PxConnection *pcp;

  PX_ErrorInit("xEventsQueued/3");
  *pcnt = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  *pcnt = XEventsQueued(pcp->xDisplay, pm);
  return FF_Ok;
}

/*
 * pxNextEvent(+Connection, +Remove, -Length, 0)
 *	integer: Connection, Remove, Length
 */
FFInteger
pxNextEvent(pc, pr, pl)
     FFInteger pc, pr, *pl;
{
  PxConnection *pcp;

  PX_ErrorInit("xNextEvent/2");
  *pl = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if (pr)
    XNextEvent(pcp->xDisplay, &XEV);
  else
    XPeekEvent(pcp->xDisplay, &XEV);
  if (PXEventToVector())
    return FF_Fail;

  *pl = pxElements;
  pxMallocBlock = NULL;
  pxVector32 = PEV;
  pxType = PX_Vector32;
  return FF_Ok;
}

/*
 * pxGetEvent(+Connection, +Window, +Mask, +Type, +Remove, +Block, -Length, 0)
 *	integer: Connection, Window, Mask, Type, Remove, Block, Length
 */
FFInteger
pxGetEvent(pc, pw, pm, pt, pr, pb, pl)
     FFInteger pc, pw, pm, pt, pr, pb, *pl;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetEvent/6");
  *pl = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  pxCheckWindow = (Window)pw;
  pxCheckMask = pm;
  pxCheckType = (int)pt;

  if (pr) {
    if (pb) 
      XIfEvent(pcp->xDisplay, &XEV, PXCheckEvent, NULL);
    else 
      if (XCheckIfEvent(pcp->xDisplay, &XEV, PXCheckEvent, NULL) == False)
	return FF_Fail;
  } else 
    if (pb) 
      XPeekIfEvent(pcp->xDisplay, &XEV, PXCheckEvent, NULL);
    else
      PX_Error("request must block or remove");
  if (PXEventToVector())
    return FF_Fail;
  *pl = pxElements;
  pxMallocBlock = NULL;
  pxVector32 = PEV;
  pxType = PX_Vector32;
  return FF_Ok;
}

/*
 * pxSelectEvent(+NConnections, +NReadFDs, +NWriteFDs, +NExceptFDs,
 *	-SConnections, -SReadFDs, -SWriteFDs, -SExceptFDs, -TimeOutFlag, 0)
 *
 *	integer: NConnections, NReadFDs, NWriteFDs, NExceptFDs, SConnections,
 *		SReadFDs, SWriteFDs, SExceptFDs, TimeOutFlag
 */
char *file_descriptor_out_of_range = "file descriptor out of range";

FFInteger
pxSelectEvent(pnc, pnr, pnw, pne, psc, psr, psw, pse, ptof)
     FFInteger pnc, pnr, pnw, pne, *psc, *psr, *psw, *pse, *ptof;
{
  int i, fd, r, cnt,
      nfds = getdtablesize(),
      rmask = 0, wmask = 0, emask = 0,
      ctable[32], rtable[32], wtable[32], etable[32];
  PxConnection *pcp;
  PxVector32 *ptr;
  struct timeval to;

  for (i = 0; i < 32; i++)
    ctable[i] = rtable[i] = wtable[i] = etable[i] = -1;

  if (nfds > 32)
    nfds = 32;

  PX_ErrorInit("xSelectEvent/10");
  *psc = *psr = *psw = *pse = *ptof = PX_Invalid;

  for (i = 0, ptr = (PxVector32 *)pxMallocBlock; i < pnc; i++, ptr++) {
    if (PXLookupConnection(*ptr, &pcp))
      return PXVectorFree();
    XFlush(pcp->xDisplay);
    fd = pcp->xDisplay->fd;
    if (fd >= nfds)
      PX_Error(file_descriptor_out_of_range);
    ctable[fd] = *ptr;		/* fd to connection table */
    rmask |= 1 << fd;		/* put into rmask */
  }

  for (i = 0; i < pnr; i++, ptr++) {
    fd = *ptr;
    if (fd < 0 || fd >= nfds)
      PX_Error(file_descriptor_out_of_range);
    rtable[fd] = 1;
    rmask |= 1 << fd;
  }
  
  for (i = 0; i < pnw; i++, ptr++) {
    fd = *ptr;
    if (fd < 0 || fd >= nfds)
      PX_Error(file_descriptor_out_of_range);
    wtable[fd] = 1;
    wmask |= 1 << fd;
  }

  
  for (i = 0; i < pne; i++, ptr++) {
    fd = *ptr;
    if (fd < 0 || fd >= nfds)
      PX_Error(file_descriptor_out_of_range);
    etable[fd] = 1;
    emask |= 1 << fd;
  }

  if (!pxUnsignedArg1) {
    if ((r = select(nfds, &rmask, &wmask, &emask, NULL)) < 0)
      return PXVectorFree();
  } 
  else {
    to.tv_sec = pxUnsignedArg1 / 1000;
    to.tv_usec = (pxUnsignedArg1 * 1000) % 1000000;
    
    if ((r = select(nfds, &rmask, &wmask, &emask, &to)) < 0)
      return PXVectorFree();
  }
  (void)PXVectorFree();
  
  if (!r) {			/* timeout */
    *psc = *psr = *psw = *pse = 0;
    *ptof = 1;
    return FF_Ok;
  }
  *ptof = 0;

  if (pxVectorAlloc(32, r, 0) == FF_Fail)
    PX_Error(no_memory);
  ptr = (PxVector32 *)pxMallocBlock;
  
  for (i = 0, cnt = 0; i < nfds && r > 0; i++) {
    if (ctable[i] == -1)
      continue;
    if (rmask & (1 << i)) {
      *ptr++ = ctable[i];
      cnt++; 
      r--;
    }
  }
  *psc = cnt;

  for (i = 0, cnt = 0; i < nfds && r > 0; i++) {
    if (rtable[i] == -1)
      continue;
    if (rmask & (1 << i)) {
      *ptr++ = i;
      cnt++; 
      r--;
    }
  }
  *psr = cnt;

  for (i = 0, cnt = 0; i < nfds && r > 0; i++) {
    if (wtable[i] == -1)
      continue;
    if (wmask & (1 << i)) {
      *ptr++ = i;
      cnt++; 
      r--;
    }
  }
  *psw = cnt;

  for (i = 0, cnt = 0; i < nfds && r > 0; i++) {
    if (etable[i] == -1)
      continue;
    if (emask & (1 << i)) {
      *ptr++ = i;
      cnt++; 
      r--;
    }
  }
  *pse = cnt;
    
  return FF_Ok;
}

/*
 * pxPutBackEvent(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxPutBackEvent(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xPutBackEvent/2");

  if (PXLookupConnection(pc, &pcp) || PXVectorToEvent())
    return PXVectorFree();

  XPutBackEvent(pcp->xDisplay, &XEV);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxSendEvent(+Connection, +Window, +Propagate, +EventMask, 0)
 *	integer: Connection, Window, Propagate, EventMask
 */
FFInteger
pxSendEvent(pc, pw, pp, pm)
     FFInteger pc, pw, pp, pm;
{
  PxConnection *pcp;
  
  PX_ErrorInit("xSendEvent/5");

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();
  if (PXVectorToEvent())
    return PXVectorFree();

  if (!XSendEvent(pcp->xDisplay, pw, pp, pm, &XEV))
    return PXVectorFree();
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxFlush(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxFlush(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xFlush/1");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  
  XFlush(pcp->xDisplay);
  return FF_Ok;
}

/*
 * pxSync(+Connection, +Discard, 0)
 *	integer: Connection, Discard
 */
FFInteger
pxSync(pc, pd)
     FFInteger pc, pd;
{
  PxConnection *pcp;

  PX_ErrorInit("xSync/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XSync(pcp->xDisplay, pd);
  return FF_Ok;
}

/*
 * pxSynchronize(+Connection, +On, 0)
 *	integer: Connection, On
 */
FFInteger
pxSynchronize(pc, po)
     FFInteger pc, po;
{
  PxConnection *pcp;

  PX_ErrorInit("xSynchronize/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  (void)XSynchronize(pcp->xDisplay, po);
  return FF_Ok;
}

/*
 * pxNormalMotion(-Time, -X, -Y, -Next, 0)
 *	integer: Time, X, Y, Next
 */

XTimeCoord *pxTimeCoord = NULL;

FFInteger
pxNormalMotion(pt, px, py, pn)
     FFInteger *pt, *px, *py, *pn;
{
  *pt = *px = *py = *pn = PX_Invalid;

  if (!pxElements)
    return FF_Fail;

  *pt = pxTimeCoord->time;
  *px = pxTimeCoord->x;
  *py = pxTimeCoord->y;
  pxTimeCoord++;

  if (--pxElements == 0) {
    pxVectorXFree();
    *pn = PX_End;
  } else
    *pn = ((pxTimeCoord->time) & pxPrecisionMask) ? PX_SplitVector : PX_Cont;
  return FF_Ok;
}

/*
 * pxSplitMotion(-Most, -Least, -X, -Y, -Next, 0)
 *	integer: DMost, Least, X, Y, Next
 */
FFInteger
pxSplitMotion(pm, pl, px, py, pn)
     FFInteger *pm, *pl, *px, *py, *pn;
{
  *pm = *pl = *px = *py = *pn = PX_Invalid;

  if (!pxElements)
    return FF_Fail;

  *pm = PX_MostSplit(pxTimeCoord->time);
  *pl = PX_LeastSplit(pxTimeCoord->time);
  *px = pxTimeCoord->x;
  *py = pxTimeCoord->y;
  pxTimeCoord++;

  if (--pxElements == 0) {
    pxVectorXFree();
    *pn = PX_End;
  } else
    *pn = ((pxTimeCoord->time) & pxPrecisionMask) ? PX_SplitVector : PX_Cont;
  return FF_Ok;
}

/*
 * pxGetMotionEvents(+Connection, +Window, -Next, 0)
 *	integer: Connection, Window, Next
 */
FFInteger
pxGetMotionEvents(pc, pw, pn)
     FFInteger pc, pw, *pn;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetMotionEvents/5");
  *pn = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if ((pxTimeCoord = XGetMotionEvents(pcp->xDisplay, pw, pxUnsignedArg1,
				      pxUnsignedArg2, &pxElements)) == NULL) 
    return FF_Fail;

  if (pxElements) {
    pxMallocBlock = (char *)pxTimeCoord;
    pxType = PX_TimeCoord;
    *pn = (pxTimeCoord->time & pxPrecisionMask) ? PX_SplitVector : PX_Cont;
  } else 
    *pn = PX_End;
  return FF_Ok;
}

/*
 * eof
 */
