;;  $Id: pce-initialisation.lisp,v 1.2 1993/01/07 10:38:20 anjo Exp $
;;
;;  File	pce-to-lisp.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.uva.psy.nl
;;  Purpose	Conversion from PCE to Lisp and back
;;  Works with	PCE 4.5,  Lisp
;;  Notice	Copyright (c) 1992, 1993  University of Amsterdam
;;
;;  History	08/02/91  (Created)
;;		29/08/91  (Adapted for PCE 4.0)
;;		29/08/91  (Define PCE/Lisp interface classes)
;;  		13/07/92  (Adapted for PCE 4.4)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")

(export '(lisp-to-pce			; LispExpression -> Object
	  pce-to-lisp			; Object -> LispExpression
	  pce-storage-class
	  ))


;;  ------------------------------------------------------
;;  Initialisation
;;  ------------------------------------------------------

(defun initialise-pce ()
  (setf *pce-object-refs* (make-hash-table :test #'eql))
  (setf *pce-object-assocs* (make-hash-table))
  (pcelisp-c-initialise)

  (setf *pce-arg1* (pce-make-special 'arg1))
  (setf *pce-arg2* (pce-make-special 'arg2))
  (setf *pce-arg3* (pce-make-special 'arg3))
  (setf *pce-arg4* (pce-make-special 'arg4))
  (setf *pce-arg5* (pce-make-special 'arg5))
  (setf *pce-arg6* (pce-make-special 'arg6))
  (setf *pce-arg7* (pce-make-special 'arg7))
  (setf *pce-arg8* (pce-make-special 'arg8))
  (setf *pce-arg9* (pce-make-special 'arg9))
  (setf *pce-arg10* (pce-make-special 'arg10))
  (setf *pce-block* (pce-make-special 'block))
  (setf *pce-default* (pce-make-special 'default))
  (setf *pce-event* (pce-make-special 'event))
  (setf *pce-nil* (pce-make-special 'nil))
  (setf *pce-on* (pce-make-special 'on))
  (setf *pce-off* (pce-make-special 'off))
  (setf *pce-receiver* (pce-make-special 'receiver))
  (setf *pce-pce* @pce)

  (pce-send @pce :rename-reference :host :lisp)
  (pce-send @pce :print-c-stack @on)
  (pce-define-class-functions)
  (special-lisp-classes)
  (pce-to-lisp-initialise)

  (pce-send @pce :home *pce-directory*)
					; Make sure interface knows
					; these names.
  (pce-send (pce-chain :abort :backtrace :break :fatal :halt) :free)
  (pce-print-banner)
  t)


;;  ------------------------------------------------------
;;  PCE banner
;;  ------------------------------------------------------

(defun pce-print-banner ()
  (format t "Welcome to PCE/Lisp~%")
  (format t "Based on PCE ~A and ~A ~A~%"
	  (pce-get @pce :version)
	  (lisp-implementation-type)
	  (lisp-implementation-version))
  (format t "Copyright (c) 1992  University of Amsterdam~%~%")
  (format t "Mail PCE/Lisp bug reports (comments ...) to pce-bugs@swi.psy.uva.nl~%"))
  

;;  ------------------------------------------------------
;;  Lisp related PCE classes
;;  ------------------------------------------------------

(defun pce-to-lisp-initialise ()
  (unless (pce-exists-p @lisp-cons-class)
	  (pce-storage-class :lisp-cons :slots
			     `((:first :any :get)
			       (:rest :any :get)))))


;;  ------------------------------------------------------
;;  Storage classes
;;  ------------------------------------------------------

;;! pce-storage-class name [:super] [:slots]
;;
;;  Defines a new PCE class which will be exclusively used for storage
;;  (i.e. no behaviour defined).  All arguments need be prefixed by
;;  colons to force lower case in order to achieve compatibility with
;;  existing PCE classes.
;;
;;  The ^slots^ is a list of three element lists.  The elements of the
;;  list are (all must be specified):
;;{
;;i slot-name
;;	Name of the slot.
;;i slot-type
;;	Type of the slot, the name of a PCE class or PCE special type.
;;	The root of the PCE class hierarchy is #:object# (i.e. anything),
;;	special types are #:name# (for atoms, symbols) and #:integer#.
;;	Default arguments are indicated by square brackets (e.g. #:[name]#).
;;i slot-access
;;	One of #:both#, #:get#, #:send# or #:none#.  Defines whether
;;	pce-send/# and pce-get/# operate on the slot.
;;}
;;
;;  Example:
;;<
;;	(pce-storage-class :name-address :slots
;;		;   slot     type  access
;;		`((:name    :name :get)
;;		  (:address :name :both)))
;;>

(defun pce-storage-class (name &key (super :object)
			       (slots nil))
  (let ((class (pce-class name super))
	(names (mapcar #'first slots))
	(args (list @arg1 @arg2 @arg3 @arg4 @arg5 @arg6 @arg7 @arg8 @arg9 @arg10))
	(types (mapcar #'second slots)))
    (pce-define-class-function name)
    (mapc #'(lambda(argument)
	      (pce-send class :instance-variable
			(pce-make :variable argument)))
	  slots)
    (pce-send class :send-method
	      (pce-send-method :initialise
			       (pce-make :vector types)
			       (pce-make :block
					 (mapcar
					  #'(lambda(name arg)
					      (pce-message @receiver :slot name arg))
					  names args))))
    class))
		       

;;  ------------------------------------------------------
;;  Lisp to PCE conversion (and vice versa)
;;  ------------------------------------------------------

;;! lisp-to-pce x ==> object
;;
;;  Converts ^x^ (a Lisp object) to a PCE object.  The conversion
;;  performed is such that converting the PCE object back to a Lisp
;;  object at least compares equal/# to the original input.  That is:
;;
;;=	(equal x (pce-to-lisp (lisp-to-pce x)))
;;
;;  always yields #T#.
;;
;;  lisp-to-pce/# handles symbols, integers, lists, keywords, strings
;;  and PCE objects.  Structures, hash-tables, vectors (etc.) are
;;  not handled properly.

(defun lisp-to-pce (x)
  (cond
   ((null x) @nil)
   ((keywordp x) x)
   ((symbolp x) x)
   ((integerp x) x)
   ((stringp x) x)
   ((floatp x) x)
   ((pce-object-p x) x)
   ((consp x) (pce-lisp-cons (lisp-to-pce (first x)) (lisp-to-pce (rest x))))
   (t (warn "LISP-TO-PCE cannot handle the lisp type ~S (~S). Returning NIL"
	    (type-of x) x)
      ;; It returned NIL before I put this warning in, so I suppose
      ;; I should make sure it still does.
      nil)
   ))


;;! pce-to-lisp object ==> x
;;
;;  Converts PCE object to a Lisp structure.  Only works if ^object^
;;  was created with lisp-to-pce/#.

(defun pce-to-lisp (x)
  (if (eq @nil x)
      nil
    (if (and (pce-object-p x)
	     (eq (pce-class-name x) :lisp-cons))
	(cons (pce-to-lisp (pce-get x :first))
	      (pce-to-lisp (pce-get x :rest)))
      x)))


(defun special-lisp-classes ()
  (let ((symbol-class (pce-class :lisp-symbol :name))
	(string-class (pce-class :lisp-string :string))
	(special-class (pce-class :lisp-special :object)))
    (pce-define-class-function :lisp-special)
    (pce-send special-class :instance-variable
	      (pce-variable :index :int :get))
    (pce-send special-class :send-method
	      (pce-send-method
		       :initialise
		       (pce-vector :int)
		       (pce-message @receiver :slot :index @arg1)))
    (pce-send symbol-class :instance-variable
	      (pce-variable :package :name :get))
    (pce-send symbol-class :send-method
	      (pce-send-method
		       :initialise
		       (pce-vector :name :name)
		       (pce-block
				(pce-message
					 @receiver :slot :package @arg2)
				(pce-message
					 @receiver
					 :send-super :initialise @arg1))))
    (pce-send symbol-class :send-method
	      (pce-send-method
		       :register
		       (pce-vector)
		       (pce-message @lisp
				:register-lisp-symbol @receiver)))
    (pce-send symbol-class :get-method
	      (pce-get-method
		       :lookup
		       :lisp-symbol
		       (pce-vector :name :name)
		       (pce-? @lisp
				:lookup-lisp-symbol @arg1 @arg2)))
    
    (pce-send string-class :send-method
	      (pce-send-method
		       :initialise
		       (pce-vector :char-array)
		       (pce-message
				@receiver
				:send-super :initialise :|%s| @arg1)))

    (pce-send string-class :get-method
	      (pce-get-method
		       :modify
		       :lisp-string
		       (pce-vector :char-array)
		       (pce-? string-class :instance @arg1)))

    (pce-send string-class :get-method
	      (pce-get-method
		       :convert
		       :lisp-string
		       (pce-vector :char-array)
		       (pce-? string-class :instance @arg1)))))


;;  ------------------------------------------------------
;;  Define class specific generator and testing functions
;;  ------------------------------------------------------

;;! pce-define-class-functions
;;
;;  Defines a Lisp function corresponding to pce-new/# for each PCE
;;  class.  The functions defined are called pce-name-of-the-class/#,
;;  for example pce-point/#.  Also defines predicates that test
;;  whether an object is an instance of a class or any of its
;;  sub-classes, for example pce-point-p/#.  All functions defined are
;;  exported.
;
;;  This function is called when PCE is initialised.

(defun pce-define-class-functions ()
  (let ((chain (pce-new :chain)))
    (pce-send @classes :for-all
	      (pce-new :message chain :append @arg1))
    (dolist (class-name (pce-chain-list chain))
	    (pce-define-class-function class-name))))


(defun pce-define-class-function (class-name)
  (let* ((class (pce-get @pce :convert class-name :class))
	 (creator (intern (string-append "PCE-" (string class-name)) "PCE"))
	 (isa (intern (string-append (string creator) "-P") "PCE")))
    (unless (fboundp creator)		; Only define when not already defined
	    (unless (eq class-name :object)
		    (compile creator
			     `(lambda (&rest args)
				(pce-make ,class-name args))))
	    (export creator "PCE")
	    (unless (eq class-name :object)
		    (compile isa
			     `(lambda (object)
				(when (pce-send object :instance-of ,class) t))))
	    (export isa "PCE"))))
