;;  $Id: pce-api.lisp,v 1.5 1993/01/25 14:07:13 anjo Exp $
;;  
;;  File	pce-api.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Primary interface functions
;;  Works with	PCE 4.5,  Lisp
;;  
;;  Notice	Copyright (c) 1992, 1993  University of Amsterdam
;;  
;;  History	02/07/92  (Created)
;;		26/10/92  (Receiver and selector are pushed as arguments)
;;  		22/01/93  (Last Modified)
;;


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")

(export '(pce-exists-p			; Object -> T | NIL
          pce-new			; [:name name] x class x ... -> Object
          pce-make			; class x arglist -> Object
          pce-get			; Object x selector x ... -> value
          pce-send			; Object x selector x ... -> Object

          pce-global			; [:name name] x class x ... -> Object
          pce-send-list			; Object x selector x arglist -> Object
	  pce-get-string		; Object x selector x ... -> string
	  pce-get-as			; type x Object x selector x ... -> type

	  pce-call-back			; ->
))


;;  ------------------------------------------------------
;;  PCE/Lisp interface error codes
;;  ------------------------------------------------------

;;  See the file "pce-lisp.c" for the C definitions of these constants.

(defconstant *pcelisp-no-error* 0)
(defconstant *pcelisp-unknown-assoc* 1)
(defconstant *pcelisp-unknown-name* 2)
(defconstant *pcelisp-unknown-reference* 3)
(defconstant *pcelisp-receiver-not-an-object* 4)
(defconstant *pcelisp-message-failed* 5)
(defconstant *pcelisp-unknown-symbol* 6) ; OBSOLETE
(defconstant *pcelisp-illegal-assoc* 7)
(defconstant *pcelisp-message-succeeded* 8)
(defconstant *pcelisp-special* 9)


;;  ------------------------------------------------------
;;  PCE/Lisp predefined interface types
;;  ------------------------------------------------------

(defconstant *pcelisp-integer* 1)	; Lisp integer 
(defconstant *pcelisp-name* 2)		; Lisp keyword
(defconstant *pcelisp-ref* 3)		; PCE numeric object reference
(defconstant *pcelisp-assoc* 4)		; PCE symbolic object reference
(defconstant *pcelisp-real* 5)		; Lisp float
(defconstant *pcelisp-native* 6)	; Known correspondence PCE / Lisp
(defconstant *pcelisp-string* 7)	; Lisp string
(defconstant *pcelisp-symbol* 8)	; Lisp symbol


;;  ------------------------------------------------------
;;  Send instruction: pce-send
;;  ------------------------------------------------------

(defun pce-send (receiver selector &rest args)
  (declare (dynamic-extent args))
  (let ((ps (alloc-pcelisp-status)))
    (set-pcelisp-status-arguments ps (+ 2 (length args)))
    (set-pcelisp-status-error ps 0)
    (pcelisp-c-send ps)
    (pcelisp-push-argument ps receiver)
    (pcelisp-push-argument ps selector)
    (dolist (i args)
	    (pcelisp-push-argument ps i))
    (let ((error (get-pcelisp-status-error ps)))
      (cond
       ((eql error *pcelisp-message-failed*)
	(free-pcelisp-status ps)
	nil)
       ((eql error *pcelisp-message-succeeded*)
	(free-pcelisp-status ps)
	receiver)
       (t
	(error "PCE-SEND (Internal)  Unknown error (~A)" error))))))


;;  ------------------------------------------------------
;;  Get instruction: pce-get
;;  ------------------------------------------------------

(defun pce-get (receiver selector &rest args)
  (declare (dynamic-extent args))
  (pcelisp-get-internal :unchecked receiver selector args))


(defun pce-get-string (receiver selector &rest args)
  (declare (dynamic-extent args))
  (pcelisp-get-internal :lisp-string receiver selector args))


(defun pce-get-as (return-as receiver selector &rest args)
  (declare (dynamic-extent args))
  (pcelisp-get-internal return-as receiver selector args))


(defun pcelisp-get-internal (return-as receiver selector args)
  (let ((ps (alloc-pcelisp-status)))
    (set-pcelisp-status-arguments ps (+ 3 (length args)))
    (set-pcelisp-status-error ps 0)
    (pcelisp-c-get ps)
    (pcelisp-push-argument ps return-as)
    (pcelisp-push-argument ps receiver)
    (pcelisp-push-argument ps selector)
    (dolist (i args)
	    (pcelisp-push-argument ps i))
    (let ((error (get-pcelisp-status-error ps)))
      (cond
       ((eql error *pcelisp-message-failed*)
	(free-pcelisp-status ps)
	nil)
       ((eql error *pcelisp-message-succeeded*)
	(prog1
	    (pcelisp-answer ps)
	(free-pcelisp-status ps)))
       (t
	(error "PCE-GET (Internal)  Unknown error (~A)" error))))))


;;  ------------------------------------------------------
;;  New instruction: pce-new
;;  ------------------------------------------------------

(defun pce-make (class args)
  (apply #'pce-new (cons class args)))


(defun pce-new (&rest args)
  (let ((ps (alloc-pcelisp-status))
	name
	class)
    (when (< (length args) 1)
	  (error "PCE-NEW  At least one argument required"))
    (when (and (eq (first args) :name)
	       (not (eql (length args) 2))) ; Class Name !!
	  (pop args)
	  (setf name (pop args))
	  (pcelisp-c-new-named (symbol-name name)))
    (setf class (pop args))
    (set-pcelisp-status-arguments ps (+ 1 (length args)))
    (set-pcelisp-status-error ps 0)
    (pcelisp-c-new ps)
    (set-pcelisp-status-name ps (if name 1 0))
    (pcelisp-push-argument ps class)
    (dolist (i args)
	    (pcelisp-push-argument ps i))
    (prog1
	(when (eql (get-pcelisp-status-error ps) *pcelisp-message-succeeded*)
	      (pcelisp-answer ps))
      (free-pcelisp-status ps))))


;;  ------------------------------------------------------
;;  Returning values
;;  ------------------------------------------------------

(defun pcelisp-answer (ps)
  (let ((type (get-pcelisp-status-return-type ps)))
    (cond
     ((eql type *pcelisp-ref*)		; @123456
      (pce-at (get-pcelisp-status-result ps)))
     ((eql type *pcelisp-special*)	; @123456 as special
      (pcelisp-special-value (pce-at (get-pcelisp-status-result ps))))
     ((eql type *pcelisp-assoc*)	; @HELLO (not known)
      (let ((name (pcelisp-c-fetch-assoc-name ps)))
	(pcelisp-new-assoc (pce-at name))))
     ((eql type *pcelisp-native*)	; @HELLO, :HELLO
      (pcelisp-c-fetch-native ps))
     ((eql type *pcelisp-integer*)	; 123456
      (get-pcelisp-status-result ps))
     ((eql type *pcelisp-name*)		; :HELLO (not known)
      (let ((name (pcelisp-c-fetch-assoc-name ps)))
	(pcelisp-new-name name)))
     ((eql type *pcelisp-real*)		; 123.456
      (pcelisp-c-fetch-real ps))
     ((eql type *pcelisp-string*)	; "Hello"
      (pcelisp-c-fetch-string ps))
#-STATIC-LISP-SYMBOLS
     ((eql type *pcelisp-symbol*)	; Hello
      (pcelisp-get-symbol ps))
     (t
      (error "PCE/Lisp (INTERNAL)  Unknown return type ~A" type)))))


#-STATIC-LISP-SYMBOLS
(defun pcelisp-get-symbol (ps)
  (let ((index (get-pcelisp-status-result ps)))
    (pcelisp-find-symbol index)))


;;  ------------------------------------------------------
;;  Argument pushing
;;  ------------------------------------------------------

(defun pcelisp-push-argument (ps arg)
  (cond ((pce-object-p arg)
	 (if (or (pce-assoc-p arg) (pce-special-p arg))
	     (pcelisp-push-assoc ps arg)
	   (pcelisp-push-ref ps arg)))
	((integerp arg) (pcelisp-c-int arg))
	((keywordp arg) (pcelisp-push-name ps arg))
	((symbolp arg) (pcelisp-push-symbol ps arg))
	((stringp arg) (pcelisp-c-string arg))
	((floatp arg) (pcelisp-c-real arg))
	(t
	 (pcelisp-push-special ps arg))))
;;	 (error "PCE/Lisp  Argument type not recognised (~A)" arg))))


(defun pcelisp-push-assoc (ps arg)
  (or (pcelisp-c-assoc arg)
      (if (eql (get-pcelisp-status-error ps) *pcelisp-unknown-assoc*)
	  (if (pcelisp-new-assoc arg)
	      (pcelisp-push-assoc ps arg)
	    (format t "PCE/Lisp  PCE object ~a does not exist~%" arg)))))


(defun pcelisp-push-ref (ps arg)
  (declare (ignore ps))
  (pcelisp-c-ref (pce-object-id arg)))


#+STATIC-LISP-SYMBOLS
(defun pcelisp-push-name (ps arg)
  (declare (ignore ps))
  (or (pcelisp-c-name arg)
      (and (pcelisp-new-name arg)
	   (pcelisp-c-name arg))))


#-STATIC-LISP-SYMBOLS
(defun pcelisp-push-name (ps arg)
  (declare (ignore ps))
  (let ((index (pcelisp-symbol-index arg)))
    (if index
	(pcelisp-c-name index)
      (pcelisp-c-name (pcelisp-add-keyword arg)))))


#+STATIC-LISP-SYMBOLS
(defun pcelisp-push-symbol (ps arg)
  (declare (ignore ps))
  (or (pcelisp-c-symbol arg)
      (progn
	(pcelisp-c-new-symbol
	 (symbol-name arg)
	 (package-name (symbol-package arg)))
	(pcelisp-c-symbol arg))))


#-STATIC-LISP-SYMBOLS
(defun pcelisp-push-symbol (ps arg)
  (declare (ignore ps))
  (let ((index (pcelisp-symbol-index arg)))
    (if (and index (pcelisp-c-symbol index))
	t
      (progn
	(pcelisp-c-new-symbol
	 (symbol-name arg)
	 (package-name (symbol-package arg)))
	(setf index (pcelisp-symbol-index arg))
	(if index
	    (pcelisp-c-symbol index)
	  (format t ";;; PCE/Lisp symbol ~A not defined~%" arg))))))


;;  ------------------------------------------------------
;;  Handling of special entries
;;  ------------------------------------------------------

(defvar *pce-specials-to-object* (make-hash-table :test #'equal))
(defvar *pce-object-to-specials* (make-hash-table :test #'equal))
(defvar *pce-specials-counter* 1)


(defun pcelisp-special-index (symbol)
  (gethash symbol *pce-specials-to-object*))


(defun pcelisp-special-value (index)
  (gethash index *pce-object-to-specials*))


(defun pcelisp-add-special (special)
  (let* ((ref (pcelisp-c-special *pce-specials-counter*))
	 (obj (pce-register (pce-at ref))))
    (setf (gethash special *pce-specials-to-object*) obj)
    (setf (gethash obj *pce-object-to-specials*) special)
    (incf *pce-specials-counter*)
    obj))


(defun pcelisp-push-special (ps arg)
  (let ((obj (or (pcelisp-special-index arg)
		 (pcelisp-add-special arg))))
    (pcelisp-push-ref ps obj)))


;;  ------------------------------------------------------
;;  Creating interface entries
;;  ------------------------------------------------------
;;
;;  In some Lisp implementations symbols and keywords are not static
;;  (i.e. can move after a garbage collection).  The code below takes
;;  this into account with the STATIC-LISP-SYMBOLS feature.  The
;;  feature is set in "load.lisp".


;;  ------------------------------------------------------
;;  Low-level non-static symbol management
;;  ------------------------------------------------------

#-STATIC-LISP-SYMBOLS
(defvar *pce-symbols-to-integer* (make-hash-table :test #'eq))
#-STATIC-LISP-SYMBOLS
(defvar *pce-integer-to-symbols* (make-hash-table :test #'eql))
#-STATIC-LISP-SYMBOLS
(defvar *pce-symbols-counter* 1)


;;  pcelisp-find-symbol index ==> symbol
;;
;;  Given index returns the corresponding symbol.  Should always
;;  succeed, so we enter the debugger if the index is not found.

#-STATIC-LISP-SYMBOLS
(defun pcelisp-find-symbol (index)
  (or (gethash index *pce-integer-to-symbols*)
      (error "PCELISP-FIND-SYMBOL ~A not found" index)))


;;  pcelisp-symbol-index symbol ==> index
;;
;;  Given symbol returns the corresponding index.  Fails when no index
;;  for the symbol is known.

#-STATIC-LISP-SYMBOLS
(defun pcelisp-symbol-index (symbol)
  (gethash symbol *pce-symbols-to-integer*))


;;  pcelisp-add-symbol symbol ==> index
;;
;;  Adds the symbol to the PCE/Lisp interface and returns the index.

#-STATIC-LISP-SYMBOLS
(defun pcelisp-add-symbol (symbol)
  (setf (gethash symbol *pce-symbols-to-integer*) *pce-symbols-counter*)
  (setf (gethash *pce-symbols-counter* *pce-integer-to-symbols*) symbol)
  (pcelisp-c-new-symbol
   (symbol-name symbol)
   (package-name (symbol-package symbol)))
  (prog1
      *pce-symbols-counter*
   (incf *pce-symbols-counter*)))


#-STATIC-LISP-SYMBOLS
(defun pcelisp-add-keyword (keyword)
  (setf (gethash keyword *pce-symbols-to-integer*) *pce-symbols-counter*)
  (setf (gethash *pce-symbols-counter* *pce-integer-to-symbols*) keyword)
  (pcelisp-c-new-name
   (symbol-name keyword)
   *pce-symbols-counter*)
  (prog1
      *pce-symbols-counter*
   (incf *pce-symbols-counter*)))


;;! pcelisp-new-name name ==> keyword
;;
;;  Creates a new keyword (Lisp) / name (PCE) association in the
;;  PCE/Lisp interface.  This function is called once for all keywords
;;  and names passed over the interface, thereafter the value is
;;  passed directly.
;;
;;  The argument may be a keyword, symbol or string.  Returns the
;;  keyword created.

#+STATIC-LISP-SYMBOLS
(defun pcelisp-new-name (name)
  (if (keywordp name)
      (progn
	(pcelisp-c-new-name (string name) name)
	name)
    (if (or (symbolp name) (stringp name))
	(progn
	  (let ((keyword (intern (string name) "KEYWORD")))
	    (pcelisp-c-new-name (string name) keyword)
	    keyword))
      (error "PCE/Lisp  Keyword or symbol expected (~A)" name))))
	

#-STATIC-LISP-SYMBOLS
(defun pcelisp-new-name (name)
  (if (keywordp name)
      (progn
	(pcelisp-add-keyword name)
	name)
    (if (or (symbolp name) (stringp name))
	(progn
	  (let ((keyword (intern (string name) "KEYWORD")))
	    (pcelisp-add-keyword keyword)
	    keyword))
      (error "PCE/Lisp  Keyword or symbol expected (~A)" name))))
	

(defun pcelisp-new-assoc (obj)
  (when (pcelisp-c-new-assoc (symbol-name (pce-object-id obj)) obj)
	obj))


;;  ------------------------------------------------------
;;  Public type testing predicates
;;  ------------------------------------------------------

;;! pce-exists-p Object ==> {t | nil}
;;
;;  Succeeds if ^Object^ is an existing PCE object.  Existence is
;;  defined by asking PCE whether the object still exists.
;;  pce-exists-p/# enables application programmers to test for objects
;;  that are automatically destroyed by PCE when their reason of
;;  existence is removed (e.g. a window part of a frame).

(defun pce-exists-p (obj)
  (when (pce-object-p obj)
	(if (pce-assoc-p obj)
	    (if (or (pcelisp-c-is-assoc obj)
		    (pcelisp-new-assoc obj))
		t
	      (progn
		(pce-unregister obj)	; This call may have generated
					;   entry in hash-table
		nil))
	  (if (pce-reference-p obj)
	      (pcelisp-c-is-ref (pce-object-id obj))
	    (progn
	      (pce-unregister obj)	; Ibid.
	      nil)))))


;;  ------------------------------------------------------
;;  Additional functions
;;  ------------------------------------------------------

;;! pce-global class arguments... ==> Object
;;
;;  Creates ^Object^ as a ``global'' object.  That is, ^Object^ cannot
;;  be garbage collected.  The arguments are the name of the ^class^
;;  and any initialising ^arguments^.

(defun pce-global (class &rest args)
  (let ((object (pce-make class args)))
    (when object
	  (pce-send object :lock-object @on))))

   
;;! pce-send-list receiver selector list ==> receiver
;;
;;  Calls pce-send/# for each element in ^list^.  Always succeeds
;;  returning ^receiver^.

(defun pce-send-list (receiver selector arguments)
  (declare (type list arguments))
  (dolist (arg arguments)
	  (pce-send receiver selector arg))
  receiver)


#|
(trace pcelisp-push-name
       pcelisp-symbol-index
       pcelisp-c-name
       pcelisp-c-new-name
       pcelisp-c-symbol
       pcelisp-c-new-symbol
       pcelisp-add-symbol
       pcelisp-add-keyword
       pcelisp-new-name
       pce-send
       pce-get
       pce-new
       get-pcelisp-status-error)
       
|#
