
/*** reversi operations ***/

:- module(play_reversi,[
	new_tab/1,		% makes an initial position
	choose_position/4,	% generates a moves for a color using a algor.
	make_move/5,		% change a Tab for a move if move is valid
	static_evaluation/4	% computes statics evaluation asn # of pieces
]).

:- use_module(matrix).


% makes an initial position

new_tab(T4) :-
	make_matrix(8,8,T0),
	change_matrix(T0,4,4,white,T1),
	change_matrix(T1,4,5,black,T2),
	change_matrix(T2,5,4,black,T3),
	change_matrix(T3,5,5,white,T4).

% choose_position(M,T,C,NT)  using method M choose new psition NT from T
%				playing with color C

choose_position(naive,T,C,NT) :- choose_naive(T,C,NT).
choose_position(heuristic,T,C,NT) :- choose_heuristic(T,C,NT).
choose_position(minimax(L),T,C,NT) :- choose_minimax(T,L,5,C,NT).
choose_position(minimax(L,B),T,C,NT) :- choose_minimax(T,L,B,C,NT).

% generation of new positions using minimax (with alfa-beta)

choose_minimax(T,L,B,black,NT) :- f_min(T,_,L,B,_Min,NT,_NV).
choose_minimax(T,L,B,white,NT) :- f_max(T,_,L,B, max,NT,_NV).
/*
	var(Min),  integer(N) -> Min @< N
	atom(max), integer(N) -> max @> N



level 3			 - f_min -
			/         \
		       /	  min ---|
		      /     	/     	 |
level 2		   f_max(1)   f_max    	f_max
		    /	     /  \
		   /	    /     max --|
		  /	   /	 / 	   |
level 1		f_min	f_min   f_min  


level ...
*/



/* first evaluation */

f_min(T,S,L,B,AB,M,V) :-
	(L=0 -> 
		V=S, M=T
	;
		K is L-1,
		(gen_ord_pos(T,black,B,[(FS,_,FT)|R]) ->
		    f_max(FT,FS,K,B,max,_,FV),
		    min(R,K,B,AB,FT,FV,M,V);
		    V=S,M=T
		)
	).
%	Tab is 20-5*L,tab(Tab), format('~w:~t result ~w ab=~w~n',[f_min,V,AB]).

f_max(T,S,L,B,AB,M,V) :-
	(L=0 -> 
		V=S, M=T
	;
		K is L-1,
		(gen_ord_pos(T,white,B,[(FS,_,FT)|R]) ->
		    f_min(FT,FS,K,B,_Min,_,FV),
		    max(R,K,B,AB,FT,FV,M,V);
		    V=S,M=T
		)
	).
%	Tab is 20-5*L,tab(Tab), format('~w:~t result ~w ab=~w~n',[f_max,V,AB]).

/* top right branch evaluation */

max([],_,_,_,M,V,M,V).
max([(FS,_,FT)|R],L,B,AB,GT,GV,M,V) :-
	(GV @> AB ->
		M=GT, V=GV
%		,Tab is 20-5*L-2,tab(Tab)
%		,format('**~w:~tcutting with ~w ab=~w~n',[max,GV,AB])
	;
		f_min(FT,FS,L,B,GV,_,FV),
		(FV > GV ->
			max(R,L,B,AB,FT,FV,M,V)
		;
			max(R,L,B,AB,GT,GV,M,V)
		)
	).
		

min([],_,_,_,M,V,M,V).
min([(FS,_,FT)|R],L,B,AB,GT,GV,M,V) :-
	(GV @< AB ->
		M=GT, V=GV
%		,Tab is 20-5*L-2,tab(Tab)
%		,format('**~w:~tcutting with ~w ab=~w~n',[min,GV,AB])
	;
		f_max(FT,FS,L,B,GV,_,FV),
		(FV < GV ->
			min(R,L,B,AB,FT,FV,M,V)
		;
			min(R,L,B,AB,GT,GV,M,V)
		)
	).


% generation of new positions using heuristic (best static evaluation)


choose_heuristic(T,C,NT) :-
	gen_ord_pos(T,C,100,[(_,_,NT)|_]).

gen_ord_pos(T,C,B,FSE) :-
	gen_positions(T,C,L),
	eval_positions(L,EL),
	sort(EL,ASE),
	(C=black -> SE=ASE; reverse(ASE,SE,[])),
	first(SE,B,FSE).

first([],_,[]).
first([X|R],N,F) :- (N==0 -> F=[]; F=[X|S], M is N-1, first(R,M,S)).

reverse([],L,L).
reverse([H|T],R,D) :-
	reverse(T,R,[H|D]).

eval_positions([],[]).
eval_positions([E|R],[(V,D,E)|VR]) :-
	static_evaluation(E,V,B,P), D is B-P,
	eval_positions(R,VR).

% naive generation of new positions (pick randomly a valid move)

choose_naive(T,C,NT) :-
	gen_positions(T,C,L),
	choose(L,NT).
	
%choose(L,C) :- len(L,N), K is integer(random*N+1),	nth(K,L,C).
choose(L,C) :- len(L,N), random(N,K),	nth(K,L,C).


% stupid random genarater 
random(N,K) :- 
	recorded(random,J,Ref), 
	findall(R,recorded(random,R,_),L),
	nth(J,L,K),
	K=<N, !,
	erase(Ref),
	recordz(random,J,_).
random(_N,_K) :- 
	recorda(random,1,_),
	recorda(random,6,_),
	recorda(random,3,_),
	recorda(random,4,_),
	recorda(random,2,_),
	recorda(random,7,_),
	recorda(random,5,_).


len([],0).
len([_|R],M) :- len(R,N), M is N+1.

nth(N,[E|R],S) :- (N=1 -> S=E; M is N-1, nth(M,R,S)).


gen_positions(Tab,C,L) :-
	findall(T,(gen(I),gen(J),make_move(Tab,I,J,C,T)),L).

gen(1).
gen(I) :- gen(J), I is J+1, (J<8 -> otherwise; !, fail).


% validation of moves and creation of a new table from T 
% with a pice of color C in the position (I,J)

make_move(T,I,J,C,NT) :- 
	matrix(T,I,J,empty),
	findall(dir(DI,DJ,LI,LJ),dir(DI,DJ,LI,LJ),LD),
	move_all_dir(LD,T,I,J,C,WT,CC),
	CC \== 0,
	change_matrix(WT,I,J,C,NT).

move_all_dir([],T,_,_,_,T,0).
move_all_dir([dir(DI,DJ,LI,LJ)|R],T,I,J,C,NT,CC) :-
	(move_dir(DI,DJ,LI,LJ,T,I,J,C,WT,C1) -> 
		otherwise
	;
		C1=0,
		WT=T
	),
	move_all_dir(R,WT,I,J,C,NT,C2),
	CC is C1+C2.
	
move_dir(DI,DJ,LI,LJ,T,I,J,C,NT,CC) :-
	NI is I+DI, NI\==LI,
	NJ is J+DJ, NJ\==LJ,
	matrix(T,NI,NJ,P),
	(P=C -> NT=T, CC=0
	;
		P \== empty,
		change_matrix(T,NI,NJ,C,WT),
		move_dir(DI,DJ,LI,LJ,WT,NI,NJ,C,NT,CD),
		CC is CD+1
	).

dir(-1,-1,0,0).	dir(-1,0,0,9).	dir(-1,1,0,9).
dir( 0,-1,9,0).			dir( 0,1,9,0).
dir( 1,-1,9,0).	dir( 1,0,9,9).	dir( 1,1,9,9).

other(black,white).
other(white,black).


% static evaluation V of a position T
% being B the number of white pieces and P the number of black pieces.

static_evaluation(T,V,B,P) :- static_evaluation(T,1,V,B,P).

static_evaluation([],_I,0,0,0).
static_evaluation([R|C],I,V,B,P) :- 
	static_evaluation(R,I,1,RV,RB,RP),
	NI is I+1,
	static_evaluation(C,NI,CV,CB,CP),
	V is RV+CV,
	B is RB+CB,
	P is RP+CP.

static_evaluation([],_I,_J,0,0,0).
static_evaluation([E|C],I,J,V,B,P) :- 
	piece_value(E,EV,EB,EP),
	pos_value(I,J,PV),
	NJ is J+1,
	static_evaluation(C,I,NJ,CV,CB,CP),
	V is CV+EV*PV,
	B is EB+CB,
	P is EP+CP.

piece_value(empty, 0,0,0).
piece_value(black,-1,0,1).
piece_value(white, 1,1,0).

pos_value(I,J,PV) :-
	(I<5 -> LI=I; LI is 9-I),
	(J<5 -> LJ=J; LJ is 9-J),
	(LI<LJ -> M=LI ; M=LJ),
	level_value(M,MV,DV),
	(LI=LJ -> PV=DV; PV=MV).


level_value(1,100,1000).
level_value(2,1,-100).
level_value(3,5,10).
level_value(4,5,5).

/*
level_value(1,8,10).
level_value(2,2,1).
level_value(3,4,6).
level_value(4,3,3).
*/

/*
level_value(1,100,200).
level_value(2,2,1).
level_value(3,10,20).
level_value(4,1,1).
*/
