;;;; Epoch related things for Edebug
;; Copyright (C) 1994 Daniel LaLiberte (liberte@cs.uiuc.edu)

;; This file contains replacements for functions in Edebug
;; to make it work better with Epoch.
;; Todo: We could use zones to highlight the current expression,
;; or to indicate frequency of use.

;; Some of this might apply to Lucid and FSF Emacs 19.

(defun edebug-get-displayed-buffer-points ()
  "Return a list of buffer point pairs, for all displayed buffers."
  (let ((buffer-point-list nil))
    (mapcar 
     (function 
      (lambda (screen)
	(select-screen screen)
	(save-excursion
	  (let* ((first-window (selected-window))
		 (next (next-window first-window)))
	    (while (not (eq next first-window))
	      (setq buffer-point-list
		    (cons (cons (window-buffer next) (point)) 
			  buffer-point-list))
	      (setq next (next-window next)))
	    buffer-point-list))))
     (epoch::screen-list 'unmapped))
    buffer-point-list
    ))

(defun edebug-pop-to-buffer (buffer)
  "Like pop-to-buffer, but select a window that buffer was shown in.
If running epoch, use the same screen too."
  (let ((edebug-window (edebug-get-buffer-window buffer)))
    (if edebug-window 
	(select-window edebug-window)
      ;; It is not currently displayed, so find some place to display it.
      ;; Select a screen that the buffer has been displayed in before
      ;; or the current screen otherwise.
      (select-screen
       ;; allowed-screens in epoch 3.2, was called screens before that
       (or (car (symbol-buffer-value 'allowed-screens buffer))
	   (epoch::current-screen)))
      (if (one-window-p)
	  (split-window))
;;      (message "next window: %s" (next-window)) (sit-for 1)
      (if (eq (get-buffer-window edebug-trace-buffer) (next-window))
	  ;; Dont select trace window
	  nil
	(select-window (next-window)))
      (set-window-buffer (selected-window) buffer)
      (set-window-hscroll (selected-window) 0)
      ))
  ;; Selecting the window does not set the buffer??  docs says it does.
  ;; But just returning the buffer is not equivalent.
  (set-buffer buffer)
  ;; buffer
  )


;; For epoch, we need to save and restore screens as well as windows.

(defun edebug-current-window-configuration ()
  "Return an object recording the current configuration of Epoch screen-list.  
The object is a list of pairs of the form (SCREEN .  CONFIGURATION)
where SCREEN has window-configuration CONFIGURATION.  The current
screen is the head of the list."
  (let ((screen-list (epoch::screen-list 'unmapped))
	(current-screen (epoch::get-screen))
	(current-buffer (current-buffer))
	)
    ;; Put current screen first.
    (setq screen-list (cons current-screen (delq current-screen screen-list)))
    (prog1
	(mapcar (function
		 (lambda (screen)
		   (cons screen
			 (progn
			   (epoch::select-screen screen)
			   (current-window-configuration)))))
		screen-list)

      ;; Recover from the excursion.
      (epoch::select-screen current-screen)
      (set-buffer current-buffer)
      )))


(defun edebug-set-window-configuration (sc)
  "Set the window-configuration for all the screens in SC.
Set the current screen to be the head of SC."
    (mapcar (function
	     (lambda (screen-conf)
	       (if (epoch::screen-p (car screen-conf))  ; still exist?
		   (progn
		     (epoch::select-screen (car screen-conf))
		     (set-window-configuration (cdr screen-conf))))))
	    sc)
    (if (epoch::screen-p (car (car sc)))
	(epoch::select-screen (car (car sc))))
    )

(fset 'edebug-get-buffer-window 'epoch::get-buffer-window)


;; Input event handling in epoch requires we do a dispatch-events
;; in order to get the full effect of sit-for and input-pending-p.

(defun edebug-sit-for (arg)
  (epoch::dispatch-events)
  (sit-for arg))

(defun edebug-input-pending-p ()
  (epoch::dispatch-events)
  (input-pending-p))


(provide 'edebug-epoch)
