/* Lisp Editor and Listener Window Class */

#include <windows.h>
#include <string.h>
#include <ctype.h>
#ifdef NOTTY
#include <dde.h>
#endif /* NOTTY */
#include "ledit.h"
#include "winutils.h"

static HFONT hFixedFont;
static HWND hTTYWnd = 0;
static void (*MainLoop)(void);
static FARPROC fpOldEditProc = NULL, fpLEditProc = NULL;

/* input and output buffers */
#define BUFSIZE 255
static char obuf[BUFSIZE], *obp = obuf;
static instart = 0, inend = 0;

/* TTY Trimming defines */
#define MAXTTYBUF 1500
#define TRIMTTYTO 1000

BOOL FAR PASCAL LEditWndProc(HWND, unsigned, WORD, LONG);
static void waitforline(void);
static BOOL check_parens(char *, int);
static BOOL at_text_end(char *, int);
static BOOL has_return(char *, int);
static BOOL input_complete(int);
static BOOL flash_matching_paren(HWND, int);
static void pardelay(void);
static void do_tab(HWND, int);
static void fix_blanks(HWND, int, int);
static int num_to_skip(char *, int);
static BOOL is_special(char *, int);
static void check_trim_buffer(int);

/**************************************************************************/
/**************************************************************************/
/**                                                                      **/
/**                         Public Routines                              **/
/**                                                                      **/
/**************************************************************************/
/**************************************************************************/

void InitLEditClass(void (*f)())
{
  MainLoop = f;

  hFixedFont = GetStockObject(ANSI_FIXED_FONT);
}

HWND CreateLEditWindow(HWND hWndParent, HMENU hMenu, HANDLE hInstance)
{
  RECT Rect;
  HWND hWnd;

  GetClientRect(hWndParent, (LPRECT) &Rect);
  hWnd = CreateWindow("Edit",
		      NULL,
		      WS_CHILD | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL |
		      ES_MULTILINE | ES_AUTOVSCROLL | ES_AUTOHSCROLL,
		      0,
		      0,
		      (Rect.right - Rect.left),
		      (Rect.bottom - Rect.top),
		      hWndParent,
		      hMenu,
		      hInstance,
		      NULL);

  if (hWnd) {
    if (fpOldEditProc == NULL) {
      fpOldEditProc = (FARPROC) GetWindowLong(hWnd, GWL_WNDPROC);
      fpLEditProc = MakeProcInstance(LEditWndProc, hInstance);
    }
    SetWindowLong(hWnd, GWL_WNDPROC, (LONG) fpLEditProc);

    /* "First" window created is TTY */
    if (hTTYWnd == NULL) hTTYWnd = hWnd;
    SendMessage(hWnd, WM_SETFONT, hFixedFont, FALSE);
  }

  return(hWnd);
}

#ifndef NOTTY
BOOL TTYHasInput(void)
{
  return(instart < inend ? TRUE : FALSE);
}

void TTYPutStr(char *s)
{
  while (*s != '\0') TTYPutC((int) *s++);
  TTYFlushOutput();
}

int TTYPutC(int c)
{
  if (obp >= obuf + BUFSIZE - 3) TTYFlushOutput();
  if (c == '\n' || c == '\r') {
    *obp++ = '\r';
    *obp++ = '\n';
    TTYFlushOutput();
  }
  else *obp++ = c;
  return(c);
}

void TTYResetInput(void)
{
  instart = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
  inend = instart;
  SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(instart, instart));
}

void TTYFlushOutput(void)
{
  int selstart, selend;

  if (obp > obuf) {
    *obp = '\0';
    obp = obuf;
    selstart = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
    selend = selstart + strlen(obuf);
    SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(selstart, selend));
    SendMessage(hTTYWnd, EM_REPLACESEL, 0, (LONG) obuf);
    TTYResetInput();
    check_trim_buffer(TRUE);
  }
}

void TTYFlush(void)
{
  TTYFlushOutput();
  TTYResetInput();
}

int TTYGetC(void)
{
  int c;
  HANDLE hText;
  PSTR pText;

  /* wait for input if there is none */
  if (! TTYHasInput()) waitforline();

  /* get the text */
  hText = (HANDLE) SendMessage(hTTYWnd, EM_GETHANDLE, 0, 0);
  if (! hText) { SysBeep(10); return('\0'); }
  pText = LocalLock(hText);
  if (! pText) { SysBeep(10); return('\0'); }

  /* skip linefeeds (or whatever they are) */
  while (pText[instart] == '\r') instart++;

  if (instart < inend)
    c = pText[instart++];
  else {
    SysBeep(10);
    c = '\0';
  }

  /* release the text buffer */
  LocalUnlock(hText);

  /* reset the enput at the end of the last line */
  if (instart >= inend) TTYResetInput();

  return((c == '\r') ? '\n' : c);
}
#endif /* NOTTY */

BOOL TTYHasSelection(void)
{
  long sel;

  sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
  return(LOWORD(sel) < HIWORD(sel) ? TRUE : FALSE);
}

void TTYSelToClip(void)
{
  SendMessage(hTTYWnd, WM_COPY, 0, 0);
}

void TTYClearSel(void)
{
#ifndef NOTTY
  long sel;
  int selstart;

  sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
  selstart = LOWORD(sel);

  if (selstart < instart) SysBeep(10);
  else SendMessage(hTTYWnd, WM_CLEAR, 0, 0);
#else
  SendMessage(hTTYWnd, WM_CLEAR, 0, 0);
#endif /* NOTTY */
}

void TTYPasteFromClip(void)
{
#ifndef NOTTY
  long sel;
  int selstart;

  sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
  selstart = LOWORD(sel);

  /* move the insertion point to the end if it is */
  /* before the input text start                  */
  if (selstart < instart) {
    int text_end = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
    SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(text_end, text_end));
  }
#endif /* NOTTY */

  SendMessage(hTTYWnd, WM_PASTE, 0, 0);

#ifndef NOTTY
  /* check for a return and a complete expression */
  if (input_complete(TRUE)) {
    inend = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
  }
#endif /* NOTTY */
}

#ifdef NOTTY
HANDLE TTYSelectionPokeData(void)
{
  HANDLE h, hData;
  int i, selstart, selend;
  long sel;
  DDEPOKE FAR *ddep;
  char *p, *s;

  sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
  selstart = LOWORD(sel);
  selend = HIWORD(sel);
  h = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE,
		  sizeof(DDEPOKE) + selend - selstart + 1);
  if (h) {
    ddep = (DDEPOKE FAR *) GlobalLock(h);
    if (ddep) {
      hData = SendMessage(hTTYWnd, EM_GETHANDLE, 0, 0);
      if (hData) {
	p = LocalLock(hData);
	if (p) {
	  ddep->fRelease = FALSE;
	  ddep->cfFormat = CF_TEXT;
	  s = ddep->Value;
	  for (i = selstart; i < selend; i++)
	    if (p[i] != '\r') *s++ = p[i];
	  *s++ = '\0';
	  LocalUnlock(hData);
	}
      }
    }
    GlobalUnlock(h);
  }
  return(h);
}
#endif /* NOTTY */
#ifndef NOTTY
void TTYTrimBuffer(void)
{
  check_trim_buffer(FALSE);
}
#endif /* NOTTY */

/**************************************************************************/
/**************************************************************************/
/**                                                                      **/
/**                    Subclass Callback Function                        **/
/**                                                                      **/
/**************************************************************************/
/**************************************************************************/

/* window function for LEdit class -- filters returns */
BOOL FAR PASCAL LEditWndProc(HWND hWnd, unsigned message, WORD wParam, LONG lParam)
{
  if (message == WM_CHAR) {
#ifndef NOTTY
    long sel;
    int selstart, selend;

    sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
    selstart = LOWORD(sel);
    selend = HIWORD(sel);

    /* ignore backspaces at the start of the input region */
    if (selstart == instart && selstart == selend && wParam == '\b') {
      SysBeep(10);
      return(FALSE);
    }

    /* move the insertion point to the end if it is before the */
    /* input text start                                        */
    if (selstart < instart) {
      int text_end = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
      SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(text_end, text_end));
    }
#endif /* NOTTY */

    /* adjust spacing on tab */
    if (wParam == '\t') {
      do_tab(hTTYWnd, instart);
      return(FALSE);
    }

#ifndef NOTTY
    /* jump to end of input on shift-return */
    if (wParam == '\r' && HIBIT(GetKeyState(VK_SHIFT))) {
      selstart = selend = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
      SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(selstart, selend));
      return(FALSE);
    }
#endif /* NOTTY */

    /* insert the character using the inherited method */
    CallWindowProc(fpOldEditProc, hWnd, message, wParam, lParam);

    /* flash matching parenthesis */
    if (wParam == ')') flash_matching_paren(hTTYWnd, instart);

#ifndef NOTTY
    /* check the character for a return */
    if (wParam == '\r' && input_complete(FALSE)) {
      inend = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
      return(TRUE);
    }
    else return(FALSE);
#else
    return(TRUE);
#endif /* NOTTY */
  }
  else return((BOOL) CallWindowProc(fpOldEditProc, hWnd, message, wParam, lParam));
}

/**************************************************************************/
/**************************************************************************/
/**                                                                      **/
/**                         Internal Routines                            **/
/**                                                                      **/
/**************************************************************************/
/**************************************************************************/

#ifndef NOTTY
static void waitforline(void)
{
  TTYFlushOutput();
  TTYResetInput();
  (*MainLoop)();
}

static BOOL input_complete(check_return)
{
  HANDLE hText;
  PSTR pText;
  BOOL result;
  int text_end, selstart, checkstart;

  text_end = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);
  hText = (HANDLE) SendMessage(hTTYWnd, EM_GETHANDLE, 0, 0);
  if (! hText) { SysBeep(10); return(FALSE); }
  pText = LocalLock(hText);
  if (! pText) { SysBeep(10); return(FALSE); }
  selstart = LOWORD(SendMessage(hTTYWnd, EM_GETSEL, 0, 0));
  checkstart = (selstart > 0) ? selstart - 1 : selstart;

  result = check_parens(pText + instart, text_end - instart)
	   && at_text_end(pText + selstart, text_end - selstart)
	   && (! check_return
	       || has_return(pText + checkstart, text_end - checkstart));

  LocalUnlock(hText);
  return(result);
}

static BOOL at_text_end(char *s, int n)
{
  int i, result = TRUE;

  for (i = 0; result && i < n; i++)
    if (! isspace(s[i]) && s[i] != '\n' && s[i] != '\r')
      result = FALSE;
  return(result);
}

static BOOL has_return(char *s, int n)
{
  int i, result = FALSE;

  for (i = 0; ! result && i < n; i++)
    if (s[i] == '\n' || s[i] == '\r')
      result = TRUE;
  return(result);
}
#endif /* NOTTY */

static BOOL check_parens(s, n)
	char *s;
	int n;
{
  int parcount = 0, inquotes = FALSE, incomment = FALSE;
  char ch;
   
  while (n-- > 0) {
    ch = *s++;
    switch (ch) {
    case  '"': inquotes = ! inquotes; break;
    case  ';': if (! inquotes) incomment = TRUE; break;
    case '\r':
    case '\n': incomment = FALSE; break;
    case  '(': if (! inquotes && ! incomment) parcount++; break;
    case  ')': if (! inquotes && ! incomment) parcount--; break;
    }
  }
  return (parcount <= 0 ? TRUE : FALSE);
}

static BOOL flash_matching_paren(HWND hWnd, int start)
{
  int parcount = 0, inquotes = FALSE, sel, par;
  char ch, *s;
  HANDLE hText;

  sel = LOWORD(SendMessage(hWnd, EM_GETSEL, 0, 0));

  hText = (HANDLE) SendMessage(hTTYWnd, EM_GETHANDLE, 0, 0);
  if (! hText) { SysBeep(10); return(FALSE); }
  s = LocalLock(hText);
  if (! s) { SysBeep(10); return(FALSE); }
  s = s + sel - 1;

  par = sel;
  do {
    par--;
    ch = *s--;
    switch (ch) {
    case  '"': inquotes = ! inquotes; break;
    case  '(': if (! inquotes) parcount--; break;
    case  ')': if (! inquotes) parcount++; break;
    }
  } while (par >= start && parcount > 0);

  LocalUnlock(hText);

  if (ch == '(') {
    SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(par, par + 1));
    pardelay();
    SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(sel, sel));
  }

  return (parcount <= 0 ? TRUE : FALSE);
}

static void pardelay(void)
{
  Delay(250);
}

#define ISRETURNCHAR(c) ((c) == '\r' || (c) == '\n')

static void do_tab(HWND hWnd, int start)
{
  int sel, curline, lastline, pos, nblanks, inquote, parcount, length;
  char *s, ch;
  HANDLE hText;

  sel = LOWORD(SendMessage(hWnd, EM_GETSEL, 0, 0));

  hText = (HANDLE) SendMessage(hTTYWnd, EM_GETHANDLE, 0, 0);
  if (! hText) { SysBeep(10); return; }
  s = LocalLock(hText);
  if (! s) { SysBeep(10); return; }
  length = (int) SendMessage(hWnd, WM_GETTEXTLENGTH, 0, 0);

  /* find beginning of the line */
  curline = sel;
  while (curline > start && ! ISRETURNCHAR(s[curline - 1])) curline--;
  if (curline == start) return;

  /* find unmatched paren */
  parcount = 0;
  inquote = FALSE;
  pos = curline;
  while (parcount >= 0 && --pos >= start) {
    ch = s[pos];
    switch (ch) {
    case ')': if (! inquote) parcount++; break;
    case '(': if (! inquote) parcount--; break;
    case '"': inquote = ! inquote; break;
    }
  }
  if (parcount == 0) return;
  
  /* find beginning of the line containing the expression start */
  lastline = pos;
  while (lastline > 0 && ! ISRETURNCHAR(s[lastline - 1])) lastline--;

  /* skip forward an s-expression or to first non blank */
  pos += num_to_skip(s + pos, curline - pos);

  if (pos > curline) pos = curline;
  nblanks = pos - lastline;

  /* adjust for the number of blanks already present, replace tabs by blanks */
  for (pos = curline; 
       pos < length && (s[pos] == ' ' || s[pos] == '\t');
       nblanks--, pos++)
    if (s[pos] == '\t') s[pos] = ' ';

  LocalUnlock(hText);

  /* insert or delete the appropriate number of blanks */
  if (nblanks == 0) return;

  sel += nblanks;
  if (pos > length) pos = length;
  SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(pos, pos));
  fix_blanks(hWnd, nblanks, curline);
  length = (int) SendMessage(hWnd, WM_GETTEXTLENGTH, 0, 0);
  if (pos > length) pos = length;
  SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(sel, sel));
}

static void fix_blanks(HWND hWnd, int nblanks, int curline)
{
  int i;

  if (nblanks > 0) {
    for (i = 0; i < nblanks && i < BUFSIZE -3; i++) obuf[i] = ' ';
    obuf[i] = '\0';
    SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(curline, curline));
    SendMessage(hWnd, EM_REPLACESEL, 0, (LONG) obuf);
  }
  else {
    obuf[0] = '\0';
    SendMessage(hWnd, EM_SETSEL, 0, MAKELONG(curline, curline - nblanks));
    SendMessage(hWnd, EM_REPLACESEL, 0, (LONG) obuf);
  }
}

static int num_to_skip(char *s, int n)
{
  char str[4];
  int i, pos, oldpos;

  pos = 0;
  
  if (n <= 0) pos = 0;
  else if (*s == '(') {
  
    s++; n--; pos = 1;
    
    /* skip blanks */
    while (n > 0 && (*s == ' ' || *s == '\t')) { s++; n--; pos++; }
    
    /* check for end of line or list or lisp comment*/
    if (n > 0 && ! ISRETURNCHAR(*s) && *s != ';' && *s != '(') {
    
      /* check for special symbols */
      for (i = 0; i < 3 && i < n; i++)
	str[i] = toupper(s[i]);
      str[i] = '\0';
      if (is_special(s, n) /* strcmp(str, "DEF") == 0 || strcmp(str, "LET") == 0 
          || strcmp(str, "FLE") == 0 */ )
        pos = 2;
      else {
        
        /* skip over the s-expression */
        oldpos = pos;
	while (n > 0 && *s != ' ' && *s != '\t' && ! ISRETURNCHAR(*s))
          { s++; n--; pos++; }
          
        /* check for another s expression */
        for (i = 0; n > 0 && (*s == ' ' || *s == '\t'); s++, n--, i++) ;
	if (n == 0 || ISRETURNCHAR(*s))
	  pos = (oldpos == pos) ? oldpos + 1 : oldpos;
        else pos += i;
      }
    }
  }
  else {
    
    /* skip over any blanks */
    for (i = 0; n > 0 && (*s == ' ' || *s == '\t'); s++, n--, i++) ;
    if (n > 0 && ! ISRETURNCHAR(*s)) pos += i;
  }
  return(pos);
}

static BOOL is_special(char *s, int n)
{
  char str[10];
  int i;
  
  for (i = 0; i < n && i < 9; i++) str[i] = toupper(s[i]);
  str[i] = '\0';

  if (n >= 5 && strncmp(str, "DEFUN", 5) == 0) return(TRUE);
  if (n >= 8 && strncmp(str, "DEFMACRO", 8) == 0) return(TRUE);
  if (n >= 7 && strncmp(str, "DEFMETH", 7) == 0) return(TRUE);
  if (n >= 8 && strncmp(str, "DEFPROTO", 8) == 0) return(TRUE);
  if (n >= 3 && strncmp(str, "LET", 3) == 0) return(TRUE);
  if (n >= 4 && strncmp(str, "FLET", 4) == 0) return(TRUE);
  if (n >= 4 && strncmp(str, "COND", 4) == 0) return(TRUE);
  if (n >= 4 && strncmp(str, "CASE", 4) == 0) return(TRUE);
  if (n >= 6 && strncmp(str, "LABELS", 6) == 0) return(TRUE);
  if (n >= 6 && strncmp(str, "LAMBDA", 6) == 0) return(TRUE);
  return(FALSE);
}

static void check_trim_buffer(int maxonly)
{
  int length, start, line;
  long sel;

  length = (int) SendMessage(hTTYWnd, WM_GETTEXTLENGTH, 0, 0);

  if (length > (maxonly ? MAXTTYBUF : TRIMTTYTO)) {
    line = (int) SendMessage(hTTYWnd, EM_LINEFROMCHAR, length - TRIMTTYTO, 0);
    start = (int) SendMessage(hTTYWnd, EM_LINEINDEX, line, 0);
    sel = SendMessage(hTTYWnd, EM_GETSEL, 0, 0);
    obuf[0] = '\0';
    SendMessage(hTTYWnd, EM_SETSEL, 0, MAKELONG(0, start));
    SendMessage(hTTYWnd, EM_REPLACESEL, 0, (LONG) obuf);
    instart -= start;
    inend -= start;
    sel = MAKELONG(LOWORD(sel) - start, HIWORD(sel) - start);
    SendMessage(hTTYWnd, EM_SETSEL, 0, sel);
  }
}
