(in-package "XLSCMP")

;;;;
;;;; Compiler Macro Facility
;;;;

;;**** move to proper place?
(defvar *cmp-macros* nil)
(defvar *cmp-setf* nil)
(defvar *cmp-structs* nil)
(defvar *cmp-global-macros* nil)

;;**** think about precedence if macro and cmpmacro both exist
;;**** may simplify setf that way?

(defun cmp-macroexpand (e &optional
			  (env (list nil *cmp-fenv* *cmp-macros* *cmp-global-macros*)))
  (macroexpand e env))

(defun cmp-macroexpand-1 (e &optional
			    (env (list nil *cmp-fenv* *cmp-macros* *cmp-global-macros*)))
  (macroexpand-1 e env))

;;****** modify to handle dotted lists
(defun fixup-macro-args (args)
  (let ((esym nil)
	(wsym nil))
    (setf args (copy-list args))
    (let ((last (last args)))
      (unless (null (cdr last))
	      (rplacd last (list '&rest (cdr last)))))
    (setf args (nsubst '&rest '&body args))
    (setf esym (getf args '&environment))
    (if esym
	(remf args '&environment)
        (setf esym (gensym "ENV")))
    (cond
     ((eq (first args) '&whole)
      (setf wsym (second args))
      (setf args (rest (rest args))))
     (t (setf wsym (gensym "WHOLE"))))
    (cons esym (cons wsym args))))

(defun destructuring-arglist-p (x)
  (consp
   (find-if #'(lambda (x) (or (member x lambda-list-keywords) (consp x))) x)))

(defun set-cmp-macro (sym fun)
  (let ((entry (assoc sym *cmp-global-macros*)))
    (if entry
	(rplacd entry fun)
        (push (cons sym fun) *cmp-global-macros*))
    sym))

(defmacro define-cmp-macro (sym args &rest body)
  (let* ((args (fixup-macro-args args))
	 (d (if (destructuring-arglist-p args) (gensym)))
	 (fexpr (if d
		    `(lambda (,(first args) ,(second args) &rest ,d)
		       (destructuring-bind ,(rest (rest args)) ,d ,@body))
		    `(lambda ,args ,@body))))
    `(set-cmp-macro ',sym (coerce-to-macro #',fexpr))))

#|
(defmacro define-cmp-macro (sym args &rest body)
  `(progn
     (push (cons ',sym
		 (coerce-to-macro #'(lambda ,(fixup-macro-args args) ,@body)))
	   *cmp-global-macros*)))
|#


;;;;
;;;; Specific Compiler Macros
;;;;

;; AND, OR, UNLESS, WHEN, and COND macros
(define-cmp-macro and (&rest args)
  (if args
      (let ((a (first args))
	    (rest (rest args)))
	(if rest `(if ,a ,(cmp-macroexpand `(and ,@rest))) a))
      t))

(define-cmp-macro or (&rest args)
  (if args
      (let ((a (first args))
	    (rest (rest args)))
	(if rest
	    (let ((s (gensym)))
	      `(let ((,s ,a))
		 (if ,s ,s ,(cmp-macroexpand `(or ,@rest)))))
	  a))
      nil))

(define-cmp-macro unless (test &rest body)
  `(if ,test nil (progn ,@body)))

(define-cmp-macro when (test &rest body)
  `(if ,test (progn ,@body)))

(define-cmp-macro cond (&rest clauses)
  (if clauses
      (let* ((fc (first clauses))
	     (rc (rest clauses))
	     (test (first fc))
	     (consequents (rest fc)))
	(if (eq test t)
	    (if consequents
		(if (consp (rest consequents))
		    `(progn ,@consequents)
		    (first consequents))
	        t)
	    (if consequents
		`(if ,test
		     ,(if (consp (rest consequents))
			  `(progn ,@consequents)
			  (first consequents))
		     ,(cmp-macroexpand `(cond ,@rc)))
	        (let ((tsym (gensym "T")))
		  `(let ((,tsym ,test))
		     (if ,tsym ,tsym ,(cmp-macroexpand `(cond ,@rc))))))))))


;; RETURN macro
(define-cmp-macro return (&rest vals) `(return-from nil ,@vals))


;; PROG, PROG*, PROG1, and PROG2 macros
#|
(defun split-declarations (x)
  (flet ((declaration-p (x) (and (consp x) (eq (first x) 'declare))))
    (do* ((body x (rest body))
	  (decls nil (cons head decls))
	  (head (first body) (first body)))
	 ((not (declaration-p head)) (list (reverse decls) body)))))
|#
(defun check-declarations (decls)
  (dolist (d decls)
    (dolist (i (rest d))
      (if (and (consp i) (eq (first i) 'special))
	  (dolist (v (rest i))
	    (warn "special declaration for ~s ignored." v))))))

(defun split-declarations (x)
  (flet ((head-is-declaration (x)
	   (and (consp (first x)) (eq (first (first x)) 'declare)))
	 (head-is-docstring (x) (and (stringp (first x)) (consp (rest x)))))
    (do ((decls nil)
	 (body x (rest body))
	 (doc nil))
	(nil)
	(cond
	 ((head-is-declaration body) (push (first body) decls))
	 ((head-is-docstring body) (setf doc (first body)))
	 (t (check-declarations decls)
	    #|(return (list (nreverse decls) body doc))|#
	    (return (list nil body doc))))))) ; drop declarations for now

(define-cmp-macro prog (vlist &rest body)
  (let ((db (split-declarations body)))
  `(block nil (let ,vlist ,@(first db) (tagbody ,@(second db))))))

(define-cmp-macro prog* (vlist &rest body)
  (let ((db (split-declarations body)))
  `(block nil (let* ,vlist ,@(first db) (tagbody ,@(second db))))))

(define-cmp-macro prog1 (first &rest rest)
  (let ((s (gensym)))
    `(let ((,s ,first)) ,@rest ,s)))

(define-cmp-macro prog2 (first &rest rest)
  `(progn ,first ,(cmp-macroexpand `(prog1 ,@rest))))


;; PSETQ macro
;;**** handle generalized variables like psetf?
(define-cmp-macro psetq (&rest pairs)
  (let ((gsyms nil)
	(syms nil)
	(vals nil))
    (loop
     (if (null pairs) (return))
     (push (gensym) gsyms)
     (push (first pairs) syms)
     (push (second pairs) vals)
     (setf pairs (rest (rest pairs))))
    (setf gsyms (reverse gsyms))
    (setf syms (reverse syms))
    (setf vals (reverse vals))
    (let ((bds (mapcar #'list gsyms vals))
	  (vsets (mapcar #'(lambda (x y) `(setq ,x ,y)) syms gsyms)))
      `(let* ,bds ,@vsets nil))))


;; PSETF macro
(define-cmp-macro psetf (&rest pairs)
  (let ((gsyms nil)
	(syms nil)
	(vals nil))
    (loop
     (if (null pairs) (return))
     (push (gensym) gsyms)
     (push (first pairs) syms)
     (push (second pairs) vals)
     (setf pairs (rest (rest pairs))))
    (setf gsyms (reverse gsyms))
    (setf syms (reverse syms))
    (setf vals (reverse vals))
    (let ((bds (mapcar #'list gsyms vals))
	  (vsets (mapcar #'(lambda (x y) `(setf ,x ,y)) syms gsyms)))
      `(let* ,bds ,@vsets nil))))


;; LOOP, DOTIMES, DOLIST, DO, and DO* macros
(define-cmp-macro loop (&rest body)
    (let ((start (gensym "LOOP")))
      `(block nil (tagbody ,start ,@body (go ,start)))))

(defun do-loop-body (body)
  (if (or (null body) (every #'consp body))
      body
      `((tagbody ,@body))))

;;**** does this work with declarations?
(define-cmp-macro dotimes (bds &rest body)
  (let* ((db (split-declarations body))
	 (decls (first db))
	 (b (second db))
	 (isym (first bds))
	 (nsym (gensym "N"))
	 (nval (second bds))
	 (res (third bds))
	 (test-sym (gensym "TEST"))
	 (loop-sym (gensym "LOOP")))
  `(block nil
	  (let ((,nsym ,nval)
		(,isym 0))
	    ,@decls
	    (tagbody
	     (go ,test-sym)
	     ,loop-sym
	     ,@(do-loop-body b)
	     (setf ,isym (1+ ,isym))
	     ,test-sym
	     (if (< ,isym ,nsym) (go ,loop-sym))
	     (return ,res))))))

;;**** not right if car is declared of special type?
(define-cmp-macro dolist (bds &rest body)
  (let* ((db (split-declarations body))
	 (decls (first db))
	 (b (second db))
	 (vsym (first bds))
	 (lsym (gensym "L"))
	 (lval (second bds))
	 (res (third bds))
	 (test-sym (gensym "TEST"))
	 (loop-sym (gensym "LOOP")))
    `(block nil
	    (let* ((,lsym ,lval)
		   (,vsym (car ,lsym)))
	      ,@decls
	      (tagbody
	       (go ,test-sym)
	       ,loop-sym
	       ,@(do-loop-body b)
	       (setf ,lsym (cdr ,lsym) ,vsym (car ,lsym))
	       ,test-sym
	       (if (consp ,lsym) (go ,loop-sym))
	       (return ,res))))))



(defun binding-variables (bds)
  (mapcar #'(lambda (x) (if (consp x) (first x) x)) bds))

(defun binding-values (bds)
  (mapcar #'(lambda (x) (if (consp x) (second x) nil)) bds))

(defun binding-steps (bds)
  (mapcar #'(lambda (x) (if (consp x) (rest (rest x)) nil)) bds))

(defun make-do-step-pairs (vars steps)
  (apply #'append
	 (delete nil (mapcar #'(lambda (x y) (if y (cons x y))) vars steps))))

(defun make-do-loop (letsym setsym bds tr body)
  (let* ((vars (binding-variables bds))
	 (ivals (binding-values bds))
	 (slist (make-do-step-pairs vars (binding-steps bds)))
	 (test (first tr))
	 (result (rest tr))
	 (db (split-declarations body)))
    `(block nil
	    (,letsym ,(mapcar #'list vars ivals)
	      ,@(first db)
	      (loop
	       ,(if (consp (rest result))
		    `(if ,test (return (progn ,@result)))
		    `(if ,test (return ,@result)))
	       ,@(do-loop-body (second db))
	       (,setsym ,@slist))))))

(define-cmp-macro do (bds tr &rest body)
  (make-do-loop 'let 'psetq bds tr body))

(define-cmp-macro do* (bds tr &rest body)
  (make-do-loop 'let* 'setq bds tr body))


;;SETF macro
;**** allow macro version
;**** is this needed at all?
(defmacro define-cmp-setf-method (name function)
  `(setf (get ',name 'cmp-setf-method) ',function))

(defun get-cmp-setf-method (sym) (get sym 'cmp-setf-method))
(defun get-xlisp-setf-method (sym) (get sym '*setf*))
(defun get-xlisp-setf-lambda (sym) (get sym '*setf-lambda*))


;;**** check this carefully
;;**** need to allow for adding these at file compile time
(define-cmp-macro setf (&rest pv)
  (let ((forms nil))
    (loop
     (unless pv (return))
     (let ((place (cmp-macroexpand (first pv)))
	   (val (second pv)))
       (setf pv (rest (rest pv)))
       (if (consp place)
	   (let ((cmpsetf (assoc (first place) *cmp-setf*))
		 (sfun (get-cmp-setf-method (first place)))
		 (sxfun (get-xlisp-setf-method (first place)))
		 (sxlam (get-xlisp-setf-lambda (first place))))
	     (push
	      (cond
	       (cmpsetf
		(if (eq (second cmpsetf) '*setf*)
		    `(,(third cmpsetf) ,@(rest place) ,val)
		    (apply (third cmpsetf) (append (rest place) (list val)))))
	       (sfun `(,sfun ,@(rest place) ,val))
	       ((and sxfun (symbolp sxfun)) `(,sxfun ,@(rest place) ,val))
	       (sxfun
		`(funcall (get ',(first place) '*setf*) ,@(rest place) ,val))
	       (sxlam (apply sxlam (append (rest place) (list val))))
	       (t (error "no setf method available for ~s" (first place))))
	      forms))
	   (push `(setq ,place ,val) forms))))
    (setf forms (reverse forms))
    (if (consp (rest forms))
	`(progn ,@forms)
        (first forms))))

(define-cmp-setf-method get %set-get)
(define-cmp-setf-method symbol-value %set-symbol-value)
(define-cmp-setf-method symbol-function %set-symbol-function)
(define-cmp-setf-method symbol-plist %set-symbol-plist)
;(define-cmp-setf-method car rplaca) ;**** wrong return value
;(define-cmp-setf-method cdr rplacd) ;**** wrong return value
(defsetf car (x) (v)
  (let ((vs (gensym)))
    `(let ((,vs ,v))
       (progn (rplaca ,x ,vs) ,vs))))
(defsetf cdr (x) (v)
  (let ((vs (gensym)))
    `(let ((,vs ,v))
       (progn (rplacd ,x ,vs) ,vs))))
(define-cmp-setf-method nth %set-nth)
(define-cmp-setf-method elt %set-elt)
(define-cmp-setf-method svref %set-svref)
(define-cmp-setf-method aref %set-aref)
(define-cmp-setf-method gethash %set-gethash)
;**** need more (second, etc)

(define-cmp-setf-method select set-select)
(define-cmp-setf-method slot-value slot-value)



;; DEFSTRUCT
;; This requires some runtime support functions in loadfsl.lsp
;; **** deal with :read-only slot option? 
(defun cmp-struct-slotinfo (p)
  (let ((si (get p '*struct-slots* 'none)))
    (if (eq si 'none)
	(let ((cmpinfo (assoc p *cmp-structs*)))
	  (unless cmpinfo (error "no slot info available for structure ~s" p))
	  (copy-list (cdr cmpinfo)))
        (mapcar #'first si))))

(defun check-structure-options-and-slots (options slots)
  (let ((include nil)
	(slotnames nil))
    (dolist (opt options)
      (unless (and (consp opt) (symbolp (first opt)))
	      (error "unknown option - ~s" opt))
      (let ((optargs (rest opt)))
	(case (first opt)
	  (:conc-name
	   (unless (and (consp optargs) (symbolp (first optargs)))
		   (error "expecting a symbol - ~s" optargs)))
	  (:include 
	   (when include (error "only one :INCLUDE option allowed"))
	   (unless (and (consp optargs) (symbolp (first optargs)))
		   (error "expecting a structure name - ~s" optargs))
	   (setf include (first optargs))
	   (let ((slotmods (rest optargs))
		 (slotinfo (cmp-struct-slotinfo include)))
	     (setf slotnames slotinfo)
	     (dolist (sp slotmods)
	       (let ((sn (if (consp sp) (first sp) sp)))
		 (unless (symbolp sn) (error "bad slot option ~s" sp))
		 (unless (member sn slotinfo)
			 (error "no inherited slot named ~s" sn))))))
	  (:print-function
	   (unless (and (consp optargs) (symbolp (first optargs)))
		   (error "expecting a print function - ~s" optargs)))
	  (:constructor
	   (unless (and (consp optargs) (symbolp (first optargs)))
		   (error "expecting a constructor function name - ~s"
			  optargs))
	   (when (consp (cdr optargs))
		 (error "BOA constructors not supported")))
	  (:predicate
	   (unless (and (consp optargs) (symbolp (first optargs)))
		   (error "expecting a predicate function name - ~s" optargs)))
	  (t (error "unknown option - ~s" (first opt))))))
    (dolist (opt slots)
      (unless (or (symbolp opt)
		  (and (consp opt)
		       (symbolp (first opt)))) ;;**** check for bad options?
	      (error "bad slot option - ~s" opt))
      (let ((sn (if (symbolp opt) opt (first opt))))
	(when (member sn slotnames)
	      (error "only one slot named ~s allowed" sn))
	(push sn slotnames)))))
    
(defun fixup-structure-options-and-slots (options slots)
  (let* ((include (assoc :include options))
	 (others (if include (remove include options) options)))
    (values
     (if include
	 (let ((iopt `(',(second include) :include list)))
	   (dolist (opt (rest (rest include)))
	     (let* ((sk (if (symbolp opt) (list opt nil) opt))
		    (sn (first sk))
		    (sd (second sk)))
	       (push `(list ',sn ,(if sd `(list #'(lambda () ,sd)) nil))
		     iopt)))
	   (reverse iopt)))
     others
     (if slots
	 (let ((sopts '(list)))
	   (dolist (opt slots)
	     (let* ((sk (if (symbolp opt) (list opt nil) opt))
		    (sn (first sk))
		    (sd (second sk)))
	       (push `(list ',sn ,(if sd `(list #'(lambda () ,sd)) nil))
		     sopts)))
	   (reverse sopts))))))

(defun cmp-register-structure (structname options slots)
  (let* ((include (second (assoc :include options)))
	 (incslots (if include (cmp-struct-slotinfo include)))
	 (slotinfo (append incslots (mapcar #'(lambda (x)
						(if (symbolp x) x (first x)))
					    slots)))
	 (slotcount 1)
	 (conc-name-opt (find :conc-name options :key #'first :from-end t))
	 (conc-name (if conc-name-opt
			(if (second conc-name-opt)
			    (symbol-name (second conc-name-opt))
			    "")
		        (concatenate 'string (symbol-name structname) "-"))))
    (push (cons structname slotinfo) *cmp-structs*)
    (dolist (s slotinfo)
      (let* ((sname (symbol-name s))
	     (setf-name (intern (concatenate 'string conc-name sname))))
	(push (list setf-name
		    '*setf-lambda*
		    (let ((i slotcount))
		      #'(lambda (x v) `(xlisp::%struct-set ,x ,i ,v))))
	      *cmp-setf*)
	(incf slotcount)))))
    
(define-cmp-macro defstruct (structspec &rest slots)
  (let ((structname (if (consp structspec) (first structspec) structspec))
	(options (if (consp structspec) (rest structspec))))
    (unless (symbolp structname) (error "bad structure name - ~s" structname))
    (check-structure-options-and-slots options slots)
    (multiple-value-bind (incopt others slotarg)
			 (fixup-structure-options-and-slots options slots)
      `(progn
	 (eval-when (:compile-toplevel)
		    (cmp-register-structure ',structname ',options ',slots))
	 (cmp-do-defstruct ',structname ,incopt ',others ,slotarg)))))

;; PROGV
#|
(define-cmp-macro progv (vars vals &rest body)
  (let ((ovalsym (gensym "D"))
	(varsym (gensym "V"))
	(valsym (gensym "V")))
    `(let* ((,varsym ,vars)
	    (,valsym ,vals)
	    (,ovalsym (mapcar #'dynamic-value ,varsym)))
       (mapc #'set ,varsym ,valsym)
       (unwind-protect (progn ,@body)
	 (mapc #'set ,varsym ,ovalsym)))))
|#
(define-cmp-macro progv (syms vals &rest body)
  `(%dynamic-bind ,syms ,vals #'(lambda () ,@body)))

;; IN-PACKAGE
(define-cmp-macro in-package (name)
  `(eval-when (:compile-toplevel :load-toplevel :execute)
	      (setf *package* (find-package ',name))))



;;;;
;;;; DEFUN and DEFMACRO
;;;;

(define-cmp-macro defun (f args &rest body)
  (let* ((db (split-declarations body))
	 (decls (first db))
	 (b (second db))
	 (doc (third db))
	 (fexpr `#'(lambda ,args ,@decls (block ,f ,@b))))
    `(progn
       (%set-symbol-function ',f ,fexpr)
       ,@(if doc `((%set-get ',f 'function-documentation ,doc))))))

(define-cmp-macro defmacro (f args &rest body)
  (let* ((db (split-declarations body))
	 (decls (first db))
	 (b (second db))
	 (doc (third db))
	 (args (fixup-macro-args args))
	 (body `(,@decls (block ,f ,@body)))
	 (d (if (destructuring-arglist-p args) (gensym)))
	 (fexpr (if d
		    `(lambda (,(first args) ,(second args) &rest ,d)
		       (destructuring-bind ,(rest (rest args)) ,d ,@body))
		    `(lambda ,args ,@body))))
    `(progn
       (eval-when (:compile-toplevel)
		  (push (cons ',f (coerce-to-macro ,fexpr)) *cmp-macros*))
       (%set-symbol-function ',f (coerce-to-macro ,fexpr))
       ,@(if doc `((%set-get ',f 'function-documentation ,doc))))))


;;;;
;;;; DEFMETH and DEFPROTO
;;;;

(define-cmp-macro defmeth (ob sym args &rest body)
  (let* ((db (split-declarations body))
	 (decls (first db))
	 (b (second db))
	 (doc (third db))
	 (fexpr `#'(lambda ,(cons 'self args) ,@decls (block ,sym ,@b)))
	 (obsym (gensym "O")))
    `(progn
       (xlisp::add-method ,ob ,sym ,fexpr ,doc))))

(define-cmp-macro defproto (name &optional ivars cvars parents doc)
  (let ((pp (gensym))
	(p (gensym))
	(d (gensym)))
    `(let* ((,pp ,parents)
	    (,p (apply #'make-object (if (listp ,pp) ,pp (list ,pp))))
	    (,d ,doc))
       (dolist (s ,cvars) (send ,p :add-slot s))
       (send ,p :make-prototype ',name ,ivars)
       (if ,d (send ,p :documentation 'proto ,d))
       (set ',name ,p)
       ',name)))

;;;;
;;;; DEFVAR, DEFPARAMETER and DEFCONSTANT
;;;;

(define-cmp-macro defvar (var &optional (value nil have-val) doc)
  `(progn 
     (eval-when (:compile-toplevel)
		(pushnew ',var *cmp-specials*))
     (mark-as-special ',var)
     ,@(if have-val `((unless (boundp ',var) (set ',var ,value))))
     ,@(if doc `((%set-get ',var 'variable-documentation ,doc)))
     ',var))

(define-cmp-macro defparameter (var value &optional doc)
  `(progn
     (eval-when (:compile-toplevel)
		(pushnew ',var *cmp-specials*))
     (mark-as-special ',var)
     (set ',var ,value)
     ,@(if doc `((%set-get ',var 'variable-documentation ,doc)))
     ',var))

;;**** is this the best way?
(define-cmp-macro defconstant (var value &optional doc)
  `(eval-when (:compile-toplevel :load-toplevel :execute)
	      (mark-as-special ',var t ,value)
	      ,@(if doc `((%set-get ',var 'variable-documentation ,doc)))
	      ',var))


;;;;
;;;; Multiple value macros
;;;;

(define-cmp-macro multiple-value-setq (variables form)
  (let* ((tvars (mapcar #'(lambda (x) (gensym "V")) variables))
	 (pairs nil))
    (mapc #'(lambda (x y) (push y pairs) (push x pairs)) variables tvars)
    `(multiple-value-bind ,tvars ,form (setq ,@pairs))))

(define-cmp-macro multiple-value-list (form)
  `(multiple-value-call #'list ,form))

#|
;;**** this could be done more efficiently -- internalize
(define-cmp-macro multiple-value-prog1 (first &rest rest)
  (let ((s (gensym)))
    `(let ((,s (multiple-value-list ,first))) ,@rest (values-list ,s))))
|#

;;**** need internal version of nth-value
#|
(define-cmp-macro nth-value (n form)
  `(nth ,n (multiple-value-list ,form)))

(define-cmp-macro nth-value (n e)
  `(%nth-value ,n (%mv-collect ,e)))
|#


;;;;
;;;; TIME
;;;;

(define-cmp-macro time (expr)
  (let ((rtsym (gensym "RUN"))
	(gtsym (gensym "GC")))
    `(let ((,rtsym (get-internal-run-time))
	   (,gtsym(get-internal-gc-time)))
       (multiple-value-prog1
	,expr
	(format t "The evaluation took ~,2f seconds; ~,2f seconds in GC.~%"
		(float (/ (- (get-internal-run-time) ,rtsym)
			  internal-time-units-per-second))
		(float (/ (- (get-internal-gc-time) ,gtsym)
			  internal-time-units-per-second)))))))


;;;;
;;;; TRACE and UNTRACE
;;;;

(define-cmp-macro trace (&rest args)
  `(dolist (s ',args *tracelist*)
     (unless (symbolp s) (error "not a symbol - ~s" s))
     (pushnew s *tracelist*)))

(define-cmp-macro untrace (&rest args)
  (if args
      `(dolist (s ',args *tracelist*)
         (unless (symbolp s) (error "not a symbol - ~s" s))
	 (delete s *tracelist*))
      `(setf *tracelist* nil)))


;;;;
;;;; THE
;;;;

;;**** should maybe do error checking for some optimize settings
(define-cmp-macro the (a b) b)


;;;;
;;;; Macros for inlining some functions
;;;; ******* more needed here -- should these be here or as symbol-call-rules??

(define-cmp-macro not (x) `(if ,x nil t))
(define-cmp-macro null (x) `(if ,x nil t))

(define-cmp-macro row-major-aref (x i) `(aref ,x ,i))
