;;; -*- Mode: LISP;  Syntax: COMMON-LISP; Package: (*SIM-I COMMON-LISP-GLOBAL); Base: 10; Muser: yes -*-

(in-package '*sim-i :use '(lisp))

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;;
;;; This file contains code which is implementation dependent.
;;; Aside from declarations, there should be no implementation dependent
;;; code anywhere in the Simulator besides here and in the
;;; specifications.lisp and the definitions.lisp and pref-setf.lisp files. (Ha!)
;;;

;;; Lispm's go very fast when vectors are declared as
;;; (sys:array-register ...).

(defmacro 1-d-array-declaration (&rest body)
  #+SYMBOLICS
  `(declare (sys:array-register ,@body))
  #-SYMBOLICS
  `(declare (type simple-vector ,@body))
  )

(defmacro bit-vector-array-declaration (&rest body)
  #+SYMBOLICS
  `(declare (sys:array-register ,@body))
  #-SYMBOLICS
  `(declare (type bit-vector ,@body))
  )

(defmacro declare-arglist (argument-list)
  #+SYMBOLICS
  `(declare (sys:arglist ,@argument-list))
  #+LUCID
  `(declare (sys::arglist ,argument-list))
  #-(OR SYMBOLICS LUCID)
  (declare (ignore argument-list))
  #-(OR SYMBOLICS LUCID)
  nil
  )


;;;
;;; This will fill ARRAY with VALUE.  This gets called quite a bit, so the
;;; idea is to do it as fast as possible.  It should be possible to handle
;;; boolean arrays much faster, but I don't know how on the 3600.
;;;
;;; This is in this file because theoretically the optimal implementation
;;; of this function is very machine and Lisp implementation dependent.
;;; I have not tried, however, to optimize it for anything other than SCL.
;;;





;;;
;;; This macro will iterate over the BODY with SYMBOL bound to selected processors.
;;;

(defmacro do-for-selected-processors ((symbol) &body body)

  #+symbolics
  `(do-for-selected-processors-with-trapping (,symbol) ., body)

  #-symbolics
  (let ((css-symbol (gensym)))
    `(let ((,css-symbol *css*))
       (declare (type (vector bit) ,css-symbol))
       (dotimes (,symbol *number-of-processors-limit*)
	 (declare (type fixnum ,symbol))
	 (when (eql 1 (sbit ,css-symbol ,symbol))
	   ,@body
	   )))))

;;;
;;; DO-FOR-SELECTED-PROCESSORS will turn into this ...-WITH-TRAPPING macro
;;; when running on a 3600.  In time, we'll put more error information
;;; printing.
;;;

#+SYMBOLICS
(DEFMACRO DO-FOR-SELECTED-PROCESSORS-WITH-TRAPPING ((SYMBOL) &BODY BODY)
  (LET ((CSS-SYMBOL (GENSYM "CSS-"))
	(ERROR-SYMBOL (GENSYM "ERROR-"))
	(EXIT-VALUE-SYMBOL (GENSYM "EXIT-")))
    `(LET ((,CSS-SYMBOL *CSS*)
	   (,SYMBOL -1))
       (DECLARE (SYS:ARRAY-REGISTER ,CSS-SYMBOL))

       (ZL::CONDITION-CASE (,ERROR-SYMBOL)

	   (DO ((,EXIT-VALUE-SYMBOL *NUMBER-OF-PROCESSORS-LIMIT*)) 
	       ((= (INCF ,SYMBOL) ,EXIT-VALUE-SYMBOL))
	     
	     (WHEN (EQL 1 (SBIT ,CSS-SYMBOL ,SYMBOL))
	       ., BODY))
	     
	 (ZETALISP-SYSTEM:ERROR		  ;catch any error at all
	   (ZL::SIGNAL 'DO-FOR-SELECTED-PROCESSORS-WITH-TRAPPING-ERROR
		      :PROCESSOR ,SYMBOL 
		      :ERROR-OBJECT ,ERROR-SYMBOL
		      ))

	 ))))

#+SYMBOLICS
(ZL:DEFFLAVOR DO-FOR-SELECTED-PROCESSORS-WITH-TRAPPING-ERROR
	((PROCESSOR NIL) (ARG NIL) (ERROR-OBJECT NIL))
	(ZETALISP-SYSTEM:ERROR)
  :INITABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  )

#+SYMBOLICS
(ZL::DEFMETHOD (SYS::PROCEED DO-FOR-SELECTED-PROCESSORS-WITH-TRAPPING-ERROR :NO-ACTION) ()
  #.(ZL:STRING "Proceed without any special action")
  ':NO-ACTION
  )

#+SYMBOLICS
(ZL::DEFMETHOD (DBG::REPORT DO-FOR-SELECTED-PROCESSORS-WITH-TRAPPING-ERROR) (STREAM)
  (LET ((ZL:PRINLENGTH NIL) (ZL:PRINLEVEL NIL))
    ZL:PRINLENGTH ZL:PRINLEVEL 
    (FORMAT STREAM "Had an error while operating on processor #~d.~@
                    Here is the actual error:~@
                    ~a"
	    PROCESSOR ERROR-OBJECT
	    )))


;;; We need a portable way to catch errors.  Of course, since Common Lisp has
;;; no such thing, the implementation is going to be non-portable but it will
;;; be confined (hopefully) to this macro and auxiliary functions.


#+VAXLISP

(progn

  (defvar *unique-value-to-throw* nil)
  (defvar *starlisp-error-trapping-catch-tag* 'xyzzy)

  (defun starlisp-error-trapping-error-handler
	 (function-name error-signalling-function &rest args)
    (when (or (eq error-signalling-function 'error)
	      (eq error-signalling-function 'cerror))
      (throw *starlisp-error-trapping-catch-tag* *unique-value-to-throw*)
     )
    (apply #'univeral-error-handler
	   function-name error-signalling-function args)
   )

 )


#+LUCID

;;; IGNORE-ERRORS evaluates BODY and returns two values, the first is the 
;;; result of evaluating the last form of BODY and the second is nil if no
;;; error has occured and T if an error has occured while evaluating
;;; the forms of body.  If the second return is T then the first return is
;;; NIL.  Written by Dan Aronson

;;; WARNING!!! this is a complete hack and is in no way a supported feature
;;; of Lucid Common Lisp.

(defmacro ignore-errors (&body body)
  `(let ((output-stream (make-string-output-stream))
	 result string
	 (tag (gentemp))
	 )
    (let ((*debug-io* (make-two-way-stream
		       (make-string-input-stream 
			(format nil "~%/(throw '~a nil)~%" tag))
		       ;;*debug-io*
		       output-stream
		       )))
      (catch tag (setq result (progn ,@body))))
    (setq string (get-output-stream-string output-stream))
    (cond ((string-equal "" string)
	   (values result nil))
	  (t (clear-input *terminal-io*)
	     (values nil t))
	)))


#+KCL

(progn

  (defvar *unique-value-to-throw* nil)
  (defvar *starlisp-error-trapping-catch-tag* 'xyzzy)

 )


(defmacro with-all-errors-trapped (protected-form on-error-form)
  
  "Evaluates protected-form and returns the result of that evaluation
   (the form must return exactly one result).  If an error is signalled,
   however, on-error-form is evaluated and the result of evaluating that
   form is returned.
  "

  #+SYMBOLICS

  (let ((error-detected-symbol (gensym)) (result-symbol (gensym)))
    `(let* ((,error-detected-symbol nil)
	    (,result-symbol
	      (zl:condition-case ()
		  ,protected-form
		(zetalisp-system:error (setq ,error-detected-symbol t)))))
       (if (not ,error-detected-symbol)
	   ,result-symbol
	   ,on-error-form
	 )))

  #+LUCID

  (let ((error-detected-symbol (gensym)) (result-symbol (gensym)))
    `(multiple-value-bind (,result-symbol ,error-detected-symbol)
	 (ignore-errors ,protected-form)
       (if (null ,error-detected-symbol) ,result-symbol ,on-error-form)
      ))

;  #+ULTRIX
;
;  (let ((throw-value (gensym)) (result-symbol (gensym)))
;    `(let ((*universal-error-handler* #'starlisp-error-trapping-error-handler))
;       (setq *unique-value-to-throw* ',throw-value)
;       (let ((,result-symbol
;	       (catch *starlisp-error-trapping-catch-tag* ,protected-form)))
;	 (if (not (eq ,result-symbol *unique-value-to-throw*))
;	     ,result-symbol
;	     ,on-error-form
;	  ))))

  #+(AND VMS DEC COMMON VAX)

  protected-form

  #+KCL

  protected-form

  #+CCL
  (let ((result (gensym)) (flag (gensym)))
    `(multiple-value-bind (,result ,flag)
	 (ccl::catch-error-quietly ,protected-form)
       (if ,flag ,on-error-form ,result)
       ))

  #+(OR :EXCL :ALLEGRO)
  (let ((protected-form-value-symbol (gensym "PROTECTED-FORM-VALUE-")))
    `(let ((,protected-form-value-symbol))
       (if (not (excl:errorset (setq ,protected-form-value-symbol ,protected-form)))
	   ,on-error-form
	   ,protected-form-value-symbol
	)))

   #+CMU
   (let ((error-detected-symbol (gensym))
	 (result-symbol (gensym)))
     `(let ((,error-detected-symbol t)
	    (,result-symbol nil))
        (ext:ignore-errors
	  (setq ,result-symbol ,protected-form)
	  (setq ,error-detected-symbol nil))
	(if (not ,error-detected-symbol)
	    ,result-symbol
	    ,on-error-form
	    )))

 )


;;; Allow C-SH-A to work for *DEFUN functions on Lisp Machines
;;; and Lucid using TMC Gmacs hacks


(defun default-arglist-declaration-from-arglist (arglist)
  "Return a declaration form for the arg list of this function"
  #+SYMBOLICS
  `(declare (si:arglist ,@arglist))
  #+LCL3.0
  `(declare (sys::arglist ,arglist))
  #-(OR SYMBOLICS LCL3.0)
  (declare (ignore arglist))
  #-(OR SYMBOLICS LCL3.0)
  nil
  )

(defun hack-declarations-for-symbolics (declarations arglist)
  #-(OR SYMBOLICS LCL3.0)
  (declare (ignore arglist))
  #-(OR SYMBOLICS LCL3.0)
  declarations
  #+SYMBOLICS
  (append declarations
	  (and (not (find 'si:arglist declarations :key #'caadr))
	       (list (default-arglist-declaration-from-arglist arglist))
	       ))
  #+LCL3.0
  (append declarations
	  (and (not (find 'sys::arglist declarations :key #'caadr))
	       (list (default-arglist-declaration-from-arglist arglist))
	       ))
 )

(defmacro argument-list-declaration (&rest arglist)
  #+SYMBOLICS
  `(declare (zl:arglist ,@arglist))
  #+LCL3.0
  `(declare (sys::arglist ,arglist))
  #-(OR SYMBOLICS LCL3.0)
  (declare (ignore arglist))
  #-(OR SYMBOLICS LCL3.0)
  nil
 )


