;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *SIM-i; MUSER: YES-*-


;;; Author.  JP Massar.


(in-package '*SIM-i)


;;; This macro provides an iteration form over all active geometries.

(defmacro do-for-active-geometries ((geometry) &body body)
  (assert (symbolp geometry))
  (let ((geometry-id-symbol (gensym "GEOMETRY-ID-")))
    `(do ((,geometry-id-symbol *first-legal-geometry-id* (1+ ,geometry-id-symbol)))
	 ((= ,geometry-id-symbol *maximum-geometries-allowed*))
       (let ((,geometry (aref *all-geometries-array* ,geometry-id-symbol)))
	 (when ,geometry
	   ,@body
	   )))))


;;; This reinitializes the geometry array and resets *next-geometry-id*.
;;; It must be called at *cold-boot time.  All previously created geometries
;;; are flushed.

(defun reinitialize-*lisp-geometries ()
  (dotimes (j *maximum-geometries-allowed*) (setf (aref *all-geometries-array* j) nil))
  (setq *next-geometry-id* *first-legal-geometry-id*)
  (setf (aref *all-geometries-array* *illegal-geometry-id*) nil)
  )


;;; Allocation and deallocation of geometries.


(defun make-geometry-at-geometry-array-slot (slot)
  (let ((new-geometry (make-geometry :id slot)))
    (setf (aref *all-geometries-array* slot) new-geometry)
    new-geometry
    ))

(defun allocate-next-geometry ()
  (when (eql *next-geometry-id* *maximum-geometries-allowed*)
    (do ((j *first-legal-geometry-id* (1+ j)))
	((eql j *maximum-geometries-allowed*))
      (when (null (aref *all-geometries-array* j))
	(return-from allocate-next-geometry (make-geometry-at-geometry-array-slot j))
	))
    (error "Error.  You have run out of geometries.  You are probably doing something wrong.  Have ~
            a *Lisp expert take a look at what you are doing.
           "
	   ))
  (incf *next-geometry-id*)
  (make-geometry-at-geometry-array-slot *next-geometry-id*)
  )

(defun deallocate-geometry (geometry)
  (let ((id (geometry-id geometry)))
    (setf (aref *all-geometries-array* id) nil)
    (setf (geometry-id geometry) *illegal-geometry-id*)
    (setf (geometry-dimensions geometry) "This geometry has been deallocated.  You should not be accessing it.")
    (deallocate-paris-geometry geometry)
    nil
    ))


(defun geometry-from-id (id)
  (let ((geometry (aref *all-geometries-array* id)))
    (when (not (geometry-p geometry))
      (error "Possible internal error.  No geometry defined for geometry id ~D" id)
      )
    geometry
    ))


(defun *cold-boot-for-geometries () (reinitialize-*lisp-geometries))
  
