;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *SIM-i; MUSER: YES-*-

(in-package '*sim-i :use '(lisp))

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;; Geometries are stored in an array of geometries, *maximum-geometries-allowed* long.
;;; At *cold-boot, all geometries are flushed and the array is initialized to NIL everywhere.
;;; Geometries can be allocated and deallocated.  Geometries are allocated sequentially
;;; until *maximum-geometries-allowed* geometries are allocated.  If none have been
;;; deallocated and *maximum-geometries-allowed* geometries have been allocated, then
;;; attempting to allocate another one will result in an error.  Otherwise the array
;;; is searched for a deallocated geometry slot.

;;; The index into the array where a particular geometry is stored in known as
;;; a geometry-id.  The geometry-id is used within *Lisp to uniquely identify
;;; geometries and to refer to them.


(defstruct geometry
  id
  rank
  dimensions
  dimension-offsets
  dimension-subsizes
  address-length
  grid-address-lengths
  spare1
  )


(eval-when (compile load eval)
  (#-KCL defconstant #+KCL defparameter *illegal-geometry-id* 0)
  (#-KCL defconstant #+KCL defparameter *first-legal-geometry-id* 1)
  (defvar *maximum-geometries-allowed* 64)
  (defvar *next-geometry-id* *first-legal-geometry-id*)
  (defvar *all-geometries-array* nil
    "An array which, at index j, contains the geometry defined
     by geometry-id j.
    "
    )
  (setq *all-geometries-array* (make-array *maximum-geometries-allowed* :initial-element nil))
  )


(defmacro vp-set-geometry-dimensions (vp-set) `(geometry-dimensions (vp-set-geometry ,vp-set)))
(defmacro vp-set-geometry-rank (vp-set) `(geometry-rank (vp-set-geometry ,vp-set)))
(defmacro vp-set-geometry-id (vp-set) `(geometry-id (vp-set-geometry ,vp-set)))

(defun vp-set-address-length (vp-set) (geometry-address-length (vp-set-geometry vp-set)))
(defun vp-set-grid-address-lengths (vp-set) (geometry-grid-address-lengths (vp-set-geometry vp-set)))
(defun vp-set-grid-address-length (vp-set n) (nth n (vp-set-grid-address-lengths vp-set)))

;;; This reinitializes the geometry array and resets *next-geometry-id*.
;;; It must be called at *cold-boot time.  All previously created geometries
;;; are flushed.



;;; Allocation and deallocation of geometries.



(defun geometry-dimension (geometry n) (elt (geometry-dimensions geometry) n))
(defun geometry-dimension-offset (geometry n) (elt (geometry-dimension-offsets geometry) n))



(defun indices-from-row-major-index (row-major-index geometry)
  (let ((dimension-subsizes (geometry-dimension-subsizes geometry)))
    (let ((indices nil))
      (dolist (subsize dimension-subsizes)
	(multiple-value-bind (dividend remainder)
	    (truncate row-major-index subsize)
	  (push dividend indices)
	  (setq row-major-index remainder)
	  ))
      (setq indices (nreverse indices))
      indices
      )))

(defun grid-index-from-row-major-index (row-major-index geometry dimension)
  (let ((dimension-subsizes (geometry-dimension-subsizes geometry)))
    (do ((dimension-index 0 (1+ dimension-index))
	 (subsizes dimension-subsizes (cdr subsizes))
	 )
	((null subsizes))
      (multiple-value-bind (dividend remainder)
	  (truncate row-major-index (car subsizes))
	(when (= dimension dimension-index)
	  (return-from grid-index-from-row-major-index dividend)
	  )
	(setq row-major-index remainder)
	))
    (error "You can't get here")
    ))


(defun dimension-subsizes-from-dimensions (dimensions)
  (let ((dimension-subsizes nil))
    (dotimes (j (length dimensions))
      (push (reduce #'* (nthcdr (1+ j) dimensions)) dimension-subsizes)
      )
    (setq dimension-subsizes (nreverse dimension-subsizes))
    dimension-subsizes
    ))


(defun define-dimensions-and-offsets-from-coordinates (coordinates)
  (if (every #'non-negative-integer-p coordinates)
      (values (mapcar '1+ coordinates) nil)
      (let ((dimensions nil) (offsets nil))
	(dolist (coordinate coordinates)
	  (if (non-negative-integer-p coordinate)
	      (progn
		(push (1+ coordinate) dimensions)
		(push 0 offsets)
		)
	      (progn
		(push (1+ (abs coordinate)) dimensions)
		(push (abs coordinate) offsets)
		)))
	(values (nreverse dimensions) (nreverse offsets))
	)))


(defun create-geometry

       (&key dimensions weights on-chip-bits on-chip-pos off-chip-bits off-chip-pos ordering)

  (declare (ignore weights on-chip-bits on-chip-pos off-chip-bits off-chip-pos ordering))

  (assert (every 'non-negative-integer-p dimensions) () "The dimensions are not all non-negative integers")

  (let ((new-geometry (allocate-next-geometry)))
    (setf (geometry-rank new-geometry) (length dimensions))
    (setf (geometry-dimensions new-geometry) dimensions)
    (setf (geometry-dimension-subsizes new-geometry) (dimension-subsizes-from-dimensions dimensions))
    (setf (geometry-address-length new-geometry) (n-bits-for-address (apply '* dimensions)))
    (setf (geometry-grid-address-lengths new-geometry) (mapcar 'n-bits-for-address dimensions))
    new-geometry
    ))


(defun make-geometry-that-can-hold-coordinates (coordinates)

  ;; Creates a new geometry that can be indexed into
  ;; successfully by the coordinates.

  (multiple-value-bind (dimensions offsets)
      (define-dimensions-and-offsets-from-coordinates coordinates)
    (let ((new-geometry (create-geometry :dimensions dimensions)))
      (setf (geometry-dimension-offsets new-geometry) offsets)
      new-geometry
      )))

(defun internal-deallocate-geometry (geometry)
  (setf (geometry-id geometry) *illegal-geometry-id*)
  (setf (geometry-dimensions geometry) "This geometry has been deallocated.  You should not be accessing it.")
  )

(defun deallocate-paris-geometry (geometry)
  (declare (ignore geometry))
  nil
  )
