;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: (*LISP-I COMMON-LISP-GLOBAL); Muser: yes -*-

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> Copyright 1986 Thinking Machines Corporation, Inc. of Cambridge, Massachusetts.
;;;> 
;;;> Permission is hereby granted to copy this source onto any machine at a site
;;;> which legitimately has this software, and to execute the resulting object
;;;> code on any machine at a site which legitimately has this software.

;;;> Permission is hereby granted to make such changes as are necessary to port this
;;;> source to a version of Common Lisp running on any machine so long as said changes
;;;> are sent back to Thinking Machines so that they may be incorporated in future
;;;> releases.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com

;;;> The *Lisp Simulator was written by JP Massar.
;;;> The *Lisp language was designed by Cliff Lasser and Steve Omohundro, with
;;;> help from may others at Thinking Machines Corporation.

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+



(in-package '*lisp-i)


;;; This file contains hardware/simulator independent code implementing
;;; the debugging functions ppp, ppp-css and other miscellaneous useful
;;; printing functions.


(defun pref-or-junk (pvar processor)
  (let ((value (pref pvar processor)))
    (if (not (eq value *junk-to-initialize-with*))
	value
	*value-to-print-when-junk-found*
     )))

(defun pref-grid-or-junk (pvar &rest addresses)
  (pref-or-junk pvar (apply #'cube-from-grid-address addresses))
  )


(defun active-processor-list ()
  (let ((list nil))
   (do-for-selected-processors (j) (push j list))
   (nreverse list)
  ))


; Make the arg list visible to c-s-A to make it easier to use - JRD

(defmacro ppp (pvar &rest keyargs)
  #+SYMBOLICS (declare (si:arglist pvar &key
			       (mode *PPP-DEFAULT-MODE*)
			       (format *PPP-DEFAULT-FORMAT*)
			       (per-line *PPP-DEFAULT-PER-LINE*)
			       (title *PPP-DEFAULT-TITLE*)
			       (start *PPP-DEFAULT-START*)
			       (end *PPP-DEFAULT-END*)
			       (grid-start *PPP-DEFAULT-GRID-START*)
			       (grid-end *PPP-DEFAULT-GRID-END*)
			       (ordering *PPP-DEFAULT-ORDERING*)
			       (processor-list *PPP-DEFAULT-PROCESSOR-LIST*)
			 ))
  `(progn (with-css-saved (ppp-internal ,pvar ,@keyargs)) (values))
 )


(defmacro pretty-print-pvar (&rest args) `(ppp ,@args))


(defun ppp-internal (
       pvar
       &rest all-key-args
       &key
       (mode *PPP-DEFAULT-MODE*)
       (format *PPP-DEFAULT-FORMAT*)
       (per-line *PPP-DEFAULT-PER-LINE*)
       (title *PPP-DEFAULT-TITLE*)
       (start *PPP-DEFAULT-START* start-provided)
       (end *PPP-DEFAULT-END* end-provided)
       (grid-start *PPP-DEFAULT-GRID-START* grid-start-provided)
       (grid-end *PPP-DEFAULT-GRID-END* grid-end-provided)
       (ordering *PPP-DEFAULT-ORDERING*)
       (processor-list *PPP-DEFAULT-PROCESSOR-LIST*)
      )

  ;; pretty print all the component values of a pvar.
  ;; The user can specify a format for printing, whether
  ;; to print in grid or cube order, and how many values
  ;; to print per line.  If the user is using grid
  ;; addressing he may also specify in which order
  ;; the dimensions are to be printed out
  ;; (for > 2 dimensions).  If the user is using cube
  ;; addressing he can specify a range of processors to
  ;; be printed out.

  (flet ((argument-error-test (condition format-string &rest format-args)
	   (when (not condition)
	     (apply #'format *error-output* format-string format-args)
	     (return-from ppp-internal nil)
	    ))
	 (integer-range-test (x low high) (and (integerp x) (>= x low) (< x high)))
	)

  (terpri)

  (argument-error-test (pvarp pvar) "Not a pvar: ~A~%" pvar)
  (argument-error-test (or (eq mode :cube) (eq mode :grid)) "Bad mode keyword: ~A~%" mode)
  (argument-error-test (stringp format) "Format keyword value is not a string: ~S~%" format)
  (argument-error-test
    (or (null per-line) (integer-range-test per-line 1 1000000))
    "Per-line keyword value not an integer: ~A~%" per-line)
  (argument-error-test
    (or (null title) (stringp title)) "Title keyword value not a string: ~A~%" title)
    
  ;; This is complicated.  If we are using cube mode then use the values of start
  ;; and end unless they were not provided and they were lists, in which case
  ;; someone set them to lists using the default values thinking grid mode.

  ;; If we are in grid mode, if start or end was provided as a list we use that
  ;; unless grid-start and grid-end were provided.

  (if (eql mode :cube)

      (progn
	(if (and (listp start) (not start-provided))
	    (setq start 0)
	    (argument-error-test
	      (integer-range-test start 0 *number-of-processors-limit*)
	      "Invalid start keyword value: ~A~%" start
	     ))
	(if (and (listp end) (not end-provided))
	    (setq end *number-of-processors-limit*)
	    (argument-error-test
	      (integer-range-test end (1+ start) (1+ *number-of-processors-limit*))
	      "Invalid end keyword value: ~A~%" end
	     )))

      ;; grid mode.

      (progn
	(flet ((check-grid-coordinate (x low limit symbol)
		 (argument-error-test
		   (integer-range-test x low limit)
		   "Bad grid coorindate for ~A: ~A~%" symbol x)
		))
	  (when (and (listp start) (not grid-start-provided)) (setq grid-start start))
	  (when (and (listp end) (not grid-end-provided)) (setq grid-end end))
	  (mapc
	    #'(lambda (start limit) (check-grid-coordinate start 0 limit 'grid-start))
	    grid-start
	    *current-cm-configuration*
	   )
	  (mapc
	    #'(lambda (start end limit)
		(check-grid-coordinate end (1+ start) (1+ limit) 'grid-end))
	    grid-start
	    grid-end
	    *current-cm-configuration*
	   )
	 )))

  ;; If there is an ordering given, we assume grid mode.
  ;; The ordering is just a list of dimensions
  ;; The dimensions are numbered from 0 up to (1- *number-of-dimensions*)
  ;; which is the default dimension ordering.
  ;; The ordering controls the iteration order over dimensions.
  ;; we ignore ordering for 2-dimensions, making our life easier.

  (when (> *number-of-dimensions* 2)
    (let ((default-ordering nil))
      (dotimes (j *number-of-dimensions*) (push j default-ordering))
      (setq default-ordering (nreverse default-ordering))
      (if (null ordering)
	  (setq ordering default-ordering)
	  (if (null (equal default-ordering (sort (copy-list ordering) #'<)))
	      (format *error-output* "Bad ordering: ~S~%" ordering)
	      (setq mode :grid)
	   ))
     ))

  (argument-error-test
    (every #'(lambda (n) (integer-range-test n start end)) processor-list)
    "Processor list contains invalid processor number: ~A~%" processor-list)

  ;; start printing!

  (and title (if per-line (format t "~A:~%" title) (format t "~A: " title)))

  ;; print using cube addressing

  (when (eq mode :cube)
    (let ((print-it nil)
	  (count 0)
	 )
      (dotimes (j *number-of-processors-limit*)
	(if processor-list
	    (setq print-it (member j processor-list))
	    (setq print-it (and (>= j start) (< j end)))
	 )
	(if print-it
	    (format t format (pref-or-junk pvar j)))
	(when per-line
	  (when (and (eql (incf count) per-line) (>= j start) (< j end))
	    (setq count 0)
	    (terpri)
	   )))))

  ;; print using grid addressing, special casing
  ;; the case where there are only 2 dimensions

  (when (eq mode :grid)
    (when (eql *number-of-dimensions* 2)
      (format t "~%     DIMENSION 0   ----->~%~%")
      (do ((y (nth 1 grid-start) (1+ y)))
	  ((= y (nth 1 grid-end)))
	(do ((x (nth 0 grid-start) (1+ x)))
	    ((= x (nth 0 grid-end)))
	  (format t format (pref-grid-or-junk pvar x y))
	 )
	(terpri)
       ))
    (when (> *number-of-dimensions* 2)
      (print-hypergrid ordering pvar format)
     )
   )

  (values)

 ))


;;; auxiliary routines to display pvar values in more than
;;; 2 dimensions.  2-dimensional slices are printed out
;;; successively, with the coordinates of the slice noted
;;; above the grid.

(defun print-hypergrid (ordering pvar format)
  (let* ((ordering-reversed (reverse ordering))
	 (last-dimension (pop ordering-reversed))
	 (second-to-last-dimension (pop ordering-reversed))
	 (dimension-ordering nil)
	)
    (dolist (dimension ordering-reversed)
      (push (list dimension nil) dimension-ordering)
     )
    (print-hypergrid-aux
      0
      dimension-ordering
      second-to-last-dimension
      last-dimension
      pvar
      format
     )))


(defun print-hypergrid-aux (
       current-dimension-index
       dimension-ordering
       second-to-last-dimension
       last-dimension
       pvar
       format
      )
  (let* ((current-dimension-info
	   (nth current-dimension-index dimension-ordering))
	 (current-dimension (car current-dimension-info))
	)
    (dotimes (j (nth current-dimension *current-cm-configuration*))
      (setf (cadr current-dimension-info) j)
      (if (eql current-dimension-index (1- (length dimension-ordering)))
	  (print-subgrid
	    pvar
	    format
	    second-to-last-dimension
	    last-dimension
	    dimension-ordering
	   )
	  (print-hypergrid-aux
	    (1+ current-dimension-index)
	    dimension-ordering
	    second-to-last-dimension
	    last-dimension
	    pvar
	    format
	   )))))


(defun print-subgrid
       (pvar format first-dimension second-dimension other-dimension-ordering)

  (terpri)
  (terpri)

  (let ((coordinate-list (make-list *number-of-dimensions*)))

    (dolist (dimension-info other-dimension-ordering)
      (let ((dimension (car dimension-info))
	    (coordinate (cadr dimension-info))
	   )
	(setf (nth dimension coordinate-list) coordinate)
	(format t "DIMENSION ~S, COORDINATE ~S~%"
		(car dimension-info) (cadr dimension-info))
       )
     )
    (format t "~%     DIMENSION ~S    ----->~%~%" first-dimension)

    (dotimes (second (nth second-dimension *current-cm-configuration*))
      (setf (nth second-dimension coordinate-list) second)
      (dotimes (first (nth first-dimension *current-cm-configuration*))
	(setf (nth first-dimension coordinate-list) first)
	(format t format (apply #'pref-grid-or-junk (cons pvar coordinate-list)))
       )
      (terpri)
     )

   ))



(defun display-active-processors ()

  "Pretty print out the active processors' cube addresses"

  (terpri)
  (when (no-processors-active)
    (format t "No active processors...~%")
    (return-from display-active-processors nil)
   )
  (let ((active-processor-list (active-processor-list))
        (count 0)
       )
    (dolist (p active-processor-list)
      (format t "~5S " p)
      (incf count)
      (when (eql count 12) (setq count 0) (terpri))
     ))
  (values)
 )


(defun list-of-active-processors () (active-processor-list))
(defmacro loap () '(list-of-active-processors))



(defun display-active-processor-values
       (pvar &key (format "~S ") (start 0) (end *number-of-processors-limit*) title)

  "Pretty print active processor cube addresses
   and contents of a pvar in those processors
  "

  (when (not (stringp format))
    (error "Format key value is not a string: ~S" format))

  (when (or (not (integerp start))
	    (not (integerp end))
	    (< end start)
	    (< start 0)
	    (> end *number-of-processors-limit*))
    (error "Start or End keyword value invalid")
   )

  (terpri)
  (when (no-processors-active)
    (format t "No active processors...~%")
    (return-from display-active-processor-values nil)
   )

  (if (stringp title) (format t "~A: " title))

  (let ((active-processor-list (active-processor-list))
	(count 0)
	(format-string (format nil "~A: ~A " "~5@S" format))
       )
    (dolist (p active-processor-list)
      (when (and (>= p start) (< p end))
      (format t format-string p (pref-or-junk pvar p))
      (incf count)
      (when (eql count 8) (setq count 0) (terpri))
     )))

  (values)

 )


(defmacro pretty-print-pvar-in-currently-selected-set (&rest args)
  `(display-active-processor-values ,@args)
 )
(defmacro ppp-css (&rest args) `(display-active-processor-values ,@args))



;;;; Two routines for run-time conditionalization of code.  Right now to
;;;; allow library stuff to be done generically.


(defun simulator-loaded-p ()
  (member (proper-symbol-for-*features* '*LISP-SIMULATOR) *features*)
 )

(defun hardware-loaded-p ()
  (member (proper-symbol-for-*features* '*LISP-HARDWARE) *features*)
 )

