;;;
;;; A route-finding program from the book 
;;; "LISP on the BBC Microcomputer" by
;;;
;;; Arthur Norman & and Gillian Cattel Section 23.7
;;;
;;; Adapted for RefLisp by Bill Birch 30 Sept 92
;;;
;;; My adaptions are in lower-case.
;;;

;"The package described here allows the user to create a database
; showing the distances between pairs of towns. Once these data have been
; entered, the package can find the shortest route between any given
; places. The method used is an example of the 'ink-blot' or 'breadth
; first' search."


(DEFUN NEW-NODE (ITEM LEFT RIGHT)
	(CONS ITEM (CONS LEFT RIGHT)))
(DEFUN GET-ITEM (HEAP) (CAR HEAP))
(DEFUN LEFT-SUBHEAP (HEAP) (CADR HEAP))
(DEFUN RIGHT-SUBHEAP (HEAP) (CDDR HEAP))

(DEFUN ADD-TO-HEAP (ITEM HEAP)
	(COND
		((NULL HEAP) (NEW-NODE ITEM NIL NIL))
		((IS-SMALLER ITEM (GET-ITEM HEAP))
			(ADD-TO-HEAP
				(GET-ITEM HEAP)
				(REPLACE-ITEM HEAP ITEM)))
		(T (NEW-NODE
			(GET-ITEM HEAP)
			(RIGHT-SUBHEAP HEAP)
			(ADD-TO-HEAP ITEM (LEFT-SUBHEAP HEAP)))) ))

(DEFUN REMOVE-RIGHTMOST (HEAP)
	(COND
		((NULL (RIGHT-SUBHEAP HEAP))
			(SETQ RIGHTMOST (GET-ITEM HEAP))
			NIL)
		(T (NEW-NODE
			(GET-ITEM HEAP)
			(REMOVE-RIGHTMOST (RIGHT-SUBHEAP HEAP))
			(LEFT-SUBHEAP HEAP)))) )

(DEFUN REMOVE-TOP-ITEM (HEAP &optional RIGHTMOST) 
	(COND
		((NULL (RIGHT-SUBHEAP HEAP)) NIL)
		(T (SETQ HEAP (REMOVE-RIGHTMOST HEAP))
			(RESTORE-HEAP
				(REPLACE-ITEM HEAP RIGHTMOST))))
)

(DEFUN REPLACE-ITEM (HEAP ITEM)
	(CONS ITEM (CDR HEAP)))

(DEFUN RESTORE-HEAP (HEAP)
	(COND
		((TOP-ITEM-IS-SMALLEST HEAP) HEAP)
		((LEFT-ITEM-IS-SMALLEST HEAP)
			(PERCOLATE-LEFT HEAP))
		(T (PERCOLATE-RIGHT HEAP))))

(DEFUN TOP-ITEM-IS-SMALLEST (HEAP)
	(OR
		(NULL (RIGHT-SUBHEAP HEAP))
		(AND 
			(IS-SMALLER
				(GET-ITEM HEAP)
				(GET-ITEM (RIGHT-SUBHEAP HEAP)))
			(OR
				(NULL (LEFT-SUBHEAP HEAP))
				(IS-SMALLER
					(GET-ITEM HEAP)
					(GET-ITEM (LEFT-SUBHEAP HEAP)))) )))

(DEFUN LEFT-ITEM-IS-SMALLEST (HEAP)
	(AND
		(LEFT-SUBHEAP HEAP)
		(IS-SMALLER
			(GET-ITEM (LEFT-SUBHEAP HEAP))
			(GET-ITEM HEAP))
		(IS-SMALLER
			(GET-ITEM (LEFT-SUBHEAP HEAP))
			(GET-ITEM (RIGHT-SUBHEAP HEAP)))) )

(DEFUN PERCOLATE-RIGHT (HEAP)
	(NEW-NODE
		(GET-ITEM (RIGHT-SUBHEAP HEAP))
		(LEFT-SUBHEAP HEAP)
		(RESTORE-HEAP
			(REPLACE-ITEM
				(RIGHT-SUBHEAP HEAP)
				(GET-ITEM HEAP)))) )

(DEFUN PERCOLATE-LEFT (HEAP)
	(NEW-NODE
		(GET-ITEM (LEFT-SUBHEAP HEAP))
		(RESTORE-HEAP
			(REPLACE-ITEM (LEFT-SUBHEAP HEAP) 
				(GET-ITEM HEAP)))
		(RIGHT-SUBHEAP HEAP)))

;;;;;
;;
;;
(DEFUN FIND-ROUTE (SOURCE DESTINATION &optional PRIORITY-QUEUE  
				CITY DISTANCE VIA W)
	(SETQ PRIORITY-QUEUE (ADD-TO-QUEUE SOURCE 0 NIL))
	(do-until	
		((cond
			((GET DESTINATION SOURCE)
				(REPORT-ROUTE SOURCE DESTINATION) 
				(print priority-queue) 
				t)
			((NULL PRIORITY-QUEUE) (REPORT-FAILURE) t)
			(t nil)
		))
		
		(SETQ W (GET-ITEM PRIORITY-QUEUE))
		(SETQ CITY (CAR W))
		(SETQ DISTANCE (CADR W))
		(SETQ VIA (CADDR W))
		(SETQ PRIORITY-QUEUE
			(REMOVE-TOP-ITEM PRIORITY-QUEUE))
		(COND
			((NULL (GET CITY SOURCE))
				(PUT CITY SOURCE (CONS DISTANCE VIA))
				(SETQ PRIORITY-QUEUE
					(ADD-TO-QUEUE
						CITY
						DISTANCE
						PRIORITY-QUEUE)))) ) 
						
)

(DEFUN NPUT (SOURCE NEIGHBOURS)
	(PUT SOURCE 'NEIGHBOURS NEIGHBOURS))

(DEFUN ADD-TO-QUEUE (CITY BASE-DISTANCE QUEUE &optional NEIGHBOURS)
	(SETQ NEIGHBOURS (GET CITY 'NEIGHBOURS))
	(do-until ((null NEIGHBOURS) QUEUE) 
		(SETQ QUEUE
			(ADD-TO-HEAP
				(LIST
					(CAAR NEIGHBOURS)
				(+ BASE-DISTANCE (CDAR NEIGHBOURS))
					CITY)
				QUEUE))
		(SETQ NEIGHBOURS (CDR NEIGHBOURS)))
)

(DEFUN IS-SMALLER (A B) (< (CADR A) (CADR B)))

(DEFUN REPORT-ROUTE (SOURCE DESTINATION &optional VIA)
	(PRINT* 'DISTANCE= (CAR (GET DESTINATION SOURCE)))
	(SETQ VIA (CONS DESTINATION VIA))
	(do-until ((EQ SOURCE DESTINATION) (PRINT* 'VIA: VIA) (terpri) t)
		(SETQ DESTINATION (CDR (GET DESTINATION SOURCE)))
		(SETQ VIA (CONS DESTINATION VIA))
	)
)

(DEFUN REPORT-FAILURE () '(NO ROUTE EXISTS))

(NPUT 'CAMBRIDGE '((BEDFORD . 15) (ROYSTON . 20)))
(NPUT 'ROYSTON '((CAMBRIDGE . 20) (WATFORD . 30) (LONDON . 50)))
(NPUT 'LONDON '((ROYSTON . 20) (WATFORD  . 25) (OXFORD . 50)))
(NPUT 'BEDFORD '((CAMBRIDGE . 15) (WATFORD . 30)))
(NPUT 'WATFORD '((BEDFORD . 30) (ROYSTON . 30) 
	(LONDON . 25) (OXFORD . 40)))
(NPUT 'OXFORD '((ROYSTON . 50) (WATFORD . 25) (LONDON . 50) 
	(hemel-hempstead . 35) ))
(nput 'hemel-hempstead '((watford . 5)))
; Example
;
;
(princ "Searching for a route from Cambridge to Oxford ...")
(terpri)
 (FIND-ROUTE 'CAMBRIDGE 'OXFORD)
; ==> DISTANCE=85
; ==> VIA:(CAMBRIDGE BEDFORD WATFORD OXFORD)
;

