(defun putprop (sym val prop ) (put sym prop val))
;----------------------------------------------------------------------------
;    OBJECT ORIENTED INTERPRETER
;
;   AUTHOR: BILL BIRCH         DATE: 10 APRIL 1990
;----------------------------------------------------------------------------
;
;   SERVICES OFFERED BY THE INTERPRETER
;
;   (defnew <name> <parent-classes> <slots> <methods>)
;
;   (dispatch <method> <object> <args>)
;
;   (self-dispatch <method> <object>)
;
; This OO extension treats LISP symbols as objects. 
; There is no distinction made between objects and 
; classes, objects inherit methods and slots from other objects.  Methods and
; variables of an object are stored in the symbol's property list; the 
; symbol's value is not used. Objects are created with the function (defnew)
; which invokes the 'new method:
;   (defnew <object name> <parent object> 
;      '((<variable-name> <initial-values>) ... )
;      '((<method-name> <method-code>)  ....  )) 
;
; More correctly the only difference between variables and methods is that 
; variables are inherited by  any child objects, whereas methods are not 
; inherited.
; 
;Thus 
;   (defnew 'fred 'person '( (age 27)) '((print  pprint)))
;
; changes the symbol 'fred into an object of class person which uses the 
; pretty printer and has an age of 27.  All the other slots and methods are 
; inherited from 'person and its super-objects.  Psychologically it does help
; to think of some objects as being classes, and to organise slots into data 
; storage and executable types. Therefore for our purposes a class is simply 
; an object that is used as a parent of many objects; and a method is a non-
; inheritable slot.
;
; Methods are invoked by calling the (dispatch) function:
;   (dispatch <method> <object> <arguments>)
;
; for example:
;   (dispatch 'print 'hello '(hello))
;
; causes the 'hello object to print itself. (dispatch) simply searches for a 
; method to execute in the current object and all of its parents. 
; The top level object is called 'classes and has default methods for 'new 
; 'print 'display and 'sets.   Above that is an object is called 'root, 
; which has no purpose but to provoke error messages. 

;------------------------------------------------------------------------------
;
;   CORE FUNCTIONS OF THE INTERPRETER
;
; Function to execute a method, searching the classes
(defun dispatch (method object args)
	(apply  (slot object method) args)
)
; Function to dispatch, but with no arguments, pass self to method
(defun self-dispatch (method object)
   (dispatch method object (list object))
)
; Function to inherit slots from all parent-classes
(defun inherit (sym class)
   (cond
      ((null class) t)
      ((null (get class 'slots)) t)
      (t   
         (inherit sym (get class 'class))
         (putprs sym (get class 'slots))
      )
   )
)

; Function to initialise a list of properties
(defun putprs (ident proplist)
   (cond
      ((null proplist) t)
      (t    
         (putprop ident  (cadar proplist)(caar proplist))
         (putprs ident (cdr proplist))
      )
   )
)
; General purpose Function to set up a new object
(defun newobj (symbol class slots methods)
         (inherit symbol class)         ; pick up slots from parents
         (putprop symbol class 'class)   ; stash class
         (putprop symbol slots 'slots)   ; save this object's own slots
         (putprs symbol slots)      ; stash slots (overwrites    
                              ; parents if included)
         (putprs symbol methods)   ; stash methods
   symbol                        ; return value
)

;------------------------------------------------------------------------
;   GENERIC FUNCTIONS AND METHODS 
;
; Shorthand Function to define a new object using the 'new method
(defun defnew (name parent-classes slots methods)
      (dispatch 'new parent-classes 
            (list name parent-classes slots methods)
      )
)
;  *BROKEN* Function to create an instance of another object
; instances do not have proper names
;(defun instance (parent-class slots methods)
;   (defnew (gensym) parent-class slots methods))
;
; GENERIC UTILITY FUNCTIONS
(defun op (x) (self-dispatch 'print x))   ; print method
(defun od (x) (self-dispatch 'display x))   ; dump out all the properties
;------------------------------------------------------------------------
;   root of the classes tree 
;
(newobj 'classes 'root nil          ; no slots

   '(                ; methods
   ; print the value of the symbol
   (print (lambda (s) (pprint (eval s))))

   ; method to set up new objects
   (new    newobj)

   ; method to display all the slots of the object
   (display (lambda (x) (pprint (symbol-plist x))))

   ; method to set the value of a slot
   (sets (lambda (this value slot) (putprop this value slot)))
))
