; COMMON.LSP		Compatibility Functions for Common LISP
;
;	This file contains definitions of Common LISP functions defined
;	in "Common LISP the Language", second edition, by Guy L. Steele
;	jnr. [CLtL2]
;
;	They will be the basis for 'C' versions in common.c one day. 
;
;	Version		Date		Comments
;	1.0		22 June 92	PSETQ, LET, MAPCAR
;	2.0		21 Oct 92	DO, DO*, LET*, IF, WHEN, UNLESS, DOTIMES, DOLIST
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun cadddr (_l) (caddr (cdr _l)))
(defun = (a b) (equal a b))

(setq first car)
(setq rest cdr)
(setq listp consp)
(setq function-value symbol-value)
;
; Parallel SETQ with a list of arguments..
;
(df psetq (&rest _args)
	(mapcar #'set (odd _args)
		(mapcar #'eval (even _args)) ) )

(defun odd (list-x)
	(cond (list-x (cons (car list-x) (odd (cddr list-x))))))
(defun even (list-x)
	(cond (list-x (cons (cadr list-x) (even (cddr list-x))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	LET - Provide temporary bindings
;
;	(LET (	(<symbol 1> <initial value 1>)
;			(<symbol 2> <initial value 2)
;			...
;			(<symbol n> <initial value n>))
;		<let body>
;	)
;
; Example:
;	> (LET ((i 3) (j 7)) (+ i j)) 
;   10
(defmacro LET (_arglist &rest _body)
	`((lambda ,(MAPCAR 'car _arglist)
		,@_body )
	,@(MAPCAR 'cadr _arglist) ) )

(defmacro LET* (_arglist &rest _body)
	`((lambda ,(cons '&optional (MAPCAR #'car _arglist))
		,@(mapcar #'(lambda  (let*-arg)
			(list 'setq (car let*-arg) (cadr let*-arg)) )
			_arglist)
		,@_body ) ) )

;*******************************************************************
;;;
;;;  SETF - A macro to set values ? not - finished
;;;
;;	Accepted forms:
;;	(setf (get <symbol> <property>) <value)
;;
;;	Examples:
;;		(setf (get 'a 's) 23)

(defmacro SETF (place value)
	(cond
		((atom place) (error "setf cannot handle " place))
		((equal (car place) 'get) 
			`(put ,(cadr place) ,(caddr place) ,value))
		(t nil) ) )
;*******************************************************************
;;;
;;;  RPLACD - destructive replacement of cdr of a cons cell.
;;;  RPLACA - destructive replacement of car of a cons cell.
;;;
(defun RPLACD (dest source)
	(_move dest (cons (car dest) source)))
(defun RPLACA (dest source)
	(_move dest (cons source (cdr dest))))

;;; *******************************************************************
;;;
;;; Auxiliary function to check a keylist
;;;
;;; (_check-key '(:print t :foo bar) ':foo) ==> bar
;;; (_check-key '(:print t :foo bar) ':verbose) ==> nil
;;;
(defun _check-key (_keylist _key)
	(cond
		((null _keylist) nil)	;
		((equal (car _keylist) key) 
			(cond
				((null (cdr _keylist))
					(error "missing key value" key) )
				(t (cadr _keylist)) ) )
		(t (_check-key (cdr _keylist) key)) ) )
;;; *******************************************************************
;;;
;;;
;;;
(defun assoc (key lis)
	(cond
		((null lis) nil)
		((equal key (caar lis)) (car lis))
		(t (assoc key (cdr lis))) ) )
;;; *******************************************************************
;;;
;;;  basic mapping function
;;;
(defun mapcar (func &rest arglists)
	(mapcar-aux func arglists '()))
			
(defun mapcar-aux (func arglists m-result)
	(do-while  (and arglists (car arglists)) 
		(setq m-result (cons (apply func (m-car arglists)) m-result))
		(setq arglists (m-cdr arglists)))
	(reverse m-result))

;;; 
;;; 
(defun m-car (lol)
	(cond ((and lol (car lol))
		(cons (caar lol) (m-car (cdr lol))))))

;;;
;;;
(defun m-cdr (lol)
	(cond ((and lol (car lol))
		(cons (cdar lol) (m-cdr (cdr lol))))))


;;;
;;;
(defun type-of (_x) (slot _x 'class))
;;;
;;;
;;; Fake pretty printer - Winston & Horn
;;;
(defun pprint (l &optional (stream *standard-output*))
   (ppaux l 0 t stream)
   (terpri stream)
   l)
(defun ppaux (l column newline stream)
	(cond (newline (terpri stream)
		(let ((i 0))
			(do-until ((equal i column))
				(princ " " stream) (setq i (+ i 1))))))

	(cond ((atom l)
		(princ l stream)
		(setq column (+ column (length 
			(cond ((symbolp l) (symbol-name l))
				((numberp l) (itoa l))
				(t l))))))
		(t (princ "(" stream)
			(setq column (+ column 1))
			(setq column (ppaux (car l) column nil stream))
			(cond ((cdr l)
				(princ " " stream)
				(setq column (+ column 1))
				(ppaux (cadr l) column nil stream)
				(mapcar #'(lambda (e) (ppaux e column t stream)) (cddr l))))

			(princ ")" stream)
			(setq column (+ column 1))))
	column)

;;;
;;;
;;;
(defun funcall (fn &rest args) (apply fn args))
;;;
;;;
;;
(defmacro do* (varlist ending &rest body)
	`((lambda ,(cons '&optional (mapcar #'car varlist))
		,@(mapcar #'(lambda (do*-arg) 
						(list 'setq (car do*-arg) (cadr do*-arg))) varlist)
		(do-until ,ending
			,@body
			,@(do-aux-stepvars varlist)))))

(defun do-aux-inits (do-arg)
	(cond ((null do-arg) nil)
		(t (cons 
			(cond
				((null (cdar do-arg)) nil)
				(t (cadar do-arg)))
			(do-aux-inits (cdr do-arg))))))



(defun do-aux-stepvars (do-arg-lst)
	(cond	((null do-arg-lst) nil)
		((and	(cdar do-arg-lst) (cddar do-arg-lst))
			(cons (list 'setq (caar do-arg-lst) (nth 2 (car do-arg-lst)))
				(do-aux-stepvars (cdr do-arg-lst))))
		(t (do-aux-stepvars (cdr do-arg-lst)))))

(defmacro do (varlist ending &rest body)
	`((lambda ,(mapcar #'car varlist)
		(do-until ,ending
			,@body
			(mapcar #'set
				',(do-aux-sv 0 varlist) 
				(mapcar #'eval ',(do-aux-sv 2 varlist))  ) )) 
	,@(do-aux-inits varlist)))

(defun do-aux-sv (index do-arg-lst)
	(cond	((null do-arg-lst) nil)
		((and	(cdar do-arg-lst) (cddar do-arg-lst))
			(cons (nth index (car do-arg-lst)) 
				(do-aux-sv index (cdr do-arg-lst))))
		(t (do-aux-sv index (cdr do-arg-lst)))))


;;;------------------------------------------------------
;;;
;;; W&H page 356
;;;
(defmacro dolist (varlist &rest rest)
	`(do* ((lst ,(cadr varlist) (cdr lst))
		(,(car varlist) (car lst) (car lst)))
	   ((null lst) ,(caddr varlist))
	   ,@rest))

(defmacro dotimes (varcount &rest rest)
	`(do* ((limit ,(cadr varcount))
		(,(car varcount) 0 (+ ,(car varcount) 1)))
	   ((= ,(car varcount) limit) ,(caddr varcount))
	   ,@rest))

;;;------------------------------------------------------
;
(defmacro if (test success-result &optional failure-result)
	`(cond (,test ,success-result)(T ,failure-result)))
;
(defmacro when (test &rest rest)
	`(cond (,test ,@rest)))
;
(defmacro unless (test &rest rest)
	`(cond ((not ,test) ,@rest)))
;;;------------------------------------------------------
;;;
;;; W&H page 379

(defmacro break (&optional message)
	`(progn
		(print (cons 'Break:  ,message))	; Print message
		(print '>)							; Prompt user
		(do ((query (read) (read)))			; Read user's expression
			((equal query 'continue) nil)	; Stop if indicated
		  (print (eval query))				; Print appropriate value
		  (print '>))))						; Prompt user again

;;;------------------------------------------------------
(defmacro remove-if (&rest args) (error "remove-if not implemented"))
(defmacro remove-if-not (&rest args) (error "remove-if-not not implemented"))

