
/*
    Copyright (c) 1994 Jeff Weisberg

    see the file "License"
*/

#ifdef RCSID
static const char *const rcsid
= "@(#)$Id: reader.c,v 1.27 94/08/23 08:51:51 weisberg Exp Locker: weisberg $";
#endif

#include <jlisp.h>
#include <stdio.h>

extern Obj makfloat(float);
extern Obj str_append(Obj, int, int);

Obj Fread();
int readchar(Obj port);
void unreadchar(Obj port, int c);

extern Obj sym_optional, sym_rest, sym_quote;
extern Obj sym_quote, sym_bquote, sym_bq_comma, sym_bq_comma_at;

extern Obj sym_iradix, sym_eof, sym_stdin;

DEFVAR(".lineno", Vlineno, ".lineno the current line number", MAKINT(1))

     
void inc_lineno(){
	/* increment line number */
	VALUE( Vlineno ) += MAKINT(1) - MAKINT(0);
}
	
int vallof(int c, int b){

	if(c>='0' && c<='9') return c - '0';
	if(c>='a' && c<='z') return c - 'a' + 0xa;
	if(c>='A' && c<='Z') return c - 'A' + 0xA;

	return 255;
}

int isvalid(int c, int b){

	return vallof(c, b) < b;
}

void eatcomment(Obj stream){
	/* eat #| comment |#
	   may be nested */
	int c=0;
	
	while( c!='#'){
		while(c!='|'){
			c = readchar(stream);
			switch(c){
			  case '#':
				c = readchar(stream);
				if(c=='|'){
					eatcomment(stream);
					c = readchar(stream);
				}
				break;
			  case '\n':
				inc_lineno();
			  default:
				break;
			}
		}
		c = readchar(stream);
	}
	return;
}

int special_char(Obj stream){
	/* handle special \escaped characters */
	int c;
	int val=0, base;
	c = readchar(stream);
	switch( c ){
	  case 'a':	c = '\a';	break;	/* yes, I know that this is the ANSI C alert char... */
	  case 'n':	c = '\n';	break;
	  case 'r':	c = '\r';	break;
	  case 'b':	c = '\b';	break;
	  case 't':	c = '\t';	break;
	  case 's':	c = ' ';	break;
	  case 'f':	c = '\f';	break;
	  case 'v':	c = '\v';	break;
	  case 'e':	c = '\033';	break;

	  case '"':	c = '"';	break;
		
	  case '0':
		base = 8;  goto rnum;
	  case 'x':
	  case 'X':
		c = readchar(stream);
		base = 16; goto rnum;
	  case '1': case '2': case '3':
	  case '4': case '5': case '6':
	  case '7': case '8': case '9':
		base = 10; goto rnum;
	  rnum:

		while( isvalid(c, base)){
			val *= base;
			val += vallof(c, base);
			c = readchar(stream);
		}
		unreadchar(stream, c);
		c = val;
		break;
		
	  case '\n':
		inc_lineno();
	  default:
		Fdisplay( makstr_c("Warning: unknown escape \\"), stderr_port);
		Fdisplay( MAKCHAR( c ),  stderr_port);
		Fdisplay( MAKCHAR('\n'), stderr_port);
		c = c;		break;

	}
	return c;
}

int getc_skipws(Obj stream){
	int c;
	
	while( 1 ){
		c = readchar(stream);
		switch(c){
		  case ';':
			while( c != '\r' && c!= '\n' ) c = readchar(stream);
			/* fall thru' */
		  case '\n':
			inc_lineno();
		  case ' ':
		  case '\t':
		  case '\r':
			continue;
		  case '#':
			c = readchar(stream);
			if(c!='|'){
				unreadchar(stream, c);
				return '#';
			}
			eatcomment(stream);
			continue;
		}
		break;
	}
	return c;
}

Obj readparen( Obj stream ){
	int c;
	Obj foo;

	c = getc_skipws( stream );
	if( c==')' ) return IC_NIL;
	unreadchar(stream, c);
	foo = Fread( stream );
	if( SYMBOLP(foo) && !strcmp( CCHARS(foo), ".")){
		/* KLUDGE ALERT */
		/* a lone .
		turn it into an improper list */
		foo = Fread( stream );
		c = getc_skipws( stream );
		if( c!=')' ) unreadchar(stream, c);
		return foo;
	}
	return Fcons( foo, readparen( stream ) );
}

DEFUN("read", Fread, Sread, 0, 1, 1,0,
      "(read [port]) read in an expression",
      (Obj stream))
{

	int c;
	char buf[1024];
	int i;
	Obj val, frac, baseo;
	int decmp, negp, base;
	Obj radix;
	Obj buffer;
	
	if( NBOUNDP( stream )) stream = getvalue(sym_stdin);
	if( NULLP(stream)){
		Fthrow(sym_eof, IC_EOF);
		return IC_EOF;
	}
	if( !RPORTP( stream )){
		return jlerror("read", stream, "wrong type of argument, inputportp");
	}

	c = getc_skipws( stream );
	
	switch( c ){
	  case EOF:
		Fthrow(sym_eof, IC_TRUE);
		return IC_EOF;
		
	  case '(':
		return readparen( stream );

	  case ')':
		return jlerror("read", stream, "unexpected ')'");

	  case '"':
		buffer = makstrn("", 0);
		i = 0;
		do {
			c = readchar(stream);
			if(c=='\\'){
				c = special_char(stream);
				str_append(buffer, i++, c);
				c = 0;
				continue;
			}
			if( c=='\n') inc_lineno();
			if( c!= '"')
				str_append(buffer, i++, c);
		}while( c != '"' );

		CCHARS(buffer)[i] = 0;
		return buffer;

	  case '?':
	  rchar:
		c = readchar(stream);
		if( c == '\\' ){
			i = special_char(stream);
		}else
			i = c;
		return MAKCHAR( i );

	  case '#':
		c = getc_skipws( stream );

		switch( c ){
		  case '\\':
			/* handle scheme-like character syntax #\x #\\n would be a newline... */
			goto rchar;
		  case 't':
		  case 'T':
			return IC_TRUE;
		  case 'f':
		  case 'F':
			return IC_FALSE;
		  case '<':
			while ( c != '>') c = getc_skipws(stream);
			return jlerror("read", IC_UNSPEC, "unreadable syntax");
		  case '(':
			return Flist_vect( readparen(stream) );
		  case 'x':
		  case 'X':
			/* _I_ like spagetti... */
			base = 16;	goto rnump;
		  case 'o':
		  case 'O':
			base = 8;	goto rnump;
		  case 'd':
		  case 'D':
			base = 10;	goto rnump;
		  case 'b':
		  case 'B':
			base = 2;
		  rnump:
			c = getc_skipws(stream);
			goto rnum;
		  case '!':
			if( VALUE(Vlineno)==MAKINT(1)){
				/* special script file handling
				   #! is a comment on the 1st line of a file
				*/
				while( c != '\n') c = readchar(stream);
				unreadchar(stream, c);
				return Fread(stream);
			}
			/* fall thru' */
		  default:
			return jlerror(Sread.name, IC_UNSPEC, "unreadable syntax");
		}
		break;
		
	  case '\'':
		return Fcons(sym_quote, Fcons(Fread( stream ), IC_NIL));
		break;

	  case '`':
		return Fcons(sym_bquote, Fcons(Fread( stream ), IC_NIL));
		break;

	  case ',':
		c = readchar(stream);
		if( c=='@') return Fcons(sym_bq_comma_at, Fcons(Fread( stream ), IC_NIL));
		unreadchar(stream, c);
		return Fcons(sym_bq_comma, Fcons(Fread( stream ), IC_NIL));
		
	  default:
		radix = getvalue( sym_iradix );
		
		if( INUMP(radix))
			base = CINT(radix);
		else
			base = 10;

	  rnum:
		baseo = MAKINT(base);

		i = 0;
		while(1){
			if( c==' ' ) break;
			if( c=='\t') break;
			if( c=='\r') break;
			if( c=='\n') break;
			if( c==')' ) break;
			if( c=='(' ) break;
			if( c==';' ) break;
			if( c=='#' ) break;	/* XXX ? ought # be allowed in a symbol name? */
			if( c==EOF ) break;

			buf[i++] = c;
			buf[i] = 0;
			c = readchar(stream);
		}
		unreadchar(stream, c);

		/* handle 2 special cases */
		if(!strcmp(buf, "&rest")) 	return sym_rest;
		if(!strcmp(buf, "&optional")) 	return sym_optional;

		val = MAKINT(0);
		frac= makfloat(1);
		decmp = negp = 0;
		i = 0;
		
		
		if( buf[0]=='-'){
			negp = 1;
			i++;
		}
		while(buf[i]){
			if( !isvalid(buf[i], base) && buf[i]!='.'){
				/* a symbol */
				return maksym( buf );
			}
			if( buf[i]=='.' ){
				if( decmp ) return maksym( buf );
				decmp = 1;
				i++;
				continue;
			}

			if( decmp ){
				frac = Fdivide(frac, baseo);
				val  = Ftimes(val, baseo);
				val  = Fplus(val, MAKINT(vallof( buf[i], base )));
			} else {
				val = Ftimes(val, baseo);
				val = Fplus(val, MAKINT(vallof( buf[i], base )));
			}
			i++;

		}
		if(negp && i==1)
			return maksym( buf );
		if(decmp && i==1)
			return maksym( buf );
		if(decmp)
			val = Ftimes(val, frac);
		if(negp)
			val = Fminus(MAKINT(0), val);
		return val;

	}
}







