
/*
    Copyright (c) 1994 Jeff Weisberg

    see the file "License"
*/

#ifdef RCSID
static const char *const rcsid
= "@(#)$Id: math.c,v 1.9 94/08/07 13:47:34 weisberg Exp Locker: weisberg $";
#endif


#include <jlisp.h>
#include <math.h>

extern Obj makfloat(float);
extern Obj makdbl(double);

#define MAX(a,b)	((a) > (b) ? (a) : (b))

double dblof(Obj a){

	if(ICHARP(a)) return CCHAR(a);
	if(INUMP(a)) return CINT(a);
	if(FLOATP(a)) return CFLOAT(a);
	if(DOUBLEP(a))return CDOUBLE(a);

	return 0.0;
}
int intof(Obj a){

	if(ICHARP(a)) return CCHAR(a);
	if(INUMP(a)) return CINT(a);
	if(FLOATP(a)) return CFLOAT(a);
	if(CDOUBLE(a)) return CDOUBLE(a);

	return 0;
}
float fltof(Obj a){

	if(ICHARP(a)) return CCHAR(a);
	if(INUMP(a)) return CINT(a);
	if(FLOATP(a)) return CFLOAT(a);
	if(CDOUBLE(a)) return CDOUBLE(a);

	return 0;
}

Obj simplemath(Obj a, Obj b, int op){
	int ta = TYPEOFX(a), tb = TYPEOFX(b);
	int mt;
	int tv;
	float tf;
	double td;
	
	mt = MAX(ta, tb);
	
	switch( mt ){
	  case TPVF_IMMED:
		switch (op ){
		  case '+':
			tv = intof(a) + intof(b);
			break;
		  case '-':
			tv = intof(a) - intof(b);
			break;
		  case '*':
			tv = intof(a) * intof(b);
			break;
		  case '/':
			if( (tv=intof(b)) == 0)
				goto divide_by_zero;
			tv = intof(a) / tv;
			break;
		}
		return MAKINT(tv);

	  case TPV_FLOAT:
		switch(op){
		  case '+':
			tf = fltof(a) + fltof(b);
			break;
		  case '-':
			tf = fltof(a) - fltof(b);
			break;
		  case '*':
			tf = fltof(a) * fltof(b);
			break;
		  case '/':
			if( (tf=fltof(b)) == 0.0)
				goto divide_by_zero;
			tf = fltof(a) / tf;
			break;
		}
		return makfloat( tf );
		
	  case TPV_DOUBLE:
		switch (op){
		  case '+':
			td = dblof(a) + dblof(b);
			break;
		  case '-':
			td = dblof(a) - dblof(b);
			break;
		  case '*':
			td = dblof(a) * dblof(b);
			break;
		  case '/':
			if( (td=dblof(b)) == 0.0)
				goto divide_by_zero;
			td = dblof(a) / td;
			break;
		}
		return makdbl( td );

	  case TPV_COMPLEX:

	  case TPV_BIGNUM:

	  default:
		return jlerror("math", Fcons(a, b), "WTA: numberp");
	}
	
  divide_by_zero:
	return jlerror("math", b, "divide by zero");
}

DEFUN("+", Fplus, Splus, 2,2,1,0, "(+ n m) addition",
      (Obj a, Obj b))
{
	return simplemath(a,b, '+');
}

DEFUN("-", Fminus, Sminus, 2,2,1,0, "(- n m) subtraction",
      (Obj a, Obj b))
{
	return simplemath(a,b, '-');
}

DEFUN("*", Ftimes, Stimes, 2,2,1,0, "(* n m) multiplication",
      (Obj a, Obj b))
{
	return simplemath(a,b, '*');
}

DEFUN("/", Fdivide, Sdivide, 2,2,1,0, "(/ n m) division",
      (Obj a, Obj b))
{
	return simplemath(a,b, '/');
}

DEFUN("<<", Fshl, Sshl, 2,2,1,0, "(<< n m) shift left)",
      (Obj a, Obj b))
{
	return MAKINT( intof(a) << intof(b) );
}

DEFUN(">>", Fshr, Sshr, 2,2,1,0, "(>> n m) shift right)",
      (Obj a, Obj b))
{
	return MAKINT( intof(a) >> intof(b) );
}

DEFUN("&", Fland, Sland, 2,2,1,0, "(& n m) bitwise and",
      (Obj a, Obj b))
{
	return MAKINT( intof(a) & intof(b) );
}

DEFUN("|", Flor, Slor, 2,2,1,0, "(| n m) bitwise or",
      (Obj a, Obj b))
{
	return MAKINT( intof(a) | intof(b) );
}

DEFUN("^", Flxor, Slxor, 2,2,1,0, "(^ n m) bitwise xor",
      (Obj a, Obj b))
{
	return MAKINT( intof(a) ^ intof(b) );
}

DEFUN("~", Flnot, Slnot, 1,1,1,0, "(~ n) bitwise not",
      (Obj a))
{
	return MAKINT( ~ intof(a));
}

#if 0
xxDEFUN("real", Freal, Sreal, 1,1,1,0, "(real c) return the real part of a complex number",
	(Obj a))
{

	if(COMPLEXP(a)) return makdbl( CCOMPLEX(a).re );
	if(INUMP(a) || FLOATP(a) || DOUBLEP(a)) return a;

	return jlerror(Sreal.name, a, "WTA: numberp");
}

xxDEFUN("imag", Fimag, Simag, 1,1,1,0, "(imag c) return the imaginary part of a complex number",
	(Obj a))
{
	
	if(COMPLEXP(a)) return makdbl( CCOMPLEX(a).im );
	if(INUMP(a) || FLOATP(a) || DOUBLEP(a)) return makdbl( (double)0.0);

	return jlerror(Simag.name, a, "WTA: numberp");
}
#endif

Obj mathbinal(Obj a, Obj b, int fnc, char* fnm){

	if(ICHARP(a)||INUMP(a)||FLOATP(a)||DOUBLEP(a)||COMPLEXP(a)
	   || ICHARP(b)||INUMP(b)||FLOATP(b)||DOUBLEP(b)||COMPLEXP(b))
		;
	else
		return jlerror(fnm, Fcons(a,b), "WTA: numberp");

	switch(fnc){
	  case '>':
		return dblof(a) > dblof(b) ? IC_TRUE : IC_FALSE;
	  case '<':
		return dblof(a) < dblof(b) ? IC_TRUE : IC_FALSE;
	  case 'g':
		return dblof(a) >= dblof(b) ? IC_TRUE : IC_FALSE;
	  case 'l':
		return dblof(a) <= dblof(b) ? IC_TRUE : IC_FALSE;
	  case '=':
		return dblof(a) == dblof(b) ? IC_TRUE : IC_FALSE;
	  case '%':
		return MAKINT( intof(a) % intof(b) );
	  case 'a':
		return makdbl( atan2( dblof(a), dblof(b)));
	  case 'p':
		return makdbl( pow( dblof(a), dblof(b)));

	}
}

DEFUN("%", Fmod, Smod, 2,2,1,0,
      "(% n1 n2) n1 modulo n2",
      (Obj a, Obj b))
{
	return mathbinal(a,b,'%', Smod.name);
}

DEFUN("=", Fmequal, Smequal, 2,2,1,0,
      "(= n1 n2) Are two numbers equal?",
      (Obj a, Obj b))
{
	return mathbinal(a,b,'=',Smequal.name);
}
DEFUN(">", Fgreater, Sgreater, 2,2,1,0,
      "(> n1 n2) Is n1 greater than n2?",
      (Obj a, Obj b))
{
	return mathbinal(a, b, '>',Sgreater.name);
}
DEFUN("<", Fless, Sless, 2,2,1,0,
      "(< n1, n2) Is n1 less than n2?",
      (Obj a, Obj b))
{
	return mathbinal(a, b, '<', Sless.name);
}
DEFUN(">=", Fgrequal, Sgrequal, 2,2,1,0,
      "(>= n1 n2) Is n1 greater or equal to n2?",
      (Obj a, Obj b))
{
	return mathbinal(a,b,'g',Sgrequal.name);
}
DEFUN("<=", Flsequal, Slsequal, 2,2,1,0,
      "(<= n1 n2) Is n1 less or equal than n2?",
      (Obj a, Obj b))
{
	return mathbinal(a,b,'l',Slsequal.name);
}

DEFUN("atan2", Fatan2, Satan2, 2,2,1,0,
      "(atan2 n1 n2)  the arctan of n2/n1",
      (Obj a, Obj b))
{
	return mathbinal(a,b,'a', Satan2.name);
}

DEFUN("pow", Fpow, Spow, 2,2,1,0,
      "(pow n1 n2) take n1 to the n2 power",
      (Obj a, Obj b))
{
	return mathbinal(a,b, 'p', Spow.name);
}

Obj trancend(Obj a, double (*fnc)(double), char* fnm){
	double d;

	if(INUMP(a)||FLOATP(a)||DOUBLEP(a)||COMPLEXP(a))
		d = dblof(a);
	else
		return jlerror(fnm, a, "WTA: numberp");

	return makdbl( fnc(d) );
}

DEFUN("exp", Fexp, Sexp, 1,1,1,0, "take e to a power",
      (Obj a))
{
	return trancend(a, exp, Sexp.name);
}

DEFUN("log", Flog, Slog, 1,1,1,0, "take the natural log",
      (Obj a))
{
	return trancend(a, log, Slog.name);
}

DEFUN("sin", Fsin, Ssin, 1,1,1,0, "take the sin",
      (Obj a))
{
	return trancend(a, sin, Ssin.name);
}

DEFUN("cos", Fcos, Scos, 1,1,1,0, "take the cosine",
      (Obj a))
{
	return trancend(a, cos, Scos.name);
}

DEFUN("tan", Ftan, Stan, 1,1,1,0, "take the tangent",
      (Obj a))
{
	return trancend(a, tan, Stan.name);
}

DEFUN("asin", Fasin, Sasin, 1,1,1,0, "take the arcsin",
      (Obj a))
{
	return trancend(a, asin, Sasin.name);
}

DEFUN("acos", Facos, Sacos, 1,1,1,0, "take the arccos",
      (Obj a))
{
	return trancend(a, acos, Sacos.name);
}
DEFUN("atan", Fatan, Satan, 1,1,1,0, "take the arctan",
      (Obj a))
{
	return trancend(a, atan, Satan.name);
}
DEFUN("sqrt", Fsqrt, Ssqrt, 1,1,1,0, "take the square root",
      (Obj a))
{
	return trancend(a, sqrt, Ssqrt.name);
}
DEFUN("sinh", Fsinh, Ssinh, 1,1,1,0, "take the hyperbolic sin",
      (Obj a))
{
	return trancend(a, sinh, Ssinh.name);
}
DEFUN("cosh", Fcosh, Scosh, 1,1,1,0, "take the hyperbolic cosine",
      (Obj a))
{
	return trancend(a, cosh, Scosh.name);
}
DEFUN("tanh", Ftanh, Stanh, 1,1,1,0, "take the hyperbolic tangent",
      (Obj a))
{
	return trancend(a, tanh, Stanh.name);
}
DEFUN("asinh", Fasinh, Sasinh, 1,1,1,0, "take the inverse hyperbolic sin",
      (Obj a))
{
	return trancend(a, asinh, Sasinh.name);
}
DEFUN("acosh", Facosh, Sacosh, 1,1,1,0, "take the inverse hyperbolic cosine",
      (Obj a))
{
	return trancend(a, acosh, Sacosh.name);
}
DEFUN("atanh", Fatanh, Satanh, 1,1,1,0, "take the inverse hyperbolic tangent",
      (Obj a))
{
	return trancend(a, atanh, Satanh.name);
}
DEFUN("cbrt", Fcbrt, Scbrt, 1,1,1,0, "take the cube root",
      (Obj a))
{
	return trancend(a, cbrt, Scbrt.name);
}
DEFUN("floor", Ffloor, Sfloor, 1,1,1,0, "take the floor",
      (Obj a))
{
	return MAKINT( (int)CDOUBLE(trancend(a, floor, Sfloor.name)) );
}
DEFUN("ceil", Fceil, Sceil, 1,1,1,0, "take the ceiling",
      (Obj a))
{
	return MAKINT( (int)CDOUBLE(trancend(a, ceil, Sceil.name)) );
}
DEFUN("abs", Fabs, Sabs, 1,1,1,0, "take the absolute value",
      (Obj a))
{
	return trancend(a, fabs, Sabs.name);
}

DEFUN("random", Frandom, Srandom, 0,0,1,0,
      "(random) return a random number",
      ())
{
	return MAKINT(random() & 0xFFFFFFF);
}

DEFUN("->int", Fto_int, Sto_int, 1,1,1,0,
      "(->int n) convert to integer",
	(Obj n))
{
	return MAKINT( intof(n));
}

DEFUN("->float", Fto_float, Sto_float, 1,1,1,0,
      "(->float n) convert to single precision float",
	(Obj n))
{
	return makfloat( fltof(n));
}

DEFUN("->double", Fto_double, Sto_double, 1,1,1,0,
      "(->double n) convert to double precision float",
	(Obj n))
{
	return makdbl( dblof(n));
}
